/*
 * @ReadLASFileXMLFile.java Version 1.0 01/29/2004
 *
 * Copyright (c) 2001 John R. Victorine
 * All Rights Reserved.
 */

package xml;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import xml.LASFileXMLStruct;
import cmn.cmnLASFileListStruct;
import cmn.cmnLASFileStruct;

/** Class ReadLASFileXMLFile()
 * <p> This Class will parse the LAS File XML Data Stream
 *     directly to the LAS File List Data Structure.
 *
 *  @version 1.1 01/29/2004
 *  @author  John R. Victorine
 */

public class ReadLASFileXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  private int                      iType       = FILE;
  private int                      iRows       = 0;
  private int                      iCount      = -1;
  private cmnLASFileListStruct     stList = new cmnLASFileListStruct();

  /** Method ReadLASFileXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadLASFileXMLFile() { iType = FILE; }

  /** Method ReadLASFileXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadLASFileXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public cmnLASFileListStruct Process(String filename)
  {
    stList.iCount = 0;
    Read(filename);
    stList.iCount = iCount+1;

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    // If the ELEMENT Well is detected then initialize the structure.

    if (eName.equals(xml.LASFileXMLStruct.LAS))
    {
      iCount++;
      if (iCount < iRows)
      {
        stList.stItem[iCount] = new cmnLASFileStruct();
        stList.stItem[iCount].iLog = new int[cmn.cmnLASFileStruct.TOTAL];
        for (int i=0; i<cmn.cmnLASFileStruct.TOTAL; i++)
        {
          stList.stItem[iCount].iLog[i] = 0;
        }
      }
    }

    // Add Attributes to the Data List Structure

    if (attrs != null)
    {
      for (int i = 0; i < attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(xml.LASFileXMLStruct.RECORDS))
        {
          iRows = cmn.cmnString.stringToInt(sOut.trim());

          if (iRows > 0)
          {
            stList.stItem = new cmnLASFileStruct[iRows];
          }
        }

        // Parse each attribute and add it to the structure

        if (iCount > -1)
        {
          if (iCount < iRows)
          {
            stList.stItem[iCount] = ParseData(aName,
                                              sOut.trim(),
                                              stList.stItem[iCount]);
          }
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    // Ignore it
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** ParseData()
   * <p> This method will parse the Data Stream for the Information
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = LAS File String
   * @param  st         = The LAS File Data Structure
   * @return st         = The LAS File Data Structure
   */

  private cmnLASFileStruct ParseData(String sIdentifier,
                                     String sData,
                                     cmnLASFileStruct st)
  {
    if (sIdentifier.equals(xml.LASFileXMLStruct.KEY)) // LAS File Primary KEY
    {
      st.sKID = new String(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.WELL_KEY)) // Well Primary KEY
    {
      st.sWellKID = new String(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.PATH_TYPE)) // Path Type
    {
      st.iType = FILE;
      if (cmn.cmnString.isNumeric(sData))
        st.iType = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.PATH)) // URL Path of LAS File
    {
      st.sDirectory = new String(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.FILENAME)) // Filename of File
    {
      st.sFilename = new String(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.VERSION)) // Version of LAS
    {
      st.sVersion = new String(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.NULL_VALUE)) // LAS Null Value
    {
      st.dNull = 0.0;
      if (cmn.cmnString.isNumeric(sData))
        st.dNull = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.START_DEPTH)) // Starting Depth
    {
      st.depthStart = 0.0;
      if (cmn.cmnString.isNumeric(sData))
        st.depthStart = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.END_DEPTH))  // Ending Depth
    {
      st.depthEnd = 0.0;
      if (cmn.cmnString.isNumeric(sData))
        st.depthEnd = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.PROPRIETARY)) // Proprietary
    {
      st.iProprietary = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.iProprietary = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.RESISTIVITY))
    {
      st.iLog[cmn.cmnLASFileStruct.RESISTIVITY] = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.iLog[cmn.cmnLASFileStruct.RESISTIVITY] =
            cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.DENSITY))
    {
      st.iLog[cmn.cmnLASFileStruct.DENSITY] = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.iLog[cmn.cmnLASFileStruct.DENSITY] =
            cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.NEUTRON))
    {
      st.iLog[cmn.cmnLASFileStruct.NEUTRON] = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.iLog[cmn.cmnLASFileStruct.NEUTRON] =
            cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.SONIC))
    {
      st.iLog[cmn.cmnLASFileStruct.SONIC] = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.iLog[cmn.cmnLASFileStruct.SONIC] = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.SP))
    {
      st.iLog[cmn.cmnLASFileStruct.SP] = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.iLog[cmn.cmnLASFileStruct.SP] = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.GAMMA_RAY))
    {
      st.iLog[cmn.cmnLASFileStruct.GAMMA_RAY] = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.iLog[cmn.cmnLASFileStruct.GAMMA_RAY] =
            cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.CALIPER))
    {
      st.iLog[cmn.cmnLASFileStruct.CALIPER] = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.iLog[cmn.cmnLASFileStruct.CALIPER] =
            cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.PE))
    {
      st.iLog[cmn.cmnLASFileStruct.PE] = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.iLog[cmn.cmnLASFileStruct.PE] = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.THOR))
    {
      st.iLog[cmn.cmnLASFileStruct.THORIUM] = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.iLog[cmn.cmnLASFileStruct.THORIUM] = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.URAN))
    {
      st.iLog[cmn.cmnLASFileStruct.URANIUM] = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.iLog[cmn.cmnLASFileStruct.URANIUM] = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(xml.LASFileXMLStruct.POTA))
    {
      st.iLog[cmn.cmnLASFileStruct.POTASSIUM] = 0;
      if (cmn.cmnString.isNumeric(sData))
        st.iLog[cmn.cmnLASFileStruct.POTASSIUM] = cmn.cmnString.stringToInt(sData);
    }

    return (st);
  }
}