/*
 * @utilLatLongConversion.java Version 1.1 06/20/2002
 *
 * Copyright (c) 2001 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package util;

import java.awt.*;
import java.io.*;
import java.lang.*;
import java.util.*;

/** Class utilLatLongConversion
 *  <p> This class will convert Latitude and Longitude to UTM Coordinates.
 *
 *  This method assumes that the Latitude and Longitude will have the proper
 *  signs to represent North (+) or South (-) for Latitude direction and
 *  East (-) or West (+) for Longitude direction.
 *
 *  @version 1.1 06/20/2002
 *  @author  John R. Victorine
 */

public class utilLatLongConversion
{
  public static final int UTM_X = 0;
  public static final int UTM_Y = 1;

  private int    iZone = 0;

  private double dAo   = 6378206.4;
  private double dE2rd = 0.00676866;
  private double dKo   = 0.9996;
  private double dMo   = 0.0;

  private double dLatitude        = 0.0;
  private double dLongitude       = 0.0;
  private double dMedianLongitude = 0.0;

  private double dUTM_X = 0.0;
  private double dUTM_Y = 0.0;

  /** Construct utilLatLongConversion()
   * <p> This is the Constructor for this class.
   *
   * @param latitude  = the Latitude in degrees to be converted
   * @param longitude = the Longitude in degrees to be converted.
   */

  public utilLatLongConversion(double latitude, double longitude)
  {
    /* --- Test Values to verify that this section works. ---
    latitude = 40.5;
    longitude = -73.5;
    dMedianLongitude = -75.0;
    */

    dLatitude        = convertDegreesToRadians(latitude); // Latitude
    dLongitude       = convertDegreesToRadians(longitude); // Longitude

    iZone = 1 + (int)((180.0+longitude) / 6.0);

    double d = -1.0 * ((180.0 - ((double)iZone * 6.0)) + 3.0);
    dMedianLongitude = convertDegreesToRadians(d);

    computeUTM();
  }

  /** Construct utilLatLongConversion()
   * <p> This is the Constructor for this class.
   *
   * @param iZone     = the new zone to make the conversion
   * @param latitude  = the Latitude in degrees to be converted
   * @param longitude = the Longitude in degrees to be converted.
   */

  public utilLatLongConversion(int iZone, double latitude, double longitude)
  {
    dLatitude        = convertDegreesToRadians(latitude); // Latitude
    dLongitude       = convertDegreesToRadians(longitude); // Longitude

    this.iZone = iZone;

    double d = -1.0 * ((180.0 - ((double)iZone * 6.0)) + 3.0);
    dMedianLongitude = convertDegreesToRadians(d);

    computeUTM();
  }

  /** Method convertDegreesToRadians()
   *  <p> This method will convert degrees to radians
   *  @param  degrees  = the value to be converted
   *  @return dRadians = the radian value.
   */

  public double convertDegreesToRadians(double degrees)
  {
    double dRadians = 0.0;

    dRadians = degrees * Math.PI / 180.0;

    return (dRadians);
  }

  /** Method getEqn_8_12()
   *
   *  <p> This method will compute the equation 8-12
   *          [e'*e' = e*e / (1 - e*e)]
   *      in Map Projects - A Working Manual by John P. Snyder
   *      U.S. Geological Survey Proffessional Paper 1395
   *      USG Printing Office, Washington: 1987
   *   @return dE2rdPrime = e'*e'
   */

  private double getEqn_8_12()
  {
    double dE2rdPrime = 0.0;

    dE2rdPrime = dE2rd / (1 - dE2rd);

    return (dE2rdPrime);
  }

  /** Method getEqn_4_20()
   *
   *  <p> This method will compute the equation 4-20
   *          [ N = a / {Square Root of (1 - e * e * sin(Phi) * sin(Phi))} ]
   *      in Map Projects - A Working Manual by John P. Snyder
   *      U.S. Geological Survey Proffessional Paper 1395
   *      USG Printing Office, Washington: 1987
   *   @return dN = N
   */

  private double getEqn_4_20()
  {
    double dN = 0.0;

    dN = dAo / (Math.sqrt(1.0-dE2rd * Math.sin(dLatitude)*Math.sin(dLatitude)));

    return (dN);
  }

  /** Method getEqn_8_13()
   *
   *  <p> This method will compute the equation 8-13
   *          [ T = tan(Phi) * tan(Phi) ]
   *      in Map Projects - A Working Manual by John P. Snyder
   *      U.S. Geological Survey Proffessional Paper 1395
   *      USG Printing Office, Washington: 1987
   *   @return dT = T
   */

  private double getEqn_8_13()
  {
    double dT = 0.0;

    dT = Math.tan(dLatitude) * Math.tan(dLatitude);

    return (dT);
  }

  /** Method getEqn_8_14()
   *
   *  <p> This method will compute the equation 8-14
   *          [ C = e' * e' * cos(Phi) * cos(Phi) ]
   *      in Map Projects - A Working Manual by John P. Snyder
   *      U.S. Geological Survey Proffessional Paper 1395
   *      USG Printing Office, Washington: 1987
   *   @return dC = C
   */

  private double getEqn_8_14()
  {
    double dC         = 0.0;
    double dE2rdPrime = getEqn_8_12();

    dC = dE2rdPrime * Math.cos(dLatitude) * Math.cos(dLatitude);

    return (dC);
  }

  /** Method getEqn_8_15()
   *
   *  <p> This method will compute the equation 8-15
   *          [ A = (Lamda - Lamda0) * cos(Phi) ]
   *      in Map Projects - A Working Manual by John P. Snyder
   *      U.S. Geological Survey Proffessional Paper 1395
   *      USG Printing Office, Washington: 1987
   *   @return dA = A
   */

  private double getEqn_8_15()
  {
    double dA = 0.0;

    dA = (dLongitude - dMedianLongitude) * Math.cos(dLatitude);

    return (dA);
  }

  /** Method getEqn_3_21()
   *
   *  <p> This method will compute the equation 3-22
   *      a simplified form of Clark 1866 ellipsoid.
   *          [ M = a * [(1 - e^2/4 - 3*e^4/64 - 5 e^6/256 - ...) * Phi -
   *                (3e^2/8 + 3*e^4/32 + 45*e^6/1024 + ...)*sin(2*Phi) +
   *                (15*e^4/256 + 45*e^6/1024 + ...) *sin(4*Phi) -
   *                (35*e^6/3072 + ...) * sin(6*Phi) + ... ] ]
   *      in Map Projects - A Working Manual by John P. Snyder
   *      U.S. Geological Survey Proffessional Paper 1395
   *      USG Printing Office, Washington: 1987
   *   @return dM = M
   */

  private double getEqn_3_21()
  {
    double dM = 0.0;

    dM = 111132.0894 * dLatitude * 180.0 / Math.PI -
            16216.94 * Math.sin(2 * dLatitude) +
               17.21 * Math.sin(4*dLatitude) -
                0.02 * Math.sin(6*dLatitude);

    return (dM);
  }

  /** Method computeUTM()
   *  <p> This method will compute the UTM.
   */

  private void computeUTM()
  {
    int    iTemp      = 0;
    double factor     = 0.0;
    double dE2rdPrime = getEqn_8_12();
    double dN         = getEqn_4_20();
    double dT         = getEqn_8_13();
    double dC         = getEqn_8_14();
    double dA         = getEqn_8_15();
    double dM         = getEqn_3_21();

    factor = 500000.0;  // Comment this line out if you use the testing values.

    dUTM_X = factor + dKo * dN * (dA + (1-dT+dC) * (Math.pow(dA, 3.0)/6.0) +
                        (5.0 - 18.0*dT + Math.pow(dT, 2.0) + 72.0*dC -
                         58.0*dE2rdPrime) * Math.pow(dA, 5.0)/120.0);
    iTemp = (int) (dUTM_X * 100.0);
    dUTM_X = (double) iTemp / 100.0;

    dUTM_Y = dKo * (dM - dMo +
                         dN * Math.tan(dLatitude) * (Math.pow(dA, 2.0)/2.0 +
                         (5.0 - dT + 9.0*dC +
                          4.0*Math.pow(dC, 2.0)) * Math.pow(dA, 4.0)/24.0 +
                         (61.0 - 58.0*dT + Math.pow(dT, 2.0) + 600.0*dC -
                          330.0*dE2rdPrime) * Math.pow(dA, 6.0)/720.0));
    iTemp = (int) (dUTM_Y * 100.0);
    dUTM_Y = (double) iTemp / 100.0;
  }

  /* ====================================================================== *
   * ---------------------------- GET METHODS ----------------------------- *
   * ====================================================================== */

  /** Method getZone()
   *  <p> This Method will retrieve the Zone of the computed UTM Value
   *  @return iZone = The Zone of the computed UTM Value.
   */

  public int getZone() { return (iZone); }

  /** Method getUTM()
   *  <p> This Method will retrieve the UTM Value
   *  @param  iUTM = The UTM Identifier
   *  @return dUTM = the UTM of the specific Identifier
   */

  public double getUTM(int iUTM)
  {
    double dUTM = 0.0;

    if (iUTM == UTM_X)
      dUTM = dUTM_X;
    else
      dUTM = dUTM_Y;

    return (dUTM);
  }

  /* ====================================================================== *
   * ---------------------------- SET METHODS ----------------------------- *
   * ====================================================================== */

  /** Method setUTM()
   *  <p> This Method will set the UTM Value
   *  @param  iUTM = The UTM Identifier
   */

  public void setUTM(int iUTM, double dUTM)
  {
    if (iUTM == UTM_X)
      dUTM_X = dUTM;
    else
      dUTM_Y = dUTM;
  }
}
