/*************************************************************************
 *  Compilation:  javac utilImageIO.java
 *  Execution:    java utilImageIO filename
 *
 *  Data type for manipulating individual pixels of an image. The original
 *  image can be read from a file in JPEG, GIF, or PNG format, or the
 *  user can create a blank image of a given size. Includes methods for
 *  displaying the image in a window on the screen or saving to a file.
 *
 *  % java utilImageIO image.jpg
 *
 *  Remarks
 *  -------
 *   - pixel (0, 0) is upper left hand corner
 *     should we change this?
 *
 *   - if JPEG read in is in grayscale, then you can only set the
 *     color to a graycale value
 *     should we change this?
 *
 *************************************************************************/

package util;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.net.*;
import java.io.File;
import java.io.IOException;
import java.awt.Color;
import java.awt.Image;
import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.JLabel;

/**
 *  This class provides methods for manipulating individual pixels of
 *  an image. The original image can be read from a file in JPEG, GIF,
 *  or PNG format, or the user can create a blank image of a given size.
 *  This class includes methods for displaying the image in a window on
 *  the screen or saving to a file.
 *  <p>
 *  By default, pixel (x, y) is column x, row y, where (0, 0) is upper left.
 *  The method setOriginLowerLeft() change the origin to the lower left.
 *  <p>
 *  For additional documentation, see
 *  <a href="http://www.cs.princeton.edu/introcs/31datatype">Section 3.1</a> of
 *  <i>Introduction to Programming in Java: An Interdisciplinary Approach</i>
 *  by Robert Sedgewick and Kevin Wayne.
 */

public class utilImageIO
{
  private BufferedImage image = null;    // the rasterized image
  private JFrame        f     = null;    // on-screen view

  // create a blank w-by-h image

  public utilImageIO(int w, int h)
  {
    image = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
  }

  // use an existing Buffered Image

  public utilImageIO(BufferedImage image)
  {
    this.image = image;
  }

  // create an image by reading in the PNG, GIF, or JPEG from a filename

  public utilImageIO(String filename)
  {
    try { image = ImageIO.read(new File(filename)); }
    catch(IOException e)
    {
      e.printStackTrace();
      throw new RuntimeException("Could not open file " + filename);
    }
  }

  // create an image by reading in the PNG, GIF, or JPEG from a file

  public utilImageIO(File file)
  {
    try { image = ImageIO.read(file); }
    catch(IOException e)
    {
      e.printStackTrace();
      throw new RuntimeException("Could not open file " + file);
    }
  }

  // create an image by reading in the PNG, GIF, or JPEG from a file

  public utilImageIO(String filename, String type)
  {
    try
    {
      if (type == "URL")
      {
        URL url = new URL(filename);
        image = ImageIO.read(url);
      }
    }
    catch(IOException e)
    {
      e.printStackTrace();
      throw new RuntimeException("Could not open file " + filename);
    }
  }

  /** Method close()
   *  <p> This method will deallocate java memory
   */

  public void close()
  {
    image = null;    // the rasterized image
    if (f != null)   // on-screen view
    {
      f.dispose();
      f = null;
	}
  }

  // to embed in a JPanel, JFrame or other GUI widget

  public JLabel getJLabel()
  {
    if (image == null) return null;         // no image available
    ImageIcon icon = new ImageIcon(image);
    return new JLabel(icon);
  }

  // view on-screen, creating new frame if necessary

  public void show(String sTitle)
  {
    // create the GUI for viewing the image if needed

    if (f == null)
    {
      f = new JFrame();
      f.setContentPane(getJLabel());
      // f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
      f.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
      if (sTitle != null)
        f.setTitle(sTitle);
      f.setResizable(false);
      f.pack();
      f.setVisible(true);
    }

    // draw

    f.repaint();
  }

  // accessor methods

  public int height() { return image.getHeight(null); }
  public int width()  { return image.getWidth(null);  }

  // Get Buffered Image

  public BufferedImage getImage() { return (image); }

  // return Color of pixel (i, j)

  public Color getColor(int i, int j) { return new Color(image.getRGB(i, j)); }

  // return grayscale equivalent of pixel (i, j)

  public int getGray(int i, int j)
  {
    Color color = getColor(i, j);
    int r = color.getRed();
    int g = color.getGreen();
    int b = color.getBlue();
    return (int) ((r + g + b) / 3.0);
  }

  // set pixel (i, j) to given grayscale value

  public void setGray(int i, int j, int c)
  {
    Color color = new Color(c, c, c);
    setColor(i, j, color);
  }

  // change color of pixel (i, j) to c

  public void setColor(int i, int j, Color c) {image.setRGB(i, j, c.getRGB());}

  // save to given filename - suffix must be png, jpg, or gif

  public void save(String filename) { save(new File(filename)); }

  // save to given filename - suffix mst be png, jpg, or gif

  public void save(File file)
  {
    String filename = file.getName();
    String suffix = filename.substring(filename.lastIndexOf('.') + 1);
    try { ImageIO.write(image, suffix, file); }
    catch (IOException e) { e.printStackTrace(); }
  }

  // test client: read in input file and display

  public static void main(String args[])
  {
    utilImageIO pic = new utilImageIO(args[0], "URL");
    pic.show("Image IO Viewer");
    pic.save("images/core.png");
  }
}
