/*
 * @utilFileIO.java Version 1.1 03/13/2001
 *
 * Copyright (c) 2001 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package util;

import java.io.*;
import java.net.*;
import java.lang.*;
import java.text.*;
import java.util.*;
import java.awt.*;
import java.util.zip.*;

/** Class utilFileIO
 *  <p> This Class is designed to open an ascii text file and load
 *      the contents into a string buffer.
 *
 *  @version 1.1 03/13/2001
 *  @author  John R. Victorine
 */

public class utilFileIO
{
  public static final int    READ      = 0;
  public static final int    WRITE     = 1;

  public static final int    URL_FILE  = 0;  // Read or Write using a URL Path
  public static final int    DIRECTORY = 1;  // Read or Write using a Directory

  public static final int    ERROR     = -1; // Error during DB Operation.
  public static final int    NOERROR   =  0; // No Error during DB Operation.

  public static final String EOF       = "EOF";  // End of File Indicator

  private int               iAction    = READ;
  private int               iFileID    = DIRECTORY;

  private String            sHOME      = new String("");
  private String            sDirectory = new String(".");
  private String            sFilename  = new String("");

  private String            sLine      = new String("");
  private URL               url        = null;
  private URLConnection     urlConn    = null;
  private BufferedReader    pIN        = null;
  private OutputStream      pOUT       = null;
  private HttpURLConnection connection = null;
  private int               ivalue     = -1;
  private int               iError     = NOERROR;
  private String            sError     = new String("");

  /** Constructor utilFileIO()
    * <p> This is the Constructor for this class.
    */

  public utilFileIO() {  }

  /** Method delete()
   * <p> This method will set all the global variables to null to force
   *     java to free the allocated memory
   */

  public void delete()
  {
    iAction    = READ;
    iFileID    = DIRECTORY;

    sHOME      = null;
    sDirectory = null;
    sFilename  = null;

    sLine      = null;
    url        = null;
    urlConn    = null;
    pIN        = null;
    pOUT       = null;
    connection = null;
    ivalue     = -1;
    iError     = NOERROR;
    sError     = null;
  }

  /** Method Open()
    * <p> This is the Constructor for this class.
    * @param iFileID   = To access the file as URL or FILE.
    * @param iAction   = Action READ from or WRITE to file.
    * @param directory = File URL or Directory Path
    * @param filename  = File Name
    */

  public void Open(int iFileID, int iAction, String directory, String filename)
  {
    this.iFileID = iFileID;
    this.iAction = iAction;

    try
    {
      sDirectory = new String(directory);
    }
    catch (Exception e)
    {
      e.printStackTrace();
      iError = ERROR;
      sError = new String("utilFileIO.Open() - Directory Path: " + directory);
    }

    try
    {
      sFilename  = new String(filename);
    }
    catch (Exception e)
    {
      e.printStackTrace();
      iError = ERROR;
      sError = new String("utilFileIO.Open() - Filename: " + filename);
    }

    if (this.iAction == READ)
    {
      if (this.iFileID == URL_FILE)
      {
        if (Exists(URL_FILE))
        {
          Read();
        }
      }
      else if (this.iFileID == DIRECTORY)
      {
        if (Exists(DIRECTORY))
        {
          ReadFile();
        }
      }
    }
  }

  /** Method Read()
    * <p> This method will Get the URL Path for the File, open a connection to
    *     the file and set up a data input stream to read the file.
    */

  public void Read()
  {
    iAction = READ;

    try
    {
      String sFile = new String(sDirectory + "/" + sFilename);

      url     = new URL(sFile);
      urlConn = url.openConnection();
      urlConn.setDoInput(true);
      urlConn.setUseCaches(false);
      pIN = new BufferedReader(new InputStreamReader(urlConn.getInputStream()));

      ivalue = 0;
    }
    catch (Exception e)
    {
      e.printStackTrace();
      iError = ERROR;
      sError = new String("utilFileIO.Read() Exception:\n " + e.toString());

      if (pIN != null)
        Close();
    }
  }

  /** Method ReadFile()
    * <p> This method will Get the URL Path for the File, open a connection to
    *     the file and set up a data input stream to read the file.
    */

  public void ReadFile()
  {
    iAction = READ;

    String sFile   = new String(sDirectory + "/" + sFilename);

    try
    {
      pIN = new BufferedReader(new FileReader(sFile));
      ivalue = 0;
    }
    catch (Exception e)
    {
      e.printStackTrace();
      iError = ERROR;
      sError = new String("utilFileIO.ReadFile() Exception:\n " + e.toString());

      if (pIN != null)
        Close();
    }
  }

  /** Method Write()
   *  <p> This method will write a String Block to a file.  This method assumes
   *      the user applied the new line characters to create a normal looking
   *      document.  This method will also open the file.
   *  @param sFileData = The File Data String containing the data to be saved.
   */

  public void Write(String sFileData)
  {
    iAction = WRITE;

    try
    {
      String sFile   = new String(sDirectory + "/" + sFilename);
      byte   bData[] = sFileData.getBytes();

      if (pOUT == null)
        pOUT = new FileOutputStream(sFile);

      pOUT.write(bData);
    }
    catch(Exception e)
    {
      e.printStackTrace();
      iError = ERROR;
      sError = new String("utilFileIO.Write() Exception:\n " + e.toString());

      if (pOUT != null)
        Close();
    }
  }

  /** Method Exists()
   *  <p> This method checks if the file exists.
   *  @param iMethod = the method being used to read the data,
   *                   URL_FILE  = 0 - URL Location used to locate file.
   *                   DIRECTORY = 1 - Directory Path used to locate file.
   *  @return bValue = true if the File Exists or false if it does not exist.
   */

  public boolean Exists(int iMethod)
  {
    boolean bValue = false;
    String  sFile  = new String(sDirectory + "/" + sFilename);

    if (iMethod == URL_FILE)
    {
      try
      {
        url = new URL(sFile);
        urlConn = url.openConnection();
        if (urlConn != null)
        {
          bValue = true;
        }
      }
      catch (Exception e)//MalformedURLException e)
      {
        e.printStackTrace();
        iError = ERROR;
        sError = new String("utilFileIO.Exists() URL:\n " + e.toString());
      }
    }
    else if (iMethod == DIRECTORY)
    {
      try
      {
        File p = new File(sFile);
        if (p.exists())
        {
          bValue = true;
        }
      }
      catch(Exception e)
      {
        e.printStackTrace();
        iError = ERROR;
        sError = new String("utilFileIO.Exists() DIRECTORY:\n "+e.toString());
      }
    }

    return (bValue);
  }

  /** Method Next()
   *  <p> This method will allow reading each line from the File.
   *
   *  @return sLine - Next Line in the file.
   */

  public String Next()
  {
    if (ivalue == 0)
    {
      try
      {
        sLine = pIN.readLine();
        if (sLine != null)
          sLine = sLine.trim();
//        if ((sLine=pIN.readLine()) == null)
        if (sLine == null)
        {
          ivalue = -1;
          sLine  = new String(EOF);
        }
      }
      catch (MalformedURLException e)
      {
        e.printStackTrace();
        iError = ERROR;
        sError = new String("utilFileIO.Next(): MALFORMED URL" + e.toString());
        if (pIN != null)
          Close();
      }
      catch (IOException e)
      {
        e.printStackTrace();
        iError = ERROR;
        sError = new String("utilFileIO.Next(): IO" + e.toString());
        if (pIN != null)
          Close();
      }
    }

    return (sLine);
  }

  /** Method Close()
   *  <p> This method closes the File.
   */

  public void Close()
  {
    try
    {
      // IF the user selected the Read() Method THEN Close the Input File.
      if (iAction == READ)
      {
        pIN.close();
      }
      // IF the user selecte the Write() Method THEN Close the Output File.
      else
      {
        pOUT.close();
        pOUT = null;
      }
    }
    catch (MalformedURLException e)
    {
      e.printStackTrace();
      iError = ERROR;
      sError = new String("utilFileIO.Close(): Malformed URL" + e.toString());
    }
    catch (IOException e)
    {
      e.printStackTrace();
      iError = ERROR;
      sError = new String("utilFileIO.Close(): IO" + e.toString());
    }
  }

  /** Method mkdir()
   * <p> This method will create a New Directory if it does not exist.
   * @param sNewDirectory = New Directory Name
   */

  public void mkdir(String sNewDirectory)
  {
    try
    {
      File p = new File(sNewDirectory);

      if (!p.exists())
      {
        p.mkdir();
      }
    }
    catch (Exception ioe)
    {
      ioe.printStackTrace();
      iError = ERROR;
      sError = new String("utilFileIO.MakeDirectory() Exception:\n " +
                           ioe.toString());
    }
  }

  /* ====================================================================== *
   * ----------------------------- GET METHOD ----------------------------- *
   * ====================================================================== */

  /** Method getCanonicalPath()
   * <p> This will return the Canonical Path
   * @param  sBase = The Base Path to start or Null if user is to use HOME
   * @return sPath = The Canonical Path
   */

  public String getCanonicalPath(String sBase)
  {
    String sPath = new String(sHOME);

    if (sBase == null)
    {
      sBase = new String(sHOME);
    }

    try
    {
      File p = new File(sBase);
      sPath  = new String(p.getCanonicalPath());
    }
    catch (Exception ioe)
    {
      ioe.printStackTrace();
      iError = ERROR;
      sError = new String("utilFileIO.getCanonicalPath() Exception:\n " +
                           ioe.toString());
    }

    return (sPath);
  }

  /** Method getUserHomePath()
   * <p> This will return the User's Home Directory Path
   * @return sPath = The User's Home Directory Path
   */

  public String getUserHomePath()
  {
    String sPath = new String("");

    try
    {
      File p = new File(System.getProperty( "user.home" ));
      sPath  = new String(p.getCanonicalPath());
    }
    catch (Exception ioe)
    {
      ioe.printStackTrace();
      iError = ERROR;
      sError = new String("utilFileIO.getUserHomePath() Exception:\n " +
                           ioe.toString());
    }

    return (sPath);
  }

  /** Method UniqueName()
   *  <p> This method will create a Unique Name based on the date and time.
   *  @return sTime = A unique name based on the date and time,
   *                  i.e., YYMMDDhhmmssAM or YYMMDDhhmmssPM
   *                  where YY is the Year
   *                        MM is the Month Number
   *                        DD is the Day
   *                        hh is the Hour
   *                        mm is the Minute
   *                        ss is the Second
   *                        AM is the Morning
   *                        PM is the Evening
   */

  public String UniqueName()
  {
    String sDirectory = new String("");

    StringTokenizer      stToken;
    String               sText;
    String               sMonth;
    String               sDay;
    String               sYear;
    String               sTime;
    String               sAMPM;
    String               sHour;
    String               sMinute;
    String               second;

    DateFormat newDate = DateFormat.getDateInstance(DateFormat.SHORT);
    DateFormat newTime = DateFormat.getTimeInstance(DateFormat.MEDIUM);

    sText     = newDate.format(new Date());
    stToken   = new StringTokenizer(sText, "/");

    sMonth    = stToken.nextToken();
    if (sMonth.length() < 2)
      sMonth  = new String("0"+sMonth);

    sDay      = stToken.nextToken();
    if (sDay.length() < 2)
      sDay    = new String("0"+sDay);

    sYear     = stToken.nextToken();
    if (sYear.length() < 2)
      sYear   = new String("0"+sYear);

    sText     = newTime.format(new Date());
    stToken   = new StringTokenizer(sText, " ");
    sTime     = stToken.nextToken();
    sAMPM     = stToken.nextToken();

    stToken   = new StringTokenizer(sTime, ":");

    sHour     = stToken.nextToken();
    if (sHour.length() < 2)
      sHour   = new String("0"+sHour);

    sMinute   = stToken.nextToken();
    if (sMinute.length() < 2)
      sMinute = new String("0"+sMinute);

    second    = stToken.nextToken();
    if (second.length() < 2)
      second  = new String("0"+second);

    /*
     *  Build the Directory Name from the Data and Time
     *  FROM DATE:
     *    sYear   = Year Number  (YY) 2 Digits
     *    sMonth  = Month Number (MM) 2 Digits
     *    sDay    = Day Number   (DD) 2 Digits
     *  FROM TIME:
     *    sHour   = Hour         (hh) 2 Digits
     *    sMinute = Minutes      (mm) 2 Digits
     *    second  = Seconds      (ss) 2 Digits
     *    sAMPM   = AM for morning, PM for afternoon/evening
     *  i.e., 020921022423AM - YYMMDDhhmmssAM or YYMMDDhhmmssPM
     */

    sDirectory = new String(
        sYear + sMonth + sDay + sHour + sMinute + second + sAMPM);

    return (sDirectory);
  }

  /** Method getString()
   *  <p> This method will return the line retrieved from the file.
   * @return sLine = The Retrieve Line from File.
   */

  public String getString() { return (sLine); }

  /** Method getHOME()
   * <p> This method will return the Canonical Path of the HOME Directory.
   * @return sHOME = The Cononical Path of the HOME Directory.
   */

  public String getHOME() { return (sHOME); }

  /* ====================================================================== *
   * ---------------------------- PRINT METHOD ---------------------------- *
   * ====================================================================== */

  /** Method Print()
   *  <p> This method will print the line retrieved from the file to the screen.
   */

  public void Print() { System.out.println(sLine); }

  /* ====================================================================== *
   * -------------------------- GET ERROR METHODS ------------------------- *
   * ====================================================================== */

  /** METHOD getErrorID()
   *  <p> This method will return the error indicator.
   * @return iError = the Error Identifier.
   */

  public int getErrorID() { return (iError); }

  /** METHOD getError()
   *  <p> This method will return the error string.
   * @return sError = The Error Message
   */

  public String getError() { return (sError); }
}