/*
 * @sedimentaryUtility.java Version 1.1 09/14/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.sedimentary;

import iqstrat.iqstratHeadersStruct;

import mask.maskSymbolsListStruct;
import rock.sedimentary.sedimentaryListStruct;
import rock.sedimentary.sedimentaryStruct;

/** Class sedimentaryUtility
 *  <p> This Class will provide basic utilities for the Rock Sedimentary
 *      data structures.
 *
 *  @version 1.1 09/14/2007
 *  @author  John R. Victorine
 */

public class sedimentaryUtility
{
  /** Method addHeaderData()
   * <p> This method will add the header data to the Sedimentary Structure data
   *     List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Rock Sedimentary Structure List Data Structure
   * @return st       = The Rock Sedimentary Structure List Data Structure
   */

  public static sedimentaryListStruct addHeaderData(iqstratHeadersStruct stHeader,
                                                    sedimentaryListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method computeDepthRange()
   * <p> This method will compute the depth range of the core data
   * @param   st = the Sedimentary structure data list structure
   * @return  st = the Sedimentary structure data list structure
   */

  public static sedimentaryListStruct computeDepthRange(sedimentaryListStruct st)
  {
    double depthStart = 0.0;
    double depthEnd   = 0.0;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        depthStart = st.stItem[i].depthStart;
        depthEnd   = st.stItem[i].depthEnd;

        if (i==0)
        {
          st.depthStart = depthStart;
          st.depthEnd   = depthEnd;
        }

        if (st.depthStart > depthStart)
          st.depthStart = depthStart;

        if (st.depthEnd < depthEnd)
          st.depthEnd   = depthEnd;
      }
    }

    return (st);
  }

  /** Method computeWidth( sedimentaryListStruct st )
   * <p> This method will compute the plot track width from the total number of
   *     objects in the observation data
   * @param  st     = the Sedimentary structure data list structure
   * @return iWidth = the plot track width
   */

  public static int computeWidth( sedimentaryListStruct st )
  {
    int iWidth = 20;
    int iCol   = 20;
    int iTemp  = 0;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        for (int i=0; i<st.iCount; i++)
        {
          iTemp = iCol*st.stItem[i].iTotal;
          if (iWidth < iTemp)
            iWidth = iTemp;
        }
      }
    }

    return (iWidth);
  }

  /** Method addStructure()
   * <p> This method will add a fossil to a specific depth this allows for
   *     the ability of inserting multiple sedimentary structures to a specific
   *     depth without having to know how many sedimentary structures are in the
   * depth.
   * @param  sID     = The ID representing a specific sedimentary structures
   * @param  sName   = the sedimentary structures name
   * @param  st      = The Old sedimentary structures Data Structure
   * @return st      = The new sedimentary structures Data Structure
   */

  public static sedimentaryStruct addStructure( String sID,
                                                String sName,
                                                sedimentaryStruct st )
  {
    int    i           = 0;
    String sTempID[]   = null;
    String sTempName[] = null;
    int    iCount      = 0;
    int    iFound      = 0;

    if (st != null)
    {
      sTempID   = new String[st.iTotal+1];
      sTempName = new String[st.iTotal+1];

      for (i=0; i<st.iTotal; i++)
      {
        if ((sName.equals(st.sText[i]) || (sID.equals(st.sID[i])))) iFound = 1;

        sTempID[iCount]   = new String(st.sID[i]);
        sTempName[iCount] = new String(st.sText[i]);
        iCount++;
      }

      if (iFound == 0)
      {
        sTempID[iCount]   = new String(sID);
        sTempName[iCount] = new String(sName);
        iCount++;
      }

      st.iTotal = iCount;
      st.sID    = new String[iCount];
      st.sText  = new String[iCount];

      for (i=0; i<iCount; i++)
      {
        st.sID[i]   = new String(sTempID[i]);
        st.sText[i] = new String(sTempName[i]);
      }
    }

    return (st);
  }

  /** Method add()
   * <p> This method will add the sedimentary structures to an existing list
   * @param  stBlock = The sedimentary structures Block List Data Structure
   * @param  st      = The Old sedimentary structures List Data Structure
   * @return st      = The new sedimentary structures List.
   */

  public static sedimentaryListStruct add( sedimentaryStruct stBlock,
                                           sedimentaryListStruct st)
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    sedimentaryListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new sedimentaryListStruct();
    stTemp.stItem = new sedimentaryStruct[iRecords];

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new sedimentaryStruct();
    stTemp.stItem[iCount] = copy(stBlock);
    iCount++;

    stTemp.iCount = iCount;

    st        = new sedimentaryListStruct();
    st.stItem = new sedimentaryStruct[iCount];
    st.iCount = iCount;

    st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
    st.source   = new String( source );    // Source of Data
    st.sCreated = new String( sCreated );  // Created Date

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a sedimentary structures in an existing list
   * @param  sKEY    = The Primary Key of the data to be removed
   * @param  stBlock = The sedimentary structure that will be replaced
   * @param  st      = The Old sedimentary structures List Data Structure
   * @return st      = The new sedimentary structures List.
   */

  public static sedimentaryListStruct modify( String sKEY,
                                              sedimentaryStruct stBlock,
                                              sedimentaryListStruct st)
  {
    int i=0;
    sedimentaryListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      stTemp        = new sedimentaryListStruct();
      stTemp.stItem = new sedimentaryStruct[st.iCount];
      stTemp.iCount = st.iCount;

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            stTemp.stItem[i] = copy(stBlock);
          }
          else
          {
            stTemp.stItem[i] = copy(st.stItem[i]);
          }
        }
      }

      st.delete();

      st        = new sedimentaryListStruct();
      st.stItem = new sedimentaryStruct[stTemp.iCount];
      st.iCount = stTemp.iCount;

      st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
      st.source   = new String( source );    // Source of Data
      st.sCreated = new String( sCreated );  // Created Date

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a sedimentary structures from an existing list
   * @param  sKEY   = The Primary Key of the data to be removed
   * @param  st     = The Old sedimentary structures List Data Structure
   * @return st     = The new sedimentary structures List.
   */

  public static sedimentaryListStruct remove( String sKEY,
                                              sedimentaryListStruct st)
  {
    int i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iFound   = -1;
    sedimentaryListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st.iCount == 1)
    {
      st.delete();
      st = null;
    }
    else
    {
      if (st != null)
      {
        sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
        source   = new String( st.source );    // Source of Data
        sCreated = new String( st.sCreated );  // Created Date

        iRecords = st.iCount-1;

        stTemp        = new sedimentaryListStruct();
        stTemp.stItem = new sedimentaryStruct[iRecords];

        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            iFound = i;
          }
        }
      }

      if (st != null)
      {
        if (st.iCount > 0)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (iCount < iRecords)
            {
              if (i != iFound)
              {
                stTemp.stItem[iCount] = copy(st.stItem[i]);
                iCount++;
              }
            }
          }
        }

        st.delete();
      }

      stTemp.iCount = iCount;

      st        = new sedimentaryListStruct();
      st.stItem = new sedimentaryStruct[iCount];
      st.iCount = iCount;

      st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
      st.source   = new String( source );    // Source of Data
      st.sCreated = new String( sCreated );  // Created Date

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method copyList()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static sedimentaryListStruct copyList(sedimentaryListStruct stOld)
  {
    sedimentaryListStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new sedimentaryListStruct();
      stNew.iCount     = stOld.iCount;
      stNew.depthStart = stOld.depthStart; // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth
      stNew.stItem     = new sedimentaryStruct[stOld.iCount];

      // LAS 3 Dependent Variable

      stNew.sKGS       = new String( stOld.sKGS ); // KGS Saved Data Indicator
      stNew.source     = new String( stOld.source );    // Source of Data
      stNew.sCreated   = new String( stOld.sCreated );  // Created Date

      for (int i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy(stOld.stItem[i]);
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static sedimentaryListStruct transfer(sedimentaryListStruct stOld)
  {
    sedimentaryListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy data from one structure to another
   * @param  stOld = The original Fossil Data Structure
   * @return stNew = The Fossil Data Structure
   */

  public static sedimentaryStruct copy(sedimentaryStruct stOld)
  {
    sedimentaryStruct stNew = new sedimentaryStruct();

    if (stOld != null)
    {
      stNew.sKEY       = new String(stOld.sKEY); // Primary Key for data set
      stNew.depthStart = stOld.depthStart; // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth

      stNew.iTotal     = stOld.iTotal;
      stNew.sID        = new String[stNew.iTotal];
      stNew.sText      = new String[stNew.iTotal];

      for (int i=0; i<stNew.iTotal; i++)
      {
        stNew.sID[i]   = new String(stOld.sID[i]);
        stNew.sText[i] = new String(stOld.sText[i]);
      }
    }

    return (stNew);
  }

  /** Method setLegend( int iSedStruct[], lithologyListStruct st )
   * <p> This method will set the legend variables
   * @param  iSedStruct = Array of indicators to plot sedimentary structure
   *                      symbol or not
   * @param  st         = Sedimentary Structures List Data Structure
   * @param  stList     = Sedimentary Structure Plot Symbols List Data Structure
   * @return iSedStruct = Array of indicators to plot sedimentary structure
   *                      symbol or not
   */

  public static int[] setLegend( int iSedStruct[], sedimentaryListStruct st,
                                 maskSymbolsListStruct stList )
  {
    if ((st != null) && (stList != null))
    {
      for (int i=0; i<st.iCount; i++)
      {
        for (int k=0; k<st.stItem[i].iTotal; k++)
        {
          for (int j=0; j<stList.iCount; j++)
          {
            if (st.stItem[i].sID[k].equals(stList.stItem[j].sID))
              iSedStruct[j] = 0;
          }
        }
      }
    }

    return (iSedStruct);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = data list structure
   * @return st = sorted data list structure
   */

  public static sedimentaryListStruct bubbleSort(sedimentaryListStruct st)
  {
    boolean swappedOnPrevRun = true;
    sedimentaryStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].depthStart > st.stItem[i+1].depthStart)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  public static void print(sedimentaryListStruct st)
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        System.out.println(st.stItem[i].sKEY+" "+
                           st.stItem[i].depthStart+" "+st.stItem[i].depthEnd);
        for (int j=0; j<st.stItem[i].iTotal; j++)
        {
          System.out.println("  "+j+" "+st.stItem[i].sID[j]+" "+st.stItem[i].sText[j]);
        }
        System.out.println(" ----------------------------------------------- ");
      }
    }
  }
}