/*
 * @sedimentaryPlotTrack.java Version 1.1 09/24/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.sedimentary.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;
import java.net.*;
import javax.swing.*;

import mask.maskSymbolsListStruct;
import mask.plot.maskPlotSymbol;
import rock.sedimentary.sedimentaryListStruct;

/** CLASS sedimentaryPlotTrack
 *  <p> This Class will create the Fossil Track.
 *
 *  @version 1.1 09/24/2007
 *  @author  John R. Victorine
 */

public class sedimentaryPlotTrack extends Canvas
{
  // Input Variables

  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale
  private maskSymbolsListStruct stSymbols = null; // Sed Struct Symbols List

  // Global Variables

  private maskPlotSymbol        plot     = null; // Plot Mask Symbol
  private sedimentaryListStruct stStruct = null;

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR sedimentaryPlotTrack( int iDataType,
   *                                    double depthStart,
   *                                    double depthEnd,
   *                                    int iScale,
   *                                    maskSymbolsListStruct stSymbols )
   *  <p> This is the constructor for this class.
   *  @param iDataType  = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart = The starting depth of plot
   *  @param depthEnd   = The ending depth of plot
   *  @param iScale     = The scale
   *  @param stSymbols  = Sedimentary Structure Plot Symbols List Data Structure
   */

  public sedimentaryPlotTrack( int iDataType,
                               double depthStart,
                               double depthEnd,
                               int iScale,
                               maskSymbolsListStruct stSymbols)
  {
    this.iDataType = iDataType;
    this.stSymbols = stSymbols;

    plot = new maskPlotSymbol( stSymbols );

    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
    stStruct = null;
    stSymbols = null;

    if (plot != null)
      plot.close();
    plot = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getData()
   * <p> This method will get an existing Fossil List Data Structure
   * @return stStruct = Fossil List Data Structure
   */

  public sedimentaryListStruct getData() { return (stStruct); }

  /** Method getNextDepth( int iRow, double diff )
   * <p> This method will retrieve the next depth data in the list in such
   *     a way that the text does not overwrite
   * @param  iRow  = the present row
   * @param  diff  = the difference between text
   * @return iNext = The next Row
   */

  private int getNextDepth( int iRow, double diff )
  {
    int    iNext  = -1;
    int    iValue = (int) diff;

    diff  = (double) iValue;

    if (iRow > -1)
    {
      if (stStruct != null)
      {

        for (int i=iRow; i<stStruct.iCount; i++)
        {
          if ((iNext == -1) &&
              (stStruct.stItem[i].depthStart >=
                 stStruct.stItem[iRow].depthEnd + diff))
          {
            iNext = i;
            break;
          }
        }
      }
    }

    return (iNext);
  }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setData( sedimentaryListStruct stStruct )
   * <p> This method will set an existing sedimentary structures List Data Structure
   * @param stStruct = sedimentary structures List Data Structure
   */

  public void setData( sedimentaryListStruct stStruct )
  {
    this.stStruct = stStruct;
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iStart    = 0;
    int    i         = 0;
    int    j         = 0;
    int    jXinc     = 0;
    int    iY1       = 0;
    int    iY2       = 0;
    int    iEndTrack = iStartTrack+iWidthTrack;
    Font        fsb  = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    iStart = iWidthTrack/20;

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES)  && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    plot.drawSymbol( g, stSymbols.stItem[7].iRows, 0,
                     stSymbols.stItem[7].symbol,
                     iStartTrack+10*(iStart-1)+1, LABELSTART+4 );

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Sedimentary", LABELSTART+20, -1*(iStartTrack+5*(iStart-1)+12));
    g2.drawString("Structures",  LABELSTART+20, -1*(iStartTrack+5*(iStart-1)+3));
    g2.rotate( -1.0 * (Math.PI / 2.0));
  }

  /** Method drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   *

  public void drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iCycle = 1;
    double dXinc  = 0.0;
    double dTemp  = 0.0;
    int    iIncrement  = iWidthTrack/5;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iY1         = 0;
    int    iY2         = 0;

    Font        fsb  = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    g.setColor(Color.black);

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if (iY2 >= PLOT_TITLES)
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.lightGray);
    g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iHeight);

    for (i=0; i<=iWidthTrack; i+=iIncrement)
    {
      if ((i==0) || (i==iWidthTrack))
      {
        g.drawLine(i+iStartTrack, LABELSTART,
                   i+iStartTrack, PLOT_TITLES+iHeight);
      }
    }

    g.setColor(Color.black);
    g.drawString("Sedimentary Structures", iStartTrack+10, LABELSTART+14);

    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, LABELSTART, iStartTrack, PLOT_TITLES+iHeight);
    g.drawLine(iEndTrack,   LABELSTART, iEndTrack,   PLOT_TITLES+iHeight);
  }
*/
  /** Method drawStructures( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the sedimentay Structure images that was selected
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawStructures( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iStart   = iStartTrack;
    int    iEnd     = iStartTrack+iWidthTrack;
    int    iFound   = -1;
    int    iDepth1  = 0;
    int    iDepth2  = 0;
    int    iDepth   = 0;
    int    iGroup   = 0;
    int    iX       = 0;
    double dRatio   = 0.0;
    double depth1   = 0.0;
    double depth2   = 0.0;
    double depth    = 0.0;
    int    iTemp    = -1;
    double diff     = iqstrat.iqstratTracksStruct.SCALE[iScale] / 10.0;

    if (stStruct != null)
    {
      for (int i=0; i<stStruct.iCount; i++)
      {
        depth1 = stStruct.stItem[i].depthStart;
        depth2 = stStruct.stItem[i].depthEnd;

        dRatio = (double) iLogHeight*(depthStart-depth1)/(depthStart-depthEnd);
        iDepth1 = PLOT_TITLES + (int) dRatio;

        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth1 = iHeight - (int) dRatio;

        dRatio = (double) iLogHeight*(depthStart-depth2)/(depthStart-depthEnd);
        iDepth2 = PLOT_TITLES + (int) dRatio;

        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth2 = iHeight - (int) dRatio;

        if (iDepth1 == iDepth2)
        {
          iDepth1 = iDepth1 - 5;
          iDepth2 = iDepth2 - 5;
        }

        iDepth = Math.abs(iDepth2 - iDepth1);
        iGroup = iDepth / 10;
        if (iGroup == 0)
          iGroup = 1;

        if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
            (iDepth1 < iHeight) && (iDepth2 <= iHeight))
        {
          for (int n=0; n<iGroup; n++)
          {
            iX = iStartTrack;
            iDepth = iDepth1 + n * 10;

            for (int k=0; k<stStruct.stItem[i].iTotal; k++)
            {
              iFound = -1;
              for (int j=0; j<stSymbols.iCount; j++)
              {
                if (stStruct.stItem[i].sID[k].equals(stSymbols.stItem[j].sID))
                {
                  if (stSymbols.stItem[j].iGroup !=
                       mask.maskSymbolsUtility.getGroupID(
                       mask.maskGroupsStruct._MNEMONIC, "B", stSymbols.stGroups))
                  {
                    iFound = j;
                  }
                }
              }

              if (iFound > -1)
              {
                plot.drawSymbol( g, stSymbols.stItem[iFound].iRows, 0,
                                 stSymbols.stItem[iFound].symbol, iX, iDepth );

                iX = iX + 20;
              }
            }
          }

          iX = iStartTrack;

          for (int k=0; k<stStruct.stItem[i].iTotal; k++)
          {
            iFound = -1;
            for (int j=0; j<stSymbols.iCount; j++)
            {
              if (stStruct.stItem[i].sID[k].equals(stSymbols.stItem[j].sID))
              {
                if (stSymbols.stItem[j].iGroup ==
                      mask.maskSymbolsUtility.getGroupID(
                      mask.maskGroupsStruct._MNEMONIC, "B", stSymbols.stGroups))
                {
                  iFound = j;
                }
              }
            }

            if (iFound > -1)
            {
              for (int p=0; p<iWidthTrack/20; p++)
              {
                plot.drawSymbol( g, stSymbols.stItem[iFound].iRows, 0,
                                 stSymbols.stItem[iFound].symbol, iX, iDepth1 );

                iX = iX + 20;
              }
            }
          }
        }

        iTemp = getNextDepth(i, diff);

        if (iTemp > 0)
        {
          if (iTemp-1 > i) i = iTemp-1;
        }
      }
    }
  }

  /** Method drawBedStructures( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the sedimentay Structure images that was selected
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawBedStructures( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iStart   = iStartTrack;
    int    iEnd     = iStartTrack+iWidthTrack;
    int    iFound   = -1;
    int    iDepth1  = 0;
    int    iDepth2  = 0;
    int    iDepth   = 0;
    int    iX       = 0;
    double dRatio   = 0.0;
    double depth1   = 0.0;
    double depth2   = 0.0;
    double depth    = 0.0;
    int    iTemp    = -1;
    double diff     = iqstrat.iqstratTracksStruct.SCALE[iScale] / 10.0;

    if (stStruct != null)
    {
      for (int i=0; i<stStruct.iCount; i++)
      {
        depth1 = stStruct.stItem[i].depthStart;
        depth2 = stStruct.stItem[i].depthEnd;

        dRatio = (double) iLogHeight*(depthStart-depth1)/(depthStart-depthEnd);
        iDepth1 = PLOT_TITLES + (int) dRatio;

        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth1 = iHeight - (int) dRatio;

        dRatio = (double) iLogHeight*(depthStart-depth2)/(depthStart-depthEnd);
        iDepth2 = PLOT_TITLES + (int) dRatio;

        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth2 = iHeight - (int) dRatio;

        if (iDepth1 == iDepth2)
        {
          iDepth1 = iDepth1 - 5;
          iDepth2 = iDepth2 - 5;
        }

        iDepth = Math.abs(iDepth2 - iDepth1);

        if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
            (iDepth1 < iHeight) && (iDepth2 <= iHeight))
        {
          iX = iStartTrack;

          for (int k=0; k<stStruct.stItem[i].iTotal; k++)
          {
            iFound = -1;
            for (int j=0; j<stSymbols.iCount; j++)
            {
              if (stStruct.stItem[i].sID[k].equals(stSymbols.stItem[j].sID))
              {
                if (stSymbols.stItem[j].iGroup ==
                      mask.maskSymbolsUtility.getGroupID(
                      mask.maskGroupsStruct._MNEMONIC, "B", stSymbols.stGroups))
                {
                  iFound = j;
                }
              }
            }

            if (iFound > -1)
            {
              for (int p=0; p<iWidthTrack/20; p++)
              {
                plot.drawSymbol( g, stSymbols.stItem[iFound].iRows, 0,
                                 stSymbols.stItem[iFound].symbol, iX, iDepth1 );

                iX = iX + 20;
              }
            }
          }
        }
      }
    }
  }

  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a Plot Track
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Track to draw
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

  public void draw( Graphics g,
                    int iSelected,
                    int iStartTrack,
                    int iWidthTrack )
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;

    if (iSelected == iqstrat.iqstratTracksStruct._SED_STRUCTURE)
    {
      g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2] ) );
      g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

      drawGrid( g, iStartTrack, iWidthTrack );
      drawStructures( g, iStartTrack, iWidthTrack );
      drawBedStructures( g, iStartTrack, iWidthTrack );
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}