/*
 * @sedimentaryPlotSymbol.java Version 1.1 09/18/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.sedimentary.plot;

import java.awt.*;
import java.awt.geom.*;
import java.util.Observable;
import java.awt.event.*;

import mask.maskSymbolsListStruct;
import mask.plot.maskPlotSymbol;
import rock.sedimentary.sedimentaryStruct;

/** Class sedimentaryPlotSymbol
 *  <p> This method will create the different symbol types for plotting data
 *      values on a plot.
 *
 *  @version 1.1 09/18/2008
 *  @author  John R. Victorine
 */

public class sedimentaryPlotSymbol extends Canvas implements MouseListener
{
  private Observable            notifier  = null; // Observable Object
  private maskSymbolsListStruct stSymbols = null; // Sed Struct Symbols List

  private static final int NO  = 0;
  private static final int YES = 1;

  private maskPlotSymbol        plot        = null; // Plot Mask Symbol
  private sedimentaryStruct     st          = null; // Sedimentary Data Struct
  private int                   iSelected[] = null; // Selected Structures
  private int                   iRow[]      = null; // Location on Plot

  private int iWidth      = 300;
  private int iHeight     = 1000;//950;

  private static final int TEXT       = 18;
  private static final int ROW        = 25;

  private static final int X_START    = 10;
  private static final int X_IMAGE    = 40;
  private static final int X_DESC     = 75;
  private static final int X_ABBREV   = 350;

  /** CONSTRUCTOR sedimentaryPlotSymbol()
   *  <p> This is the constructor for this class.
   *  @param notifier  = Observable
   *  @param stSymbols = Sedimentary Structure Plot Symbols List Data Structure
   */

  public sedimentaryPlotSymbol( Observable notifier,
                                maskSymbolsListStruct stSymbols)
  {
    this.notifier  = notifier;
    this.stSymbols = stSymbols;

    plot = new maskPlotSymbol( stSymbols );

    this.setBackground(Color.white);
    addMouseListener(this);

    iSelected = new int[stSymbols.iCount];
    iRow      = new int[stSymbols.iCount];

    iHeight = (stSymbols.iCount + 12) * ROW;

    for (int i=0; i<stSymbols.iCount; i++)
    {
      iSelected[i] = NO;
    }
  }

  /** Method close()
   * <p> This method will force Java to deallocate memory
   */

  public void close()
  {
    notifier  = null;
    stSymbols = null;

    if (plot != null)
      plot.close();
    plot = null;

    st        = null;
    iSelected = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Width of the plot panel.
   * @return WIDTH = the width of the plot panel
   */

  public int getPlotWidth() { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Width of the plot panel.
   * @return the height of the plot panel
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getData()
   * <p> This method will get the array of sedimentary structures selected
   * @return st = sedimentary data structure
   */

  public sedimentaryStruct getData() { return (st); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setData()
   * <p> This method will set the array of sedimentary structures selected
   * @param st = sedimentary data Structure
   */

  public void setData(sedimentaryStruct st)
  {
    this.st = st;

    for (int i=0; i<stSymbols.iCount; i++)
    {
      iSelected[i] = NO;
      if (isSelected(i))
        iSelected[i] = YES;
    }

    repaint();
  }

  /* =============================================================== *
   * ------------------------ MISC METHODS ------------------------- *
   * =============================================================== */

  /** Method isSelected()
   * <p> This method will test the ID's to see if they have been selected
   *     and return true if they have and false if not.
   * @param  iValue = the id to check
   * @return bValue = true/false if the Sedimentary Structure ID's selected or not
   */

  private boolean isSelected(int iValue)
  {
    boolean bValue = false;

    if (iValue > -1)
    {
      if (st != null)
      {
        if (st.iTotal > 0)
        {
          for (int i=0; i<st.iTotal; i++)
          {
            if (st.sID[i].equals(stSymbols.stItem[iValue].sID))
              bValue = true;
//                rock.sedimentary.sedimentarySymbolsStruct.SYMBOLS[iValue][0]))
          }
        }
      }
    }

    return (bValue);
  }

  /** Method toggleStructure()
   * <p> This method will toggle a sedimentary structure from one state to the other.
   * @param iSedStruct = the sedimentary structure number to toggle.
   */

  private void toggleStructure(int iSedStruct)
  {
    if (iSelected[iSedStruct] == NO)
      iSelected[iSedStruct] = YES;
    else
      iSelected[iSedStruct] = NO;
  }

  /** Method modifyStructureList()
   * <p> This method will check add or remove sedimentary structures from the
   *     sedimentary structure list depending if it is selected
   */

  private void modifyStructureList()
  {
    int i      = 0;
    int iCount = 0;

    if (st == null)
    {
      st = new sedimentaryStruct();
      st.sKEY = new String(cmn.cmnString.UniqueName());
    }

    for (i=0; i<stSymbols.iCount; i++)
    {
      if (iSelected[i] == YES)
        iCount++;
    }

    if (iCount > 0)
    {
      st.iTotal = iCount;
      st.sID    = new String[iCount];
      st.sText  = new String[iCount];

      iCount = 0;
      for (i=0; i<stSymbols.iCount; i++)
      {
        if (iSelected[i] == YES)
        {
          if (iCount < st.iTotal)
          {
            st.sID[iCount] = new String( stSymbols.stItem[i].sID );
//                rock.sedimentary.sedimentarySymbolsStruct.SYMBOLS[i][0]);
            st.sText[iCount] =new String( stSymbols.stItem[i].sName );
//                rock.sedimentary.sedimentarySymbolsStruct.SYMBOLS[i][3]);
            iCount++;
          }
        }
      }
    }
    else
    {
      st.iTotal = 0;
      st.sID    = null;
      st.sText  = null;
    }
  }

  /* =============================================================== *
   * -------------------- MouseListener Methods -------------------- *
   * =============================================================== */

  public void mouseMoved(MouseEvent event) { }
  public void mouseEntered(MouseEvent event) { }
  public void mouseExited(MouseEvent event) { }
  public void mousePressed(MouseEvent event) { }
  public void mouseDragged(MouseEvent event) {}
  public void mouseReleased(MouseEvent event) {}

  /** Method mouseClicked()
   *  <p> Mouse Clicked Method retrieve the location of the mouse pointer
   *      location.
   * @param event - Mouse Clicked Event
   */

  public void mouseClicked(MouseEvent event)
  {
    int iXm = event.getX();
    int iYm = event.getY();

    for (int i=0; i<stSymbols.iCount; i++)
    {
      if ((iYm >= iRow[i]) && (iYm < iRow[i]+ROW))
      {
        toggleStructure(i);
        modifyStructureList();
        repaint();

        if (notifier != null)
          notifier.notifyObservers(new String("Sed Structure Symbol Selected"));
      }
    }
  }

  /* ======================================================================== *
   * ------------------------------- PAINT ---------------------------------- *
   * ======================================================================== */

  /** Method drawStructureGroup()
   * <p> This method will draw the Sedimentary Structure Group
   *  @param  g      - Graphics Handle;
   *  @param  iStart - Starting height of the sedimentary structure group
   *  @param  iGroup - Group ID
   *  @param  iFrom  - Begining sedimentary structure ID
   *  @param  iTo    - Ending sedimentary structure ID
   *  @return iEnd   - Ending height of the sedimentary structure group
   *

  public int drawStructureGroup(Graphics g, int iStart,
                                int iGroup, int iFrom, int iTo)
  {
    int         iEnd  = iStart;
    int         i     = 0;
    int         j     = 0;
    int         k     = 0;
    String      sText = "";
    int         iLen  = 0;
    Font        fsb   = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm  = g.getFontMetrics(fsb);

    g.setFont( fsb );

    sText = new String(rock.sedimentary.sedimentarySymbolsStruct.GROUP[iGroup]);
    iLen  = sText.length();

    g.setColor(Color.black);
    g.drawString(sText, (iWidth - 4*iLen)/2, iStart);

    iEnd = iEnd + TEXT;

    for (i=iFrom; i<iTo; i++)
    {
      iRow[i] = iEnd;

      if (notifier != null)
      {
        if (isSelected(i))
        {
          g.setColor(Color.red);
          g.drawLine(X_START,      iEnd, X_START + 10, iEnd + 10);
          g.drawLine(X_START + 10, iEnd, X_START,      iEnd + 10);
        }

        g.setColor(Color.black);
        g.drawRect( X_START, iEnd, 10, 10);
      }

      plot.plotMaskSymbol.drawSymbol( g, 10,
           rock.sedimentary.plot.sedimentaryPlotSymbolStruct.getSymbol(i),
           X_IMAGE, iEnd );

      g.setColor(Color.black);
      g.drawString(rock.sedimentary.sedimentarySymbolsStruct.SYMBOLS[i][3],
                   X_DESC,   iEnd+10);
      g.drawString(rock.sedimentary.sedimentarySymbolsStruct.SYMBOLS[i][4],
                   X_ABBREV, iEnd+10);

      iEnd = iEnd + ROW;
    }

    return (iEnd);
  }
*/

  /** Method drawStructureGroup()
   * <p> This method will draw the Sedimentary Structure Group
   *  @param  g      = Graphics Handle;
   *  @param  iStart = Starting height of the sedimentary structure group
   *  @param  iGroup = Group ID
   *  @return iEnd   = Ending height of the sedimentary structure group
   */

  public int drawStructureGroup( Graphics g, int iStart, int iGroup )
  {
    int         iEnd   = iStart;
    int         j      = 0;
    String      sText  = "";
    int         iLen   = 0;
    int         iCont  = 0;
    Font        fsb    = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm   = g.getFontMetrics(fsb);

    g.setFont( fsb );

    for (j=0; j<stSymbols.iCount; j++)
    {
      if (stSymbols.stItem[j].iGroup == iGroup)
      {
        if (iCont == 0)
        {
          sText = new String( stSymbols.stGroups.sGroups[iGroup][1] );
          iLen  = sText.length();

          g.setColor(Color.black);
          g.drawString(sText, iWidth/2 - 4*iLen, iStart);

          iEnd = iEnd + TEXT;

          iCont = 1;
        }

        iRow[j] = iEnd;

        if (notifier != null)
        {
          if (isSelected(j))
          {
            g.setColor(Color.red);
            g.drawLine(X_START,      iEnd, X_START + 10, iEnd + 10);
            g.drawLine(X_START + 10, iEnd, X_START,      iEnd + 10);
          }

          g.setColor(Color.black);
          g.drawRect( X_START, iEnd, 10, 10);
        }

        plot.drawSymbol( g, stSymbols.stItem[j].iRows, 0,
                         stSymbols.stItem[j].symbol, X_IMAGE, iEnd );

        g.setColor(Color.black);
        g.drawString(stSymbols.stItem[j].sName,  X_DESC,   iEnd + 10);
        g.drawString(stSymbols.stItem[j].sAbrev, X_ABBREV, iEnd + 10);

        iEnd = iEnd + ROW;
      }
    }

    return (iEnd);
  }

  /** Method draw()
   * <p> This method will draw all the sedimentary structure symbols.
   *  @param g = Graphics Handle;
   */

  public void draw(Graphics g)
  {
    int    i         = 0;
    int    j         = 0;
    int    iStart    = 10;
    String sText     = "Sedimentary Structure Symbols";
    int    iLen      = 0;
    int    iGroup    = 0;

    g.setFont( new Font("Serif", Font.BOLD, 14) );
    g.setColor(Color.black);

    g.drawString(sText, 0, 14);

    iStart = iStart + TEXT;

    if (stSymbols != null)
    {
      if (stSymbols.stGroups != null)
      {
        iStart = iStart + 10;

        g.setFont( new Font("Serif", Font.BOLD, 12) );

        sText = new String( stSymbols.stGroups.sGroups[0][1] );
        iLen  = sText.length();

        g.setColor(Color.black);
        g.drawString(sText, iWidth/2 - 4*iLen, iStart);

        iStart = iStart + TEXT;

        for (i=1; i<stSymbols.stGroups.iGroups; i++)
        {
          // Draw Sedimentary Structure

          iStart = drawStructureGroup(g, iStart+10, i);
        }
      }
    }
  }

  /** Method paint(Graphics g)
   * <p> This method will paint Porosity & Permeability Plot
   * @param g - Graphics Parameter.
   */

  public void paint(Graphics g)
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);

    draw(g);
  }
}

/*
 *  @version 1.1 09/18/2008
 *  @author  John Victorine
 */
