/*
 * @sedimentaryPlotLegend.java Version 1.1 03/10/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.sedimentary.plot;

import java.awt.*;
import java.awt.geom.*;
import java.util.Observable;
import java.awt.event.*;

import mask.maskSymbolsListStruct;
import mask.plot.maskPlotSymbol;

/** Class sedimentaryPlotLegend
 *  <p> This method will display the sedimentary structure plot legend.
 *
 *  @version 1.1 03/10/2009
 *  @author  John R. Victorine
 */

public class sedimentaryPlotLegend extends Canvas
{
  private static final int NO  = 0;
  private static final int YES = 1;

  private int                   iSelected[] = null; // Selected Sed Structures
  private maskSymbolsListStruct stSymbols   = null; // Sed Struct Symbols List
  private maskPlotSymbol        plot        = null; // Plot Mask Symbol

//  private int iSedStruct  = -1;

  private int iWidth      = 300;
  private int iHeight     = 1000;//950;

  private static final int TEXT       = 15; //18;
  private static final int ROW        = 18; //25;

  private static final int X_START    = 10;
  private static final int X_IMAGE    = 10;  //40;
  private static final int X_DESC     = 45;  //75;
  private static final int X_ABBREV   = 320; //350;

  /** CONSTRUCTOR sedimentaryPlotLegend( int iSelected[] )
   *  <p> This is the constructor for this class.
   *  @param iSelected = the sedimentary structures selected
   *  @param stSymbols = Sedimentary Structure Plot Symbols List Data Structure
   */

  public sedimentaryPlotLegend( int iSelected[],
                                maskSymbolsListStruct stSymbols )
  {
    this.iSelected = iSelected;
    this.stSymbols = stSymbols;

    plot = new maskPlotSymbol( stSymbols );

    computeHeight();
  }

  /** Method close()
   * <p> This method will force Java to deallocate memory
   */

  public void close()
  {
    iSelected = null;
    stSymbols = null;

    if (plot != null)
      plot.close();
    plot = null;
  }

  /** Method countMembers( int iStart, int iGroup,  )
   * <p> This method will count the number of members in the group
   *  @param  iStart = Starting height of the lithology group
   *  @param  iGroup = Group ID
   *  @return iEnd   = Ending heith of the lithology group
   */

  public int countMembers( int iStart, int iGroup )
  {
    int iEnd  = iStart;
    int iCont = 0;

    for (int j=0; j<stSymbols.iCount; j++)
    {
      if (stSymbols.stItem[j].iGroup == iGroup)
      {
        if (iSelected != null)
        {
          if (iSelected[j] > -1)
          {
            if (iCont == 0)
            {
              iEnd = iEnd + TEXT;

              iCont = 1;
            }

            iEnd = iEnd + ROW;
          }
        }
      }
    }

    return (iEnd);
  }

  /** Method computeHeight()
   * <p> This method will compute the height of the plot
   */

  private void computeHeight()
  {
    int    i         = 0;
    int    j         = 0;
    int    iStart    = 10;
    int    iContinue = 0;
    int    iGroup    = 0;

    iStart = iStart + TEXT;

    if (stSymbols != null)
    {
      if (stSymbols.stGroups != null)
      {
        for (i=0; i<stSymbols.stGroups.iGroups; i++)
        {
          iContinue = -1;
          for (j=0; j<stSymbols.iCount; j++)
          {
            if (stSymbols.stItem[j].iGroup == i)
            {
              if (iSelected != null)
              {
                if (iSelected[j] > -1)
                  iContinue = j;
              }
            }
          }

          if (iContinue > -1)
          {
            // Depositional Structures Super Group

            if (stSymbols.stGroups.sGroups[
                  stSymbols.stItem[iContinue].iGroup][0].length() < 3)
            {
              // IS this the first occurrence of a Depositional Structure

              if (iGroup == 0)
              {
                iStart = iStart + 10;
                iStart = iStart + TEXT;
                iGroup = 1;
              }
            }

            // Draw Sedimentary Structure

            iStart = countMembers(iStart+10, i);
          }
        }
      }
    }

    iHeight = iStart+50;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Width of the plot panel.
   * @return WIDTH = the width of the plot panel
   */

  public int getPlotWidth() { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Width of the plot panel.
   * @return the height of the plot panel
   */

  public int getPlotHeight() { return (iHeight); }

  /* ======================================================================== *
   * ------------------------------- PAINT ---------------------------------- *
   * ======================================================================== */

  /** Method drawStructureGroup()
   * <p> This method will draw the Sedimentary Structure Group
   *  @param  g      = Graphics Handle;
   *  @param  iStart = Starting height of the sedimentary structure group
   *  @param  iGroup = Group ID
   *  @return iEnd   = Ending height of the sedimentary structure group
   */

  public int drawStructureGroup( Graphics g, int iStart, int iGroup )
  {
    int         iEnd   = iStart;
    int         i      = 0;
    int         j      = 0;
    int         k      = 0;
    String      sText  = "";
    int         iLen   = 0;
    int         iCont  = 0;
    Font        fsb    = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm   = g.getFontMetrics(fsb);

    g.setFont( fsb );

    for (j=0; j<stSymbols.iCount; j++)
    {
      if (stSymbols.stItem[j].iGroup == iGroup)
      {
        if (iSelected[j] > -1)
        {
          if (iCont == 0)
          {
            sText = new String( stSymbols.stGroups.sGroups[iGroup][1] );
            iLen  = sText.length();

            g.setColor(Color.black);
            g.drawString(sText, iWidth/2 - 4*iLen, iStart);

            iEnd = iEnd + TEXT;

            iCont = 1;
          }

          plot.drawSymbol( g, stSymbols.stItem[j].iRows, 0,
                           stSymbols.stItem[j].symbol, X_IMAGE, iEnd );

          g.setColor(Color.black);
          g.drawString(stSymbols.stItem[j].sName,  X_DESC,   iEnd + 10);
          g.drawString(stSymbols.stItem[j].sAbrev, X_ABBREV, iEnd + 10);

          iEnd = iEnd + ROW;
        }
      }
    }

    return (iEnd);
  }

  /** Method draw()
   * <p> This method will draw all the sedimentary structure symbols.
   *  @param g = Graphics Handle;
   */

  public void draw(Graphics g)
  {
    int    i         = 0;
    int    j         = 0;
    int    iStart    = 10;
    String sText     = "Sedimentary Structure Symbols";
    int    iLen      = 0;
    int    iContinue = 0;
    int    iGroup    = 0;

    g.setFont( new Font("Serif", Font.BOLD, 14) );
    g.setColor(Color.black);

    g.drawString(sText, 0, 14);

    iStart = iStart + TEXT;

    if (stSymbols != null)
    {
      if (stSymbols.stGroups != null)
      {
        for (i=0; i<stSymbols.stGroups.iGroups; i++)
        {
          iContinue = -1;
          for (j=0; j<stSymbols.iCount; j++)
          {
            if (stSymbols.stItem[j].iGroup == i)
            {
              if (iSelected != null)
              {
                if (iSelected[j] > -1)
                  iContinue = j;
              }
            }
          }

          if (iContinue > -1)
          {
            // Depositional Structures Super Group

            if (stSymbols.stGroups.sGroups[
                  stSymbols.stItem[iContinue].iGroup][0].length() < 3)
            {
              // IS this the first occurrence of a Depositional Structure

              if (iGroup == 0)
              {
                iStart = iStart + 10;

                g.setFont( new Font("Serif", Font.BOLD, 12) );

                sText = new String( stSymbols.stGroups.sGroups[0][1] );
                iLen  = sText.length();

                g.setColor(Color.black);
                g.drawString(sText, iWidth/2 - 4*iLen, iStart);

                iStart = iStart + TEXT;
                iGroup = 1;
              }
            }

            // Draw Sedimentary Structure

            iStart = drawStructureGroup(g, iStart + 10, i);
          }
        }
      }
    }
  }

  /** Method paint(Graphics g)
   * <p> This method will paint Sedimentary Structre Legend Plot
   * @param g = Graphics Parameter.
   */

  public void paint(Graphics g)
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);

    draw(g);
  }
}

/*
 *  @version 1.1 03/10/2009
 *  @author  John Victorine
 */
