/*
 * @sedimentaryPlotSymbolFrame.java Version 1.1 09/18/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.sedimentary.gui;

import java.util.Observable;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import mask.maskSymbolsListStruct;

import rock.sedimentary.gui.sedimentaryPlotSymbolPanel;
import rock.sedimentary.sedimentaryStruct;

import util.utilImageIO;

/** Class sedimentaryPlotSymbolFrame
 *  <p> This Class will display the sedimentary images available
 *
 *  @version 1.1 09/18/2008
 *  @author  John R. Victorine
 */

public class sedimentaryPlotSymbolFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable                 notifier  = null; // Observable Object
  private maskSymbolsListStruct      stSymbols = null; // Sed Struct Symbols List
  private sedimentaryPlotSymbolPanel plot      = null; // Symbol Plot Panel

  private sedimentaryStruct          stStruct  = null; // sedimentary data structure
  private String                     sKEY      = "0";  // Unique key for data

  // Global Variables

  // -- Depth range variables

  private double dStart = 0.0; // Starting Depth
  private double dEnd   = 0.0; // Ending Depth

  // Global Frame Widgets

  // -- Depth Range Textfields

  private JTextField txtStart  = new JTextField(); // Starting Depth
  private JTextField txtEnd    = new JTextField(); // Ending Depth

  // -- Global Frame Buttons

  private JButton    btnAdd    = new JButton();    // Add Data Button
  private JButton    btnClear  = new JButton();    // Clear Data Fields
  private JButton    btnCancel = new JButton();    // Cancel Dialog

  private JMenuItem  mExit     = null;             // Exit Profile Plot

  // -- Dialog Position & Size

  private int iXStart = 0;
  private int iYStart = 0;
  private int iWidth  = 0;
  private int iHeight = 0;

  /** Constructor sedimentaryPlotSymbolFrame()
   *  <p> This is the Constructor for this class.
   *  @param stSymbols = Sedimentary Structure Plot Symbols List Data Structure
   */

  public sedimentaryPlotSymbolFrame( maskSymbolsListStruct stSymbols )
  {
    try
    {
      this.stSymbols = stSymbols;

      jbInit();
      addWindowListener( new sedimentaryPlotSymbolFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Constructor sedimentaryPlotSymbolFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param stSymbols = Sedimentary Structure Plot Symbols List Data Structure
   */

  public sedimentaryPlotSymbolFrame( Observable notifier,
                                     maskSymbolsListStruct stSymbols )
  {
    try
    {
      this.notifier  = notifier;
      this.stSymbols = stSymbols;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Constructor sedimentaryPlotSymbolFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param iXStart = the x-coordinate of Data entry dialog
   *  @param iYStart = the y-coordinate of Data entry dialog
   *  @param iWidth  = the width of Data entry dialog
   *  @param iHeight = the height of Data entry dialog
   *  @param stSymbols = Sedimentary Structure Plot Symbols List Data Structure
   */

  public sedimentaryPlotSymbolFrame( Observable notifier,
                                     int iXStart, int iYStart,
                                     int iWidth, int iHeight,
                                     maskSymbolsListStruct stSymbols )
  {
    try
    {
      this.notifier  = notifier;
      this.iXStart   = iXStart;
      this.iYStart   = iYStart;
      this.iWidth    = iWidth;
      this.iHeight   = iHeight;
      this.stSymbols = stSymbols;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
	JPanel pnlTop        = new JPanel();
    JPanel pnlDepthRange = new JPanel();
    JPanel pnlStart      = new JPanel();
    JPanel pnlEnd        = new JPanel();

    JPanel pnlAdd        = new JPanel();
    JPanel pnlClear      = new JPanel();

    TitledBorder titledBorderStart = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Starting Depth:");
    titledBorderStart.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderEnd = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Ending Depth:");
    titledBorderEnd.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    JMenuBar    menuBar  = new JMenuBar();
    JMenu       menuFile = new JMenu("Close Dialog");

    // Build Menu

    mExit = new JMenuItem("Close");
    mExit.addActionListener(this);

    menuFile.add( mExit );

    menuBar.add(menuFile);

    this.setJMenuBar(menuBar);

    // Build Dialog

    this.setTitle("Sedimentary Structures");
    this.getContentPane().setLayout(new BorderLayout());

    // Top Panel

    pnlTop.setLayout(new GridLayout());

    // -- Add Button Panel

    pnlAdd.setLayout(new GridLayout(2,1));

    btnAdd.setFont(new java.awt.Font("Dialog", 1, 11));
    btnAdd.setText("Add");
    btnAdd.addActionListener(this);

    pnlClear.setLayout(new GridLayout());

    btnClear.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClear.setText("Clear");
    btnClear.addActionListener(this);

    btnCancel.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnCancel.setEnabled(false);
    btnCancel.setText("Cancel");
    btnCancel.addActionListener(this);

    // -- Depth Range Panel

    pnlDepthRange.setPreferredSize(new Dimension(100, 10));
    pnlDepthRange.setLayout(new GridLayout());

    // .. Starting Depth Panel

    pnlStart.setBorder(titledBorderStart);
    pnlStart.setLayout(new BorderLayout());

    txtStart.setText("0.0");
    txtStart.setEditable(false);
    txtStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtStart.addFocusListener(new sedimentaryPlotSymbolFrameFocusAdapter(this));

    // .. Endind Depth Panel

    pnlEnd.setBorder(titledBorderEnd);
    pnlEnd.setLayout(new BorderLayout());

    txtEnd.setText("0.0");
    txtEnd.setEditable(false);
    txtEnd.setHorizontalAlignment(SwingConstants.TRAILING);
    txtEnd.addFocusListener(new sedimentaryPlotSymbolFrameFocusAdapter(this));

    // Create Sedimentary Structure Plot Panel

    plot = new sedimentaryPlotSymbolPanel( this, notifier, stSymbols );

    // Attach Widgets to Frame

    this.getContentPane().add(pnlTop,  BorderLayout.NORTH);

    pnlTop.add( pnlDepthRange,         BorderLayout.NORTH );
    pnlDepthRange.add( pnlStart,       null );
    pnlStart.add( txtStart,            BorderLayout.CENTER );

    pnlDepthRange.add( pnlEnd,         null );
    pnlEnd.add( txtEnd,                BorderLayout.CENTER );

    pnlTop.add(pnlAdd,                 BorderLayout.SOUTH);
    pnlAdd.add(btnAdd,                 null);

    pnlAdd.add(pnlClear,               null);
//    pnlClear.add(btnCancel,            null);
    pnlClear.add(btnClear,             null);

    this.getContentPane().add(plot,    BorderLayout.CENTER);

    // Display the Frame

    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setSize(new Dimension(450, d.height-50));

    int iXPos = (d.width  - this.getSize().width) / 2;
    if (iWidth > 0) iXPos = iXStart + iWidth;
    this.setLocation(iXPos, (d.height - this.getSize().height) / 2);

//    this.setResizable(false);
    this.setVisible(true);
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getData()
   * <p> This method will get the array of sedimentary structures selected
   * @return st = sedimentary data Structure
   */

  public sedimentaryStruct getData()
  {
    if (stStruct == null)
    {
      stStruct = new sedimentaryStruct();
      stStruct.sKEY = new String(cmn.cmnString.UniqueName());
    }

    if (dEnd == 0.0) dEnd = dStart;

    if (plot != null) stStruct = plot.getData();

    stStruct.sKEY       = new String( sKEY );
    stStruct.depthStart = dStart;
    stStruct.depthEnd   = dEnd;

    return (stStruct);
  }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setData()
   * <p> This method will set the array of sedimentary structures selected
   * @param st = sedimentary data Structure
   */

  public void setData(sedimentaryStruct st)
  {
    this.stStruct = st;

    if (st != null)
    {
	  this.sKEY   = new String( st.sKEY );
      this.dStart = st.depthStart;
      this.dEnd   = st.depthEnd;

      txtStart.setText( "" + this.dStart );
      txtEnd.setText( "" + this.dEnd );
    }

    if (plot != null) plot.setData(st);

    btnAdd.setText("Modify");
  }

  /** Method setDepthRange(double depthStart, double depthEnd)
   * <p> This method will set the depth range of the data
   * @param depthStart = the starting depth
   * @param depthEnd   = the ending depth
   */

  public void setDepthRange(double depthStart, double depthEnd)
  {
    this.dStart = depthStart;
    txtStart.setText( "" + this.dStart );

    this.dEnd = depthEnd;
    txtEnd.setText( "" + this.dEnd );
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method add()
   * <p> This method will build the sedimentary data structure and signal to add or
   *     modify depending on the settings.
   */

  private void add()
  {
    notifier.notifyObservers(new String("ADD or MODIFY Sed Struct Data"));
  }

  /** Method cancel()
   * <p> This method will cancel the edit
   */

  public void cancel()
  {
	if (notifier != null)
      notifier.notifyObservers(new String("Cancel Edit"));
  }

  /** Method clear()
   * <p> This method will clear the textfields
   */

  public void clear()
  {
    if (stStruct != null)
      stStruct.delete();
    stStruct = null;

    if (plot       != null) plot.setData(stStruct);

//    btnAdd.setText("Add");
  }

  /** METHOD sedimentaryPlotSymbolFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class sedimentaryPlotSymbolFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
  	  if (notifier != null)
        notifier.notifyObservers(new String("Cancel Edit"));
    }
  }

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    notifier  = null; // Observable Object
    stSymbols = null; // Sedimentary Structures Plot Symbol Data List Structure

    if (plot != null)
      plot.close();
    plot = null;

    // -- Depth Range Textfields

    txtStart  = null; // Starting Depth
    txtEnd    = null; // Ending Depth

    // -- Global Frame Buttons

    btnAdd    = null; // Add Data Button
    btnClear  = null; // Clear Data Fields
    btnCancel = null; // Cancel Dialog

    mExit     = null; // Exit Profile Plot

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if ( event.getSource() == btnAdd  )   { add(); }
    if ( event.getSource() == btnClear )  { clear(); }
    if ( event.getSource() == btnCancel ) { cancel(); }

    if ( event.getSource() == mExit )     { close(); }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtStart)
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Start Depth Value is a Numeric Field");
    }

    if (e.getSource() == txtEnd)
    {
      iNumeric = 1;
      sTemp = txtEnd.getText();
      sMessage = new String("End Depth Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
        if (e.getSource() == txtStart)
          dStart = cmn.cmnString.stringToDouble(txtStart.getText());

        if (e.getSource() == txtEnd)
          dEnd = cmn.cmnString.stringToDouble(txtEnd.getText());

//        setDepthRange( dStart, dEnd );
      }
    }
  }
}

/** CLASS sedimentaryPlotSymbolFrameFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class sedimentaryPlotSymbolFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  sedimentaryPlotSymbolFrame adaptee;

  sedimentaryPlotSymbolFrameFocusAdapter(sedimentaryPlotSymbolFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 09/18/2008
 *  @author  John Victorine
 */
