/*
 * @rockPlotLimitsUtility.java Version 1.1 04/15/2016
 *
 * Copyright (c) 2016 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock;

import rock.rockDataListStruct;

/** Class rockPlotLimitsUtility
 *  <p> This Class will provide basic utilities to build the
 *      Core Plot Limits Data Array. This array allows the
 *      users to override the Plot Limits on the LAS Plot Track
 *
 *  @version 1.1 04/15/2016
 *  @author  John R. Victorine
 */

public class rockPlotLimitsUtility
{
  // Rock & Core Porosity
  public static final int _PCORE    = rock.rockStandardTools._PCORE;
  public static final int _PPLUG    = rock.rockStandardTools._PPLUG;
  public static final int _P800     = rock.rockStandardTools._P800;
  public static final int _PINSI    = rock.rockStandardTools._PINSI;
  public static final int _PEFF     = rock.rockStandardTools._PEFF;
  // Rock Permeability
  public static final int _KMAX     = rock.rockStandardTools._KMAX;
  public static final int _K90      = rock.rockStandardTools._K90;
  public static final int _KVRT     = rock.rockStandardTools._KVRT;
  public static final int _KPLG     = rock.rockStandardTools._KPLG;
  public static final int _KKL      = rock.rockStandardTools._KKL;
  public static final int _KINSI    = rock.rockStandardTools._KINSI;
  public static final int _KKLIN    = rock.rockStandardTools._KKLIN;
  public static final int _KPVRT    = rock.rockStandardTools._KPVRT;
  // Saturation
  public static final int _SOIL     = rock.rockStandardTools._SOIL;
  public static final int _SW       = rock.rockStandardTools._SW;
  // Density
  public static final int _GMCC     = rock.rockStandardTools._GMCC;
  public static final int _RHOD     = rock.rockStandardTools._RHOD;
  public static final int _RHOW     = rock.rockStandardTools._RHOW;
  // Archie Constants
  public static final int _MAMB     = rock.rockStandardTools._MAMB;
  public static final int _MINSI    = rock.rockStandardTools._MINSI;
  public static final int _NAMB     = rock.rockStandardTools._NAMB;
  public static final int _NINSI    = rock.rockStandardTools._NINSI;
  // Gamma Ray
  public static final int _GR       = rock.rockStandardTools._GR;
  public static final int _CGR      = rock.rockStandardTools._CGR;
  public static final int _PGR      = rock.rockStandardTools._PGR;
  public static final int _TH       = rock.rockStandardTools._TH;
  public static final int _U        = rock.rockStandardTools._U;
  public static final int _K        = rock.rockStandardTools._K;

  public static final int _LIN_1    = rock.rockStandardTools._LIN_1;
  public static final int _LIN_2    = rock.rockStandardTools._LIN_2;
  public static final int _LIN_3    = rock.rockStandardTools._LIN_3;
  public static final int _LIN_4    = rock.rockStandardTools._LIN_4;

  public static final int _LOG_1    = rock.rockStandardTools._LOG_1;
  public static final int _LOG_2    = rock.rockStandardTools._LOG_2;
  public static final int _LOG_3    = rock.rockStandardTools._LOG_3;
  public static final int _LOG_4    = rock.rockStandardTools._LOG_4;

  public static final int ROWS      = rock.rockStandardTools.TOTAL_DATA;

  public static final int MINIMUM   = 0;
  public static final int MAXIMUM   = 1;
  public static final int GROUP     = 2;
  public static final int LAS       = 3;
  public static final int COLUMNS   = 4;

  /** Method initialize()
   *  <p> This method will initialize the LAS Plot Limits data Array
   *      There are 3 columns Minimum, Maximum, Group, Log Curve Present.
   *      The Group is used to force other curves that reside on the same
   *      plot track, i.e. resistivity, permeability, porosity and
   *      the Log Curve Present is an indicator to include this curve.
   *
   * @param   st   = LAS File Data Structure
   * @return  dOUT = Array of data limits
   */

  public static double[][] initialize(rockDataListStruct st)
  {
    double dOUT[][] = null;
    double dMinimum = 0.0;
    double dMaximum = 0.0;

    dOUT = new double[ROWS][COLUMNS];

    for (int i=0; i<_PCORE; i++)
    {
      dOUT[i][MINIMUM] = cmn.cmnString.stringToDouble(
                           rock.rockStandardTools.ROCK_TOOLS[i][4]);
      dOUT[i][MAXIMUM] = cmn.cmnString.stringToDouble(
                           rock.rockStandardTools.ROCK_TOOLS[i][5]);
      dOUT[i][GROUP]   = -1.0;
      dOUT[i][LAS]     = -1.0;
	}

    for (int i=_PCORE; i<ROWS; i++)
    {
      dOUT[i][MINIMUM] = cmn.cmnString.stringToDouble(
                           rock.rockStandardTools.ROCK_TOOLS[i][4]);
      dOUT[i][MAXIMUM] = cmn.cmnString.stringToDouble(
                           rock.rockStandardTools.ROCK_TOOLS[i][5]);
      dOUT[i][GROUP]   = -1.0;
      dOUT[i][LAS]     = -1.0;

      // Set Curves with Data that will allow change of Limits
      // Add Minimum & Maximums that deviate from standard values

  	  if (st != null)
  	  {
        if (st.checkData(i))
        {
	      dOUT[i][LAS] = 1.0;

          switch (i)
          {
            case _PCORE:
            case _PPLUG:
            case _P800:
            case _PINSI:
            case _PEFF:
              if (st.isPercent(i))
              {
                dOUT[i][MINIMUM] = -10.0;
                dOUT[i][MAXIMUM] = 30.0;
			  }
              break;

            case _LIN_1:
              dOUT[i][MINIMUM] = st.dLIN_1Min;
              dOUT[i][MAXIMUM] = st.dLIN_1Max;
              break;

            case _LIN_2:
              dOUT[i][MINIMUM] = st.dLIN_2Min;
              dOUT[i][MAXIMUM] = st.dLIN_2Max;
              break;

            case _LIN_3:
              dOUT[i][MINIMUM] = st.dLIN_3Min;
              dOUT[i][MAXIMUM] = st.dLIN_3Max;
              break;

            case _LIN_4:
              dOUT[i][MINIMUM] = st.dLIN_4Min;
              dOUT[i][MAXIMUM] = st.dLIN_4Max;
              break;

            case _LOG_1:
              dOUT[i][MINIMUM] = st.dLOG_1Min;
              dOUT[i][MAXIMUM] = st.dLOG_1Max;
              break;

            case _LOG_2:
              dOUT[i][MINIMUM] = st.dLOG_2Min;
              dOUT[i][MAXIMUM] = st.dLOG_2Max;
              break;

            case _LOG_3:
              dOUT[i][MINIMUM] = st.dLOG_3Min;
              dOUT[i][MAXIMUM] = st.dLOG_3Max;
              break;

            case _LOG_4:
              dOUT[i][MINIMUM] = st.dLOG_4Min;
              dOUT[i][MAXIMUM] = st.dLOG_4Max;
              break;
   		  }
	    }
	  }

      // Identify Groups for Curves

      switch (i)
      {
		// Rock Porosity
        case _PCORE:
        case _PPLUG:
        case _P800:
        case _PINSI:
        case _PEFF:
          dOUT[i][GROUP] = (double) _PCORE;
          break;

        // Rock Permeability
        case _KMAX:
        case _K90:
        case _KVRT:
        case _KPLG:
        case _KKL:
        case _KINSI:
        case _KKLIN:
        case _KPVRT:
          dOUT[i][GROUP] = (double) _KMAX;
          break;

        // Saturation
        case _SOIL:
        case _SW:
          break;

        // Density
        case _GMCC:
        case _RHOD:
        case _RHOW:
          dOUT[i][GROUP] = (double) _GMCC;
          break;

        // Archie Constants
        case _MAMB:
        case _MINSI:
        case _NAMB:
        case _NINSI:
          break;

        // Gamma Ray
        case _GR:
        case _CGR:
        case _PGR:
          dOUT[i][GROUP] = (double) _GR;
          break;

        case _TH:
        case _U:
        case _K:
          break;

        case _LIN_1:
        case _LIN_2:
        case _LIN_3:
        case _LIN_4:
          break;

        case _LOG_1:
        case _LOG_2:
        case _LOG_3:
        case _LOG_4:
          dOUT[i][GROUP] = (double) _LOG_1;
          break;
	  }
    }

    return (dOUT);
  }

  /** Method copy()
   * <p>  This method will copy the LAS Plot Limits data Array
   *      There are 3 columns Minimum, Maximum, Group, Log Curve Present.
   *      The Group is used to force other curves that reside on the same
   *      plot track, i.e. resistivity, permeability, porosity and
   *      the Log Curve Present is an indicator to include this curve.
   *
   * @param   dIN  = Array of data limits
   * @return  dOUT = Array of data limits copied from dIN
   */

  public static double[][] copy( double dIN[][] )
  {
    double dOUT[][] = null;
    double iColumns = 3;

    if (dIN != null)
    {
      dOUT = new double[ROWS][COLUMNS];

      for (int i=0; i<ROWS; i++)
      {
		for (int j=0; j<COLUMNS; j++)
		{
		  dOUT[i][j] = dIN[i][j];
		}
      }
    }

    return (dOUT);
  }

  /** Method setByGroup()
   * <p>  This method will copy the LAS Plot Limits data Array
   *      There are 3 columns Minimum, Maximum, Group, Log Curve Present.
   *      The Group is used to force other curves that reside on the same
   *      plot track, i.e. resistivity, permeability, porosity and
   *      the Log Curve Present is an indicator to include this curve.
   *
   * @param   dGroup = Group ID
   * @param   dMin   = Minimum Value for Group
   * @param   dMax   = Maximum Value for Group
   * @param   data   = Array of data limits
   * @return  data   = Array of data limits normalized by group
   */

  public static double[][] setByGroup( double dGroup, double dMin, double dMax, double data[][] )
  {
    if (data != null)
    {
      for (int i=0; i<ROWS; i++)
      {
		if ((dGroup == data[i][GROUP]) && (data[i][GROUP] > -1))
		{
          data[i][MINIMUM] = dMin;
          data[i][MAXIMUM] = dMax;
		}
      }
    }

    return (data);
  }
}

/*
 *  @version 1.1 04/15/2016
 *  @author  John Victorine
 */
