/*
 * @rockImagesUtility.java Version 1.1 11/07/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock;

import rock.rockImagesStruct;
import rock.rockImagesListStruct;

import iqstrat.iqstratHeadersStruct;

/** Class rockImagesUtility
 *  <p> This Class will provide basic utilities for the Rock data structures.
 *
 *  @version 1.1 11/07/2007
 *  @author  John R. Victorine
 */

public class rockImagesUtility
{
  /** Method computeDepthRange()
   * <p> This method will compute the depth range of the rock image
   * @param   st = the Rock image list structure
   * @return  st = the Rock image list structure
   */

  public static rockImagesListStruct computeDepthRange(rockImagesListStruct st)
  {
    double depthStart = 0.0;
    double depthEnd   = 0.0;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        depthStart = st.stItem[i].depthStart;
        depthEnd   = st.stItem[i].depthEnd;

        if (i==0)
        {
          st.depthStart = depthStart;
          st.depthEnd   = depthEnd;
        }

        if (st.depthStart > depthStart)
          st.depthStart = depthStart;

        if (st.depthEnd < depthEnd)
          st.depthEnd   = depthEnd;
      }
    }

    return (st);
  }

  /** Method copyList()
   * <p> This method will copy one core image list structure to another
   * @param  stOld = Old core image list data structure
   * @return stNew = New core image list data structure
   */

  public static rockImagesListStruct copyList(rockImagesListStruct stOld)
  {
    rockImagesListStruct stNew = null;

    if (stOld != null)
    {
      if (stOld.iCount > 0)
      {
        stNew        = new rockImagesListStruct();
        stNew.iCount = stOld.iCount;
        stNew.stItem = new rockImagesStruct[stOld.iCount];

        // Identification Information

        stNew.sKID   = new String(stOld.sKID); // Location Header KID
        stNew.sKEY   = new String(stOld.sKEY); // Primary Key created on user
        stNew.iType  = stOld.iType;
        stNew.sAPI   = new String(stOld.sAPI); // API-Number of Location
        stNew.sName  = new String(stOld.sName); // Well Name or Outcrop Name
        stNew.status = new String(stOld.status); // Status of well or Outcrop Type

        // XY Position Information

        stNew.dLatitude  = stOld.dLatitude; // Latitude
        stNew.dLongitude = stOld.dLongitude; // Longitude

        // Z Position - Depth, Elevation

        stNew.depth = stOld.depth; // Total Depth of Well or Height of Outcrop
        stNew.dGL   = stOld.dGL; // Ground Level

        stNew.depthStart = stOld.depthStart;  // Starting Depth of the Rock Data
        stNew.depthEnd   = stOld.depthEnd;  // Ending Depth of the Rock Data

        // LAS 3 Required Fields

        stNew.sKGS        = new String( stOld.sKGS );        // KGS Saved Data Indicator

        stNew.sIMAGE_SRC  = new String( stOld.sIMAGE_SRC );  // Image Source
        stNew.sIMAGE_DATE = new String( stOld.sIMAGE_DATE ); // Image Date

        // LAS 3 File Parameters

        stNew.iParamRows = stOld.iParamRows;    // Total number of rows
        stNew.iParamCols = stOld.iParamCols;    // Total number of columns
        stNew.sParams    = new String[stNew.iParamRows][stNew.iParamCols];

        for (int i=0; i<stOld.iCount; i++)
        {
          stNew.stItem[i] = copy(stOld.stItem[i]);
        }
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static rockImagesListStruct transfer(rockImagesListStruct stOld)
  {
    rockImagesListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy one rock image structure to another
   * @param  stOld = old rock image structure
   * @return stNew = new rock image structure
   */

  public static rockImagesStruct copy(rockImagesStruct stOld)
  {
    rockImagesStruct stNew = new rockImagesStruct();

    if (stOld != null)
    {
      stNew.sKID         = new String(stOld.sKID);
      stNew.sKEY         = new String(stOld.sKEY);
      stNew.sLocation    = new String(stOld.sLocation);
      stNew.sURL         = new String(stOld.sURL);
      stNew.sDirectory   = new String(stOld.sDirectory);
      stNew.sFilename    = new String(stOld.sFilename);
      stNew.sOriginal    = new String(stOld.sOriginal);
      stNew.sFormat      = new String(stOld.sFormat);
      stNew.sType        = new String(stOld.sType);
      stNew.depthStart   = stOld.depthStart;
      stNew.depthEnd     = stOld.depthEnd;
      stNew.sAuthors     = new String(stOld.sAuthors);
      stNew.sYear        = new String(stOld.sYear);
      stNew.sTitle       = new String(stOld.sTitle);
      stNew.source       = new String(stOld.source);
      stNew.sDate        = new String(stOld.sDate);
      stNew.sThumbnail   = new String(stOld.sThumbnail);
      stNew.sDescription = new String(stOld.sDescription);
      stNew.sComments    = new String(stOld.sComments);
    }

    return (stNew);
  }

  /** Method print()
   * <p> This method will print the contents of the image
   *     list structure.
   */

  public static void print( rockImagesListStruct st )
  {
	String sTemp = "";

    if (st != null)
    {
	  for (int i=0; i<st.iCount; i++)
	  {
	    sTemp = new String( sTemp + "Start: " + st.stItem[i].depthStart + "\n"  );
	    sTemp = new String( sTemp + "End: "   + st.stItem[i].depthEnd + "\n"  );

		if (st.stItem[i].sLocation.length() > 0)
		  sTemp = new String( sTemp + "LOCATION: " + st.stItem[i].sLocation + "\n" );

		if (st.stItem[i].sURL.length() > 0)
		  sTemp = new String( sTemp + "URL: " + st.stItem[i].sURL + "\n"  );

        if (st.stItem[i].sDirectory.length() > 0)
		  sTemp = new String( sTemp + "DIR: " + st.stItem[i].sDirectory + "\n"  );

        if (st.stItem[i].sFilename.length() > 0)
		  sTemp = new String( sTemp + "File: " + st.stItem[i].sFilename + "\n"  );

        if (st.stItem[i].sOriginal.length() > 0)
		  sTemp = new String( sTemp + "Orig: " + st.stItem[i].sOriginal + "\n"  );

        if (st.stItem[i].sFormat.length() > 0)
		  sTemp = new String( sTemp + "Format: " + st.stItem[i].sFormat + "\n"  );

        if (st.stItem[i].sType.length() > 0)
		  sTemp = new String( sTemp + "Type: " + st.stItem[i].sType + "\n"  );

        if (st.stItem[i].sAuthors.length() > 0)
		  sTemp = new String( sTemp + "Authors: " + st.stItem[i].sAuthors + "\n"  );

        if (st.stItem[i].sYear.length() > 0)
		  sTemp = new String( sTemp + "Year: " + st.stItem[i].sYear + "\n"  );

        if (st.stItem[i].sTitle.length() > 0)
		  sTemp = new String( sTemp + "Title: " + st.stItem[i].sTitle + "\n"  );

        if (st.stItem[i].source.length() > 0)
		  sTemp = new String( sTemp + "Source: " + st.stItem[i].source + "\n"  );

        if (st.stItem[i].sDate.length() > 0)
		  sTemp = new String( sTemp + "Date: " + st.stItem[i].sDate + "\n"  );

        if (st.stItem[i].sThumbnail.length() > 0)
		  sTemp = new String( sTemp + "Thumb: " + st.stItem[i].sThumbnail + "\n"  );

        if (st.stItem[i].sDescription.length() > 0)
		  sTemp = new String( sTemp + "Desc: " + st.stItem[i].sDescription + "\n"  );

        if (st.stItem[i].sComments.length() > 0)
		  sTemp = new String( sTemp + "Comment: " + st.stItem[i].sComments + "\n"  );

		System.out.println(sTemp);
		sTemp = new String("");
	  }
	}
  }

}