/*
 * @rockFileDataUtility.java Version 1.1 09/21/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock;

import iqstrat.iqstratHeadersStruct;

import rock.rockFileDataStruct;
import rock.rockDataListStruct;

import lith.texture.textureListStruct;
import rock.phi.phiListStruct;
import rock.phi.phiStruct;

/** Class rockFileDataUtility
 *  <p> This Class will provide basic utilities for the Rock File
 *      data structures.
 *
 *  @version 1.1 09/21/2009
 *  @author  John R. Victorine
 */

public class rockFileDataUtility
{
  /* ==================================================================== *
   * --------------- LAS File Well Data Structure Methods --------------- *
   * ==================================================================== */

  /** Method initialize()
   *  <p> This method will initialize the Rock File Data Structure
   *   @return st = The Rock File Data Structure
   */

  public static rockFileDataStruct initialize()
  {
    rockFileDataStruct st = new rockFileDataStruct();

    if (st != null)
    {
      st.iSource    = las3.las3Constants._SOURCE_NONE;

      st.sKID       = new String(""); // Primary Key for the KGS Well
      st.sKEY       = new String(""); // Primary Key created by user
      st.iType      = iqstrat.iqstratHeadersStruct.WELL; // Type of Data

      st.sAPI       = new String(""); // API-Number of Well
      st.sName      = new String(""); // Well Name

      // Other Fields not really displayed

      st.state      = new String(""); // State Name
      st.sCounty    = new String(""); // County Name

      // Well Location Information

      st.dLatitude  = 0.0; // Latitude of Well
      st.dLongitude = 0.0; // Longitude of Well

      // Well Parameter Information

      st.dKB        = 0.0; // Kelly Bushing of Well
      st.dGL        = 0.0; // Ground Level of Well
      st.dTD        = 0.0; // Total Depth of Well

      // Well Status

      st.status     = new String(""); // Status of Well

      // Unknown Curve Mnemonics

      st.sUnknown = new String[4][2];
      for (int i=0; i<4; i++)
      {
		for (int j=0; j<2; j++)
		{
		  st.sUnknown[i][j] = new String( "" );
		}
	  }

      // LAS File Information

      st.dNull      = 0.0; // Null Value of Data
      st.depthStart = 0.0; // Start Depth of LAS Data
      st.depthEnd   = 0.0; // Ending Depth of LAS Data
      st.depthStep  = 0.0; // Depth Step of LAS DAta

      st.iProprietary = 0; // Proprietary Data Identfier

      // Gamma Ray Curves

      st.iGR   = -1; // Gamma Ray Curve Number
      st.iCGR  = -1; // Gamma Ray Minus Uranium Curve Number
      st.iPGR  = -1; // Pseudo Gamma Ray Value

      // -- Density Data

      st.iGMCC  = -1; // Grain Density (gm/cc)
      st.iRHOD  = -1; // Density of Rock Dry (gm/cc)
      st.iRHOW  = -1; // Density of Rock Wet (gm/cc)

      // -- Porosity Data

      st.iPCORE = -1; // Porosity whole routine (PU)
      st.iPPLUG = -1; // Porosity Plug routine (PU)
      st.iP800  = -1; // Porosity Plug 800 PSI (PU)
      st.iPINSI = -1; // Porosity Plug Insitu (PU)
      st.iPEFF  = -1; // Effective Rock Porosity (PU)

      // -- Spectral Gamma Ray Data

      st.iTh    = -1; // Thorium Curve Number
      st.iU     = -1; // Uranium Curve Number
      st.iK     = -1; // Potasium Curve Number

      // Spectral Gamma Ray Ratios

      st.iThU   = -1; // Th/U
      st.iThK   = -1; // Th/K

      // -- Lithology Data

      st.iGRL   = -1; // Gamma Ray Curve Number
      st.iPHI   = -1; // Porosity Curve Number
      st.iRHOG  = -1; // "Grain" Density Curve Number

      // Unknown Linear Curves

      st.iLIN_1  = -1;  // Linear Curve 1
      st.iLIN_2  = -1;  // Linear Curve 2
      st.iLIN_3  = -1;  // Linear Curve 3
      st.iLIN_4  = -1;  // Linear Curve 4

      // Unknown Log Curves

      st.iLOG_1  = -1;  // Log Curve 1
      st.iLOG_2  = -1;  // Log Curve 2
      st.iLOG_3  = -1;  // Log Curve 3
      st.iLOG_4  = -1;  // Log Curve 4

      // Lithofacies Curves

      st.iRows  = 0;    // Number of Rows
      st.depths = null; // Array of Depth Data Values

      st.dGR    = null; // Array of Gamma Ray Values
      st.dCGR   = null; // Array of Gamma Ray minus Uranium Values
      st.dPGR   = null; // Array of Pseudo Gamma Ray Values

      // Density Data

      st.dGMCC  = null; // Array of Grain Density (gm/cc) Values
      st.dRHOD  = null; // Array of Density of Rock Dry (gm/cc) Values
      st.dRHOW  = null; // Array of Density of Rock Wet (gm/cc) Values

      // Porosity Data

      st.dPCORE = null; // Array of Porosity whole routine (PU) Values
      st.dPPLUG = null; // Array of Porosity Plug routine (PU) Values
      st.dP800  = null; // Array of Porosity Plug 800 PSI (PU) Values
      st.dPINSI = null; // Array of Porosity Plug Insitu (PU) Values
      st.dPEFF  = null; // Array of Effective Rock Porosity (PU) Values

      // Spectral Gamma Ray Data

      st.dTh    = null; // Array of Thorium Values
      st.dU     = null; // Array of Uranium Values
      st.dK     = null; // Array of Potassium Values

      // Spectral Gamma Ray Lithology Curves

      st.dThU   = null; // Array of Th/U Ratio Values
      st.dThK   = null; // Array of Th/K Ratio Values

      // -- Lithology Data

      st.dGRL   = null; // Array of Gamma Ray Values
      st.dPHI   = null; // Array of Porosity Values
      st.dRHOG  = null; // Array of "Grain" Density Values

      // -- Unknown Linear Curves

      st.dLIN_1 = null; // Linear Curve 1
      st.dLIN_2 = null; // Linear Curve 2
      st.dLIN_3 = null; // Linear Curve 3
      st.dLIN_4 = null; // Linear Curve 4

      // -- Unknown Log Curves

      st.dLOG_1 = null; // Log Curve 1
      st.dLOG_2 = null; // Log Curve 2
      st.dLOG_3 = null; // Log Curve 3
      st.dLOG_4 = null; // Log Curve 4

      st.dPHIMin  = 0.0;
      st.dPHIMax  = 0.3;

      // .. Linear Curves

      st.dLIN_1Min = 0.0; // Linear Curve 1 Minimum
      st.dLIN_2Min = 0.0; // Linear Curve 2 Minimum
      st.dLIN_3Min = 0.0; // Linear Curve 3 Minimum
      st.dLIN_4Min = 0.0; // Linear Curve 4 Minimum

      st.dLIN_1Max = 1.0; // Linear Curve 1 Maximum
      st.dLIN_2Max = 1.0; // Linear Curve 2 Maximum
      st.dLIN_3Max = 1.0; // Linear Curve 3 Maximum
      st.dLIN_4Max = 1.0; // Linear Curve 4 Maximum

      // .. Semilog Curves

      st.dLOG_1Min = 0.1;    // Log Curve 1 Minimum
      st.dLOG_2Min = 0.1;    // Log Curve 2 Minimum
      st.dLOG_3Min = 0.1;    // Log Curve 3 Minimum
      st.dLOG_4Min = 0.1;    // Log Curve 4 Minimum

      st.dLOG_1Max = 1000.0; // Log Curve 1 Maximum
      st.dLOG_2Max = 1000.0; // Log Curve 2 Maximum
      st.dLOG_3Max = 1000.0; // Log Curve 3 Maximum
      st.dLOG_4Max = 1000.0; // Log Curve 4 Maximum
    }

    return (st);
  }

  /** Method buildNew( int iUnit, double depthStart, double depthEnd )
   * <p> This method will create a new Rock File Data Structure based on the
   *     depth range.
   * @param  iUnit      = Depth Units Scale factor (Default is 1/2 foot)
   * @param  depthStart = The Starting Depth of data
   * @param  depthEnd   = The Ending Depth of Data
   * @return st         = The new Rock File Data Struture
   */

  public static rockFileDataStruct buildNew( int iUnit,
                                             double depthStart,
                                             double depthEnd )
  {
    int    iTotal  = 0;
    int    iCount  = 0;
    double depth1  = 0;
    double depth2  = 0;
    double dScale  = 0;

    rockFileDataStruct st = initialize();

    depthStart = (double) ( (int) depthStart );
    depthEnd   = 1.0 + (double) ( (int) depthEnd );
    dScale     = iqstrat.iqstratTracksStruct._SCALE[iUnit];
    iCount     = 1 + (int) (iqstrat.iqstratTracksStruct._SCALE_FACTOR[iUnit] *
                              Math.abs( depthEnd - depthStart ));

    st.depthStart = depthStart;
    st.depthEnd   = depthEnd;
    st.depthStep  = iqstrat.iqstratTracksStruct._SCALE[iUnit];

    st.depths = new double[iCount];

    st.dGR    = new double[iCount];
    st.dCGR   = new double[iCount];
    st.dPGR   = new double[iCount];

    // Density Data

    st.dGMCC  = new double[iCount];
    st.dRHOD  = new double[iCount];
    st.dRHOW  = new double[iCount];

    // Porosity Data

    st.dPCORE = new double[iCount];
    st.dPPLUG = new double[iCount];
    st.dP800  = new double[iCount];
    st.dPINSI = new double[iCount];
    st.dPEFF  = new double[iCount];

    // Spectral Gamma Ray Data

    st.dTh    = new double[iCount];
    st.dU     = new double[iCount];
    st.dK     = new double[iCount];

    // Spectral Gamma Ray Lithology Curves

    st.dThU   = new double[iCount];
    st.dThK   = new double[iCount];

    // -- Lithology Data

    st.dGRL   = new double[iCount];
    st.dPHI   = new double[iCount];
    st.dRHOG  = new double[iCount];

    // -- Unknown Linear Curves

    st.dLIN_1  = new double[iCount]; // Linear Curve 1
    st.dLIN_2  = new double[iCount]; // Linear Curve 2
    st.dLIN_3  = new double[iCount]; // Linear Curve 3
    st.dLIN_4  = new double[iCount]; // Linear Curve 4

    // -- Unknown Log Curves

    st.dLOG_1  = new double[iCount]; // Log Curve 1
    st.dLOG_2  = new double[iCount]; // Log Curve 2
    st.dLOG_3  = new double[iCount]; // Log Curve 3
    st.dLOG_4  = new double[iCount]; // Log Curve 4

    for (int i=0; i<iCount; i++)
    {
      if (iTotal < iCount)
      {
        st.depths[iTotal] = depthStart + i * dScale;

        st.dGR[iTotal]    = st.dNull;
        st.dCGR[iTotal]   = st.dNull;
        st.dPGR[iTotal]   = st.dNull;

        // Density Data

        st.dGMCC[iTotal]  = st.dNull;
        st.dRHOD[iTotal]  = st.dNull;
        st.dRHOW[iTotal]  = st.dNull;

        // Porosity Data

        st.dPCORE[iTotal] = st.dNull;
        st.dPPLUG[iTotal] = st.dNull;
        st.dP800[iTotal]  = st.dNull;
        st.dPINSI[iTotal] = st.dNull;
        st.dPEFF[iTotal]  = st.dNull;

        // Spectral Gamma Ray Data

        st.dTh[iTotal]    = st.dNull;
        st.dU[iTotal]     = st.dNull;
        st.dK[iTotal]     = st.dNull;

        // Spectral Gamma Ray Lithology Curves

        st.dThU[iTotal]   = st.dNull;
        st.dThK[iTotal]   = st.dNull;

        // -- Lithology Data

        st.dGRL[iTotal]   = st.dNull;
        st.dPHI[iTotal]   = st.dNull;
        st.dRHOG[iTotal]  = st.dNull;

        // -- Unknown Linear Curves

        st.dLIN_1[iTotal] = st.dNull; // Linear Curve 1
        st.dLIN_2[iTotal] = st.dNull; // Linear Curve 2
        st.dLIN_3[iTotal] = st.dNull; // Linear Curve 3
        st.dLIN_4[iTotal] = st.dNull; // Linear Curve 4

        // -- Unknown Log Curves

        st.dLOG_1[iTotal] = st.dNull; // Log Curve 1
        st.dLOG_2[iTotal] = st.dNull; // Log Curve 2
        st.dLOG_3[iTotal] = st.dNull; // Log Curve 3
        st.dLOG_4[iTotal] = st.dNull; // Log Curve 4

        iTotal++;
      }
    }

    st.iRows = iTotal;

    st.dPHIMin  = 0.0;
    st.dPHIMax  = 0.3;

    // .. Linear Curves

    st.dLIN_1Min = 0.0; // Linear Curve 1 Minimum
    st.dLIN_2Min = 0.0; // Linear Curve 2 Minimum
    st.dLIN_3Min = 0.0; // Linear Curve 3 Minimum
    st.dLIN_4Min = 0.0; // Linear Curve 4 Minimum

    st.dLIN_1Max = 1.0; // Linear Curve 1 Maximum
    st.dLIN_2Max = 1.0; // Linear Curve 2 Maximum
    st.dLIN_3Max = 1.0; // Linear Curve 3 Maximum
    st.dLIN_4Max = 1.0; // Linear Curve 4 Maximum

    // .. Semilog Curves

    st.dLOG_1Min = 0.1;    // Log Curve 1 Minimum
    st.dLOG_2Min = 0.1;    // Log Curve 2 Minimum
    st.dLOG_3Min = 0.1;    // Log Curve 3 Minimum
    st.dLOG_4Min = 0.1;    // Log Curve 4 Minimum

    st.dLOG_1Max = 1000.0; // Log Curve 1 Maximum
    st.dLOG_2Max = 1000.0; // Log Curve 2 Maximum
    st.dLOG_3Max = 1000.0; // Log Curve 3 Maximum
    st.dLOG_4Max = 1000.0; // Log Curve 4 Maximum

    return (st);
  }

  /** Method copy( rockFileDataStruct stOld )
   * <p> This method will copy the Rock Data from one file to another.
   * @param   stOld = Rock File Data Structure
   * @return  stNew = Rock File Data Structure
   */

  public static rockFileDataStruct copy( rockFileDataStruct stOld )
  {
	int i,j;
    rockFileDataStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new rockFileDataStruct();

      stNew.iSource    = stOld.iSource;

      stNew.sKID       = new String(stOld.sKID); // Primary Key for the KGS Well
      stNew.sKEY       = new String(stOld.sKEY); // Primary Key created by user
      stNew.iType      = stOld.iType;            // Type of Data

      stNew.sAPI       = new String(stOld.sAPI);  // API-Number of Well
      stNew.sName      = new String(stOld.sName); // Well Name

      // Other Fields not really displayed

      stNew.state      = new String(stOld.state);    // State Name
      stNew.sCounty    = new String(stOld.sCounty); // County Name

      // Well Location Information

      stNew.dLatitude  = stOld.dLatitude;  // Latitude of Well
      stNew.dLongitude = stOld.dLongitude; // Longitude of Well

      // Well Parameter Information

      stNew.dKB        = stOld.dKB; // Kelly Bushing of Well
      stNew.dGL        = stOld.dGL; // Ground Level of Well
      stNew.dTD        = stOld.dTD; // Total Depth of Well

      // Well Status

      stNew.status     = new String(stOld.status); // Status of Well

      // Unknown Curve Mnemonics

      stNew.sUnknown = new String[4][2];
      for (i=0; i<4; i++)
      {
		for (j=0; j<2; j++)
		{
		  stNew.sUnknown[i][j] = new String( stOld.sUnknown[i][j] );
		}
	  }

      // LAS File Information

      stNew.dNull      = stOld.dNull;      // Null Value of Data
      stNew.depthStart = stOld.depthStart; // Start Depth of LAS Data
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth of LAS Data
      stNew.depthStep  = stOld.depthStep;  // Depth Step of LAS DAta

      stNew.iProprietary = stOld.iProprietary; // Proprietary Data Identfier

      // Gamma Ray Curves

      stNew.iGR   = stOld.iGR; // Gamma Ray Curve Number
      stNew.iCGR  = stOld.iCGR; // Gamma Ray Minus Uranium Curve Number
      stNew.iPGR  = stOld.iPGR; // Pseudo Gamma Ray Value

      // -- Density Data

      stNew.iGMCC  = stOld.iGMCC; // Grain Density (gm/cc)
      stNew.iRHOD  = stOld.iRHOD; // Density of Rock Dry (gm/cc)
      stNew.iRHOW  = stOld.iRHOW; // Density of Rock Wet (gm/cc)

      // -- Porosity Data

      stNew.iPCORE = stOld.iPCORE; // Porosity whole routine (PU)
      stNew.iPPLUG = stOld.iPPLUG; // Porosity Plug routine (PU)
      stNew.iP800  = stOld.iP800; // Porosity Plug 800 PSI (PU)
      stNew.iPINSI = stOld.iPINSI; // Porosity Plug Insitu (PU)
      stNew.iPEFF  = stOld.iPEFF; // Effective Rock Porosity (PU)

      // -- Spectral Gamma Ray Data

      stNew.iTh    = stOld.iTh; // Thorium Curve Number
      stNew.iU     = stOld.iU; // Uranium Curve Number
      stNew.iK     = stOld.iK; // Potasium Curve Number

      // -- Spectral Gamma Ray Ratios

      stNew.iThU   = stOld.iThU; // Th/U
      stNew.iThK   = stOld.iThK; // Th/K

      // -- Lithology Data

      stNew.iGRL   = stOld.iGRL; // Gamma Ray Curve Number
      stNew.iPHI   = stOld.iPHI; // Porosity Curve Number
      stNew.iRHOG  = stOld.iRHOG; // "Grain" Density Curve Number

      // -- Linear Curve Data

      stNew.iLIN_1 = stOld.iLIN_1; // Linear Curve 1
      stNew.iLIN_2 = stOld.iLIN_2; // Linear Curve 2
      stNew.iLIN_3 = stOld.iLIN_3; // Linear Curve 3
      stNew.iLIN_4 = stOld.iLIN_4; // Linear Curve 4

      // -- Log Curve Data

      stNew.iLOG_1 = stOld.iLOG_1; // Log Curve 1
      stNew.iLOG_2 = stOld.iLOG_2; // Log Curve 2
      stNew.iLOG_3 = stOld.iLOG_3; // Log Curve 3
      stNew.iLOG_4 = stOld.iLOG_4; // Log Curve 4

      // Lithofacies Curves

      stNew.iRows  = stOld.iRows;    // Number of Rows
      stNew.depths = new double[stNew.iRows];

      stNew.dGR    = new double[stNew.iRows];
      stNew.dCGR   = new double[stNew.iRows];
      stNew.dPGR   = new double[stNew.iRows];

      // Density Data

      stNew.dGMCC  = new double[stNew.iRows];
      stNew.dRHOD  = new double[stNew.iRows];
      stNew.dRHOW  = new double[stNew.iRows];

      // Porosity Data

      stNew.dPCORE = new double[stNew.iRows];
      stNew.dPPLUG = new double[stNew.iRows];
      stNew.dP800  = new double[stNew.iRows];
      stNew.dPINSI = new double[stNew.iRows];
      stNew.dPEFF  = new double[stNew.iRows];

      // Spectral Gamma Ray Data

      stNew.dTh    = new double[stNew.iRows];
      stNew.dU     = new double[stNew.iRows];
      stNew.dK     = new double[stNew.iRows];

      // Spectral Gamma Ray Lithology Curves

      stNew.dThU   = new double[stNew.iRows];
      stNew.dThK   = new double[stNew.iRows];

      // -- Lithology Data

      stNew.dGRL   = new double[stNew.iRows];
      stNew.dPHI   = new double[stNew.iRows];
      stNew.dRHOG  = new double[stNew.iRows];

      // -- Linear Curve Data

      stNew.dLIN_1 = new double[stNew.iRows]; // Linear Curve 1
      stNew.dLIN_2 = new double[stNew.iRows]; // Linear Curve 2
      stNew.dLIN_3 = new double[stNew.iRows]; // Linear Curve 3
      stNew.dLIN_4 = new double[stNew.iRows]; // Linear Curve 4

      // -- Log Curve Data

      stNew.dLOG_1 = new double[stNew.iRows]; // Log Curve 1
      stNew.dLOG_2 = new double[stNew.iRows]; // Log Curve 2
      stNew.dLOG_3 = new double[stNew.iRows]; // Log Curve 3
      stNew.dLOG_4 = new double[stNew.iRows]; // Log Curve 4

      for (i=0; i<stOld.iRows; i++)
      {
        stNew.depths[i] = stOld.depths[i];   // Depth Data Values

        stNew.dGR[i]    = stOld.dGR[i];   // Gamma Ray Values
        stNew.dCGR[i]   = stOld.dCGR[i];  // Gamma Ray minus Uranium Values
        stNew.dPGR[i]   = stOld.dPGR[i];  // Pseudo Gamma Ray Values

        // Density Data

        stNew.dGMCC[i]  = stOld.dGMCC[i]; // Grain Density (gm/cc) Values
        stNew.dRHOD[i]  = stOld.dRHOD[i]; // Density of Rock Dry (gm/cc) Values
        stNew.dRHOW[i]  = stOld.dRHOW[i]; // Density of Rock Wet (gm/cc) Values

        // Porosity Data

        stNew.dPCORE[i] = stOld.dPCORE[i]; // Porosity whole routine (PU) Values
        stNew.dPPLUG[i] = stOld.dPPLUG[i]; // Porosity Plug routine (PU) Values
        stNew.dP800[i]  = stOld.dP800[i];  // Porosity Plug 800 PSI (PU) Values
        stNew.dPINSI[i] = stOld.dPINSI[i]; // Porosity Plug Insitu (PU) Values
        stNew.dPEFF[i]  = stOld.dPEFF[i];  // Effective Rock Porosity (PU) Values

        // Spectral Gamma Ray Data

        stNew.dTh[i]    = stOld.dTh[i];   // Thorium Values
        stNew.dU[i]     = stOld.dU[i];    // Uranium Values
        stNew.dK[i]     = stOld.dK[i];    // Potassium Values

        // Spectral Gamma Ray Lithology Curves

        stNew.dThU[i]   = stOld.dThU[i];  // Th/U Ratio Values
        stNew.dThK[i]   = stOld.dThK[i];  // Th/K Ratio Values

        // -- Lithology Data

        stNew.dGRL[i]   = stOld.dGRL[i];  // Gamma Ray Values
        stNew.dPHI[i]   = stOld.dPHI[i];  // Porosity Values
        stNew.dRHOG[i]  = stOld.dRHOG[i]; // "Grain" Density Values

        // -- Linear Curve Data

        stNew.dLIN_1[i] = stOld.dLIN_1[i]; // Linear Curve 1
        stNew.dLIN_2[i] = stOld.dLIN_2[i]; // Linear Curve 2
        stNew.dLIN_3[i] = stOld.dLIN_3[i]; // Linear Curve 3
        stNew.dLIN_4[i] = stOld.dLIN_4[i]; // Linear Curve 4

        // -- Log Curve Data

        stNew.dLOG_1[i] = stOld.dLOG_1[i]; // Log Curve 1
        stNew.dLOG_2[i] = stOld.dLOG_2[i]; // Log Curve 2
        stNew.dLOG_3[i] = stOld.dLOG_3[i]; // Log Curve 3
        stNew.dLOG_4[i] = stOld.dLOG_4[i]; // Log Curve 4
      }

      stNew.dPHIMin   = stOld.dPHIMin;
      stNew.dPHIMax   = stOld.dPHIMax;

      // .. Linear Curves

      stNew.dLIN_1Min = stOld.dLIN_1Min; // Linear Curve 1 Minimum
      stNew.dLIN_2Min = stOld.dLIN_2Min; // Linear Curve 2 Minimum
      stNew.dLIN_3Min = stOld.dLIN_3Min; // Linear Curve 3 Minimum
      stNew.dLIN_4Min = stOld.dLIN_4Min; // Linear Curve 4 Minimum

      stNew.dLIN_1Max = stOld.dLIN_1Max; // Linear Curve 1 Maximum
      stNew.dLIN_2Max = stOld.dLIN_2Max; // Linear Curve 2 Maximum
      stNew.dLIN_3Max = stOld.dLIN_3Max; // Linear Curve 3 Maximum
      stNew.dLIN_4Max = stOld.dLIN_4Max; // Linear Curve 4 Maximum

      // .. Semilog Curves

      stNew.dLOG_1Min = stOld.dLOG_1Min; // Log Curve 1 Minimum
      stNew.dLOG_2Min = stOld.dLOG_2Min; // Log Curve 2 Minimum
      stNew.dLOG_3Min = stOld.dLOG_3Min; // Log Curve 3 Minimum
      stNew.dLOG_4Min = stOld.dLOG_4Min; // Log Curve 4 Minimum

      stNew.dLOG_1Max = stOld.dLOG_1Max; // Log Curve 1 Maximum
      stNew.dLOG_2Max = stOld.dLOG_2Max; // Log Curve 2 Maximum
      stNew.dLOG_3Max = stOld.dLOG_3Max; // Log Curve 3 Maximum
      stNew.dLOG_4Max = stOld.dLOG_4Max; // Log Curve 4 Maximum
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static rockFileDataStruct transfer(rockFileDataStruct stOld)
  {
    rockFileDataStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copy( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method addWellData()
   * <p> This method will add the data to the Rock File Data Structure
   * @param  stW = Well Information List Data Structure
   * @param  stC = Rock File Data Structure
   * @return stC = Rock File Data Structure
   */

  public static rockFileDataStruct addWellData( iqstratHeadersStruct stW,
                                                rockFileDataStruct   stC )
  {
    if ((stC != null) && (stW !=null))
    {
      stC.sKID      = new String(stW.sKID);  // Primary Key created by KGS
      stC.sKEY      = new String(stW.sKEY);  // Primary Key created by user
      stC.iType     = stW.iType;             // Type of Data
      stC.sAPI      = new String(stW.sAPI);  // API-Number of Well
      stC.sName     = new String(stW.sName); // Well Name

      // Other Fields not really displayed

      stC.sCounty = new String(stW.sCounty); // County Name
      stC.state   = new String(stW.state);   // State Name

      // Well Location Information

      stC.dLatitude  = stW.dLatitude;
      stC.dLongitude = stW.dLongitude;

      // Well Parameter Information

      stC.dKB   = stW.dKB;
      stC.dGL   = stW.dGL;
      stC.dTD   = stW.depth;

      // Well Status

      stC.status = new String(stW.status);
    }

    return (stC);
  }

  /** Method addData()
   * <p> This method will add the rock data to the Rock File Data Structure
   * @param  st  = Rock Data List Data Structure
   * @param  stC = Rock File Data Structure
   * @return stC = Rock File Data  Structure
   */

  public static rockFileDataStruct addData( rockDataListStruct st,
                                            rockFileDataStruct stC )
  {
    int    i,j;
    double dMin   = 9999.0;
    double del    = 0.0;
    double depth1 = 0.0;
    double depth2 = 0.0;
    double dStart = 0.0;
    double dEnd   = 0.0;
    double dRange[] = { 0.0, 0.0 };
    double dNULL  = 0.0;

    if (st != null)
    {
      if (st._BASE == -1)
      {
        for (i=0; i<st.iCount-1; i++)
        {
          del = Math.abs( st.stItem[i].dTOP - st.stItem[i+1].dTOP );
          if (del != 0.0)
          {
            if (del < dMin)
              dMin = del;
          }
        }
      }
    }

    if ((st != null) && (stC != null))
    {
      dNULL = st.stItem[0].dNULL;

      // Unknown Curve Mnemonics

      stC.sUnknown = new String[4][2];
      for (i=0; i<4; i++)
      {
		for (j=0; j<2; j++)
		{
		  stC.sUnknown[i][j] = new String( st.sUnknown[i][j] );
		}
	  }

      for (i=0; i<st.iCount; i++)
      {
        dStart = st.stItem[i].dTOP  + st.stItem[i].dCORR;
        if (st._BASE == -1)
          dEnd   = dStart + dMin;
        else
          dEnd   = st.stItem[i].dBASE + st.stItem[i].dCORR;

        for (j=0; j<stC.iRows-1; j++)
        {
          depth1 = stC.depths[j];
          depth2 = stC.depths[j+1];

          if (((dStart >= depth1) && (dStart < depth2)) ||
              ((depth1 >= dStart) && (depth1 < dEnd)))
          {
            if (st.checkData(rock.rockStandardTools._GR))
            {
              if ((st.stItem[i].dGR != st.stItem[i].dNULL) &&
                  (st.stItem[i].dGR != 0.0))
              {
                if (stC.iGR == -1) stC.iGR = 0;
                stC.iGR++;
                stC.dGR[j] = st.stItem[i].dGR; // Gamma Ray Values
              }
            }

            if (st.checkData(rock.rockStandardTools._CGR))
            {
              if ((st.stItem[i].dCGR != st.stItem[i].dNULL) &&
                  (st.stItem[i].dCGR != 0.0))
              {
                if (stC.iCGR == -1) stC.iCGR = 0;
                stC.iCGR++;
                stC.dCGR[j] = st.stItem[i].dCGR; // Gamma Ray minus Uranium Values
              }
            }

            // Density Data

            if (st.checkData(rock.rockStandardTools._GMCC))
            {
              if ((st.stItem[i].dGMCC != st.stItem[i].dNULL) &&
                  (st.stItem[i].dGMCC != 0.0))
              {
                if (stC.iGMCC == -1) stC.iGMCC = 0;
                stC.iGMCC++;
                stC.dGMCC[j] = st.stItem[i].dGMCC; // Grain Density (gm/cc) Values
              }
            }

            if (st.checkData(rock.rockStandardTools._RHOD))
            {
              if ((st.stItem[i].dRHOD != st.stItem[i].dNULL) &&
                  (st.stItem[i].dRHOD != 0.0))
              {
                if (stC.iRHOD == -1) stC.iRHOD = 0;
                stC.iRHOD++;
                stC.dRHOD[j] = st.stItem[i].dRHOD; // Density of Rock Dry (gm/cc) Values
              }
            }

            if (st.checkData(rock.rockStandardTools._RHOW))
            {
              if ((st.stItem[i].dRHOW != st.stItem[i].dNULL) &&
                  (st.stItem[i].dRHOW != 0.0))
              {
                if (stC.iRHOW == -1) stC.iRHOW = 0;
                stC.iRHOW++;
                stC.dRHOW[j] = st.stItem[i].dRHOW; // Density of Rock Wet (gm/cc) Values
              }
            }

            // Porosity Data

            if (st.checkData(rock.rockStandardTools._PCORE))
            {
              if ((st.stItem[i].dPCORE != st.stItem[i].dNULL) &&
                  (st.stItem[i].dPCORE != 0.0))
              {
                if (stC.iPCORE == -1) stC.iPCORE = 0;
                stC.iPCORE++;
                stC.dPCORE[j] = st.stItem[i].dPCORE; // Porosity whole routine (PU) Values
              }
            }

            if (st.checkData(rock.rockStandardTools._PPLUG))
            {
              if ((st.stItem[i].dPPLUG != st.stItem[i].dNULL) &&
                  (st.stItem[i].dPPLUG != 0.0))
              {
                if (stC.iPPLUG == -1) stC.iPPLUG = 0;
                stC.iPPLUG++;
                stC.dPPLUG[j] = st.stItem[i].dPPLUG; // Porosity Plug routine (PU) Values
              }
            }

            if (st.checkData(rock.rockStandardTools._P800))
            {
              if ((st.stItem[i].dP800 != st.stItem[i].dNULL) &&
                  (st.stItem[i].dP800 != 0.0))
              {
                if (stC.iP800 == -1) stC.iP800 = 0;
                stC.iP800++;
                stC.dP800[j] = st.stItem[i].dP800; // Porosity Plug 800 PSI (PU) Values
              }
            }

            if (st.checkData(rock.rockStandardTools._PINSI))
            {
              if ((st.stItem[i].dPINSI != st.stItem[i].dNULL) &&
                  (st.stItem[i].dPINSI != 0.0))
              {
                if (stC.iPINSI == -1) stC.iPINSI = 0;
                stC.iPINSI++;
                stC.dPINSI[j] = st.stItem[i].dPINSI; // Porosity Plug Insitu (PU) Values
              }
            }

            if (st.checkData(rock.rockStandardTools._PEFF))
            {
              if ((st.stItem[i].dPEFF != st.stItem[i].dNULL) &&
                  (st.stItem[i].dPEFF != 0.0))
              {
                if (stC.iPEFF == -1) stC.iPEFF = 0;
                stC.iPEFF++;
                stC.dPEFF[j] = st.stItem[i].dPEFF; // Effective Rock Porosity (PU) Values
              }
            }

            // Spectral Gamma Ray Data

            if (st.checkData(rock.rockStandardTools._TH))
            {
              if ((st.stItem[i].dTh != st.stItem[i].dNULL) &&
                  (st.stItem[i].dTh != 0.0))
              {
                if (stC.iTh == -1) stC.iTh = 0;
                stC.iTh++;
                stC.dTh[j] = st.stItem[i].dTh; // Thorium Values
              }
            }

            if (st.checkData(rock.rockStandardTools._U))
            {
              if ((st.stItem[i].dU != st.stItem[i].dNULL) &&
                  (st.stItem[i].dU != 0.0))
              {
                if (stC.iU == -1) stC.iU = 0;
                stC.iU++;
                stC.dU[j] = st.stItem[i].dU; // Uranium Values
              }
            }

            if (st.checkData(rock.rockStandardTools._K))
            {
              if ((st.stItem[i].dK != st.stItem[i].dNULL) &&
                  (st.stItem[i].dK != 0.0))
              {
                if (stC.iK == -1) stC.iK = 0;
                stC.iK++;
                stC.dK[j] = st.stItem[i].dK; // Potassium Values
              }
            }

            // Linear Curves

            if (st.checkData(rock.rockStandardTools._LIN_1))
            {
              if ((st.stItem[i].dLIN_1 != st.stItem[i].dNULL) &&
                  (st.stItem[i].dLIN_1 != 0.0))
              {
                if (stC.iLIN_1 == -1) stC.iLIN_1 = 0;
                stC.iLIN_1++;
                stC.dLIN_1[j] = st.stItem[i].dLIN_1;
              }
            }
            if (st.checkData(rock.rockStandardTools._LIN_2))
            {
              if ((st.stItem[i].dLIN_2 != st.stItem[i].dNULL) &&
                  (st.stItem[i].dLIN_2 != 0.0))
              {
                if (stC.iLIN_2 == -1) stC.iLIN_2 = 0;
                stC.iLIN_2++;
                stC.dLIN_2[j] = st.stItem[i].dLIN_2;
              }
            }
            if (st.checkData(rock.rockStandardTools._LIN_3))
            {
              if ((st.stItem[i].dLIN_3 != st.stItem[i].dNULL) &&
                  (st.stItem[i].dLIN_3 != 0.0))
              {
                if (stC.iLIN_3 == -1) stC.iLIN_3 = 0;
                stC.iLIN_3++;
                stC.dLIN_3[j] = st.stItem[i].dLIN_3;
              }
            }
            if (st.checkData(rock.rockStandardTools._LIN_4))
            {
              if ((st.stItem[i].dLIN_4 != st.stItem[i].dNULL) &&
                  (st.stItem[i].dLIN_4 != 0.0))
              {
                if (stC.iLIN_4 == -1) stC.iLIN_4 = 0;
                stC.iLIN_4++;
                stC.dLIN_4[j] = st.stItem[i].dLIN_4;
              }
            }

            // Log Curves

            if (st.checkData(rock.rockStandardTools._LOG_1))
            {
              if ((st.stItem[i].dLOG_1 != st.stItem[i].dNULL) &&
                  (st.stItem[i].dLOG_1 != 0.0))
              {
                if (stC.iLOG_1 == -1) stC.iLOG_1 = 0;
                stC.iLOG_1++;
                stC.dLOG_1[j] = st.stItem[i].dLOG_1;
              }
            }
            if (st.checkData(rock.rockStandardTools._LOG_2))
            {
              if ((st.stItem[i].dLOG_1 != st.stItem[i].dNULL) &&
                  (st.stItem[i].dLOG_1 != 0.0))
              {
                if (stC.iLOG_2 == -1) stC.iLOG_2 = 0;
                stC.iLOG_2++;
                stC.dLOG_2[j] = st.stItem[i].dLOG_2;
              }
            }
            if (st.checkData(rock.rockStandardTools._LOG_3))
            {
              if ((st.stItem[i].dLOG_3 != st.stItem[i].dNULL) &&
                  (st.stItem[i].dLOG_3 != 0.0))
              {
                if (stC.iLOG_3 == -1) stC.iLOG_3 = 0;
                stC.iLOG_3++;
                stC.dLOG_3[j] = st.stItem[i].dLOG_3;
              }
            }
            if (st.checkData(rock.rockStandardTools._LOG_4))
            {
              if ((st.stItem[i].dLOG_4 != st.stItem[i].dNULL) &&
                  (st.stItem[i].dLOG_4 != 0.0))
              {
                if (stC.iLOG_4 == -1) stC.iLOG_4 = 0;
                stC.iLOG_4++;
                stC.dLOG_4[j] = st.stItem[i].dLOG_4;
              }
            }
          }
        }
      }

      if (stC.iLIN_1 > -1)
      {
        dRange = math.mathLAS.computeCurveMinMax(stC.dLIN_1, dNULL);
        stC.dLIN_1Min = dRange[0];
        stC.dLIN_1Max = dRange[1];
System.out.println("LIN 1 "+stC.dLIN_1Min+" "+stC.dLIN_1Max);
      }

      if (stC.iLIN_2 > -1)
      {
        dRange = math.mathLAS.computeCurveMinMax(stC.dLIN_2, dNULL);
        stC.dLIN_2Min = dRange[0];
        stC.dLIN_2Max = dRange[1];
System.out.println("LIN 2 "+stC.dLIN_2Min+" "+stC.dLIN_2Max);
	  }

      if (stC.iLIN_3 > -1)
      {
        dRange = math.mathLAS.computeCurveMinMax(stC.dLIN_3, dNULL);
        stC.dLIN_3Min = dRange[0];
        stC.dLIN_3Max = dRange[1];
	   }

      if (stC.iLIN_4 > -1)
      {
        dRange = math.mathLAS.computeCurveMinMax(stC.dLIN_4, dNULL);
        stC.dLIN_4Min = dRange[0];
        stC.dLIN_4Max = dRange[1];
      }

      if (stC.iLOG_1 > -1)
      {
        dRange = math.mathLAS.computeLogMinMax(stC.dLOG_1, dNULL);
        stC.dLOG_1Min = dRange[0];
        stC.dLOG_1Max = dRange[1];
System.out.println("LOG 1 "+stC.dLOG_1Min+" "+stC.dLOG_1Max);
      }

      if (stC.iLOG_2 > -1)
      {
        dRange = math.mathLAS.computeLogMinMax(stC.dLOG_2, dNULL);
        stC.dLOG_2Min = dRange[0];
        stC.dLOG_2Max = dRange[1];
System.out.println("LOG 2 "+stC.dLOG_2Min+" "+stC.dLOG_2Max);
	  }

      if (stC.iLOG_3 > -1)
      {
        dRange = math.mathLAS.computeLogMinMax(stC.dLOG_3, dNULL);
        stC.dLOG_3Min = dRange[0];
        stC.dLOG_3Max = dRange[1];
	  }

      if (stC.iLOG_4 > -1)
      {
        dRange = math.mathLAS.computeLogMinMax(stC.dLOG_4, dNULL);
        stC.dLOG_4Min = dRange[0];
        stC.dLOG_4Max = dRange[1];
      }
    }

    return (stC);
  }

  /** Method addData()
   * <p> This method will add the rock texture data to the Rock File Data Structure
   * @param  st  = Rock Texture Data List Data Structure
   * @param  stC = Rock File Data Structure
   * @return stC = Rock File Data  Structure
   */

  public static rockFileDataStruct addData( textureListStruct st,
                                            rockFileDataStruct stC )
  {
    int    i      = 0;
    int    iCell  = 0;
    double dMin   = 9999.0;
    double del    = 0.0;
    double depth1 = 0.0;
    double depth2 = 0.0;
    double dStart = 0.0;
    double dEnd   = 0.0;

    if ((st != null) && (stC != null))
    {
      for (i=0; i<st.iCount; i++)
      {
        dStart = st.stItem[i].depthStart;
        dEnd   = st.stItem[i].depthEnd;

        for (int j=0; j<stC.iRows-1; j++)
        {
          depth1 = stC.depths[j];
          depth2 = stC.depths[j+1];

          if (((dStart >= depth1) && (dStart < depth2)) ||
              ((depth1 >= dStart) && (depth1 < dEnd)))
          {
            iCell = st.stItem[i].iEnergy;
            if (iCell == 0) iCell = 16;

            if (stC.iPGR == -1) stC.iPGR = 0;
            stC.iPGR++;
            stC.dPGR[j] = 80.0 * (1.0 - iCell / 16.0);
          }
        }
      }
    }

    return (stC);
  }

  /** Method computeGR( rockFileDataStruct st )
   * <p> This method will compute the gamma ray from spectral gamma ray
   * @param  st = Rock File Data Structure
   * @return st = Rock File Data  Structure
   */

  public static rockFileDataStruct computeGR( rockFileDataStruct st )
  {
    int i = 0;

    if (st != null)
    {
      /* IF Computed Gamma Ray does not exist and or
       *    Gamma Ray does not exist
       *   Uranium:    1.0 ppm =  8.0   (8.09) API
       *   Thorium:    1.0 ppm =  4.0   (3.93) API
       *   Potassium:  1.0 %   = 16.0   (16.32) API
       */

      if (st.iGR == -1)
      {
        if ((st.iTh > -1) && (st.iU > -1) && (st.iK > -1))
        {
          st.iGR = 100;
          for (i=0; i<st.iRows; i++)
          {
            st.dGR[i] = st.dNull;
            if ((st.dU[i] != st.dNull) ||
                (st.dTh[i] != st.dNull) ||
                (st.dK[i] != st.dNull))
            {
              st.dGR[i] = 8.0 * st.dU[i] + 4.0 * st.dTh[i] + 16.0 * st.dK[i];
            }
          }
        }
      }

      if (st.iCGR == -1)
      {
        if ((st.iTh > -1) && (st.iU > -1) && (st.iK > -1))
        {
          st.iCGR = 100;
          for (i=0; i<st.iRows; i++)
          {
            st.dCGR[i] = st.dNull;
            if ((st.dTh[i] != st.dNull) || (st.dK[i] != st.dNull))
              st.dCGR[i] = 4.0 * st.dTh[i] + 16.0 * st.dK[i];
          }
        }
      }
    }

    return (st);
  }

  /** Method computeSPRatio( rockFileDataStruct st )
   * <p> This method will compute the spectral gamma ray ratio
   * @param  st = Rock File Data Structure
   * @return st = Rock File Data  Structure
   */

  public static rockFileDataStruct computeSPRatio( rockFileDataStruct st )
  {
    double dOut[][] = null;

    if (st != null)
    {
      if ((st.iTh > -1) && (st.iU > -1) && (st.iK > -1))
      {
        dOut = math.mathLAS.computeSpectralGR( st.iRows,
                                               st.dTh,
                                               st.dU,
                                               st.dK,
                                               st.dNull );

        if (dOut != null)
        {
          for (int i=0; i<st.iRows; i++)
          {
            if (st.iThU == -1) st.iThU = 0;
            if (st.iThU == -1) st.iThU = 0;

            st.dThU[i] = dOut[0][i]; // Thorium-Uranium Ratio
            st.dThK[i] = dOut[1][i]; // Thorium-Potassium Ratio

            st.iThU++; // Th/U
            st.iThK++; // Th/K
         }
        }
      }
    }

    return (st);
  }

  /** Method lithologyCurves( rockFileDataStruct st )
   * <p> This method will automatically assign the lithology curves.
   * @param  st = Rock File Data Structure
   * @return st = Rock File Data  Structure
   */

  public static rockFileDataStruct lithologyCurves( rockFileDataStruct st )
  {
    int i     = 0;
    int iGR   = -1;
    int iRHO  = -1;
    int iPHI  = -1;
    int idata = 0;

    if (st != null)
    {
      if (st.iPGR    > 0) iGR = rock.rockStandardTools._PGR;
      if (st.iGR     > 0) iGR = rock.rockStandardTools._GR;

      if (st.iPEFF   > 0) iPHI = rock.rockStandardTools._PEFF;
      if (st.iP800   > 0) iPHI = rock.rockStandardTools._P800;
      if (st.iPINSI  > 0) iPHI = rock.rockStandardTools._PINSI;
      if ((st.iPPLUG > 0) && (st.iPPLUG > st.iPINSI))
        iPHI = rock.rockStandardTools._PPLUG;
      if ((st.iPCORE > 0) &&
          ((st.iPCORE > st.iPPLUG) && (st.iPCORE > st.iPINSI)))
        iPHI = rock.rockStandardTools._PCORE;

      if (st.iRHOW > 0) iRHO = rock.rockStandardTools._RHOW;
      if (st.iGMCC > 0) iRHO = rock.rockStandardTools._GMCC;

      if (iGR  > -1)
      {
        st.dGRL = st.getData(iGR);
        st.iGRL = 1;
      }

      if (iPHI > -1)
      {
        st.dPHI = st.getData(iPHI);
        st.iPHI = 1;
      }

      if (iRHO == rock.rockStandardTools._GMCC)
      {
        st.dRHOG = st.getData(iRHO);
        st.iRHOG = 1;
      }
      else if ((iRHO == rock.rockStandardTools._RHOW) && (iPHI > -1))
      {
        st.dRHOG = math.mathLAS.computeRhomaa( math.mathLAS._FRESH,
                                               st.iRows,
                                               st.getData(iPHI),
                                               st.getData(iRHO),
                                               st.dNull );
        st.iRHOG = 1;
      }

      idata = 0;
      for (i=0; i<st.iRows; i++)
      {
        if (st.dPHI[i] > 2.0)
          idata = 1;
      }

      st.dPHIMin = 1.0;
      st.dPHIMax = 0.0;

      if (idata > 0)
      {
        for (i=0; i<st.iRows; i++)
        {
          if (st.dPHI[i] != st.dNull)
            st.dPHI[i] = st.dPHI[i] / 100.0;

          if ((st.dPHI[i] < st.dPHIMin) && (st.dPHI[i] != 0.0))
            st.dPHIMin = st.dPHI[i];

          if (st.dPHI[i] > st.dPHIMax)
            st.dPHIMax = st.dPHI[i];
        }
      }
    }

    return (st);
  }

  /** Method getPorosity()
   * <p> This method will get the porosity Data List Structure from the
   *     Rock File Data Structure
   * @param   st = Rock File Data Structure
   * @return stList = The Porosity Data List Structure
   */

  public static phiListStruct getPorosity( rockFileDataStruct st )
  {
    phiListStruct stList = null;

    if (st != null)
    {
      if (st.iRows > 0)
      {
        stList        = new phiListStruct();
        stList.stItem = new phiStruct[st.iRows];
        stList.iCount = st.iRows;

        for (int i=0; i<st.iRows; i++)
        {
          stList.stItem[i] = new phiStruct();

          stList.stItem[i].sKEY = new String( cmn.cmnString.UniqueName()+i );
          stList.stItem[i].depthStart = st.depths[i];
          stList.stItem[i].depthEnd   =
              stList.stItem[i].depthStart + st.depthStep;
          if (st.dPHI[i] != rock.rockDataStruct.dNULL)
            stList.stItem[i].dPHI     = st.dPHI[i] * 100.0;
        }
      }
    }

    return (stList);
  }

  /** Method computeBulkDensity()
   * <p> This method will compute the bulk density from the core grain density
   *     and the primary porosity
   * @param   st = Rock File Data Structure
   * @return  dRHOB = the Bulk Density
   */

  public static double[] computeBulkDensity( rockFileDataStruct st )
  {
    double dRHOB[] = null;
    double dPHI    = 0.0;
    double dRHO    = 0.0;
    double dRhoH2O = 1.0;

    if (st != null)
    {
      dRHOB = new double[st.iRows];
      for (int i=0; i<st.iRows; i++)
      {
        if (st.dPHI[i] != st.dNull)
        {
          if (st.dRHOG[i] != st.dNull)
          {
            dPHI = st.dPHI[i];
            dRHO = st.dRHOG[i];

            dRHOB[i] = dRHO - dPHI * (dRHO - dRhoH2O);
          }
        }
      }
    }

    return (dRHOB);
  }

  /** Method print()
   * <p> This method will print the
   * @param st = Rock File Data Structure
   */

  public static void print( rockFileDataStruct st )
  {
    if (st != null)
    {
      for (int i=0; i<st.iRows; i++)
      {
        System.out.println("--> "+st.depths[i]);

        System.out.println(st.dGR[i]+" "+st.dCGR[i]+" "+st.dPGR[i]);

        // Density Data

        System.out.println(st.dGMCC[i]+" "+st.dRHOD[i]+" "+st.dRHOW[i]);

        // Porosity Data

        System.out.println(st.dPCORE[i]+" "+st.dPPLUG[i]+" "+st.dP800[i]+" "+st.dPINSI[i]+" "+st.dPEFF[i]);

        // Spectral Gamma Ray Data

        System.out.println(st.dTh[i]+" "+st.dU[i]+" "+st.dK[i]);

        // Spectral Gamma Ray Lithology Curves

        System.out.println(st.dThU[i]+" "+st.dThK[i]);

        // -- Lithology Data

        System.out.println(st.dGRL[i]+" "+st.dPHI[i]+" "+st.dRHOG[i]);
      }
    }
  }
}