/*
 * @rockDataUtility.java Version 1.1 11/07/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock;

import rock.rockDataStruct;
import rock.rockDataListStruct;
import las3.las3Struct;

import iqstrat.iqstratHeadersStruct;

import lith.texture.textureListStruct;

/** Class rockDataUtility
 *  <p> This Class will provide basic utilities for the Rock data structures.
 *
 *  @version 1.1 11/07/2007
 *  @author  John R. Victorine
 */

public class rockDataUtility
{
  /** Method addHeaderData()
   * <p> This method will add the header data to the rock data List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Rock List Data Structure
   * @return st       = The Rock List Data Structure.
   */

  public static rockDataListStruct addHeaderData(iqstratHeadersStruct stHeader,
                                                 rockDataListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method computeDepthRange()
   * <p> This method will compute the depth range of the core data
   * @param   st = the Rock data list structure
   * @return  st = the Rock data list structure
   */

  public static rockDataListStruct computeDepthRange(rockDataListStruct st)
  {
    double depthStart = 0.0;
    double depthEnd   = 0.0;
    int    iBegin     = 0;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        depthStart = st.stItem[i].dTOP;
        depthEnd   = st.stItem[i].dBASE;
        if (depthEnd  == rock.rockDataStruct.dNULL)
          depthEnd = depthStart;

//        if (i==0)
        if ((iBegin==0) && (depthStart != rock.rockDataStruct.dNULL))
        {
          st.depthStart = depthStart;
          st.depthEnd   = depthEnd;
          iBegin = 1;
        }
        else
        {
          if (st.depthStart > depthStart)
            st.depthStart = depthStart;

          if (st.depthEnd < depthEnd)
            st.depthEnd   = depthEnd;
        }
      }
    }

    return (st);
  }

  /** Method computeBulkDensity()
   * <p> This method will compute the bulk density from the core grain density
   *     and the primary porosity
   * @param   st = the Rock data list structure
   * @return  st = the Rock data list structure
   */

  public static rockDataListStruct computeBulkDensity(rockDataListStruct st)
  {
    int    i       = 0;
    int    idata   = 0;
    double dPHI    = 0.0;
    double dRhoH2O = 1.1;
    double dRHO    = 0.0;

    if (st != null)
    {
      idata = 0;
      for (i=0; i<st.iCount; i++)
      {
        if (st.stItem[i].dNPHI > 2.0)
          idata = 1;
      }

      if (idata > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          st.stItem[i].dNPHI = st.stItem[i].dNPHI / 100.0;
        }
      }

      for (i=0; i<st.iCount; i++)
      {
        if (st.stItem[i].dNPHI != rock.rockDataStruct.dNULL)
        {
          if (st.stItem[i].dGMCC != rock.rockDataStruct.dNULL)
          {
            dPHI = st.stItem[i].dNPHI;
            dRHO = st.stItem[i].dGMCC;

            st.stItem[i].dRHOB = dRHO - dPHI * (dRHO - dRhoH2O);
          }
        }
      }
    }

    return (st);
  }

  /** Method computeGR( rockDataListStruct st )
   * <p> This method will compute the gamma ray from spectral gamma ray
   * @param  st = Rock File Data Structure
   * @return st = Rock File Data  Structure
   */

  public static rockDataListStruct computeGR( rockDataListStruct st )
  {
    int i = 0;

    if (st != null)
    {
      /* IF Computed Gamma Ray does not exist and or
       *    Gamma Ray does not exist
       *   Uranium:    1.0 ppm =  8.0   (8.09) API
       *   Thorium:    1.0 ppm =  4.0   (3.93) API
       *   Potassium:  1.0 %   = 16.0   (16.32) API
       */

      if (st._GR == -1)
      {
        if ((st._TH > -1) && (st._U > -1) && (st._K > -1))
        {
          st._GR = 1;
          for (i=0; i<st.iCount; i++)
          {
            st.stItem[i].dGR = st.stItem[i].dNULL;
            if ((st.stItem[i].dU  != st.stItem[i].dNULL) ||
                (st.stItem[i].dTh != st.stItem[i].dNULL) ||
                (st.stItem[i].dK  != st.stItem[i].dNULL))
            {
              st.stItem[i].dGR = 8.0 * st.stItem[i].dU +
                                 4.0 * st.stItem[i].dTh +
                                 16.0 * st.stItem[i].dK;
            }
          }
        }
      }

      if (st._CGR == -1)
      {
        if ((st._TH > -1) && (st._U > -1) && (st._K > -1))
        {
          st._CGR = 1;
          for (i=0; i<st.iCount; i++)
          {
            st.stItem[i].dCGR = st.stItem[i].dNULL;
            if ((st.stItem[i].dTh != st.stItem[i].dNULL) ||
                (st.stItem[i].dK  != st.stItem[i].dNULL))
            {
              st.stItem[i].dCGR = 4.0 * st.stItem[i].dTh +
                                  16.0 * st.stItem[i].dK;
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method addBaseDepth()
   * <p> This method will add a base depth if it is null by determining the
   *     difference from one sample depth with the next and picking the smallest
   *     interval between them
   * @param  stList = core data list data structure
   * @return stList = core data list data structure
   */

  public static rockDataListStruct addBaseDepth( rockDataListStruct stList )
  {
    int    i     = 0;
    double diff  = 0.0;
    double diffo = 2.0;

    if (stList != null)
    {
      if (stList.iCount > 0)
      {
        for (i=0; i<stList.iCount; i++)
        {
          if (i < stList.iCount-1)
          {
            diff = stList.stItem[i+1].dTOP - stList.stItem[i].dTOP;
            if (diff < diffo) diffo = diff;
          }
        }

        for (i=0; i<stList.iCount; i++)
        {
          if ((stList.stItem[i].dBASE == rock.rockDataStruct.dNULL) &&
              (stList.stItem[i].dTOP != rock.rockDataStruct.dNULL))
          {
            stList.stItem[i].dBASE = stList.stItem[i].dTOP + diffo;
          }
        }
      }
    }

    return (stList);
  }

  /** Method addPorosity()
   * <p> This method will add the Porosity by hierarchy of Porosity data types
   * @param  stList = core data list data structure
   * @return stList = core data list data structure
   */

  public static rockDataListStruct addPorosity( rockDataListStruct stList )
  {
    int i=0;
    int iData[] = { 0, 0, 0, 0, 0 };  // { PCORE, PPLUG, PINSI, PEFF, P800 }
    int iPHI    = 4;
    int iValue  = 0;

    if (stList != null)
    {
      if (stList.iCount > 0)
      {
        for (i=0; i<stList.iCount; i++)
        {
          if (stList._PCORE > -1) iData[0]++;
          if (stList._PPLUG > -1) iData[1]++;
          if (stList._PINSI > -1) iData[2]++;
          if (stList._PEFF  > -1) iData[3]++;
          if (stList._P800  > -1) iData[4]++;
        }

        for (i=4; i>=0; i--)
        {
          if (iData[i] >= iData[iPHI])
            iPHI = i;
        }

        for (i=0; i<stList.iCount; i++)
        {
          switch (iPHI)
          {
            case 0:
              stList.stItem[i].dNPHI = stList.stItem[i].dPCORE;
              break;
            case 1:
              stList.stItem[i].dNPHI = stList.stItem[i].dPPLUG;
              break;
            case 2:
              stList.stItem[i].dNPHI = stList.stItem[i].dPINSI;
              break;
            case 3:
              stList.stItem[i].dNPHI = stList.stItem[i].dPEFF;
              break;
            case 4:
              stList.stItem[i].dNPHI = stList.stItem[i].dP800;
              break;
          }

          if (stList.stItem[i].dNPHI > 2.0)
            iValue = 1;
        }

        if (iValue == 1)
        {
          for (i=0; i<stList.iCount; i++)
          {
            if (stList.stItem[i].dNPHI != rock.rockDataStruct.dNULL)
              stList.stItem[i].dNPHI = stList.stItem[i].dNPHI / 100.0;
          }
        }
      }
    }

    return (stList);
  }

  /** Method copyList()
   * <p> This method will copy one Core Data list structure to another
   * @param  stOld = Old core data list data structure
   * @return stNew = New core data list data structure
   */

  public static rockDataListStruct copyList(rockDataListStruct stOld)
  {
    int i,j;
    rockDataListStruct stNew = null;

    if (stOld != null)
    {
      if (stOld.iCount > 0)
      {
        stNew        = new rockDataListStruct();
        stNew.iCount = stOld.iCount;
        stNew.stItem = new rockDataStruct[stOld.iCount];

        stNew.iSource = stOld.iSource;

        // Identification Information

        stNew.sKID   = new String(stOld.sKID); // Location Header KID
        stNew.sKEY   = new String(stOld.sKEY); // Primary Key created on user
        stNew.iType  = stOld.iType;
        stNew.sAPI   = new String(stOld.sAPI); // API-Number of Location
        stNew.sName  = new String(stOld.sName); // Well Name or Outcrop Name
        stNew.status = new String(stOld.status); // Status of well or Outcrop Type

        // XY Position Information

        stNew.dLatitude  = stOld.dLatitude; // Latitude
        stNew.dLongitude = stOld.dLongitude; // Longitude

        // Z Position - Depth, Elevation

        stNew.depth = stOld.depth; // Total Depth of Well or Height of Outcrop
        stNew.dGL   = stOld.dGL; // Ground Level

        // -- LAS 3 Output Required Variables

        stNew.source     = new String(stOld.source);     // Core Source
        stNew.sType      = new String(stOld.sType);      // Core Type
        stNew.sRecovery  = new String(stOld.sRecovery);  // Recovery Date
        stNew.depthStart = stOld.depthStart;             // Core Top Depth
        stNew.depthEnd   = stOld.depthEnd;               // Core Base Depth
        stNew.dLength    = stOld.dLength;                // Recovered Amount
        stNew.sFormation = new String(stOld.sFormation); // Primary Formation
        stNew.diameter   = stOld.diameter;               // Core Diameter
        stNew.sCompany   = new String(stOld.sCompany);   // Analyzing Company
        stNew.sAnalyzed  = new String(stOld.sAnalyzed);  // Analysis Date

        // LAS 3 File Parameters

        stNew.iParamRows = stOld.iParamRows;    // Total number of rows
        stNew.iParamCols = stOld.iParamCols;    // Total number of columns
        stNew.sParams    = new String[stNew.iParamRows][stNew.iParamCols];

        for (i=0; i<stNew.iParamRows; i++)
        {
          for (j=0; j<stNew.iParamCols; j++)
          {
            // Array holding the Parameter Definitions
            stNew.sParams[i][j] = new String( stOld.sParams[i][j] );
          }
        }

        stNew.iLogs = stOld.iLogs;  // Total number of LAS3 Data Structures
        if (stOld.iLogs > 0)
        {
          stNew.stLAS3 = new las3Struct[stOld.iLogs];
          for (i=0; i<stOld.iLogs; i++)
          {
            stNew.stLAS3[i] = las3.las3Utility.copyWOData( stOld.stLAS3[i] );
          }
        }

        // Unknown Curve Mnemonics

        stNew.sUnknown = new String[4][2];
        for (i=0; i<4; i++)
        {
  		  for (j=0; j<2; j++)
  		  {
		    stNew.sUnknown[i][j] = new String( stOld.sUnknown[i][j] );
		  }
	    }

        // -- Depth information

        stNew._TOP      = stOld._TOP; // Top of Rock Depth
        stNew._BASE     = stOld._BASE; // Base of Rock Depth
        stNew._CORR     = stOld._CORR; // Depth Correction

        // -- Descriptive fields to define rock

        stNew._STU      = stOld._STU; // Stratigraphic Unit
        stNew._STN      = stOld._STN; // Stratigraphic Name
        stNew._ENV      = stOld._ENV; // Environment
        stNew._LITHO    = stOld._LITHO; // Lithofacies

        // -- Porosity Data

        stNew._PCORE    = stOld._PCORE; // Porosity whole routine (PU)
        stNew._PPLUG    = stOld._PPLUG; // Porosity Plug routine (PU)
        stNew._P800     = stOld._P800; // Porosity Plug 800 PSI (PU)
        stNew._PINSI    = stOld._PINSI; // Porosity Plug Insitu (PU)
        stNew._PEFF     = stOld._PEFF; // Effective Rock Porosity (PU)

        // -- Permeability Data

        stNew._KMAX     = stOld._KMAX; // Permeability Whole Max (md)
        stNew._K90      = stOld._K90; // Permeability Whole 90 (md)
        stNew._KVRT     = stOld._KVRT; // Permeability Whole Vertical (md)
        stNew._KPLG     = stOld._KPLG; // Permeability Plug Routine (md)
        stNew._KKL      = stOld._KKL; // Permeability Plug Klinkenberg Routine (md)
        stNew._KINSI    = stOld._KINSI; // Permeability Plug Insitu (md)
        stNew._KKLIN    = stOld._KKLIN; // Permeability Plug Klinkenberg Insitu (md)
        stNew._KPVRT    = stOld._KPVRT; // Permeability Plug Vertical (md)

        // -- Saturation Data

        stNew._SOIL     = stOld._SOIL; // Saturation Oil (%)
        stNew._SW       = stOld._SW; // Saturation Water (%)

        // -- Density Data

        stNew._GMCC     = stOld._GMCC; // Grain Density (gm/cc)
        stNew._RHOD     = stOld._RHOD; // Density of Rock Dry (gm/cc)
        stNew._RHOW     = stOld._RHOW; // Density of Rock Wet (gm/cc)

        // -- Archie Constants

        stNew._MAMB     = stOld._MAMB;  // Archie Cementation (M) Ambient
        stNew._MINSI    = stOld._MINSI; // Archie Cementation (M) Insitu
        stNew._NAMB     = stOld._NAMB;  // Archie Saturation (N) Ambient
        stNew._NINSI    = stOld._NINSI; // Archie Saturation (N) Insitu

        // -- misc data

        stNew._LTHCD    = stOld._LTHCD; // Lithofacies Code
        stNew._FRACTURE = stOld._FRACTURE; // Fractures

        // -- Measured Gamma Ray Data

        stNew._GR       = stOld._GR;  // Gamma Ray Value
        stNew._CGR      = stOld._CGR; // Gamma Ray minus Uranium Value

        stNew._TH       = stOld._TH;  // Thorium Values
        stNew._U        = stOld._U;   // Uranium Values
        stNew._K        = stOld._K;   // Potassium Values

        // Unknown Linear Curves

        stNew._LIN_1    = stOld._LIN_1;     // Linear Curve 1
        stNew._LIN_2    = stOld._LIN_2;     // Linear Curve 2
        stNew._LIN_3    = stOld._LIN_3;     // Linear Curve 3
        stNew._LIN_4    = stOld._LIN_4;     // Linear Curve 4

        // Unknown Log Curves

        stNew._LOG_1    = stOld._LOG_1;     // Log Curve 1
        stNew._LOG_2    = stOld._LOG_2;     // Log Curve 2
        stNew._LOG_3    = stOld._LOG_3;     // Log Curve 3
        stNew._LOG_4    = stOld._LOG_4;     // Log Curve 4

        // .. Linear Curves

        stNew.dLIN_1Min = stOld.dLIN_1Min; // Linear Curve 1 Minimum
        stNew.dLIN_2Min = stOld.dLIN_2Min; // Linear Curve 2 Minimum
        stNew.dLIN_3Min = stOld.dLIN_3Min; // Linear Curve 3 Minimum
        stNew.dLIN_4Min = stOld.dLIN_4Min; // Linear Curve 4 Minimum

        stNew.dLIN_1Max = stOld.dLIN_1Max; // Linear Curve 1 Maximum
        stNew.dLIN_2Max = stOld.dLIN_2Max; // Linear Curve 2 Maximum
        stNew.dLIN_3Max = stOld.dLIN_3Max; // Linear Curve 3 Maximum
        stNew.dLIN_4Max = stOld.dLIN_4Max; // Linear Curve 4 Maximum

        // .. Semilog Curves

        stNew.dLOG_1Min = stOld.dLOG_1Min; // Log Curve 1 Minimum
        stNew.dLOG_2Min = stOld.dLOG_2Min; // Log Curve 2 Minimum
        stNew.dLOG_3Min = stOld.dLOG_3Min; // Log Curve 3 Minimum
        stNew.dLOG_4Min = stOld.dLOG_4Min; // Log Curve 4 Minimum

        stNew.dLOG_1Max = stOld.dLOG_1Max; // Log Curve 1 Maximum
        stNew.dLOG_2Max = stOld.dLOG_2Max; // Log Curve 2 Maximum
        stNew.dLOG_3Max = stOld.dLOG_3Max; // Log Curve 3 Maximum
        stNew.dLOG_4Max = stOld.dLOG_4Max; // Log Curve 4 Maximum

        for (i=0; i<stOld.iCount; i++)
        {
          stNew.stItem[i] = copy(stOld.stItem[i]);
        }
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static rockDataListStruct transfer(rockDataListStruct stOld)
  {
    rockDataListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copyRockHeader()
   * <p> This method will copy one Core Data list structure to another
   * @param  stOld = Old core data list data structure
   * @return stNew = New core data list data structure
   */

  public static rockDataListStruct copyRockHeader(rockDataListStruct stOld)
  {
    int i,j;
    rockDataListStruct stNew = new rockDataListStruct();

    if (stOld != null)
    {
        // Identification Information

        stNew.sKID   = new String(stOld.sKID); // Location Header KID
        stNew.sKEY   = new String(stOld.sKEY); // Primary Key created on user
        stNew.iType  = stOld.iType;
        stNew.sAPI   = new String(stOld.sAPI); // API-Number of Location
        stNew.sName  = new String(stOld.sName); // Well Name or Outcrop Name
        stNew.status = new String(stOld.status); // Status of well or Outcrop Type

        // XY Position Information

        stNew.dLatitude  = stOld.dLatitude; // Latitude
        stNew.dLongitude = stOld.dLongitude; // Longitude

        // Z Position - Depth, Elevation

        stNew.depth = stOld.depth; // Total Depth of Well or Height of Outcrop
        stNew.dGL   = stOld.dGL; // Ground Level

        // -- LAS 3 Output Required Variables

        stNew.source     = new String(stOld.source);     // Core Source
        stNew.sType      = new String(stOld.sType);      // Core Type
        stNew.sRecovery  = new String(stOld.sRecovery);  // Recovery Date
        stNew.depthStart = stOld.depthStart;             // Core Top Depth
        stNew.depthEnd   = stOld.depthEnd;               // Core Base Depth
        stNew.dLength    = stOld.dLength;                // Recovered Amount
        stNew.sFormation = new String(stOld.sFormation); // Primary Formation
        stNew.diameter   = stOld.diameter;               // Core Diameter
        stNew.sCompany   = new String(stOld.sCompany);   // Analyzing Company
        stNew.sAnalyzed  = new String(stOld.sAnalyzed);  // Analysis Date

        // LAS 3 File Parameters

        stNew.iParamRows = stOld.iParamRows;    // Total number of rows
        stNew.iParamCols = stOld.iParamCols;    // Total number of columns
        stNew.sParams    = new String[stNew.iParamRows][stNew.iParamCols];

        for (i=0; i<stNew.iParamRows; i++)
        {
          for (j=0; j<stNew.iParamCols; j++)
          {
            // Array holding the Parameter Definitions
            stNew.sParams[i][j] = new String( stOld.sParams[i][j] );
          }
        }

        stNew.iLogs = stOld.iLogs;  // Total number of LAS3 Data Structures
        if (stOld.iLogs > 0)
        {
          stNew.stLAS3 = new las3Struct[stOld.iLogs];
          for (i=0; i<stOld.iLogs; i++)
          {
            stNew.stLAS3[i] = las3.las3Utility.copyWOData( stOld.stLAS3[i] );
          }
        }

        // Unknown Curve Mnemonics

        stNew.sUnknown = new String[4][2];
        for (i=0; i<4; i++)
        {
		  for (j=0; j<2; j++)
		  {
		    stNew.sUnknown[i][j] = new String( stOld.sUnknown[i][j] );
		  }
	    }

        // -- Depth information

        stNew._TOP      = stOld._TOP; // Top of Rock Depth
        stNew._BASE     = stOld._BASE; // Base of Rock Depth
        stNew._CORR     = stOld._CORR; // Depth Correction

        // -- Descriptive fields to define rock

        stNew._STU      = stOld._STU; // Stratigraphic Unit
        stNew._STN      = stOld._STN; // Stratigraphic Name
        stNew._ENV      = stOld._ENV; // Environment
        stNew._LITHO    = stOld._LITHO; // Lithofacies

        // -- Porosity Data

        stNew._PCORE    = stOld._PCORE; // Porosity whole routine (PU)
        stNew._PPLUG    = stOld._PPLUG; // Porosity Plug routine (PU)
        stNew._P800     = stOld._P800; // Porosity Plug 800 PSI (PU)
        stNew._PINSI    = stOld._PINSI; // Porosity Plug Insitu (PU)
        stNew._PEFF     = stOld._PEFF; // Effective Rock Porosity (PU)

        // -- Permeability Data

        stNew._KMAX     = stOld._KMAX; // Permeability Whole Max (md)
        stNew._K90      = stOld._K90; // Permeability Whole 90 (md)
        stNew._KVRT     = stOld._KVRT; // Permeability Whole Vertical (md)
        stNew._KPLG     = stOld._KPLG; // Permeability Plug Routine (md)
        stNew._KKL      = stOld._KKL; // Permeability Plug Klinkenberg Routine (md)
        stNew._KINSI    = stOld._KINSI; // Permeability Plug Insitu (md)
        stNew._KKLIN    = stOld._KKLIN; // Permeability Plug Klinkenberg Insitu (md)
        stNew._KPVRT    = stOld._KPVRT; // Permeability Plug Vertical (md)

        // -- Saturation Data

        stNew._SOIL     = stOld._SOIL; // Saturation Oil (%)
        stNew._SW       = stOld._SW; // Saturation Water (%)

        // -- Density Data

        stNew._GMCC     = stOld._GMCC; // Grain Density (gm/cc)
        stNew._RHOD     = stOld._RHOD; // Density of Rock Dry (gm/cc)
        stNew._RHOW     = stOld._RHOW; // Density of Rock Wet (gm/cc)

        // -- Archie Constants

        stNew._MAMB     = stOld._MAMB;  // Archie Cementation (M) Ambient
        stNew._MINSI    = stOld._MINSI; // Archie Cementation (M) Insitu
        stNew._NAMB     = stOld._NAMB;  // Archie Saturation (N) Ambient
        stNew._NINSI    = stOld._NINSI; // Archie Saturation (N) Insitu

        // -- misc data

        stNew._LTHCD    = stOld._LTHCD; // Lithofacies Code
        stNew._FRACTURE = stOld._FRACTURE; // Fractures

        // -- Measured Gamma Ray Data

        stNew._GR       = stOld._GR;  // Gamma Ray Value
        stNew._CGR      = stOld._CGR; // Gamma Ray minus Uranium Value
        stNew._PGR      = stOld._PGR; // Pseudo Gamma Ray Value

        stNew._TH       = stOld._TH;  // Thorium Values
        stNew._U        = stOld._U;   // Uranium Values
        stNew._K        = stOld._K;   // Potassium Values

        // Unknown Linear Curves

        stNew._LIN_1    = stOld._LIN_1;     // Linear Curve 1
        stNew._LIN_2    = stOld._LIN_2;     // Linear Curve 2
        stNew._LIN_3    = stOld._LIN_3;     // Linear Curve 3
        stNew._LIN_4    = stOld._LIN_4;     // Linear Curve 4

        // Unknown Log Curves

        stNew._LOG_1    = stOld._LOG_1;     // Log Curve 1
        stNew._LOG_2    = stOld._LOG_2;     // Log Curve 2
        stNew._LOG_3    = stOld._LOG_3;     // Log Curve 3
        stNew._LOG_4    = stOld._LOG_4;     // Log Curve 4

        // .. Linear Curves

        stNew.dLIN_1Min = stOld.dLIN_1Min; // Linear Curve 1 Minimum
        stNew.dLIN_2Min = stOld.dLIN_2Min; // Linear Curve 2 Minimum
        stNew.dLIN_3Min = stOld.dLIN_3Min; // Linear Curve 3 Minimum
        stNew.dLIN_4Min = stOld.dLIN_4Min; // Linear Curve 4 Minimum

        stNew.dLIN_1Max = stOld.dLIN_1Max; // Linear Curve 1 Maximum
        stNew.dLIN_2Max = stOld.dLIN_2Max; // Linear Curve 2 Maximum
        stNew.dLIN_3Max = stOld.dLIN_3Max; // Linear Curve 3 Maximum
        stNew.dLIN_4Max = stOld.dLIN_4Max; // Linear Curve 4 Maximum

        // .. Semilog Curves

        stNew.dLOG_1Min = stOld.dLOG_1Min; // Log Curve 1 Minimum
        stNew.dLOG_2Min = stOld.dLOG_2Min; // Log Curve 2 Minimum
        stNew.dLOG_3Min = stOld.dLOG_3Min; // Log Curve 3 Minimum
        stNew.dLOG_4Min = stOld.dLOG_4Min; // Log Curve 4 Minimum

        stNew.dLOG_1Max = stOld.dLOG_1Max; // Log Curve 1 Maximum
        stNew.dLOG_2Max = stOld.dLOG_2Max; // Log Curve 2 Maximum
        stNew.dLOG_3Max = stOld.dLOG_3Max; // Log Curve 3 Maximum
        stNew.dLOG_4Max = stOld.dLOG_4Max; // Log Curve 4 Maximum
    }

    return (stNew);
  }

  /** Method add()
   * <p> This method will add the Rock Data to an existing list
   * @param  stRock = The Rock Data Structure
   * @param  st     = The Old Rock Data List Data Structure
   * @return st     = The new Rock Data List Data Structure.
   */

  public static rockDataListStruct add( rockDataStruct stRock,
                                        rockDataListStruct st )
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iSource  = -1;
    rockDataListStruct stTemp = null;

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

//    stTemp        = new rockDataListStruct();
    stTemp        = copyRockHeader( st );
    stTemp.stItem = new rockDataStruct[iRecords];

    iSource       = stTemp.iSource;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new rockDataStruct();
    stTemp.stItem[iCount] = copy(stRock);
    iCount++;

    stTemp.iCount = iCount;

//    st        = new rockDataListStruct();
    st        = copyRockHeader( stTemp );
    st.stItem = new rockDataStruct[iCount];
    st.iCount = iCount;

    st.iSource = iSource;

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method copy()
   * <p> This method will copy one rock data structure to another
   * @param  stOld = old rock data structure
   * @return stNew = new rock data structure
   */

  public static rockDataStruct copy (rockDataStruct stOld)
  {
    rockDataStruct stNew = new rockDataStruct();

    if (stOld != null)
    {
      stNew.sKID   = new String(stOld.sKID); // Primary KEY of Record
      stNew.sKEY   = new String(stOld.sKEY); // User defined Primary KEY of Record

      stNew.sProprietary = new String(stOld.sProprietary);  // Proprietary data

      // Descriptive fields to define rock

      stNew.sUNIT   = new String(stOld.sUNIT);   // Stratigraphic Unit
      stNew.sNAME   = new String(stOld.sNAME);   // Stratigraphic Name
      stNew.sENV    = new String(stOld.sENV);    // DEPOSITIONAL_ENVIRONMENT
      stNew.sLITHO  = new String(stOld.sLITHO);  // LITHOFACIES

      // Depth information

      stNew.dTOP   = stOld.dTOP;   // Depth Top of Rock (ft)
      stNew.dBASE  = stOld.dBASE;  // Depth Base of Rock (ft)
      stNew.dCORR  = stOld.dCORR;  // Correction added to depth (ft)

      // Porosity Data

      stNew.dPCORE = stOld.dPCORE; // Porosity whole routine (PU)
      stNew.dPPLUG = stOld.dPPLUG; // Porosity Plug routine (PU)
      stNew.dP800  = stOld.dP800;  // Porosity Plug 800 PSI (PU)
      stNew.dPINSI = stOld.dPINSI; // Porosity Plug Insitu (PU)
      stNew.dPEFF  = stOld.dPEFF;  // Effective Rock Porosity (PU)

      // Permeability Data

      stNew.dKMAX  = stOld.dKMAX;  // Permeability Whole Max (md)
      stNew.dK90   = stOld.dK90;   // Permeability Whole 90 (md)
      stNew.dKVRT  = stOld.dKVRT;  // Permeability Whole Vertical (md)
      stNew.dKPLG  = stOld.dKPLG;  // Permeability Plug Routine (md)
      stNew.dKKL   = stOld.dKKL;   // Permeability Plug Klinkenberg Routine (md)
      stNew.dKINSI = stOld.dKINSI; // Permeability Plug Insitu (md)
      stNew.dKKLIN = stOld.dKKLIN; // Permeability Plug Klinkenberg Insitu (md)
      stNew.dKPVRT = stOld.dKPVRT; // Permeability Plug Vertical (md)

      // Saturation Data

      stNew.dSOIL  = stOld.dSOIL;  // Saturation Oil (%)
      stNew.dSW    = stOld.dSW;    // Saturation Water (%)

      // Density Data

      stNew.dGMCC  = stOld.dGMCC;  // Grain Density (gm/cc)
      stNew.dRHOD  = stOld.dRHOD;  // Density of Rock Dry (gm/cc)
      stNew.dRHOW  = stOld.dRHOW;  // Density of Rock Wet (gm/cc)

      // Archie Constants

      stNew.dMAMB  = stOld.dMAMB;  // Archie Cementation (M) Ambient
      stNew.dMINSI = stOld.dMINSI; // Archie Cementation (M) Insitu
      stNew.dNAMB  = stOld.dNAMB;  // Archie Saturation (N) Ambient
      stNew.dNINSI = stOld.dNINSI; // Archie Saturation (N) Insitu

      // Measured Gamma Ray Data

      stNew.dGR    = stOld.dGR;    // Gamma Ray Value
      stNew.dCGR   = stOld.dCGR;   // Gamma Ray minus Uranimum Value
      stNew.dPGR   = stOld.dPGR;   // Pseudo Gamma Ray Value

      stNew.dTh    = stOld.dTh;    // Thorium Values
      stNew.dU     = stOld.dU;     // Uranium Values
      stNew.dK     = stOld.dK;     // Potassium Values

      // Unknown Linear Curves

      stNew.dLIN_1 = stOld.dLIN_1; // Linear Curve 1
      stNew.dLIN_2 = stOld.dLIN_2; // Linear Curve 2
      stNew.dLIN_3 = stOld.dLIN_3; // Linear Curve 3
      stNew.dLIN_4 = stOld.dLIN_4; // Linear Curve 4

      // Unknown Log Curves

      stNew.dLOG_1 = stOld.dLOG_1; // Log Curve 1
      stNew.dLOG_2 = stOld.dLOG_2; // Log Curve 2
      stNew.dLOG_3 = stOld.dLOG_3; // Log Curve 3
      stNew.dLOG_4 = stOld.dLOG_4; // Log Curve 4

      // misc data

      stNew.iLITH_CD  = stOld.iLITH_CD;              // Lithofacies Code
      stNew.sFracture = new String(stOld.sFracture); // Fractures

      stNew.source    = new String(stOld.source);    // source
      stNew.sDate     = new String(stOld.sDate);     // date of data
      stNew.sComments = new String(stOld.sComments); // Comments

      // Computed Curves for building Lithology

      stNew.dRHOB = stOld.dRHOB;
      stNew.dNPHI = stOld.dNPHI;
    }

    return (stNew);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = rock data list structure
   * @return st = sorted rock data list structure
   */

  public static rockDataListStruct bubbleSort(rockDataListStruct st)
  {
    boolean swappedOnPrevRun  = true;
    rockDataStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].dTOP > st.stItem[i+1].dTOP)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

}