/*
 * @rockDataListStruct.java Version 1.1 11/06/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock;

import rock.rockDataStruct;
import rock.rockColumnsStruct;
import las3.las3Struct;

/** CLASS rockDataListStruct
 *  <p> This Class holds the Rock Data List Information.
 *
 *  @version 1.1 11/06/2007
 *  @author  John R. Victorine
 */

public class rockDataListStruct
{
  public int    iSource = las3.las3Constants._SOURCE_NONE;

  // Stratigraphic Headers Data Information

  public String sKID       = "0";  // Primary Key of the Well
  public String sKEY       = "";   // Primary Key created on user's PC
  public int    iType      = iqstrat.iqstratHeadersStruct.WELL; // Type of Data
  public String sAPI       = "";   // API-Number OR UWI
  public String sName      = "";   // Well Name or Outcrop Name
  public String status     = iqstrat.iqstratStatusStruct.STATUS[0][0]; // Status
  public double dLatitude  = 0.0;  // Latitude of Location
  public double dLongitude = 0.0;  // Longitude of Location
  public double depth      = 0.0;  // Total Depth of Well or Height of Outcrop
  public double dGL        = 0.0;  // Ground Level

  public String source     = "";   // Core Source {S}
  public String sType      = "";   // Core Type {S}
  public String sRecovery  = "";   // Recovery Date (Date Core Cut) {D}
  public double depthStart = 0.0;  // Core Top Depth {F}
  public double depthEnd   = 0.0;  // Core Base Depth {F}
  public double dLength    = 0.0;  // Recovered Amount (Length) {F}
  public String sFormation = "";   // Primary Formation Cored {S}
  public double diameter   = 0.0;  // Core Diameter {F}
  public String sCompany   = "";   // Analyzing Company {S}
  public String sAnalyzed  = "";   // Analysis Date {D}

  // LAS 3 Data Structure without the data section

  public int        iLogs    = 0;
  public las3Struct stLAS3[] = null;

  // Other Core Parameters that are not in the Required List

  public int    iParamRows  = 0;      // Total number of rows
  public int    iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
  public String sParams[][] = null;   // Array holding the Parameter Definitions

  // Rock Data

  public int               iCount      = 0;    // Total Number of Records
  public rockDataStruct    stItem[]    = null; // Array of Rock Data Structures

  // Rock Data Columns Names

  public int               iColumns    = 0;    // Total Number of Columns
  public rockColumnsStruct stColumns[] = null; // Array of Rock Columns Structures

  // Unknown Curve Mnemonics

  public String sUnknown[][] = {
    {"", ""},
    {"", ""},
    {"", ""},
    {"", ""},
  };

  // -- Depth information

  public int    _TOP      = -1; // Top of Rock Depth
  public int    _BASE     = -1; // Base of Rock Depth
  public int    _CORR     = -1; // Depth Correction

  // -- Descriptive fields to define rock

  public int    _STU      = -1; // Stratigraphic Unit
  public int    _STN      = -1; // Stratigraphic Name
  public int    _ENV      = -1; // Environment
  public int    _LITHO    = -1; // Lithofacies

  // -- Porosity Data

  public int    _PCORE    = -1; // Porosity whole routine (PU)
  public int    _PPLUG    = -1; // Porosity Plug routine (PU)
  public int    _P800     = -1; // Porosity Plug 800 PSI (PU)
  public int    _PINSI    = -1; // Porosity Plug Insitu (PU)
  public int    _PEFF     = -1; // Effective Rock Porosity (PU)

  // -- Permeability Data

  public int    _KMAX     = -1; // Permeability Whole Max (md)
  public int    _K90      = -1; // Permeability Whole 90 (md)
  public int    _KVRT     = -1; // Permeability Whole Vertical (md)
  public int    _KPLG     = -1; // Permeability Plug Routine (md)
  public int    _KKL      = -1; // Permeability Plug Klinkenberg Routine (md)
  public int    _KINSI    = -1; // Permeability Plug Insitu (md)
  public int    _KKLIN    = -1; // Permeability Plug Klinkenberg Insitu (md)
  public int    _KPVRT    = -1; // Permeability Plug Vertical (md)

  // -- Saturation Data

  public int    _SOIL     = -1; // Saturation Oil (%)
  public int    _SW       = -1; // Saturation Water (%)

  // -- Density Data

  public int    _GMCC     = -1; // Grain Density (gm/cc)
  public int    _RHOD     = -1; // Density of Rock Dry (gm/cc)
  public int    _RHOW     = -1; // Density of Rock Wet (gm/cc)

  // -- Archie Constants

  public int    _MAMB     = -1; // Archie Cementation (M) Ambient
  public int    _MINSI    = -1; // Archie Cementation (M) Insitu
  public int    _NAMB     = -1; // Archie Saturation (N) Ambient
  public int    _NINSI    = -1; // Archie Saturation (N) Insitu

  // -- misc data

  public int    _LTHCD    = -1; // Lithofacies Code
  public int    _FRACTURE = -1; // Fractures

  // -- Measured Gamma Ray Data

  public int    _GR       = -1; // Gamma Ray Value
  public int    _CGR      = -1; // Gamma Ray minus Uranium Value
  public int    _PGR      = -1; // Pseudo Gamma Ray Value

  public int    _TH       = -1; // Thorium Values
  public int    _U        = -1; // Uranium Values
  public int    _K        = -1; // Potassium Values

  // Unknown Linear Curves

  public int    _LIN_1  = -1;     // Linear Curve 1
  public int    _LIN_2  = -1;     // Linear Curve 2
  public int    _LIN_3  = -1;     // Linear Curve 3
  public int    _LIN_4  = -1;     // Linear Curve 4

  // Unknown Log Curves

  public int    _LOG_1  = -1;     // Log Curve 1
  public int    _LOG_2  = -1;     // Log Curve 2
  public int    _LOG_3  = -1;     // Log Curve 3
  public int    _LOG_4  = -1;     // Log Curve 4

  // .. Linear Curves

  public double dLIN_1Min = 0.0; // Linear Curve 1 Minimum
  public double dLIN_2Min = 0.0; // Linear Curve 2 Minimum
  public double dLIN_3Min = 0.0; // Linear Curve 3 Minimum
  public double dLIN_4Min = 0.0; // Linear Curve 4 Minimum

  public double dLIN_1Max = 1.0; // Linear Curve 1 Maximum
  public double dLIN_2Max = 1.0; // Linear Curve 2 Maximum
  public double dLIN_3Max = 1.0; // Linear Curve 3 Maximum
  public double dLIN_4Max = 1.0; // Linear Curve 4 Maximum

  // .. Semilog Curves

  public double dLOG_1Min = 0.1;    // Log Curve 1 Minimum
  public double dLOG_2Min = 0.1;    // Log Curve 2 Minimum
  public double dLOG_3Min = 0.1;    // Log Curve 3 Minimum
  public double dLOG_4Min = 0.1;    // Log Curve 4 Minimum

  public double dLOG_1Max = 1000.0; // Log Curve 1 Maximum
  public double dLOG_2Max = 1000.0; // Log Curve 2 Maximum
  public double dLOG_3Max = 1000.0; // Log Curve 3 Maximum
  public double dLOG_4Max = 1000.0; // Log Curve 4 Maximum

  /** Method delete()
   * <p> This method will force set the Objects to null and force Java to
   *     free unwanted memory
   */

  public void delete()
  {
    int i = 0;

    sKID   = null; // Primary Key of the Well
    sKEY   = null; // Primary Key created on user's PC
    sAPI   = null; // API-Number OR UWI
    sName  = null; // Well Name or Outcrop Name
    status = null; // Status

    source     = null; // Core Source {S}
    sType      = null; // Core Type {S}
    sRecovery  = null; // Recovery Date (Date Core Cut) {D}
    depthStart = 0.0;  // Core Top Depth {F}
    depthEnd   = 0.0;  // Core Base Depth {F}
    dLength    = 0.0;  // Recovered Amount (Length) {F}
    sFormation = null; // Primary Formation Cored {S}
    diameter   = 0.0;  // Core Diameter {F}
    sCompany   = null; // Analyzing Company {S}
    sAnalyzed  = null; // Analysis Date {D}

    // LAS 3 File Data Structure

    if (iLogs > 0)
    {
      for (i=0; i<iLogs; i++)
      {
        if (stLAS3[i] != null)
          stLAS3[i].delete();
        stLAS3[i] = null;
      }
    }

    // LAS 3 File Parameters

    iParamRows = 0;    // Total number of rows
    iParamCols = 0;    // Total number of columns
    sParams    = null; // Array holding the Parameter Definitions

    // Rock Data

    for (i=0; i<iCount; i++)
    {
      if (stItem[i] != null)
        stItem[i].delete();
      stItem[i] = null;
    }

    iCount = 0;

    // Rock Data Names

    for (i=0; i<iColumns; i++)
    {
      if (stColumns[i] != null)
        stColumns[i].delete();
      stColumns[i] = null;
    }

    iColumns = 0;
  }

  /** Method isPercent(int iCurve)
   * <p> This method will return a true if the porosity value is greater than 2.0
   * @param  iCurve = the requested curve type
   * @return bValue true the data is present and false it is not there
   */

  public boolean isPercent(int iCurve)
  {
    boolean bValue = false;
    double  data[] = null;

    if (checkData(iCurve))
    {
	  data = getData(iCurve);

      for (int i=0; i<iCount; i++)
      {
		if (data[i] > 2.0)
		  bValue = true;
	  }
	}

	return (bValue);
  }

  /** Method checkData(int iCurve)
   * <p> This method will return a true if the data is present
   * @param iCurve = the requested curve type
   * @return bValue true the data is present and false it is not there
   */

  public boolean checkData(int iCurve)
  {
    boolean bValue = false;

    switch (iCurve)
    {
      case rock.rockStandardTools._TOP:
        if (_TOP > -1) bValue = true;
        break;
      case rock.rockStandardTools._BASE:
        if (_BASE > -1) bValue = true;
        break;
      case rock.rockStandardTools._CORR:
        if (_CORR > -1) bValue = true;
        break;

      case rock.rockStandardTools._STU:
        if (_STU > -1) bValue = true;
        break;
      case rock.rockStandardTools._STN:
        if (_STN > -1) bValue = true;
        break;
      case rock.rockStandardTools._ENV:
        if (_ENV > -1) bValue = true;
        break;
      case rock.rockStandardTools._LITHO:
        if (_LITHO > -1) bValue = true;
        break;

      case rock.rockStandardTools._PCORE:
        if (_PCORE > -1) bValue = true;
        break;
      case rock.rockStandardTools._PPLUG:
        if (_PPLUG > -1) bValue = true;
        break;
      case rock.rockStandardTools._P800:
        if (_P800 > -1) bValue = true;
        break;
      case rock.rockStandardTools._PINSI:
        if (_PINSI > -1) bValue = true;
        break;
      case rock.rockStandardTools._PEFF:
        if (_PEFF > -1) bValue = true;
        break;

      case rock.rockStandardTools._KMAX:
        if (_KMAX > -1) bValue = true;
        break;
      case rock.rockStandardTools._K90:
        if (_K90 > -1) bValue = true;
        break;
      case rock.rockStandardTools._KVRT:
        if (_KVRT > -1) bValue = true;
        break;
      case rock.rockStandardTools._KPLG:
        if (_KPLG > -1) bValue = true;
        break;
      case rock.rockStandardTools._KKL:
        if (_KKL > -1) bValue = true;
        break;
      case rock.rockStandardTools._KINSI:
        if (_KINSI > -1) bValue = true;
        break;
      case rock.rockStandardTools._KKLIN:
        if (_KKLIN > -1) bValue = true;
        break;
      case rock.rockStandardTools._KPVRT:
        if (_KPVRT > -1) bValue = true;
        break;

      case rock.rockStandardTools._SOIL:
        if (_SOIL > -1) bValue = true;
        break;
      case rock.rockStandardTools._SW:
        if (_SW > -1) bValue = true;
        break;

      case rock.rockStandardTools._GMCC:
        if (_GMCC > -1) bValue = true;
        break;
      case rock.rockStandardTools._RHOD:
        if (_RHOD > -1) bValue = true;
        break;
      case rock.rockStandardTools._RHOW:
        if (_RHOW > -1) bValue = true;
        break;

      case rock.rockStandardTools._MAMB:
        if (_MAMB > -1) bValue = true;
        break;
      case rock.rockStandardTools._MINSI:
        if (_MINSI > -1) bValue = true;
        break;
      case rock.rockStandardTools._NAMB:
        if (_NAMB > -1) bValue = true;
        break;
      case rock.rockStandardTools._NINSI:
        if (_NINSI > -1) bValue = true;
        break;

      case rock.rockStandardTools._GR:
        if (_GR > -1) bValue = true;
        break;
      case rock.rockStandardTools._CGR:
        if (_CGR > -1) bValue = true;
        break;
      case rock.rockStandardTools._PGR:
        if (_PGR > -1) bValue = true;
        break;

      case rock.rockStandardTools._TH:
        if (_TH > -1) bValue = true;
        break;
      case rock.rockStandardTools._U:
        if (_U > -1) bValue = true;
        break;
      case rock.rockStandardTools._K:
        if (_K > -1) bValue = true;
        break;

      case rock.rockStandardTools._LIN_1:
        if (_LIN_1 > -1) bValue = true;
        break;
      case rock.rockStandardTools._LIN_2:
        if (_LIN_2 > -1) bValue = true;
        break;
      case rock.rockStandardTools._LIN_3:
        if (_LIN_3 > -1) bValue = true;
        break;
      case rock.rockStandardTools._LIN_4:
        if (_LIN_4 > -1) bValue = true;
        break;

      case rock.rockStandardTools._LOG_1:
        if (_LOG_1 > -1) bValue = true;
        break;
      case rock.rockStandardTools._LOG_2:
        if (_LOG_2 > -1) bValue = true;
        break;
      case rock.rockStandardTools._LOG_3:
        if (_LOG_3 > -1) bValue = true;
        break;
      case rock.rockStandardTools._LOG_4:
        if (_LOG_4 > -1) bValue = true;
        break;

      case rock.rockStandardTools._LTHCD:
        if (_LTHCD > -1) bValue = true;
        break;
      case rock.rockStandardTools._FRACTURE:
        if (_FRACTURE > -1) bValue = true;
        break;
    }

    return (bValue);
  }

  /** Method setData(int iRow)
   * <p> This method will return a true if the data is present
   * @param iRow = the rock Data Structure recored
   * @return bValue true the data is present and false it is not there
   */

  public void setData(int iRow)
  {
	int    iCurve = -1;
    double data   = rock.rockDataStruct.dNULL;

    for (int i=0; i<rock.rockStandardTools.TOTAL; i++)
    {
      iCurve = i;
      data   = getData( iCurve, iRow );

      switch (iCurve)
      {
        case rock.rockStandardTools._TOP:
          if (data != rock.rockDataStruct.dNULL) _TOP = 1;
          break;
        case rock.rockStandardTools._BASE:
          if (data != rock.rockDataStruct.dNULL) _BASE = 1;
          break;
        case rock.rockStandardTools._CORR:
          if (data != rock.rockDataStruct.dNULL) _CORR = 1;
          break;

        case rock.rockStandardTools._STU:
          if (stItem[iCurve].sUNIT.length() > 0) _STU = 1;
          break;
        case rock.rockStandardTools._STN:
          if (stItem[iCurve].sNAME.length() > 0) _STN = 1;
          break;
        case rock.rockStandardTools._ENV:
          if (stItem[iCurve].sENV.length()  > 0) _ENV = 1;
          break;
        case rock.rockStandardTools._LITHO:
          if (stItem[iCurve].sLITHO.length() > 0) _LITHO = 1;
          break;

        case rock.rockStandardTools._PCORE:
          if (data != rock.rockDataStruct.dNULL) _PCORE = 1;
          break;
        case rock.rockStandardTools._PPLUG:
          if (data != rock.rockDataStruct.dNULL) _PPLUG = 1;
          break;
        case rock.rockStandardTools._P800:
          if (data != rock.rockDataStruct.dNULL) _P800 = 1;
          break;
        case rock.rockStandardTools._PINSI:
          if (data != rock.rockDataStruct.dNULL) _PINSI = 1;
          break;
        case rock.rockStandardTools._PEFF:
          if (data != rock.rockDataStruct.dNULL) _PEFF = 1;
          break;

        case rock.rockStandardTools._KMAX:
          if (data != rock.rockDataStruct.dNULL) _KMAX = 1;
          break;
        case rock.rockStandardTools._K90:
          if (data != rock.rockDataStruct.dNULL) _K90 = 1;
          break;
        case rock.rockStandardTools._KVRT:
          if (data != rock.rockDataStruct.dNULL) _KVRT = 1;
          break;
        case rock.rockStandardTools._KPLG:
          if (data != rock.rockDataStruct.dNULL) _KPLG = 1;
          break;
        case rock.rockStandardTools._KKL:
          if (data != rock.rockDataStruct.dNULL) _KKL = 1;
          break;
        case rock.rockStandardTools._KINSI:
          if (data != rock.rockDataStruct.dNULL) _KINSI = 1;
          break;
        case rock.rockStandardTools._KKLIN:
          if (data != rock.rockDataStruct.dNULL) _KKLIN = 1;
          break;
        case rock.rockStandardTools._KPVRT:
          if (data != rock.rockDataStruct.dNULL) _KPVRT = 1;
          break;

        case rock.rockStandardTools._SOIL:
          if (data != rock.rockDataStruct.dNULL) _SOIL = 1;
          break;
        case rock.rockStandardTools._SW:
          if (data != rock.rockDataStruct.dNULL) _SW = 1;
          break;

        case rock.rockStandardTools._GMCC:
          if (data != rock.rockDataStruct.dNULL) _GMCC = 1;
          break;
        case rock.rockStandardTools._RHOD:
          if (data != rock.rockDataStruct.dNULL) _RHOD = 1;
          break;
        case rock.rockStandardTools._RHOW:
          if (data != rock.rockDataStruct.dNULL) _RHOW = 1;
          break;

        case rock.rockStandardTools._MAMB:
          if (data != rock.rockDataStruct.dNULL) _MAMB = 1;
          break;
        case rock.rockStandardTools._MINSI:
          if (data != rock.rockDataStruct.dNULL) _MINSI = 1;
          break;
        case rock.rockStandardTools._NAMB:
          if (data != rock.rockDataStruct.dNULL) _NAMB = 1;
          break;
        case rock.rockStandardTools._NINSI:
          if (data != rock.rockDataStruct.dNULL) _NINSI = 1;
          break;

        case rock.rockStandardTools._GR:
          if (data != rock.rockDataStruct.dNULL) _GR = 1;
          break;
        case rock.rockStandardTools._CGR:
          if (data != rock.rockDataStruct.dNULL) _CGR = 1;
          break;
        case rock.rockStandardTools._PGR:
          if (data != rock.rockDataStruct.dNULL) _PGR = 1;
          break;

        case rock.rockStandardTools._TH:
          if (data != rock.rockDataStruct.dNULL) _TH = 1;
          break;
        case rock.rockStandardTools._U:
          if (data != rock.rockDataStruct.dNULL) _U = 1;
          break;
        case rock.rockStandardTools._K:
          if (data != rock.rockDataStruct.dNULL) _K = 1;
          break;

        case rock.rockStandardTools._LIN_1:
          if (data != rock.rockDataStruct.dNULL) _LIN_1 = 1;
          break;
        case rock.rockStandardTools._LIN_2:
          if (data != rock.rockDataStruct.dNULL) _LIN_2 = 1;
          break;
        case rock.rockStandardTools._LIN_3:
          if (data != rock.rockDataStruct.dNULL) _LIN_3 = 1;
          break;
        case rock.rockStandardTools._LIN_4:
          if (data != rock.rockDataStruct.dNULL) _LIN_4 = 1;
          break;

        case rock.rockStandardTools._LOG_1:
          if (data != rock.rockDataStruct.dNULL) _LOG_1 = 1;
          break;
        case rock.rockStandardTools._LOG_2:
          if (data != rock.rockDataStruct.dNULL) _LOG_2 = 1;
          break;
        case rock.rockStandardTools._LOG_3:
          if (data != rock.rockDataStruct.dNULL) _LOG_3 = 1;
          break;
        case rock.rockStandardTools._LOG_4:
          if (data != rock.rockDataStruct.dNULL) _LOG_4 = 1;
          break;

        case rock.rockStandardTools._LTHCD:
          if (data != rock.rockDataStruct.dNULL) _LTHCD = 1;
          break;
        case rock.rockStandardTools._FRACTURE:
          if (data != rock.rockDataStruct.dNULL) _FRACTURE = 1;
          break;
	  }
    }
  }

  /** Method getData(int iCurve)
   * <p> This method will retrive the curve data requested
   * @param  iCurve = the requested curve type
   * @return data   = the data requested.
   */

  public double[] getData(int iCurve)
  {
    double data[] = null;

    data = new double[iCount];

    for (int i=0; i<iCount; i++)
    {
      switch (iCurve)
      {
        case rock.rockStandardTools._TOP:
          data[i] = stItem[i].dTOP;
          break;
        case rock.rockStandardTools._BASE:
          data[i] = stItem[i].dBASE;
          break;
        case rock.rockStandardTools._CORR:
          data[i] = stItem[i].dCORR;
          break;

        case rock.rockStandardTools._STU:
        case rock.rockStandardTools._STN:
        case rock.rockStandardTools._ENV:
        case rock.rockStandardTools._LITHO:
          break;

        case rock.rockStandardTools._PCORE:
          data[i] = stItem[i].dPCORE;
          break;
        case rock.rockStandardTools._PPLUG:
          data[i] = stItem[i].dPPLUG;
          break;
        case rock.rockStandardTools._P800:
          data[i] = stItem[i].dP800;
          break;
        case rock.rockStandardTools._PINSI:
          data[i] = stItem[i].dPINSI;
          break;
        case rock.rockStandardTools._PEFF:
          data[i] = stItem[i].dPEFF;
          break;

        case rock.rockStandardTools._KMAX:
          data[i] = stItem[i].dKMAX;
          break;
        case rock.rockStandardTools._K90:
          data[i] = stItem[i].dK90;
          break;
        case rock.rockStandardTools._KVRT:
          data[i] = stItem[i].dKVRT;
          break;
        case rock.rockStandardTools._KPLG:
          data[i] = stItem[i].dKPLG;
          break;
        case rock.rockStandardTools._KKL:
          data[i] = stItem[i].dKKL;
          break;
        case rock.rockStandardTools._KINSI:
          data[i] = stItem[i].dKINSI;
          break;
        case rock.rockStandardTools._KKLIN:
          data[i] = stItem[i].dKKLIN;
          break;
        case rock.rockStandardTools._KPVRT:
          data[i] = stItem[i].dKPVRT;
          break;

        case rock.rockStandardTools._SOIL:
          data[i] = stItem[i].dSOIL;
          break;
        case rock.rockStandardTools._SW:
          data[i] = stItem[i].dSW;
          break;

        case rock.rockStandardTools._GMCC:
          data[i] = stItem[i].dGMCC;
          break;
        case rock.rockStandardTools._RHOD:
          data[i] = stItem[i].dRHOD;
          break;
        case rock.rockStandardTools._RHOW:
          data[i] = stItem[i].dRHOW;
          break;

        case rock.rockStandardTools._MAMB:
          data[i] = stItem[i].dMAMB;
          break;
        case rock.rockStandardTools._MINSI:
          data[i] = stItem[i].dMINSI;
          break;
        case rock.rockStandardTools._NAMB:
          data[i] = stItem[i].dNAMB;
          break;
        case rock.rockStandardTools._NINSI:
          data[i] = stItem[i].dNINSI;
          break;

        case rock.rockStandardTools._GR:
          data[i] = stItem[i].dGR;
          break;
        case rock.rockStandardTools._CGR:
          data[i] = stItem[i].dCGR;
          break;
        case rock.rockStandardTools._PGR:
          data[i] = stItem[i].dPGR;
          break;

        case rock.rockStandardTools._TH:
          data[i] = stItem[i].dTh;
          break;
        case rock.rockStandardTools._U:
          data[i] = stItem[i].dU;
          break;
        case rock.rockStandardTools._K:
          data[i] = stItem[i].dK;
          break;

        case rock.rockStandardTools._LIN_1:
          data[i] = stItem[i].dLIN_1;
          break;
        case rock.rockStandardTools._LIN_2:
          data[i] = stItem[i].dLIN_2;
          break;
        case rock.rockStandardTools._LIN_3:
          data[i] = stItem[i].dLIN_3;
          break;
        case rock.rockStandardTools._LIN_4:
          data[i] = stItem[i].dLIN_4;
          break;

        case rock.rockStandardTools._LOG_1:
          data[i] = stItem[i].dLOG_1;
          break;
        case rock.rockStandardTools._LOG_2:
          data[i] = stItem[i].dLOG_2;
          break;
        case rock.rockStandardTools._LOG_3:
          data[i] = stItem[i].dLOG_3;
          break;
        case rock.rockStandardTools._LOG_4:
          data[i] = stItem[i].dLOG_4;
          break;

        case rock.rockStandardTools._LTHCD:
          data[i] = (double) stItem[i].iLITH_CD;
          break;
        case rock.rockStandardTools._FRACTURE:
          break;
      }
    }

    return (data);
  }

  /** Method getData(int iCurve, int iRow)
   * <p> This method will retrive the curve data requested
   * @param  iCurve = the requested curve type
   * @param  iRow   = the row
   * @return data   = the data requested.
   */

  public double getData( int iCurve, int iRow )
  {
    double data = rock.rockDataStruct.dNULL;

    switch (iCurve)
    {
      case rock.rockStandardTools._TOP:
        data = stItem[iRow].dTOP;
        break;
      case rock.rockStandardTools._BASE:
        data = stItem[iRow].dBASE;
        break;
      case rock.rockStandardTools._CORR:
        data = stItem[iRow].dCORR;
        break;

      case rock.rockStandardTools._STU:
      case rock.rockStandardTools._STN:
      case rock.rockStandardTools._ENV:
      case rock.rockStandardTools._LITHO:
        break;

      case rock.rockStandardTools._PCORE:
        data = stItem[iRow].dPCORE;
        break;
      case rock.rockStandardTools._PPLUG:
        data = stItem[iRow].dPPLUG;
        break;
      case rock.rockStandardTools._P800:
        data = stItem[iRow].dP800;
        break;
      case rock.rockStandardTools._PINSI:
        data = stItem[iRow].dPINSI;
        break;
      case rock.rockStandardTools._PEFF:
        data = stItem[iRow].dPEFF;
        break;

      case rock.rockStandardTools._KMAX:
        data = stItem[iRow].dKMAX;
        break;
      case rock.rockStandardTools._K90:
        data = stItem[iRow].dK90;
        break;
      case rock.rockStandardTools._KVRT:
        data = stItem[iRow].dKVRT;
        break;
      case rock.rockStandardTools._KPLG:
        data = stItem[iRow].dKPLG;
        break;
      case rock.rockStandardTools._KKL:
        data = stItem[iRow].dKKL;
        break;
      case rock.rockStandardTools._KINSI:
        data = stItem[iRow].dKINSI;
        break;
      case rock.rockStandardTools._KKLIN:
        data = stItem[iRow].dKKLIN;
        break;
      case rock.rockStandardTools._KPVRT:
        data = stItem[iRow].dKPVRT;
        break;

      case rock.rockStandardTools._SOIL:
        data = stItem[iRow].dSOIL;
        break;
      case rock.rockStandardTools._SW:
        data = stItem[iRow].dSW;
        break;

      case rock.rockStandardTools._GMCC:
        data = stItem[iRow].dGMCC;
        break;
      case rock.rockStandardTools._RHOD:
        data = stItem[iRow].dRHOD;
        break;
      case rock.rockStandardTools._RHOW:
        data = stItem[iRow].dRHOW;
        break;

      case rock.rockStandardTools._MAMB:
        data = stItem[iRow].dMAMB;
        break;
      case rock.rockStandardTools._MINSI:
        data = stItem[iRow].dMINSI;
        break;
      case rock.rockStandardTools._NAMB:
        data = stItem[iRow].dNAMB;
        break;
      case rock.rockStandardTools._NINSI:
        data = stItem[iRow].dNINSI;
        break;

      case rock.rockStandardTools._GR:
        data = stItem[iRow].dGR;
        break;
      case rock.rockStandardTools._CGR:
        data = stItem[iRow].dCGR;
        break;
      case rock.rockStandardTools._PGR:
        data = stItem[iRow].dPGR;
        break;

      case rock.rockStandardTools._TH:
        data = stItem[iRow].dTh;
        break;
      case rock.rockStandardTools._U:
        data = stItem[iRow].dU;
        break;
      case rock.rockStandardTools._K:
        data = stItem[iRow].dK;
        break;

      case rock.rockStandardTools._LIN_1:
        data = stItem[iRow].dLIN_1;
        break;
      case rock.rockStandardTools._LIN_2:
        data = stItem[iRow].dLIN_2;
        break;
      case rock.rockStandardTools._LIN_3:
        data = stItem[iRow].dLIN_3;
        break;
      case rock.rockStandardTools._LIN_4:
        data = stItem[iRow].dLIN_4;
        break;

      case rock.rockStandardTools._LOG_1:
        data = stItem[iRow].dLOG_1;
        break;
      case rock.rockStandardTools._LOG_2:
        data = stItem[iRow].dLOG_2;
        break;
      case rock.rockStandardTools._LOG_3:
        data = stItem[iRow].dLOG_3;
        break;
      case rock.rockStandardTools._LOG_4:
        data = stItem[iRow].dLOG_4;
        break;

      case rock.rockStandardTools._LTHCD:
        data = (double) stItem[iRow].iLITH_CD;
        break;
      case rock.rockStandardTools._FRACTURE:
        break;
    }

    return (data);
  }

  public void print()
  {
    if (iCount > 0)
    {
      for (int i=0; i<iCount; i++)
      {
        stItem[i].print();
      }
    }
  }
}
