/*
 * @rockPlotTrack.java Version 1.1 11/06/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.plot;

import java.awt.*;
import java.awt.geom.*;

import rock.rockDataListStruct;

/** CLASS rockPlotTrack
 *  <p> This Class will create the IQSTRAT - Stratigraphic Plot Rock Tracks.
 *
 *  @version 1.1 11/06/2007
 *  @author  John R. Victorine
 */

public class rockPlotTrack extends Canvas
{
  // Input Variables

  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables

  private double dIncrementDepth  = 0.0;
  private int    iStartDepthLabel = 10;

  public static final Color cRED     = new Color( 180,   0,   0 );
  public static final Color cGREEN   = new Color( 0,   180,   0 );
  public static final Color cBLUE    = new Color( 0,     0, 180 );
  public static final Color cCYAN    = new Color(  0,  180, 180 );
  public static final Color cMAGENTA = new Color( 180,   0, 180 );
  public static final Color cYELLOW  = new Color( 180, 180,   0 );
  public static final Color cORANGE  = new Color( 204, 102,   0 );
  public static final Color cBROWN   = new Color( 102, 102,   0 );
//  public static final Color cBROWN   = new Color(  51,  51,   0 );

  public static final Color COLORS[] =  // Color for Log Curves
      { cRED, cGREEN, cBLUE, cBROWN, cMAGENTA, cORANGE, cYELLOW, cCYAN };

//  public static final Color COLORS[] =   // Color for Log Curves
//            {Color.red,     Color.green,  Color.blue,   Color.cyan,
//             Color.magenta, Color.yellow, Color.orange, Color.pink};

  private rockDataListStruct stRock  = null; // Rock Data List Structure

  private double dLIMITS[][] = null; // Minimum & Maximum Limits to CORE Plot Track

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR rockPlotTrack( int iDataType,
   *                             double depthStart,
   *                             double depthEnd,
   *                             int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   */

  public rockPlotTrack( int iDataType,
                        double depthStart,
                        double depthEnd,
                        int iScale )
  {
    this.iDataType = iDataType;
    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
	stRock  = null;
    dLIMITS = null; // Minimum & Maximum Limits to CORE Plot Track
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
    dIncrementDepth = iqstrat.iqstratTracksStruct.SCALE[iScale];
  }

  /** Method setRockData( rockDataListStruct stRock )
   * <p> This method will set the LAS File Data Structure
   * @param stRock - LAS File Data Structure
   */

  public void setRockData(rockDataListStruct stRock) { this.stRock = stRock; }

  /** Method setPlotLimits()
   *  <p> This method will override the default Plot Limits
   *  @param dLIMITS = Array of Minimum & Maximum Plot Limits
   */

  public void setPlotLimits( double dLIMITS[][] ) {	this.dLIMITS = dLIMITS; }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g,
   *                   int iROCK,
   *                   int iStartTrack,
   *                   int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iROCK       = The LAS Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g,
                        int iROCK,
                        int iStartTrack,
                        int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iCycle = 1;
    double dXinc  = 0.0;
    double dTemp  = 0.0;
    int    iIncrement  = iWidthTrack/4;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iTrack      = iROCK - iqstrat.iqstratTracksStruct.ROCK_START;
    int    iCurve      = iqstrat.iqstratTracksStruct.ROCK_TRACK[iTrack][0];
    int    iValue      = 0;
    int    iY1         = 0;
    int    iY2         = 0;

    double dMinimum = cmn.cmnString.stringToDouble(
                        rock.rockStandardTools.ROCK_TOOLS[iCurve][4]);
    double dMaximum = cmn.cmnString.stringToDouble(
                        rock.rockStandardTools.ROCK_TOOLS[iCurve][5]);

    if (dLIMITS != null)
    {
	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	  dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	}

    if (rock.rockStandardTools.ROCK_TOOLS[iCurve][2].equals("MD"))
    {
       iLog   = 1;
       dMinimum = Math.round(Math.log(dMinimum) / Math.log(10));
       dMaximum = Math.round(Math.log(dMaximum) / Math.log(10));
       iCycle = (int) dMaximum - (int) dMinimum;
    }

    if (rock.rockStandardTools.ROCK_TOOLS[iCurve][2].equals("UNL"))
    {
      if (iCurve == rock.rockStandardTools._LOG_1)
      {
        dMinimum = stRock.dLOG_1Min;
        dMaximum = stRock.dLOG_1Max;

        if (dLIMITS != null)
        {
    	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
    	  dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
    	}
      }

      if (iCurve == rock.rockStandardTools._LOG_2)
      {
        dMinimum = stRock.dLOG_2Min;
        dMaximum = stRock.dLOG_2Max;

        if (dLIMITS != null)
        {
    	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
    	  dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
    	}
      }

      if (iCurve == rock.rockStandardTools._LOG_3)
      {
        dMinimum = stRock.dLOG_3Min;
        dMaximum = stRock.dLOG_3Max;

        if (dLIMITS != null)
        {
    	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
    	  dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
    	}
      }

      if (iCurve == rock.rockStandardTools._LOG_4)
      {
        dMinimum = stRock.dLOG_4Min;
        dMaximum = stRock.dLOG_4Max;

        if (dLIMITS != null)
        {
    	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
    	  dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
    	}
      }

      iLog = 1;
      dMinimum = Math.round(Math.log(dMinimum) / Math.log(10));
      dMaximum = Math.round(Math.log(dMaximum) / Math.log(10));
      iCycle = (int) dMaximum - (int) dMinimum;
    }

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES)  && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    if (iLog == 0)
    {
      g.setColor(Color.gray);
      g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

      for (i=0; i<=iWidthTrack; i+=iIncrement)
      {
        g.setColor(Color.gray);
        if ((i==0) || (i==iWidthTrack))
        {
          g.drawLine(i+iStartTrack, LABELSTART,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }
        else
        {
          g.drawLine(i+iStartTrack, PLOT_TITLES,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }

//        g.setColor(Color.lightGray);
        g.setColor(new Color(230, 230, 230));
        for (j=1; j<5; j++)
        {
          dTemp = (double) iIncrement / 5.0;
          jXinc = i + iStartTrack + j * (int) dTemp;

          if (jXinc <= iEndTrack)
          {
            g.drawLine(jXinc, PLOT_TITLES,
                       jXinc, PLOT_TITLES+iLogHeight); //iHeight);
          }
        }
      }
    }
    else
    {
      iIncrement = iWidthTrack / iCycle;

      g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

      for (i=0; i<iWidthTrack; i+=iIncrement)
      {
        g.setColor(Color.gray);
        if (i==0)
        {
          g.drawLine(i+iStartTrack, LABELSTART,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }

        g.setColor(Color.gray);
//        g.drawLine(i+iStartTrack, HEIGHT_START,
        g.drawLine(i+iStartTrack, PLOT_TITLES,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);

        g.setColor(Color.lightGray);
        for (j=1; j<5; j++)
        {
          dXinc = (double) j * 2.0;
          dTemp = (double) iIncrement * Math.log(dXinc) / Math.log(10.0);
          jXinc = i + iStartTrack + (int) dTemp;

          if (jXinc <= iEndTrack)
          {
            g.drawLine(jXinc, PLOT_TITLES,
                       jXinc, PLOT_TITLES+iLogHeight); //iHeight);
          }
        }
      }
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawTrackLabels( Graphics g,
   *                          int iROCK,
   *                          int iStartTrack,
   *                          int iWidthTrack )
   *  <p> This method will draw the labels on the LAS Plot.
   *   @param g           = Graphics Parameter.
   *   @param iROCK       = The LAS Track to draw
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTrackLabels( Graphics g,
                               int iROCK,
                               int iStartTrack,
                               int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    iCurve      = -1;
    int    iContinue   = 0;
    int    iLine       = -1;
    int    length      = 0;
    String sLabel      = "";
    String sTemp       = "";
    String sUnits      = "";
    double dMinimum    = 0.0;
    double dMaximum    = 0.0;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iTrack      = iROCK - iqstrat.iqstratTracksStruct.ROCK_START;
    double dLog[]      = null;

    Font fsb = new Font("Serif", Font.BOLD, 10);
    g.setFont( fsb );

    for (i=0; i<5; i++)
    {
      iCurve = iqstrat.iqstratTracksStruct.ROCK_TRACK[iTrack][i];

      iContinue = 1;
      if (stRock != null)
      {
        if (!stRock.checkData(iCurve))
        {
          iContinue = 0;
        }
        else
        {
          iLine++;
        }
      }

      if ((iCurve > -1) && (iContinue == 1))
      {
        sLabel = new String(rock.rockStandardTools.ROCK_TOOLS[iCurve][0]);
        length = sLabel.length();

        sUnits = new String(rock.rockStandardTools.ROCK_TOOLS[iCurve][2]);
        if ((sUnits.equals("PU")) || (sUnits.equals("USEC/FT")))
        {
          dMaximum = cmn.cmnString.stringToDouble(
              rock.rockStandardTools.ROCK_TOOLS[iCurve][4]);
          dMinimum = cmn.cmnString.stringToDouble(
              rock.rockStandardTools.ROCK_TOOLS[iCurve][5]);

          if (dLIMITS != null)
          {
	        dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	        dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	      }

          if (stRock != null)
          {
            dLog = stRock.getData(iCurve);

            if (sUnits.equals("PU"))
            {
              for (j=0; j<stRock.iCount; j++)
              {
                if ((dLog[j] > 2.0) &&
                    ((iCurve >= rock.rockStandardTools._PCORE) &&
                     (iCurve <= rock.rockStandardTools._PEFF)))
                {
                  dMinimum = 30.0;
                  dMaximum = -10.0;

                  if (dLIMITS != null)
                  {
        	        dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
        	        dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
         	      }
                }
              }
            }
          }
        }
        else
        {
          dMinimum = cmn.cmnString.stringToDouble(
              rock.rockStandardTools.ROCK_TOOLS[iCurve][4]);
          dMaximum = cmn.cmnString.stringToDouble(
              rock.rockStandardTools.ROCK_TOOLS[iCurve][5]);

          if (dLIMITS != null)
          {
      	    dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	      }
        }

        if (sUnits.equals("UNI"))
        {
          if (iCurve == rock.rockStandardTools._LIN_1)
          {
            sLabel = new String(stRock.sUnknown[0][0]);
            length = sLabel.length();
            dMinimum = stRock.dLIN_1Min;
            dMaximum = stRock.dLIN_1Max;

            if (dLIMITS != null)
            {
        	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	          dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	        }
          }

          if (iCurve == rock.rockStandardTools._LIN_2)
          {
            sLabel = new String(stRock.sUnknown[1][0]);
            length = sLabel.length();
            dMinimum = stRock.dLIN_2Min;
            dMaximum = stRock.dLIN_2Max;

            if (dLIMITS != null)
            {
        	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	          dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	        }
          }

          if (iCurve == rock.rockStandardTools._LIN_3)
          {
            sLabel = new String(stRock.sUnknown[2][0]);
            length = sLabel.length();
            dMinimum = stRock.dLIN_3Min;
            dMaximum = stRock.dLIN_3Max;

            if (dLIMITS != null)
            {
        	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	          dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	        }
          }

          if (iCurve == rock.rockStandardTools._LIN_4)
          {
            sLabel = new String(stRock.sUnknown[3][0]);
            length = sLabel.length();
            dMinimum = stRock.dLIN_4Min;
            dMaximum = stRock.dLIN_4Max;

            if (dLIMITS != null)
            {
        	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	          dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	        }
          }
        }

        if (sUnits.equals("UNL"))
        {
          if (iCurve == rock.rockStandardTools._LOG_1)
          {
            sLabel = new String(stRock.sUnknown[0][1]);
            length = sLabel.length();
            dMinimum = stRock.dLOG_1Min;
            dMaximum = stRock.dLOG_1Max;

            if (dLIMITS != null)
            {
        	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	          dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	        }
          }

          if (iCurve == rock.rockStandardTools._LOG_2)
          {
            sLabel = new String(stRock.sUnknown[1][1]);
            length = sLabel.length();
            dMinimum = stRock.dLOG_2Min;
            dMaximum = stRock.dLOG_2Max;

            if (dLIMITS != null)
            {
        	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	          dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	        }
          }

          if (iCurve == rock.rockStandardTools._LOG_3)
          {
            sLabel = new String(stRock.sUnknown[2][1]);
            length = sLabel.length();
            dMinimum = stRock.dLOG_3Min;
            dMaximum = stRock.dLOG_3Max;

            if (dLIMITS != null)
            {
        	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	          dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	        }
          }

          if (iCurve == rock.rockStandardTools._LOG_4)
          {
            sLabel = new String(stRock.sUnknown[3][1]);
            length = sLabel.length();
            dMinimum = stRock.dLOG_4Min;
            dMaximum = stRock.dLOG_4Max;

            if (dLIMITS != null)
            {
        	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	          dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	        }
          }
        }

        if (iLine > -1)
        {
          g.setColor(COLORS[i]);
          g.drawString(sLabel,
                       (iStartTrack + iWidthTrack / 2 - length * 4),
                       LABELSTART + (iLine + 2) * 14);

          g.drawString("" + dMinimum,
                       iStartTrack + 1,
                       LABELSTART + (iLine + 2) * 14);

          sTemp = new String("" + dMaximum);
          length = sTemp.length();
          g.drawString("" + dMaximum, iEndTrack - length * 5,
                       LABELSTART + (iLine + 2) * 14);
        }
      }
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);

    sLabel = new String( iqstrat.iqstratTracksStruct.ROCK_TRACK_DESC[iTrack] );
//    sLabel = new String("Rock Data Log Curves");
    length = sLabel.length();
//    g.drawString(sLabel, iStartTrack+7, LABELSTART+14);
    g.drawString(sLabel, iStartTrack+iWidthTrack/2-(9*length)/4, LABELSTART+14);
  }

  /** Method drawTrackCurves( Graphics g,
   *                          int iROCK,
   *                          int iStartTrack,
   *                          int iWidthTrack )
   *  <p> This method will draw the curves on the LAS Plot.
   *   @param g           = Graphics Parameter.
   *   @param iROCK       = The LAS Track to draw
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTrackCurves( Graphics g,
                               int iROCK,
                               int iStartTrack,
                               int iWidthTrack )
  {
    int    j = 0;
    int    iColumns    = 0;
    int    iPlot       = 0;
    int    iX1         = 0;
    int    iX2         = 0;
    int    iY1         = 0;
    int    iY2         = 0;
    int    iLog        = 0;
    int    iColor      = 0;
    double dRatio      = 0.0;
    double dMinimum    = 0.0;
    double dMaximum    = 0.0;
    double dMin        = 0.0;
    double dMax        = 0.0;
    String sUnits      = "";
    double dLog[]      = null;
    double depth       = 0.0;
    double data        = 0.0;
    double data1       = 0.0;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iTrack      = iROCK - iqstrat.iqstratTracksStruct.ROCK_START;
    int    iCurve      = iqstrat.iqstratTracksStruct.ROCK_TRACK[iTrack][0];

    if (rock.rockStandardTools.ROCK_TOOLS[iCurve][2].equals("MD"))
    {
      iLog = 1;
      dMin = cmn.cmnString.stringToDouble(
                rock.rockStandardTools.ROCK_TOOLS[iCurve][4]);
      dMax = cmn.cmnString.stringToDouble(
                rock.rockStandardTools.ROCK_TOOLS[iCurve][5]);

      if (dLIMITS != null)
      {
   	    dMin = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	    dMax = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	  }

      dMin = Math.round(Math.log(dMin) / Math.log(10));
      dMax = Math.round(Math.log(dMax) / Math.log(10));
    }

    if (rock.rockStandardTools.ROCK_TOOLS[iCurve][2].equals("UNL"))
    {
      if (iCurve == rock.rockStandardTools._LOG_1)
      {
        dMin = stRock.dLOG_1Min;
        dMax = stRock.dLOG_1Max;
      }

      if (iCurve == rock.rockStandardTools._LOG_2)
      {
        dMin = stRock.dLOG_2Min;
        dMax = stRock.dLOG_2Max;
      }

      if (iCurve == rock.rockStandardTools._LOG_3)
      {
        dMin = stRock.dLOG_3Min;
        dMax = stRock.dLOG_3Max;
      }

      if (iCurve == rock.rockStandardTools._LOG_4)
      {
        dMin = stRock.dLOG_4Min;
        dMax = stRock.dLOG_4Max;
      }

      if (dLIMITS != null)
      {
    	dMin = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	    dMax = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	  }

//System.out.println("a "+dMin+" "+dMax);
      iLog = 1;
      dMin = Math.round(Math.log(dMin) / Math.log(10));
      dMax = Math.round(Math.log(dMax) / Math.log(10));
//System.out.println("b "+dMin+" "+dMax);
    }

    if (stRock != null)
    {
      for (int i=0; i<5; i++)
      {
        dLog   = null;
        iCurve = iqstrat.iqstratTracksStruct.ROCK_TRACK[iTrack][i];

        if ((iCurve > -1) && (stRock.checkData(iCurve)))
        {
          iColor = i;
          iColumns = stRock.iCount;

          dLog = stRock.getData(iCurve);

          if (dLog != null)
          {
            dMinimum = cmn.cmnString.stringToDouble(
                rock.rockStandardTools.ROCK_TOOLS[iCurve][4]);
            dMaximum = cmn.cmnString.stringToDouble(
                rock.rockStandardTools.ROCK_TOOLS[iCurve][5]);
            sUnits = new String(rock.rockStandardTools.ROCK_TOOLS[iCurve][2]);

            if (sUnits.equals("PU"))
            {
              for (j=0; j<iColumns; j++)
              {
                if ((dLog[j] > 2.0) &&
                    ((iCurve >= rock.rockStandardTools._PCORE) &&
                     (iCurve <= rock.rockStandardTools._PEFF)))
                {
                  dMinimum = -10.0;
                  dMaximum = 30.0;
                }
              }
            }

            if (iCurve == rock.rockStandardTools._LIN_1)
            {
              dMinimum = stRock.dLIN_1Min;
              dMaximum = stRock.dLIN_1Max;
            }

            if (iCurve == rock.rockStandardTools._LIN_2)
            {
              dMinimum = stRock.dLIN_2Min;
              dMaximum = stRock.dLIN_2Max;
            }

            if (iCurve == rock.rockStandardTools._LIN_3)
            {
              dMinimum = stRock.dLIN_3Min;
              dMaximum = stRock.dLIN_3Max;
            }

            if (iCurve == rock.rockStandardTools._LIN_4)
            {
              dMinimum = stRock.dLIN_4Min;
              dMaximum = stRock.dLIN_4Max;
            }

            if (dLIMITS != null)
            {
        	  dMinimum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MINIMUM];
	          dMaximum = dLIMITS[iCurve][rock.rockPlotLimitsUtility.MAXIMUM];
	        }

            if (iLog == 1)
            {
              dMinimum = dMin;
              dMaximum = dMax;
            }

            for (j=0; j<iColumns; j++)
            {
              if (dLog[j] != rock.rockDataStruct.dNULL)
              {
                depth = stRock.stItem[j].dTOP;
                if (stRock.stItem[j].dCORR != rock.rockDataStruct.dNULL)
                  depth = stRock.stItem[j].dTOP + stRock.stItem[j].dCORR;
                data  = dLog[j];

                if (iLog == 1)
                {
                  if (data <= 0.0)  data = .00001;
                  data = Math.log(data) / Math.log(10);
                }

                dRatio = (double) iLogHeight * (depth - depthStart) /
                    (depthEnd - depthStart);
                iY1 = PLOT_TITLES + (int) dRatio;
                if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                  iY1 = iHeight - (int) dRatio;

                iPlot = 0;
                if ( (iY1 > PLOT_TITLES) && (iY1 < iHeight))
                {
                  iPlot = 1;
                }

                if ( (sUnits.equals("PU")) || (sUnits.equals("USEC/FT")))
                {
                  dRatio = (double) iWidthTrack *
                      (dMaximum - data) / (dMaximum - dMinimum);
                  iX1 = iStartTrack + (int) dRatio;
                }
                else
                {
                  dRatio = (double) iWidthTrack *
                      (data - dMinimum) / (dMaximum - dMinimum);
                  iX1 = iStartTrack + (int) dRatio;
                }

                if ((iX1 >= iStartTrack) && (iX1 <= iEndTrack))
                {
                  g.setColor(COLORS[iColor]);
                  if (iPlot == 1)
                    g.fillRect(iX1-2, iY1-2, 4, 4);
                }
              }
            }
          }
        }
      }
    }
  }

  /** Method draw( Graphics g, int iROCK, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iROCK       = The LAS Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw( Graphics g, int iROCK, int iStartTrack, int iWidthTrack )
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;

    g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][2] ) );
    g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

    drawGrid( g, iROCK, iStartTrack, iWidthTrack );
    drawTrackLabels( g, iROCK, iStartTrack, iWidthTrack );
    drawTrackCurves( g, iROCK, iStartTrack, iWidthTrack );
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}
