/*
 * @rockPlotLimitsFrame.java Version 1.1 04/15/2016
 *
 * Copyright (c) 2016 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.plot;

import java.awt.*;
import java.awt.event.*;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

/** Class rockPlotLimitsFrame
 *  <p> This Class will allow the user to change the Scale on the LAS & PfEFFER Plot
 *      track
 *
 *  @version 1.1 04/15/2016
 *  @author  John R. Victorine
 */

public class rockPlotLimitsFrame extends JFrame implements ActionListener
{
  private static final int   _C_CORE    = iqstrat.iqstratControlStruct._C_CORE;

  private static final int   MINIMUM    = las.lasPlotLimitsUtility.MINIMUM;
  private static final int   MAXIMUM    = las.lasPlotLimitsUtility.MAXIMUM;
  private static final int   GROUP      = las.lasPlotLimitsUtility.GROUP;
  private static final int   DATA       = las.lasPlotLimitsUtility.LAS;
  private static final int   COLUMNS    = las.lasPlotLimitsUtility.COLUMNS;

  private static final Color cDEPTH     = new Color( 165,  42,  42 ); // brown
  private static final Color cOHM       = new Color( 255, 165,   0 ); // orange
  private static final Color cVSH       = new Color( 150, 150, 200 ); // slate blue
  private static final Color cPHI       = new Color(   0, 255, 255 ); // cyan
  private static final Color cGR        = new Color( 255,   0,   0 ); // red
  private static final Color cRHOMAA    = new Color(   0, 255,   0 ); // green
  private static final Color cDTMAA     = new Color(  34, 139,  34 ); // forest green
  private static final Color cUMAA      = new Color( 173, 255,  47 ); // green yellow
  private static final Color cMD        = new Color(   0,   0, 255 ); // blue
  private static final Color cDEGF      = new Color( 255, 105, 180 ); // hot pink
  private static final Color cFRAC      = new Color( 255, 255, 128 ); // middle yellow
  private static final Color cUNI       = new Color( 148,   0, 211 ); // dark violet
  private static final Color cUNL       = new Color( 148, 130, 238 ); // violet
  private static final Color cMISC      = new Color( 255, 255,   0 ); // yellow

  // Input Variables

  private Observable notifier = null; // Observable

  private int        iData        = _C_CORE;

  private double     orig_core[][] = null; // Original Plot Limits for CORE
  private double     data_core[][] = null; // Plot Limits for CORE

  // Global Frame Variables

  // Global Frame Widgets

  private JButton    btnClose     = null; // Close Dialog

  private JTextField txtCORE[][]  = null; // Array of CORE Curves Minimum & Maximum

  private JButton    btnCORE[]    = null; // Array of Reset Buttons
  private JButton    btnXCORE[]   = null; // Array of change in Limits Buttons

  /** Constructor rockPlotLimitsFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param orig     = Original Plot Track Limits
   *  @param data     = Modified Plot Track Limits
   */

  public rockPlotLimitsFrame( Observable notifier,
                              double orig[][],  double data[][] )
  {
    try
    {
      this.notifier  = notifier;
      this.orig_core = orig;
      this.data_core = data;

      if (data != null)
        jbInit();

      setColor(_C_CORE);

      addWindowListener( new rockPlotLimitsFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlTop       = new JPanel();
    JPanel pnlCenter    = new JPanel();

    // Build Frame

    this.setTitle("Change Plot Limits");
    this.getContentPane().setLayout(new BorderLayout());

    // Top Panel

    pnlTop.setLayout(new BorderLayout());

    btnClose = new JButton(createImageIcon(cmn.cmnStruct.CLOSE));
    btnClose.setPreferredSize(new Dimension(27, 27));
    btnClose.setToolTipText("Close");
    btnClose.addActionListener(this);

    // Center Panel

    pnlCenter = buildCOREPanel();

    // Attach Widgets to frame

    this.getContentPane().add(pnlTop,        BorderLayout.NORTH);
    pnlTop.add(btnClose,                     BorderLayout.WEST);

    this.getContentPane().add(pnlCenter,     BorderLayout.CENTER);

    // Display the Frame

    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setSize(new Dimension(600, 750));
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

    this.setResizable(true);
    this.setVisible(true);
  }

  /** Method buildCOREPanel()
   *  <p> This method will build the change CORE Curves Limits Panel
   *  @return panel = The CORE Curves Limits Panel
   */

  private JPanel buildCOREPanel()
  {
	int i,j;
	int    cnt   = 0;
	int    ittl  = 0;
    String sUnit = "";

	JPanel panel         = new JPanel();

	JPanel panelCORE     = new JPanel();

    JPanel pnlCORE       = new JPanel();
    JPanel pnlCORElbl    = new JPanel();
    JPanel pnlCORElbl1   = new JPanel();
    JPanel pnlCORElbl2   = new JPanel();
    JPanel pnlCOREEast   = new JPanel();
    JPanel pnlCOREtxt    = new JPanel();
    JPanel pnlCOREtxt1   = new JPanel();
    JPanel pnlCOREtxt2   = new JPanel();
    JPanel pnlCOREReset  = new JPanel();

    JLabel lblCOREMnem   = new JLabel();
    JLabel lblCOREDesc   = new JLabel();
    JLabel lblCOREMin    = new JLabel();
    JLabel lblCOREMax    = new JLabel();
    JLabel lblCOREReset  = new JLabel();

    JPanel pnlL[]       = null;
    JPanel pnlLLabel[]  = null;
    JPanel pnlLLabel1[] = null;
    JPanel pnlLLabel2[] = null;
    JPanel pnlLEast[]   = null;
    JPanel pnlLText[]   = null;
    JPanel pnlLMin[]    = null;
    JPanel pnlLMax[]    = null;

    JLabel lblCORE[][]   = null;

    JScrollPane scrollC = new JScrollPane();

    TitledBorder titledBorderCORE = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "CORE Log Plot Limits:");
    titledBorderCORE.setTitleFont(new java.awt.Font("Dialog", 1, 13));

    ittl = 0;
    for (i=0; i<rock.rockStandardTools.TOTAL_DATA; i++)
    {
	  if (data_core[i][DATA] != -1.0)
	  {
		ittl++;
	  }
	}

    // Build CORE Panel

    panel.setLayout(new BorderLayout());
    panel.setBorder(titledBorderCORE);

    // -- CORE Panel

    panelCORE.setLayout(new GridLayout(ittl, 1));

    // .. CORE Header Panel

    pnlCORE.setLayout(new BorderLayout());

    // .. .. CORE Header Labels Panel

    pnlCORElbl.setLayout(new BorderLayout());

    lblCOREMnem.setFont(new java.awt.Font("Dialog", 1, 13));
    lblCOREMnem.setText("Mnem");

    lblCOREDesc.setFont(new java.awt.Font("Dialog", 1, 13));
    lblCOREDesc.setText("Curve Description");

    // .. .. CORE Header Texts Panel

    pnlCOREtxt.setLayout(new GridLayout());

    lblCOREMin.setFont(new java.awt.Font("Dialog", 1, 13));
    lblCOREMin.setText("Minimum");

    lblCOREMax.setFont(new java.awt.Font("Dialog", 1, 13));
    lblCOREMax.setText("Maximum");

    // .. .. CORE Header Reset Panel

    pnlCOREReset.setLayout(new BorderLayout());

    lblCOREReset.setFont(new java.awt.Font("Dialog", 1, 13));
    lblCOREReset.setText("Reset");

    // .. Attach CORE Header Widgets

    pnlCORE.add(pnlCORElbl,       BorderLayout.CENTER);

    pnlCORElbl.add(pnlCORElbl1,   BorderLayout.WEST);
    pnlCORElbl1.add(lblCOREMnem,  null);

    pnlCORElbl.add(pnlCORElbl2,   BorderLayout.CENTER);
    pnlCORElbl2.add(lblCOREDesc,  null);

    pnlCORE.add(pnlCOREEast,      BorderLayout.EAST);
    pnlCOREEast.add(pnlCOREtxt,   BorderLayout.CENTER);

    pnlCOREtxt.add(pnlCOREtxt1,   null);
    pnlCOREtxt1.add(lblCOREMin,   null);

    pnlCOREtxt.add(pnlCOREtxt2,   null);
    pnlCOREtxt2.add(lblCOREMax,   null);

    pnlCOREEast.add(lblCOREReset, BorderLayout.EAST);

    // .. CORE List Panel

    pnlL       = new JPanel[rock.rockStandardTools.TOTAL_DATA];
	pnlLLabel  = new JPanel[rock.rockStandardTools.TOTAL_DATA];
	pnlLLabel1 = new JPanel[rock.rockStandardTools.TOTAL_DATA];
	pnlLLabel2 = new JPanel[rock.rockStandardTools.TOTAL_DATA];
	pnlLEast   = new JPanel[rock.rockStandardTools.TOTAL_DATA];
	pnlLText   = new JPanel[rock.rockStandardTools.TOTAL_DATA];
	pnlLMin    = new JPanel[rock.rockStandardTools.TOTAL_DATA];
	pnlLMax    = new JPanel[rock.rockStandardTools.TOTAL_DATA];

    lblCORE     = new JLabel[rock.rockStandardTools.TOTAL_DATA][2];
    txtCORE     = new JTextField[rock.rockStandardTools.TOTAL_DATA][2];

    btnCORE     = new JButton[rock.rockStandardTools.TOTAL_DATA];
    btnXCORE    = new JButton[rock.rockStandardTools.TOTAL_DATA];

    for (i=0; i<rock.rockStandardTools.TOTAL_DATA; i++)
    {
      pnlL[i]      = new JPanel();
      pnlL[i].setLayout(new BorderLayout());

  	  pnlLLabel[i] = new JPanel();
  	  pnlLLabel[i].setLayout(new GridLayout());

      pnlLLabel1[i] = new JPanel();
      pnlLLabel1[i].setLayout(new BorderLayout());

      lblCORE[i][0] = new JLabel();
      lblCORE[i][0].setFont(new java.awt.Font("Dialog", 1, 12));
      lblCORE[i][0].setText(rock.rockStandardTools.ROCK_TOOLS[i][0]);

      pnlLLabel2[i] = new JPanel();
      pnlLLabel2[i].setLayout(new BorderLayout());

      lblCORE[i][1] = new JLabel();
      lblCORE[i][1].setFont(new java.awt.Font("Dialog", 1, 12));
      lblCORE[i][1].setText(rock.rockStandardTools.ROCK_TOOLS[i][1]);

      pnlLEast[i]  = new JPanel();
	  pnlLEast[i].setLayout(new BorderLayout());

	  pnlLText[i]  = new JPanel();
	  pnlLText[i].setLayout(new GridLayout());

	  pnlLMin[i]   = new JPanel();
	  pnlLMin[i].setLayout(new BorderLayout());

      txtCORE[i][0] = new JTextField();
      txtCORE[i][0].setColumns(6);
      txtCORE[i][0].setFont(new java.awt.Font("Dialog", 1, 12));
      txtCORE[i][0].setHorizontalAlignment(SwingConstants.TRAILING);
      txtCORE[i][0].setText(""+data_core[i][MINIMUM]);
      txtCORE[i][0].addFocusListener(new rockPlotLimitsFrameFocusAdapter(this));

	  pnlLMax[i]   = new JPanel();
	  pnlLMax[i].setLayout(new BorderLayout());

      txtCORE[i][1] = new JTextField();
      txtCORE[i][1].setColumns(6);
      txtCORE[i][1].setFont(new java.awt.Font("Dialog", 1, 12));
      txtCORE[i][1].setHorizontalAlignment(SwingConstants.TRAILING);
      txtCORE[i][1].setText(""+data_core[i][MAXIMUM]);
      txtCORE[i][1].addFocusListener(new rockPlotLimitsFrameFocusAdapter(this));

      btnCORE[i] = new JButton();
      btnCORE[i].setPreferredSize(new Dimension(27, 27));
      btnCORE[i].addActionListener(this);

      sUnit = rock.rockStandardTools.ROCK_TOOLS[i][2];

      // Porosity
	  if (sUnit.equals("PU"))
        btnCORE[i].setBackground( cPHI ); // cyan
      // Photoelectric
	  else if (sUnit.equals("BARNS/E"))
        btnCORE[i].setBackground( cUMAA ); // green yellow
      // Grain Density
	  else if ((sUnit.equals("GM/CC"))  || (sUnit.equals("GMCC")))
        btnCORE[i].setBackground( cRHOMAA ); // green
      // Acoustic transit time
	  else if (sUnit.equals("USEC/FT"))
        btnCORE[i].setBackground( cDTMAA ); // forest green
      // Radioactive
	  else if ((sUnit.equals("PPM")) || (sUnit.equals("%")) || (sUnit.equals("API")))
        btnCORE[i].setBackground( cGR );     // red
      // Permeability
	  else if (sUnit.equals("MD"))
        btnCORE[i].setBackground( cMD ); // blue
      // Feet Inches
	  else if ((sUnit.equals("F")) || (sUnit.equals("FT")) || (sUnit.equals("IN")))
        btnCORE[i].setBackground( cDEPTH ); // brown
      // Fraction
	  else if (sUnit.equals("FRAC"))
        btnCORE[i].setBackground( cFRAC );
      // Unknown Linear Curve
	  else if (sUnit.equals("UNI"))
        btnCORE[i].setBackground( cUNI );
      // Unknown Log Curve
	  else if (sUnit.equals("UNL"))
        btnCORE[i].setBackground( cUNL );
      else if (sUnit.equals("DEGF"))
        btnCORE[i].setBackground( cDEGF );
      // Default
      else
        btnCORE[i].setBackground( cMISC ); // yellow

      btnXCORE[i] = new JButton();
      btnXCORE[i].setPreferredSize(new Dimension(10, 27));
      btnXCORE[i].setBackground(Color.lightGray);
      btnXCORE[i].addActionListener(this);

      // .. .. Attach Row

  	  if (data_core[i][DATA] != -1.0)
	  {
        panelCORE.add(pnlL[i],            null);
	  }

      pnlL[i].add(btnXCORE[i],            BorderLayout.WEST);

      pnlL[i].add(pnlLLabel[i],           BorderLayout.CENTER);

      pnlLLabel[i].add(pnlLLabel1[i],     null); //BorderLayout.WEST);
      pnlLLabel1[i].add(lblCORE[i][0],    BorderLayout.WEST);

      pnlLLabel[i].add(pnlLLabel2[i],     null); //BorderLayout.CENTER);
      pnlLLabel2[i].add(lblCORE[i][1],    BorderLayout.WEST);

      pnlL[i].add(pnlLEast[i],            BorderLayout.EAST);
      pnlLEast[i].add(pnlLText[i],        BorderLayout.CENTER);

      pnlLText[i].add(pnlLMin[i],         null);
      pnlLMin[i].add(txtCORE[i][0],       BorderLayout.CENTER);

      pnlLText[i].add(pnlLMax[i],         null);
      pnlLMax[i].add(txtCORE[i][1],       BorderLayout.CENTER);

      pnlLEast[i].add(btnCORE[i],         BorderLayout.EAST);
	}

    // Add Widgets to panel

    panel.add(pnlCORE,                    BorderLayout.NORTH);
    panel.add(scrollC,                    BorderLayout.CENTER);
    scrollC.getViewport().add( panelCORE, null);

	return (panel);
  }

  /** METHOD createImageIcon()
   *  <p> This Method will create the Frame for this class.
   *  @param path = the path to the image
   *  @return image icon
   */

  protected ImageIcon createImageIcon(String path)
  {
    java.net.URL imgURL = getClass().getResource(path);
    ImageIcon    image  = null;

    if (imgURL != null)
    {
      image = new ImageIcon(imgURL);
    }
    else
    {
       System.err.println("Couldn't find file: " + path);
    }

    return (image);
  }

  /* ===================================================================== *
   * -------------------------- CLOSE DIALOG ----------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will set global variables to null in order to force JAVA
   *     to free unwanted memory.
   */

  public void close()
  {
    // Input Variables

    notifier  = null; // Observable
    orig_core = null; // Original Plot Limits for CORE
    data_core = null; // Plot Limits for CORE

    // Global Frame Widgets

    btnClose = null;

    if (txtCORE != null)
    {
      for (int i=0; i<rock.rockStandardTools.TOTAL_DATA; i++)
      {
	    for (int j=0; j<2; j++)
	    {
		  txtCORE[i][j] = null;
	    }
	    btnXCORE[i] = null;
    	btnCORE[i]  = null;
      }
	}
    txtCORE  = null; // CORE Textfields
    btnCORE  = null;
    btnXCORE = null;

    dispose();
  }

  /** METHOD rockPlotLimitsFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class rockPlotLimitsFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
//      close();
      if (notifier != null)
        notifier.notifyObservers(new String("Close CORE Plot Limits"));
    }
  }

  /* ====================================================================== *
   * ---------------------------- GET METHODS ----------------------------- *
   * ====================================================================== */

  /** Method getCORELimits()
   *  <p> this method will retrieve the CORE Limits data array
   *  @return data_core = array of the CORE Limits data
   */

  public double[][] getCORELimits() { return(data_core); } // Plot Limits for CORE

  /** Method getCOREOriginal()
   *  <p> this method will retrieve the Standard CORE Limits data array
   *  @return data_core = array of the Standard CORE Limits data
   */

  public double[][] getCOREOriginal() { return(orig_core); } // Plot Limits for CORE

  /* ====================================================================== *
   * ---------------------------- SET METHODS ----------------------------- *
   * ====================================================================== */

  /** Method setColor()
   *  <p> This method will set the Button Change Colors
   *  @param iAction = the List selected
   */

  private void setColor(int iAction)
  {
	int i,j;
	String sUnit = "";

    if ((iAction == _C_CORE) && (data_core != null))
    {
      for (i=0; i<rock.rockStandardTools.TOTAL_DATA; i++)
      {
	    for (j=0; j<2; j++)
	    {
          txtCORE[i][j].setText(""+ data_core[i][j]);

	  	  if (data_core[i][j] != orig_core[i][j])
		  {
            sUnit = rock.rockStandardTools.ROCK_TOOLS[i][2];

            // Porosity
	        if (sUnit.equals("PU"))
              btnXCORE[i].setBackground( cPHI ); // cyan
            // Photoelectric
	        else if (sUnit.equals("BARNS/E"))
              btnXCORE[i].setBackground( cUMAA ); // green yellow
            // Grain Density
	        else if ((sUnit.equals("GM/CC"))  || (sUnit.equals("GMCC")))
              btnXCORE[i].setBackground( cRHOMAA ); // green
            // Acoustic transit time
	        else if (sUnit.equals("USEC/FT"))
              btnXCORE[i].setBackground( cDTMAA ); // forest green
            // Radioactive
	        else if ((sUnit.equals("PPM")) || (sUnit.equals("%")) || (sUnit.equals("API")))
              btnXCORE[i].setBackground( cGR );     // red
            // Permeability
	        else if (sUnit.equals("MD"))
              btnXCORE[i].setBackground( cMD ); // blue
            // Feet Inches
	        else if ((sUnit.equals("F")) || (sUnit.equals("FT")) || (sUnit.equals("IN")))
              btnXCORE[i].setBackground( cDEPTH ); // brown
            // Fraction
	        else if (sUnit.equals("FRAC"))
              btnXCORE[i].setBackground( cFRAC );
            // Unknown Linear Curve
	        else if (sUnit.equals("UNI"))
              btnXCORE[i].setBackground( cUNI );
            // Unknown Log Curve
	        else if (sUnit.equals("UNL"))
              btnXCORE[i].setBackground( cUNL );
            else if (sUnit.equals("DEGF"))
              btnXCORE[i].setBackground( cDEGF );
            // Default
            else
              btnXCORE[i].setBackground( cMISC ); // yellow
		  }
		}
	  }
	}
  }

  /* ====================================================================== *
   * ---------------------------- ACTIONS --------------------------------- *
   * ====================================================================== */

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
	int i,j,k;
    double dGroup    = -1;
    double dMin      = 0.0;
    double dMax      = 0.0;

    for (i=0; i<rock.rockStandardTools.TOTAL_DATA; i++)
    {
	  if (event.getSource() == btnCORE[i])
	  {
		dGroup = data_core[i][GROUP];
		dMin   = orig_core[i][MINIMUM];
		dMax   = orig_core[i][MAXIMUM];

        data_core[i][MINIMUM] = dMin;
        data_core[i][MAXIMUM] = dMax;

        if (dGroup != -1.0)
          data_core = rock.rockPlotLimitsUtility.setByGroup(dGroup, dMin, dMax, data_core );

        for (k=0; k<rock.rockStandardTools.TOTAL_DATA; k++)
        {
		  for (j=0; j<2; j++)
		  {
            txtCORE[k][j].setText(""+ data_core[k][j]);
		  }

          if ((data_core[k][MINIMUM] == orig_core[k][MINIMUM]) &&
              (data_core[k][MAXIMUM] == orig_core[k][MAXIMUM]))
          {
	        btnXCORE[k].setBackground(Color.lightGray);
		  }
		}

        if (notifier != null)
          notifier.notifyObservers(new String("CORE Plot Limits Changed"));
	  }
	}

    if (event.getSource() == btnClose)   // Close Dialog
    {
      if (notifier != null)
        notifier.notifyObservers(new String("Close CORE Plot Limits"));
	}
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
	int    i,j;
    int    iNumeric  = 0;
    int    k_row     = -1;
    int    k_column  = -1;
    int    iAction   = -1;
    String sMnemonic = "";
    String sMessage  = "";
    String sTemp     = "";
    double dTemp     = 0.0;
    int    iCycle    = 0;
    double dGroup    = -1;
    double dMin      = 0.0;
    double dMax      = 0.0;

    for (i=0; i<rock.rockStandardTools.TOTAL_DATA; i++)
    {
	  for (j=0; j<2; j++)
	  {
		if (txtCORE !=  null)
		{
          if (e.getSource() == txtCORE[i][j])
          {
		    iAction   = _C_CORE;
            iNumeric  = 1;
            sTemp     = txtCORE[i][j].getText();
            k_row     = i;
            k_column  = j;
            sMnemonic = new String(rock.rockStandardTools.ROCK_TOOLS[i][1]);
            dGroup    = data_core[k_row][GROUP];

	        switch (j)
	        {
              case MINIMUM: // Minimum Data Value
                sMessage = new String("The Minimum " + sMnemonic + " is a Numeric Field");
                break;
              case MAXIMUM: // Maximum Data Value
                sMessage = new String("The Maximum " + sMnemonic + " is a Numeric Field");
                break;
			}
		  }
		}
	  }
	}

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        txtCORE[k_row][k_column].setText(""+data_core[k_row][k_column]);

        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
		dTemp = cmn.cmnString.stringToDouble( sTemp );

        if (((k_row >= rock.rockStandardTools._KMAX)  && (k_row <= rock.rockStandardTools._KPVRT)) ||
            ((k_row >= rock.rockStandardTools._LOG_1) && (k_row <= rock.rockStandardTools._LOG_4)))
        {

	      switch (k_column)
	      {
            case MINIMUM: // Minimum Data Value
      	      iCycle  = (int) Math.floor(Math.log(dTemp) / Math.log(10.0));
              break;
            case MAXIMUM: // Maximum Data Value
      	      iCycle  = (int) Math.ceil(Math.log(dTemp) / Math.log(10.0));
              break;
		  }

          dTemp   = Math.pow(10.0, iCycle);
          if (dTemp > 100000) { dTemp = 100000; }

          txtCORE[k_row][k_column].setText(""+ dTemp);
		}

    	data_core[k_row][k_column] = dTemp;
		dMin = data_core[k_row][MINIMUM];
		dMax = data_core[k_row][MAXIMUM];

        data_core = rock.rockPlotLimitsUtility.setByGroup(dGroup, dMin, dMax, data_core );

        setColor(iAction);

        if (notifier != null)
          notifier.notifyObservers(new String("CORE Plot Limits Changed"));

        iAction  = -1;
        k_row    = -1;
        k_column = -1;

//        if (notifier != null)
//          notifier.notifyObservers(new String("Profile Plot Limits Changed"));
      }
    }
  }
}

/** CLASS rockPlotLimitsFrameFocusAdapter()
  *  <p> This Class will handle Actions when focus is lost on a textfield
  */

class rockPlotLimitsFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  rockPlotLimitsFrame adaptee;

  rockPlotLimitsFrameFocusAdapter(rockPlotLimitsFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 04/15/2016
 *  @author  John Victorine
 */
