/*
 * @rockPlotImageTrack.java Version 1.1 11/09/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;

import cmn.cmnStruct;
import rock.rockImagesListStruct;
import util.utilBrowser;

/** CLASS rockPlotImageTrack
 *  <p> This Class will create the Rock Texture Plot Track Selection.
 *
 *  @version 1.1 11/09/2007
 *  @author  John R. Victorine
 */

public class rockPlotImageTrack extends Canvas
{
  // Input Variables

  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables

  private cmnStruct            stCMN    = null;
  private rockImagesListStruct stImages = null;
  private utilBrowser          pBrowser = null;

  // Core Image Variables

  private int  iSelectedImage[] = null;

  private int  iTotal     = 0;
  private int  iYMouse[]  = null;
  private int  iYMouse2[] = null;

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  private double dIncrementDepth  = 0.0;

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR rockPlotImageTrack( int iDataType,
   *                                  double depthStart,
   *                                  double depthEnd,
   *                                  int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   */

  public rockPlotImageTrack( int iDataType,
                             double depthStart,
                             double depthEnd,
                             int iScale )
  {
    this.iDataType = iDataType;
    this.stImages  = stImages;

    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
    stImages       = null;
    iSelectedImage = null;
    iYMouse        = null;
    iYMouse2       = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getData()
   * <p> This method will set the rock column that will be shown
   * @return stImages = Rock Images
   */

  public rockImagesListStruct getData() { return (stImages); }

  /** Method getKEY( int iXm, int iYm, int iView,
   *                 int iStartTrack, int iWidthTrack )
   * <p> This method will return the Core Image Primary KEY
   * @param iXm         = the x position
   * @param iYm         = the y position
   * @param iView       = the core image type
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   * @return sKEY       = the primary key of core image
   */

  public String getKEY( int iXm, int iYm, int iView,
                        int iStartTrack, int iWidthTrack )
  {
    String sKEY  = "0";
    int    iDraw = 0;

    for (int k=0; k<iTotal; k++)
    {
      iDraw = 0;
      if ((iXm > iStartTrack && iXm < iStartTrack+iWidthTrack) &&
          (iYm > iYMouse[k] && iYm < iYMouse2[k]))
      {
        if (stImages.stItem[k].sType.equals(cmn.cmnStruct.CORE_IMAGES[iView]))
          iDraw = 1;

        if (iDraw == 1)
          sKEY = stImages.stItem[k].sKID;
      }
    }

    return (sKEY);
  }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
    dIncrementDepth = iqstrat.iqstratTracksStruct.SCALE[iScale];

    convertCoreToPixels();
  }

  /** Method setData( rockImagesListStruct st )
   * <p> This method will set the rock column that will be shown
   * @param st = Rock Images.
   */

  public void setData( rockImagesListStruct st )
  {
    this.stImages = st;
    convertCoreToPixels();
  }

  /** Method setCommon()
   * <p> This method will set the Global Common Data Structure
   *  @param stCMN  = the Global Common Data Structure
   */

  public void setCommon(cmnStruct stCMN) { this.stCMN = stCMN; }

  /* =============================================================== *
   * --------------------------- COMPUTE --------------------------- *
   * =============================================================== */

  /** Method convertCoreToPixels()
   *  <p> This method will convert the postion of the core into pixel location
   *      so the core can be represented by a button.
   */

  private void convertCoreToPixels()
  {
    double dStart = 0.0;
    double dEnd = 0.0;
    double dRatio = 0.0;
    int    iTemp  = 0;

    if (stImages != null)
    {
      iTotal = stImages.iCount;

      iYMouse  = new int[iTotal];
      iYMouse2 = new int[iTotal];

      iSelectedImage = new int[iTotal];

      for (int i=0; i<iTotal; i++)
      {
        iSelectedImage[i] = 0;

        dStart = stImages.stItem[i].depthStart;
        dEnd   = stImages.stItem[i].depthEnd;

        dRatio = (double) iLogHeight * (dStart-depthStart)/(depthEnd-depthStart);
        iYMouse[i] = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iYMouse[i] = iHeight - (int) dRatio;

        dRatio = (double) iLogHeight * (dEnd-depthStart)/(depthEnd-depthStart);
        iYMouse2[i] = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iYMouse2[i] = iHeight - (int) dRatio;

        if ((iYMouse2[i]-iYMouse[i]) < 10)
        {
          iYMouse2[i] = iYMouse[i] + 10;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iYMouse2[i] = iYMouse[i] - 10;
        }

        if (iYMouse2[i] < iYMouse[i])
        {
          iTemp       = iYMouse[i];
          iYMouse[i]  = iYMouse2[i];
          iYMouse2[i] = iTemp;
        }
      }
    }
  }

  /** Method computeMouseDepth( int iYm )
   * <p> This method will compute the Mouse Depth
   * @param iYm = the cursor y-coordinate
   * @return depth = depth computed from mouse cursor location
   */

  public double computeMouseDepth( int iYm )
  {
    double depth = 0.0;

    if (iYm > PLOT_TITLES)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        depth = depthStart + (depthEnd-depthStart) * (iHeight-iYm)/iLogHeight;
      else
        depth = depthStart + (depthEnd-depthStart) * (iYm-PLOT_TITLES)/iLogHeight;
    }

    return (depth);
  }

  /** Method displayCore( int iXm, int iYm,
   *                      int iView,
   *                      int iStartTrack, int iWidthTrack )
   * <p> This method will display the core image if it is selected
   * @param iXm         = the x position
   * @param iYm         = the y position
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

  public void displayCore( int iXm, int iYm,
                           int iView,
                           int iStartTrack, int iWidthTrack )
  {
    int iFound = -1;

    for (int k=0; k<iTotal; k++)
    {
      if ((iXm > iStartTrack && iXm < iStartTrack+iWidthTrack) &&
          (iYm > iYMouse[k] && iYm < iYMouse2[k]))
      {
        switch (iView)
        {
          case cmn.cmnStruct.CORE_BOX:
            if (stImages.stItem[k].sType.equals(cmn.cmnStruct.CORE_IMAGES[iView]))
              iFound = k;
            break;
          case cmn.cmnStruct.CORE_SLAB:
            if (stImages.stItem[k].sType.equals(cmn.cmnStruct.CORE_IMAGES[iView]))
              iFound = k;
            break;
          case cmn.cmnStruct.CORE_THIN:
            if (stImages.stItem[k].sType.equals(cmn.cmnStruct.CORE_IMAGES[iView]))
              iFound = k;
            break;
          case cmn.cmnStruct.CORE_SEM:
            if (stImages.stItem[k].sType.equals(cmn.cmnStruct.CORE_IMAGES[iView]))
              iFound = k;
            break;
          case cmn.cmnStruct.ROCK_IMAGE:
            if (stImages.stItem[k].sType.equals(cmn.cmnStruct.CORE_IMAGES[iView]))
              iFound = k;
            break;
        }
      }
    }

    if (iFound > -1)
    {

      iSelectedImage[iFound] = 1;

      String sCoreImageFile =
          new String(stImages.stItem[iFound].sURL +
                     stImages.stItem[iFound].sFilename);

      if (!stCMN.bStandalone)
      {
        pBrowser =
            new utilBrowser(stCMN.appletContext,
                            cmn.cmnStruct.DISPLAY_CORE +
                            stImages.stItem[iFound].sKID);
      }
      else
      {
        util.BrowserControl.displayURL(cmn.cmnStruct.DISPLAY_CORE +
                                       stImages.stItem[iFound].sKID);
      }
    }
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawCores( Graphics g,
   *                    int iView,
   *                    int iStartTrack,
   *                    int iWidthTrack )
   *  <p> This method will draw the correlation Grid.
   * @param g           = Graphics Parameter.
   * @param iView       = The Core Image Track to draw
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

  public void drawCores( Graphics g,
                         int iView,
                         int iStartTrack,
                         int iWidthTrack )
  {
    int iEndTrack = iStartTrack + iWidthTrack;
    int iDraw     = 0;

    for (int i=0; i<iTotal; i++)
    {
      iDraw = 0;
      if (stImages.stItem[i].sType.equals(cmn.cmnStruct.CORE_IMAGES[iView]))
        iDraw = 1;

      switch (iView)
      {
        case cmn.cmnStruct.CORE_BOX:
          if (iSelectedImage[i] == 0)
            g.setColor(Color.blue);
          else
            g.setColor(Color.cyan);
          break;
        case cmn.cmnStruct.CORE_SLAB:
          if (iSelectedImage[i] == 0)
            g.setColor(Color.green);
          else
            g.setColor(Color.yellow);
          break;
        case cmn.cmnStruct.CORE_THIN:
          if (iSelectedImage[i] == 0)
            g.setColor(Color.red);
          else
            g.setColor(Color.pink);
          break;
        case cmn.cmnStruct.CORE_SEM:
          if (iSelectedImage[i] == 0)
            g.setColor(Color.gray);
          else
            g.setColor(Color.lightGray);
          break;
        case cmn.cmnStruct.ROCK_IMAGE:
          if (iSelectedImage[i] == 0)
            g.setColor(new Color( 51, 51, 0 ));
          else
            g.setColor(new Color( 102, 102, 0 ));
          break;
      }

      if (iDraw == 1)
      {
        if ((iYMouse[i] >= PLOT_TITLES) && (iYMouse2[i] > PLOT_TITLES) &&
	                (iYMouse[i] < iHeight) && (iYMouse2[i] <= iHeight))
	    {
		  iDraw = 1;
	    }
	    else
	    {
		  iDraw = 0;
		}
      }

      if (iDraw == 1)
      {
        g.fill3DRect( iStartTrack,
                      iYMouse[i],
                      iWidthTrack,
                      (iYMouse2[i] - iYMouse[i]),
                      true );

        g.setColor(Color.darkGray);
        g.drawLine(iStartTrack + 1,               iYMouse[i]  + 1,
                   iStartTrack + iWidthTrack - 1, iYMouse[i]  + 1);
        g.drawLine(iStartTrack + 1,               iYMouse[i]  + 1,
                   iStartTrack + 1,               iYMouse2[i] - 1);
        g.drawLine(iStartTrack + 1,               iYMouse2[i] - 1,
                   iStartTrack + iWidthTrack - 1, iYMouse2[i] - 1);
        g.drawLine(iStartTrack + iWidthTrack - 1, iYMouse[i]  + 1,
                   iStartTrack + iWidthTrack - 1, iYMouse2[i] - 1);
      }
    }
  }

  /** Method drawGrid( Graphics g,
   *                   int iView,
   *                   int iStartTrack,
   *                   int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iView       = The Core Image Track to draw
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

  public void drawGrid( Graphics g,
                        int iView,
                        int iStartTrack,
                        int iWidthTrack )
  {
    int    i=0;
    int    iDepth = 0;
    int    iY1    = 0;
    double depth  = 0.0;
    String sTitle = "";
    int    iEndTrack = iStartTrack + iWidthTrack;

    g.setColor(Color.black);

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      depth  = depthStart + i * dIncrementDepth / iIncrementY;
      iDepth = (int) (10.0 * depth);
      depth  = (double) iDepth / 10.0;

      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      g.setColor(Color.black);
      g.drawLine(iStartTrack, iY1, iStartTrack+iWidthTrack, iY1);
    }

    switch (iView)
    {
      case cmn.cmnStruct.CORE_BOX:
        sTitle = "Core Box";
        break;
      case cmn.cmnStruct.CORE_SLAB:
        sTitle = "Core Slab";
        break;
      case cmn.cmnStruct.CORE_THIN:
        sTitle = "Thin Section";
        break;
      case cmn.cmnStruct.CORE_SEM:
        sTitle = "Core SEM";
        break;
      case cmn.cmnStruct.ROCK_IMAGE:
        sTitle = "Rock Image";
        break;
    }

    g.setColor(Color.black);

    Font        fsb  = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString(sTitle, LABELSTART+14, -1*(iStartTrack+2*iWidthTrack/3));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method draw( Graphics g,
   *               int iSelected,
   *               int iStartTrack,
   *               int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g - Graphics Parameter.
   * @param iSelected   = The Image Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw( Graphics g,
                    int iSelected,
                    int iStartTrack,
                    int iWidthTrack )
  {
    int iView = -1;
    int iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;

    if ((iSelected == iqstrat.iqstratTracksStruct._ROCK_BOX) ||
        (iSelected == iqstrat.iqstratTracksStruct._ROCK_SLAB) ||
        (iSelected == iqstrat.iqstratTracksStruct._ROCK_THINSECTION) ||
        (iSelected == iqstrat.iqstratTracksStruct._ROCK_SEM) ||
        (iSelected == iqstrat.iqstratTracksStruct._ROCK_IMG_OUTCROP))
    {
      switch (iSelected)
      {
        case iqstrat.iqstratTracksStruct._ROCK_BOX:
          iView = 0;
          break;
        case iqstrat.iqstratTracksStruct._ROCK_SLAB:
          iView = 1;
          break;
        case iqstrat.iqstratTracksStruct._ROCK_THINSECTION:
          iView = 2;
          break;
        case iqstrat.iqstratTracksStruct._ROCK_SEM:
          iView = 3;
          break;
        case iqstrat.iqstratTracksStruct._ROCK_IMG_OUTCROP:
          iView = 4;
          break;
      }

      g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2] ) );
      g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

      drawGrid( g, iView, iStartTrack, iWidthTrack );
      drawCores( g, iView, iStartTrack, iWidthTrack );
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g - Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}