/*
 * @rockObsEntryPlot.java Version 1.1 04/11/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.plot;

import java.awt.*;
import java.awt.geom.*;

import iqstrat.iqstratStruct;

import lith.rock.plot.rockColumnPlotTrack;
import lith.rock.rockColumnListStruct;

import lith.lithology.plot.lithologyPlotTrack;
import lith.lithology.lithologyListStruct;
import lith.lithology.lithologySymbolsListStruct;

import lith.texture.plot.texturePlotTrack;
import lith.texture.textureListStruct;

import rock.color.plot.rockColorPlotTrack;
import rock.color.rockColorListStruct;

import rock.fossil.plot.fossilPlotTrack;
import rock.fossil.fossilListStruct;

import rock.sedimentary.plot.sedimentaryPlotTrack;
import rock.sedimentary.sedimentaryListStruct;

import rock.phi.plot.phiPlotTrack;
import rock.phi.phiListStruct;

/** Class rockObsEntryPlot
 *  <p> This method will display the sedimentary structures selected
 *
 *  @version 1.1 04/11/2011
 *  @author  John R. Victorine
 */

public class rockObsEntryPlot extends Canvas
{
  public static final int _SOURCE      = iqstrat.iqstratTracksStruct._SOURCE;
  public static final int _TYPE        = iqstrat.iqstratTracksStruct._TYPE;
  public static final int _DESCRIPTION = iqstrat.iqstratTracksStruct._DESCRIPTION;
  public static final int _STATUS      = iqstrat.iqstratTracksStruct._STATUS;

  // Input Variables

  private double depthStart  = 0.0;   // Starting Depth
  private double depthEnd    = 0.0;   // Ending Depth
  private int    iScale      = -1;    // Plot Scale
  private int    iSelected[] = null;  // Selected Tracks

  // Global Variables

  private int    iUnit       = 0;     // Lithology/Texture Data Plot Sampling
  private int    iDataType   = 0;     // Format of Data, LAS File or Outcrop

  private int    iStart[]    = null;  // Array of Track Starting X-Position
  private int    iWide[]     = null;  // Array of Track Widths
  private int    iButtons    = 0;     // Total Number of Tracks

  private int    iBio        = 10;    // Total number of fossils/depth/well
  private int    iSed        = 20;    // Total number of structurs/depth/well

  private phiPlotTrack          plotPhi         = null; // Porosity Type Track
  private rockColumnPlotTrack   pRockColumn     = null; // Rock Column Plot Track
  private lithologyPlotTrack    plotLithology   = null; // Rock Column Track
  private texturePlotTrack      plotTexture     = null; // Rock Texture Track
  private fossilPlotTrack       plotFossils     = null; // Fossils Plot Track
  private sedimentaryPlotTrack  plotSedimentary = null; // Sedimentary Track
  private rockColorPlotTrack    plotRockColor   = null; // Rock Color Track

  // Plot Variables

  private int    iWidth           = 650;  // Width of Plot
  private int    iHeight          = 100;  // Height of Plot with the Titles
  private int    iLogHeight       = 100;  // Height of Plot without the Titles
  private int    iIncrementY      = 100;  // Increment Height

  private double dIncrementDepth  = 0.0;  // Increment Depth
  private int    iStartDepthLabel = 10;   // Start Depth Label at

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int DEPTH_TRACK = iqstrat.iqstratTracksStruct.DEPTH_TRACK;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR rockObsEntryPlot()
   *  <p> This is the constructor for this class.
   *  @param stStruct   = Global IQSTRAT Data Structure
   *  @param iDataType  = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart = The starting depth of plot
   *  @param depthEnd   = The ending depth of plot
   *  @param iScale     = The scale
   *  @param iSelected  = The selected tracks for plotting
   */

  public rockObsEntryPlot( iqstratStruct stStruct,
                           double depthStart,
                           double depthEnd,
                           int iScale,
                           int iSelected[] )
  {
    iUnit = stStruct.iUnit;

    setPlotWidth(iSelected);
    setPlotHeight(iScale, depthEnd, depthStart);

    plotPhi       = new phiPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stPorosity);
    pRockColumn   = new rockColumnPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stParseRock);
    plotLithology = new lithologyPlotTrack(
        iDataType, depthStart, depthEnd, iScale, iUnit, stStruct.stSymbols);
    plotTexture   = new texturePlotTrack(
        iDataType, depthStart, depthEnd, iScale, iUnit, stStruct.stSymbols);
    plotFossils   = new fossilPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stFossils);
    plotSedimentary = new sedimentaryPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stSedStruct);
    plotRockColor = new rockColorPlotTrack(iDataType, depthStart, depthEnd, iScale);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
    iSelected = null;
    iStart    = null;
    iWide     = null;

    if (plotPhi != null)
      plotPhi.close();
    plotPhi         = null;

    if (pRockColumn != null)
      pRockColumn.close();
    pRockColumn = null;

    if (plotLithology != null)
      plotLithology.close();
    plotLithology = null;

    if (plotTexture != null)
      plotTexture.close();
    plotTexture = null; // Rock Texture Track

    if (plotFossils != null)
      plotFossils.close();
    plotFossils = null; // Fossils Plot Track

    if (plotSedimentary != null)
      plotSedimentary.close();
    plotSedimentary = null; // Sedimentary Track

    if (plotRockColor != null)
      plotRockColor.close();
    plotRockColor = null; // Rock Color Track
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setSelected()
   * <p> This method will change the selected tracks that are plotted
   *  @param iSelected  = The Tracks that will be displayed
   */

  public void setSelected(int iSelected[])
  {
    setPlotWidth(iSelected);
    repaint();
  }

  /** Method setPlotWidth()
   * <p> This method will set the Plot Width
   *  @param iSelected  = The Tracks that will be displayed
   */

  public void setPlotWidth(int iSelected[])
  {
    int iTrackWidth = 0;

    iWidth = DEPTH_TRACK;

    this.iSelected = iSelected;
    if (iSelected != null)
      this.iButtons = iSelected.length;

    if (this.iButtons > 0)
    {
      iStart = new int[this.iButtons];
      iWide  = new int[this.iButtons];

      for (int i=0; i<this.iButtons; i++)
      {
        iTrackWidth = 0;
        for (int j=0; j<iqstrat.iqstratTracksStruct.TOTAL_TYPES; j++)
        {
          if (iSelected[i] != -1)
          {
            if (iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_TYPE].equals(
                iqstrat.iqstratTracksStruct.TYPE[j][0]))
            {
              if ( iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_TYPE].equals(
                   "ICON"))
              {
                if (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_FOSSILS)
                  iTrackWidth = iBio;

                if (iSelected[i] == iqstrat.iqstratTracksStruct._SED_STRUCTURE)
                  iTrackWidth = iSed;
              }
              else
              {
                iTrackWidth = cmn.cmnString.stringToInt(
                    iqstrat.iqstratTracksStruct.TYPE[j][2]);
              }
            }
          }
        }

        iStart[i] = iWidth;
        iWide[i]  = iTrackWidth;
        iWidth    = iWidth + iTrackWidth;
      }
    }
  }

  /** Method setPlotWidth()
   * <p> This method will set the Plot Width
   */

  public void setPlotWidth()
  {
    int iTrackWidth = 0;

    iWidth = DEPTH_TRACK;

    if (iButtons > 0)
    {
      iStart = new int[iButtons];
      iWide  = new int[iButtons];

      for (int i=0; i<iButtons; i++)
      {
        iTrackWidth = 0;
        for (int j=0; j<iqstrat.iqstratTracksStruct.TOTAL_TYPES; j++)
        {
          if (iSelected[i] != -1)
          {
            if (iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_TYPE].equals(
                iqstrat.iqstratTracksStruct.TYPE[j][0]))
            {
              if ( iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_TYPE].equals(
                   "ICON"))
              {
                if (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_FOSSILS)
                  iTrackWidth = iBio;

                if (iSelected[i] == iqstrat.iqstratTracksStruct._SED_STRUCTURE)
                  iTrackWidth = iSed;
              }
              else
                iTrackWidth = cmn.cmnString.stringToInt(
                    iqstrat.iqstratTracksStruct.TYPE[j][2]);
            }
          }
        }

        iStart[i] = iWidth;
        iWide[i]  = iTrackWidth;
        iWidth    = iWidth + iTrackWidth;
      }
    }
  }

  /** Method setPlotHeight(int iScale, double dMaximum, double dMinimum)
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight(int iScale, double dMaximum, double dMinimum)
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
    dIncrementDepth = iqstrat.iqstratTracksStruct.SCALE[iScale];
  }

  /** Method setScale(int iScale, double dMaximum, double dMinimum)
   * <p> This method will set the Plot scale in the
   *     Cross Plot Data List Structure
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setScale(int iScale, double dMaximum, double dMinimum)
  {
    setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotPhi != null)
      plotPhi.setPlotHeight( iScale, dMaximum, dMinimum );

    if (pRockColumn != null)
      pRockColumn.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotLithology != null)
      plotLithology.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotTexture != null)
      plotTexture.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotRockColor != null)
      plotRockColor.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotSedimentary != null)
      plotSedimentary.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotFossils != null)
      plotFossils.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotRockColor != null)
      plotRockColor.setPlotHeight(iScale, dMaximum, dMinimum);

    repaint();
  }

  /** Method modifyWidths( int iBio, int iSed )
   * <p> This method will modify the plot track width depending on the
   *     maximum number of fossils or sedimentary structures per depth.
   * @param iBio = The total number of fossils per depth per well
   * @param iSed = The total number of structures per depth per well
   */

  public void modifyWidths( int iBio, int iSed )
  {
    this.iBio = iBio;
    this.iSed = iSed;

    setPlotWidth();
    repaint();
  }

  /* ------------------------------------------------------------------ *
   * ---------------- Thin Porosity Plot Track Actions ---------------- *
   * ------------------------------------------------------------------ */

  /* ------------------------------------------------------------------ *
   * ------------------- Porosity Plot Track Actions -------------------- *
   * ------------------------------------------------------------------ */

  /** Method setPHI()
   * <p> This method will set the Porosity Data
   * @param st = The Porosity Data List Structure
   */

  public void setPHI(phiListStruct st)
  {
    if (plotPhi != null) plotPhi.setPHIData( st );
  }

  /* ------------------------------------------------------------------ *
   * ---------------- Rock Column Plot Track Actions ----------------- *
   * ------------------------------------------------------------------ */

  /** Method setRockColumn( rockColumnListStruct st)
   *  <p> This method will set the rock column lithology
   *  @param st = Rock Column List Data Structure
   */

  public void setRockColumn( rockColumnListStruct st )
  {
	if (pRockColumn != null) pRockColumn.setRockColumn( st );
  }

  /* ------------------------------------------------------------------ *
   * ------------ Rock Lithology Column Plot Track Actions ------------ *
   * ------------------------------------------------------------------ */

  /** Method setRockLithology(lithologyListStruct st)
   * <p> This method will set the rock column that will be shown
   * @param stRockLithology = Rock Column determined by user from rock data.
   *

  public void setRockLithology(lithologyListStruct stRockLithology)
  {
    plotLithology.setRockLithology( stRockLithology );
  }
*/
  /* ------------------------------------------------------------------ *
   * ---------------- Rock Texture Plot Track Actions ----------------- *
   * ------------------------------------------------------------------ */

  /** Method setRockTexture(textureListStruct st)
   * <p> This method will set the rock texture that will be shown
   * @param st = Rock Texture determined by user from rock data.
   *

  public void setRockTexture(textureListStruct st)
  {
    plotTexture.setRockTexture( st );
  }
*/
  /* ------------------------------------------------------------------ *
   * ----------- Sedimentary Structure Plot Track Actions ------------- *
   * ------------------------------------------------------------------ */

  /** Method setStructures()
   * <p> This method will set the sedimentary structures Data
   * @param st = The sedimentary structures Data List Structure
   */

  public void setStructures(sedimentaryListStruct st)
  {
    if (plotSedimentary != null) plotSedimentary.setData( st );
  }

  /* ------------------------------------------------------------------ *
   * ------------------- Fossil Plot Track Actions -------------------- *
   * ------------------------------------------------------------------ */

  /** Method setFossils()
   * <p> This method will set the Fossil Data
   * @param st = The Fossil Data List Structure
   */

  public void setFossils(fossilListStruct st)
  {
    if (plotFossils != null) plotFossils.setFossilData( st );
  }

  /* ------------------------------------------------------------------ *
   * -------------- Rock Color Column Plot Track Actions -------------- *
   * ------------------------------------------------------------------ */

  /** Method setRockColorData()()
   * <p> This method will set the Rock Color Data.
   * @param st = the Rock Color List Data Structure
   */

  public void setRockColorData(rockColorListStruct st)
  {
    if (plotRockColor != null) plotRockColor.setRockColorData(st);
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawDepth(Graphics g)
   * <p> This method will draw the depth scale
   * @param g - Graphics Parameter.
   */

  public void drawDepth(Graphics g)
  {
    int    i=0;
    int    j=0;
    int    iDepth;
    int    jXinc;
    int    iStartTrack = 0;
    int    iWidthTrack = DEPTH_TRACK;
    int    iY1 = 0;
    int    iY2 = 0;

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      iDepth = (int) (depthStart +
                      (double)i * dIncrementDepth / (double) iIncrementY);

      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if (iY2 >= PLOT_TITLES)
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iStartTrack + iWidthTrack, iY2);
        }
      }

      g.setColor(Color.black);
      g.drawLine(iStartTrack, iY1, iStartTrack+iWidthTrack, iY1);

      g.drawString("" + iDepth, iStartDepthLabel, iY1);
    }

    g.drawLine(iStartTrack, PLOT_TITLES, iStartTrack+iWidthTrack, PLOT_TITLES);
  }

  /** Method draw(graphics g)
   * <p> This method will draw the stratigraphic plot
   * @param g - Graphics Parameter.
   */

  public void draw(Graphics g)
  {
    int iStartWidth = DEPTH_TRACK;
    int iTrackWidth = 0;

    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);

//    drawDepth(g);

    for (int i=0; i<iButtons; i++)
    {
      if (iSelected[i] != -1)
      {
        plotPhi.draw(g, iSelected[i], iStart[i], iWide[i]);
        pRockColumn.draw(g, iSelected[i], iStart[i], iWide[i]);
        plotLithology.draw(g, iSelected[i], iStart[i], iWide[i]);
        plotTexture.draw(g, iSelected[i], iStart[i], iWide[i]);
        plotSedimentary.draw(g, iSelected[i], iStart[i], iWide[i]);
        plotFossils.draw(g, iSelected[i], iStart[i], iWide[i]);
        plotRockColor.draw(g, iSelected[i], iStart[i], iWide[i]);
      }
    }
  }

  /** Method paint(Graphics g)
   * <p> This method will paint stratigraphic Plot
   * @param g - Graphics Parameter.
   */

  public void paint(Graphics g) { draw(g); }
}

/*
 *  @version 1.1 04/11/2011
 *  @author  John Victorine
 */
