/*
 * @phiPlotTrack.java Version 1.1 09/07/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.phi.plot;

import java.awt.*;
import java.awt.geom.*;

import mask.maskSymbolsListStruct;
import mask.plot.maskPlotSymbol;
import rock.phi.phiListStruct;

/** CLASS phiPlotTrack
 *  <p> This Class will create the IQSTRAT - Stratigraphic Plot Thin Phi Tracks.
 *
 *  @version 1.1 09/07/2007
 *  @author  John R. Victorine
 */

public class phiPlotTrack extends Canvas
{
  // Input Variables

  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Data Variables
  // Porosity Data List Structure

  private phiListStruct         stPHI     = null;
  private maskSymbolsListStruct stSymbols = null;
  private maskPlotSymbol        plot      = null;

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR phiPlotTrack( int iDataType,
   *                            double depthStart,
   *                            double depthEnd,
   *                            int iScale,
   *                            maskSymbolsListStruct stSymbols )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   *  @param stSymbols   = Porosity Type Plot Symbols List Data Structure
   */

  public phiPlotTrack( int iDataType,
                       double depthStart,
                       double depthEnd,
                       int iScale,
                       maskSymbolsListStruct stSymbols )
  {
    this.iDataType = iDataType;
    this.stSymbols = stSymbols;

    plot  = new maskPlotSymbol( stSymbols );

    setPlotHeight( iScale, depthEnd, depthStart );

    this.setBackground( Color.white );
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
    stPHI     = null;
    stSymbols = null;

    if (plot != null)
      plot.close();
    plot      = null;
  }

  /** Method isOverlap( int iStart, int iEnd, int iRow )
   * <p> This method will determine if the cell will overlap
   * @param  iStart = starting depth pixel
   * @param  iEnd   = ending depth pixel
   * @param  iRow   = the row number
   * @return bValue = true it overlaps; false it does not
   */

  private boolean isOverlap( int iStart, int iEnd, int iRow )
  {
    boolean bValue  = false;
    int     iDepth1 = 0;
    int     iDepth2 = 0;
    double  dRatio  = 0.0;
    double  depth1  = 0.0;
    double  depth2  = 0.0;

    for (int i=0; i<iRow; i++)
    {
      depth1 = stPHI.stItem[i].depthStart;
      depth2 = stPHI.stItem[i].depthEnd;

      dRatio = (double) iLogHeight*(depthStart-depth1)/(depthStart-depthEnd);
      iDepth1 = PLOT_TITLES + (int) dRatio;

      dRatio = (double) iLogHeight*(depthStart-depth2)/(depthStart-depthEnd);
      iDepth2 = PLOT_TITLES + (int) dRatio;

      if ((iDepth2-iDepth1) < 10)
      {
        if (iStart < iDepth1+10)
          bValue = true;
      }
    }

    return (bValue);
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getNextDepth( int iRow, double diff )
   * <p> This method will retrieve the next depth data in the list in such
   *     a way that the text does not overwrite
   * @param  iRow  = the present row
   * @param  diff  = the difference between text
   * @return iNext = The next Row
   */

  private int getNextDepth( int iRow, double diff )
  {
    int iNext = -1;
    int iValue = (int) diff;

    diff = (double) iValue;

    if (iRow > -1)
    {
      if (stPHI != null)
      {
        for (int i=iRow; i<stPHI.iCount; i++)
        {
          if ((iNext == -1) &&
              (stPHI.stItem[i].depthStart >=
                 stPHI.stItem[iRow].depthEnd + diff))
          {
            iNext = i;
            break;
          }
        }
      }
    }

    return (iNext);
  }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setPHIData( phiListStruct stPHI )
   * <p> This method will set the Porosity data for the rock porosity track
   * @param stPHI - porosity data List Structure
   */

  public void setPHIData( phiListStruct stPHI ) { this.stPHI = stPHI; }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iCycle = 1;
    double dXinc  = 0.0;
    double dTemp  = 0.0;
    int    iIncrement  = iWidthTrack/6;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iY1         = 0;
    int    iY2         = 0;

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES) && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.lightGray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    if (iLog == 0)
    {
      g.setColor(Color.lightGray);
      g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

      for (i=0; i<=iWidthTrack; i+=iIncrement)
      {
        if ((i==0) || (i==iWidthTrack))
        {
          g.drawLine(i+iStartTrack, LABELSTART,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }
        else
        {
          g.drawLine(i+iStartTrack, PLOT_TITLES,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }
      }
    }
    else
    {
      iIncrement = iWidthTrack / iCycle;

      g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

      for (i=0; i<iWidthTrack; i+=iIncrement)
      {
        g.setColor(Color.lightGray);
        if (i==0)
        {
          g.drawLine(i+iStartTrack, LABELSTART,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }

        g.setColor(Color.lightGray);
        g.drawLine(i+iStartTrack, PLOT_TITLES,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);

        g.setColor(Color.lightGray);
        for (j=1; j<5; j++)
        {
          dXinc = (double) j * 2.0;
          dTemp = (double) iIncrement * Math.log(dXinc) / Math.log(10.0);
          jXinc = i + iStartTrack + (int) dTemp;

          if (jXinc <= iEndTrack)
          {
            g.drawLine(jXinc, PLOT_TITLES,
                       jXinc, PLOT_TITLES+iLogHeight); //iHeight);
          }
        }
      }
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, LABELSTART, iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART, iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawRockPorosity( Graphics g, int iStartTrack, int iWidthTrack )
   *   <p> This track will plot the porosity in black color fill
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawRockPorosity( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int         length  = 0;
    int         iStart  = 0;
    int         iEnd    = 0;
    int         iWide   = 0;
    int         iDepth1 = 0;
    int         iDepth2 = 0;
    int         iDepth  = 0;
    double      dRatio  = 0.0;
    double      depth1  = 0.0;
    double      depth2  = 0.0;
    String      sTemp   = "";
    Font        fsb     = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm    = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    iStart = iStartTrack;
    iEnd   = iStartTrack+iWidthTrack;

    if (stPHI != null)
    {
      for (int i=0; i<stPHI.iCount; i++)
      {
        depth1 = stPHI.stItem[i].depthStart;
        depth2 = stPHI.stItem[i].depthEnd;

        dRatio = (double) iLogHeight * (depthStart - depth1) /
                                       (depthStart - depthEnd);
        iDepth1 = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth1 = iHeight - (int) dRatio;

        dRatio = (double) iLogHeight * (depthStart - depth2) /
                                       (depthStart - depthEnd);
        iDepth2 = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth2 = iHeight - (int) dRatio;

        iDepth = Math.abs(iDepth2 - iDepth1);

        if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
            (iDepth1 < iHeight)     && (iDepth2 <= iHeight))
        {
          iWide = (int) ( iWidthTrack * stPHI.stItem[i].dPHI / 30.0 );

          if (iWide < 0)           iWide = 0;
          if (iWide > iWidthTrack) iWide = iWidthTrack;

          if (iWide > 0) g.fillRect(iStart, iDepth1, iWide, iDepth);
        }
      }
    }

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Porosity", LABELSTART+14, -1*(iStart+2*iWidthTrack/3));
//    g2.drawString(sPHI, LABELSTART+14, -1*(iStart+5));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.drawString("0", iStart+1, LABELSTART+74);

    sTemp = new String("30");
    length = sTemp.length();
    g.drawString("30", iEnd-length*5, LABELSTART+74);

    g.drawLine(iStart, LABELSTART,  iEnd,   LABELSTART);
    g.drawLine(iStart, PLOT_TITLES, iEnd,   PLOT_TITLES);
    g.drawLine(iStart, LABELSTART,  iStart, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEnd,   LABELSTART,  iEnd,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawLithology( Graphics g, int iStartTrack, int iWidthTrack )
   *  <p> This method will draw the rock mineral composition percentage track
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawLithology( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    i       = 0;
    int    j       = 0;
    int    k       = 0;
    int    n       = 0;
    int    iX      = 0;
    int    iY      = 0;
    int    iRGB[]  = { 255, 255, 255 };
    int    iBKGD   = -1;
    int    iPlus   = 1;
    int    iDepth1 = 0;
    int    iDepth2 = 0;
    int    iDepth  = 0;
    int    iCount  = 0;
    int    iGroup  = 0;
    double diff    = 0;
    double depth1  = 0.0;
    double depth2  = 0.0;
    double dRatio  = 0.0;

    if (stPHI != null)
    {
      if (stPHI.iCount > 0)
      {
        iCount = -1;
        for (k=0; k<stPHI.iCount; k++)
        {
          if (stPHI.stItem[k] != null)
          {
            depth1 = stPHI.stItem[k].depthStart;
            depth2 = stPHI.stItem[k].depthEnd;

            dRatio  = (double) iLogHeight * (depthStart - depth1) /
                                            (depthStart - depthEnd);
            iDepth1 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth1 = iHeight - (int) dRatio;

            dRatio  = (double) iLogHeight * (depthStart - depth2) /
                                            (depthStart - depthEnd);
            iDepth2 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth2 = iHeight - (int) dRatio;

            iDepth = Math.abs( iDepth2 - iDepth1 );

            if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
                (iDepth1 < iHeight)      && (iDepth2 <= iHeight))
            {
              if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                iDepth1 =  iDepth2;

              iGroup = iDepth / 5;
              if (iGroup == 0) iGroup = 1;

              for (n=0; n<iGroup; n++)
              {
                if (stPHI.stItem[k].iRockType != rock.phi.phiLookupStruct._NONE)
                {
                  iCount++;
                  for (j=0; j<3; j++)
                  {
                    iBKGD = stPHI.stItem[k].iRockType;
                    iX    = stSymbols.stModifiers[iBKGD].iColumns;
                    iY    = stSymbols.stModifiers[iBKGD].iRows;
                    iRGB  = stSymbols.stModifiers[iBKGD].iRGB;

                    plot.drawBackground( g, iRGB[0], iRGB[1], iRGB[2],
                                         iStartTrack + j*iX, iDepth1 + n*iY,
                                         iX, iY);
                    plot.drawSymbol( g, iY, plot.getOrder(iCount),
                                     stSymbols.stModifiers[iBKGD].symbol,
                                     iStartTrack + j*iX, iDepth1 + n*iY );
                  }
                }
              }
            }
          }
        }
      }
    }
  }

  /** Method drawPorosityGrid( Graphics g, int iStartTrack, int iWidthTrack )
   *   <p> This track will plot the porosity grid
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawPorosityGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int         iStart = iStartTrack;
    int         iEnd   = iStartTrack+iWidthTrack;
    Font        fsb    = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm   = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Porosity", LABELSTART+14, -1*(iStart+2*iWidthTrack/3));
    g2.drawString("Type",     LABELSTART+20, -1*(iStart+5));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.drawLine(iStart, LABELSTART,  iEnd,   LABELSTART);
    g.drawLine(iStart, PLOT_TITLES, iEnd,   PLOT_TITLES);
    g.drawLine(iStart, LABELSTART,  iStart, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEnd,   LABELSTART,  iEnd,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawPorosity( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the porosity images that were selected
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawPorosity( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iStart  = iStartTrack;
    int    iEnd    = iStartTrack+iWidthTrack;
    int    iFound  = -1;
    int    iDepth1 = 0;
    int    iDepth2 = 0;
    int    iDepth  = 0;
    int    iGroup  = 0;
    int    iX      = 0;
    int    iRow    = 0;
    double dRatio  = 0.0;
    double depth1  = 0.0;
    double depth2  = 0.0;
    double depth   = 0.0;
    int    iTemp   = -1;
    double diff    = iqstrat.iqstratTracksStruct.SCALE[iScale] / 10.0;

    if (stPHI != null)
    {
      for (int i=0; i<stPHI.iCount; i++)
      {
        depth1 = stPHI.stItem[i].depthStart;
        depth2 = stPHI.stItem[i].depthEnd;

        dRatio = (double) iLogHeight*(depthStart-depth1)/(depthStart-depthEnd);
        iDepth1 = PLOT_TITLES + (int) dRatio;

        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth1 = iHeight - (int) dRatio;

        dRatio = (double) iLogHeight*(depthStart-depth2)/(depthStart-depthEnd);
        iDepth2 = PLOT_TITLES + (int) dRatio;

        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth2 = iHeight - (int) dRatio;

        if (iDepth1 == iDepth2)
        {
          iDepth1 = iDepth1 - 5;
          iDepth2 = iDepth2 - 5;
        }

        iDepth = Math.abs(iDepth2 - iDepth1);
        iGroup = iDepth / 10;
        if (iGroup == 0)
          iGroup = 1;

        if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
            (iDepth1 < iHeight) && (iDepth2 <= iHeight))
        {
          for (int n=0; n<iGroup; n++)
          {
            iX = iStartTrack;
            iDepth = iDepth1 + n * 10;

            for (int k=0; k<stPHI.stItem[i].iTotal; k++)
            {
              iFound = -1;
              for (int j=0; j<stSymbols.iCount; j++)
              {
                if (stPHI.stItem[i].sID[k].equals(stSymbols.stItem[j].sID))
                {
                  iFound = j;
                }
              }

              if ((iFound > -1) && (iX < iStartTrack+30))
              {
                if (!isOverlap(iDepth, iDepth+10, iRow))
                {
                  plot.drawSymbol( g, stSymbols.stItem[iFound].iRows, 0,
                                   stSymbols.stItem[iFound].symbol, iX, iDepth );
                  iRow = i+1;
                }
                iX = iX + 10;
              }
            }
          }
        }

//        iTemp = getNextDepth(i, diff);
//        if (iTemp > 0)
//        {
//          if (iTemp-1 > i) i = iTemp-1;
//        }
      }
    }

    g.setColor(Color.black);
    g.drawLine(iStart, LABELSTART, iStart, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEnd,   LABELSTART, iEnd,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Porosity Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_LAS;

    if (iSelected == iqstrat.iqstratTracksStruct._ROCK_THIN_PHI)
    {
      iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;
      g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2] ));
      g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

      drawLithology( g, iStartTrack, iWidthTrack );
      drawGrid( g, iStartTrack, iWidthTrack );
      drawRockPorosity( g, iStartTrack, iWidthTrack );
    }

    if (iSelected == iqstrat.iqstratTracksStruct._ROCK_POROSITY)
    {
      iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;
      g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2] ));
      g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

      drawPorosityGrid( g, iStartTrack, iWidthTrack );
      drawPorosity( g, iStartTrack, iWidthTrack );
    }
  }

  /** Method paint( Graphics g )
   * <p> This method is a dummy method for the plot package - NOT USED
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}