/*
 * @phiUtility.java Version 1.1 09/29/2008
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.phi;

import iqstrat.iqstratHeadersStruct;

import mask.maskSymbolsListStruct;
import rock.phi.phiListStruct;
import rock.phi.phiStruct;

/** Class phiUtility
 *  <p> This Class will provide basic utilities for the Rock Porosity
 *      data structures.
 *
 *  @version 1.1 09/29/2008
 *  @author  John R. Victorine
 */

public class phiUtility
{
  /** Method addHeaderData()
   * <p> This method will add the header data to the Porosity data List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Rock Porosity List Data Structure
   * @return st       = The Rock Porosity List Data Structure
   */

  public static phiListStruct addHeaderData(iqstratHeadersStruct stHeader,
                                               phiListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method computeDepthRange()
   * <p> This method will compute the depth range of the core data
   * @param   st = the Rock Porosity data list structure
   * @return  st = the Rock Porosity data list structure
   */

  public static phiListStruct computeDepthRange(phiListStruct st)
  {
    double depthStart = 0.0;
    double depthEnd   = 0.0;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        depthStart = st.stItem[i].depthStart;
        depthEnd   = st.stItem[i].depthEnd;

        if (i==0)
        {
          st.depthStart = depthStart;
          st.depthEnd   = depthEnd;
        }

        if (st.depthStart > depthStart)
          st.depthStart = depthStart;

        if (st.depthEnd < depthEnd)
          st.depthEnd   = depthEnd;
      }
    }

    return (st);
  }

  /** Method addPHI()
   * <p> This method will add a phi to a specific depth this allows for
   *     the ability of inserting multiple phis to a specific depth without
   *     having to know how many phis are in the depth.
   * @param  sID     = The ID representing a specific phi
   * @param  sType   = Type of Porosity, Primary or Secondary
   * @param  sName   = the porosity name
   * @param  st      = The Old Porosity Data Structure
   * @return st      = The new Porosity Data Structure
   */

  public static phiStruct addPHI( String sID, String sType,
                                  String sName, phiStruct st )
  {
    int    i           = 0;
    String sTempID[]   = null;
    String sTempType[] = null;
    String sTempName[] = null;
    int    iCount      = 0;
    int    iFound      = 0;

    if (st != null)
    {
      sTempID   = new String[st.iTotal+1];
      sTempType = new String[st.iTotal+1];
      sTempName = new String[st.iTotal+1];

      for (i=0; i<st.iTotal; i++)
      {
        if ((sName.equals(st.sText[i]) || (sID.equals(st.sID[i])))) iFound = 1;

        sTempID[iCount]   = new String(st.sID[i]);
        sTempType[iCount] = new String(st.sType[i]);
        sTempName[iCount] = new String(st.sText[i]);
        iCount++;
      }

      if (iFound == 0)
      {
        sTempID[iCount]   = new String(sID);
        sTempType[iCount] = new String(sType);
        sTempName[iCount] = new String(sName);
        iCount++;
      }

      st.iTotal = iCount;
      st.sID    = new String[iCount];
      st.sType  = new String[iCount];
      st.sText  = new String[iCount];

      for (i=0; i<iCount; i++)
      {
        st.sID[i]   = new String(sTempID[i]);
        st.sType[i] = new String(sTempType[i]);
        st.sText[i] = new String(sTempName[i]);
      }
    }

    return (st);
  }

  /** Method add()
   * <p> This method will add the Porosity to an existing list
   * @param  stBlock = The Porosity Block List Data Structure
   * @param  st      = The Old Porosity List Data Structure
   * @return st      = The new Porosity List.
   */

  public static phiListStruct add( phiStruct stBlock, phiListStruct st )
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    phiListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new phiListStruct();
    stTemp.stItem = new phiStruct[iRecords];

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new phiStruct();
    stTemp.stItem[iCount] = copy(stBlock);
    iCount++;

    stTemp.iCount = iCount;

    st        = new phiListStruct();
    st.stItem = new phiStruct[iCount];
    st.iCount = iCount;

    st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
    st.source   = new String( source );    // Source of Data
    st.sCreated = new String( sCreated );  // Created Date

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Rock porosity in an existing list
   * @param  sKEY    = The Primary Key of the data to be removed
   * @param  stBlock = The phi structure that will be replaced
   * @param  st      = The Old Porosity List Data Structure
   * @return st      = The new Porosity List.
   */

  public static phiListStruct modify( String sKEY,
                                      phiStruct stBlock,
                                      phiListStruct st )
  {
    int i=0;
    phiListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      stTemp        = new phiListStruct();
      stTemp.stItem = new phiStruct[st.iCount];
      stTemp.iCount = st.iCount;

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            stTemp.stItem[i] = copy(stBlock);
          }
          else
          {
            stTemp.stItem[i] = copy(st.stItem[i]);
          }
        }
      }

      st.delete();

      st        = new phiListStruct();
      st.stItem = new phiStruct[stTemp.iCount];
      st.iCount = stTemp.iCount;

      st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
      st.source   = new String( source );    // Source of Data
      st.sCreated = new String( sCreated );  // Created Date

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Rock porosity from an existing list
   * @param  sKEY   = The Primary Key of the data to be removed
   * @param  st     = The Old Porosity List Data Structure
   * @return st     = The new Porosity List.
   */

  public static phiListStruct remove( String sKEY, phiListStruct st )
  {
    int i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iFound   = -1;
    phiListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st.iCount == 1)
    {
      st.delete();
      st = null;
    }
    else
    {
      if (st != null)
      {
        sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
        source   = new String( st.source );    // Source of Data
        sCreated = new String( st.sCreated );  // Created Date

        iRecords = st.iCount-1;

        stTemp        = new phiListStruct();
        stTemp.stItem = new phiStruct[iRecords];

        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            iFound = i;
          }
        }
      }

      if (st != null)
      {
        if (st.iCount > 0)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (iCount < iRecords)
            {
              if (i != iFound)
              {
                stTemp.stItem[iCount] = copy(st.stItem[i]);
                iCount++;
              }
            }
          }
        }

        st.delete();
      }

      stTemp.iCount = iCount;

      st        = new phiListStruct();
      st.stItem = new phiStruct[iCount];
      st.iCount = iCount;

      st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
      st.source   = new String( source );    // Source of Data
      st.sCreated = new String( sCreated );  // Created Date

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method copyList()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static phiListStruct copyList(phiListStruct stOld)
  {
    phiListStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new phiListStruct();

      stNew.depthStart = stOld.depthStart; // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth

      // LAS 3 Dependent Variable

      stNew.sKGS       = new String( stOld.sKGS ); // KGS Saved Data Indicator
      stNew.source     = new String( stOld.source );    // Source of Data
      stNew.sCreated   = new String( stOld.sCreated );  // Created Date

      stNew.iCount     = stOld.iCount;
      stNew.stItem     = new phiStruct[stOld.iCount];

      for (int i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy(stOld.stItem[i]);
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static phiListStruct transfer(phiListStruct stOld)
  {
    phiListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy data from one structure to another
   * @param  stOld = The original Porosity Data Structure
   * @return stNew = The Porosity Data Structure
   */

  public static phiStruct copy( phiStruct stOld )
  {
    phiStruct stNew = new phiStruct();

    if (stOld != null)
    {
      stNew.sKEY       = new String(stOld.sKEY); // Primary Key for data set
      stNew.depthStart = stOld.depthStart;       // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;         // Ending Depth

      stNew.dPHI       = stOld.dPHI;

      stNew.iRockType  = stOld.iRockType;
      stNew.iProcess   = stOld.iProcess;
      stNew.iStage     = stOld.iStage;
      stNew.iTime      = stOld.iTime;
      stNew.iSize      = stOld.iSize;

      stNew.iTotal     = stOld.iTotal;
      stNew.sID        = new String[stNew.iTotal];
      stNew.sType      = new String[stNew.iTotal];
      stNew.sText      = new String[stNew.iTotal];

      for (int i=0; i<stNew.iTotal; i++)
      {
        stNew.sID[i]   = new String(stOld.sID[i]);
        stNew.sType[i] = new String(stOld.sType[i]);
        stNew.sText[i] = new String(stOld.sText[i]);
      }
    }

    return (stNew);
  }

  /** Method setLegend( int iFossils[], lithologyListStruct st )
   * <p> This method will set the legend variables
   * @param  iPHITypes = Array of indicators to plot porosity Types symbol or not
   * @param  st        = Lithology List Data Structure
   * @param  stList    = Porosity Type Symbols List Data Structure
   * @return iPHITypes = Array of indicators to plot porosity Types symbol or not
   */

  public static int[] setLegend( int iPHITypes[], phiListStruct st,
                                 maskSymbolsListStruct stList )
  {
    if ((st != null) && (stList != null))
    {
      for (int i=0; i<st.iCount; i++)
      {
        for (int k=0; k<st.stItem[i].iTotal; k++)
        {
          for (int j=0; j<stList.iCount; j++)
          {
            if (st.stItem[i].sID[k].equals(stList.stItem[j].sID))
              iPHITypes[j] = 0;
          }
        }
      }
    }

    return (iPHITypes);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = data list structure
   * @return st = sorted data list structure
   */

  public static phiListStruct bubbleSort(phiListStruct st)
  {
    boolean swappedOnPrevRun = true;
    phiStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].depthStart > st.stItem[i+1].depthStart)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  public static void print(phiListStruct st)
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        System.out.println(st.stItem[i].sKEY+" "+
                           st.stItem[i].depthStart+" "+
                           st.stItem[i].depthEnd+" "+
                           st.stItem[i].dPHI);
        for (int j=0; j<st.stItem[i].iTotal; j++)
        {
          System.out.println("  "+j+" "+st.stItem[i].sID[j]+" "+
                             st.stItem[i].sText[j]);
        }
        System.out.println(" ----------------------------------------------- ");
      }
    }
  }
}