/*
 * @phiPlotSymbolFrame.java Version 1.1 09/29/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.phi.gui;

import java.util.Observable;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import mask.maskSymbolsListStruct;
import rock.phi.gui.phiPlotSymbolPanel;
import rock.phi.phiStruct;

import util.utilImageIO;

/** Class phiPlotSymbolFrame
 *  <p> This Class will display the phi images available
 *
 *  @version 1.1 09/29/2008
 *  @author  John R. Victorine
 */

public class phiPlotSymbolFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable            notifier  = null; // Observable Object
  private maskSymbolsListStruct stSymbols = null; // Porosity Type Symbols

  // Global Variables

  private phiPlotSymbolPanel    plot      = null;
  private phiStruct             stPHI     = null; // porosity data structure

  private String sKEY   = "0";

  // -- Depth range variables

  private double dStart = 0.0;
  private double dEnd   = 0.0;

  // -- Porosity Value

  private double dPHI   = 0.0; // Porosity (%)

  // Global Frame Widgets

  // -- Depth Range Textfields

  private JTextField   txtStart   = new JTextField();    // Starting Depth
  private JTextField   txtEnd     = new JTextField();    // Ending Depth

  // -- Porosity Value

  private JTextField   txtPHI     = new JTextField();    // Porosity (%)

  // -- Global Frame Buttons

  private JButton      btnAdd     = new JButton();       // Add Fossils to List
  private JButton      btnClear   = new JButton();       // Clear Fossils data
  private JButton      btnCancel  = new JButton();       // Cancel Modify

  private JMenuItem    mExit      = null;                // Exit Profile Plot

  // Dialog Position & Size

  private int iXStart = 0;
  private int iYStart = 0;
  private int iWidth  = 0;
  private int iHeight = 0;

  /** Constructor phiPlotSymbolFrame()
   *  <p> This is the Constructor for this class.
   *  @param stSymbols = Porosity Types Plot Symbols List Data Structure
   */

  public phiPlotSymbolFrame( maskSymbolsListStruct stSymbols )
  {
    try
    {
      this.stSymbols = stSymbols;
      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Constructor phiPlotSymbolFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier  = Observable
   *  @param stSymbols = Porosity Types Plot Symbols List Data Structure
   */

  public phiPlotSymbolFrame( Observable notifier,
                             maskSymbolsListStruct stSymbols )
  {
    try
    {
      this.notifier  = notifier;
      this.stSymbols = stSymbols;

      jbInit();
      addWindowListener( new phiPlotSymbolFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Constructor phiPlotSymbolFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier  = Observable
   *  @param iXStart   = the x-coordinate of Data entry dialog
   *  @param iYStart   = the y-coordinate of Data entry dialog
   *  @param iWidth    = the width of Data entry dialog
   *  @param iHeight   = the height of Data entry dialog
   *  @param stSymbols = Porosity Types Plot Symbols List Data Structure
   */

  public phiPlotSymbolFrame( Observable notifier,
                             int iXStart, int iYStart,
                             int iWidth, int iHeight,
                             maskSymbolsListStruct stSymbols )
  {
    try
    {
      this.notifier  = notifier;
      this.iXStart   = iXStart;
      this.iYStart   = iYStart;
      this.iWidth    = iWidth;
      this.iHeight   = iHeight;
      this.stSymbols = stSymbols;

      jbInit();
      addWindowListener( new phiPlotSymbolFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlNorth      = new JPanel();
    JPanel pnlWest       = new JPanel();
    JPanel pnlData       = new JPanel();
    JPanel pnlPHI        = new JPanel();
    JPanel pnlDepthRange = new JPanel();
    JPanel pnlStart      = new JPanel();
    JPanel pnlEnd        = new JPanel();
    JPanel pnlAdd        = new JPanel();
    JPanel pnlClear      = new JPanel();

    TitledBorder titledBorderStart = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Starting Depth:");
    titledBorderStart.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderEnd = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Ending Depth:");
    titledBorderEnd.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderPhi = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Porosity (%)");
    titledBorderPhi.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Build Menu

    JMenuBar    menuBar  = new JMenuBar();
    JMenu       menuFile = new JMenu("Close Dialog");

    mExit = new JMenuItem("Close");
    mExit.addActionListener(this);

    menuFile.add( mExit );

    menuBar.add(menuFile);

    this.setJMenuBar(menuBar);

    // Build Dialog

    this.setTitle("Porosity Symbols");
    this.getContentPane().setLayout(new BorderLayout());

    // Data Entry Panel

    pnlWest.setLayout(new BorderLayout());

    // Enter Fossil Data Panel

    pnlData.setLayout(new BorderLayout());

    // -- Depth Range Panel

    pnlDepthRange.setPreferredSize(new Dimension(100, 120));
    pnlDepthRange.setLayout(new GridLayout(2,1));

    // .. Starting Depth Panel

    pnlStart.setBorder(titledBorderStart);
    pnlStart.setLayout(new BorderLayout());

    txtStart.setText("0.0");
    txtStart.setEditable(false);
    txtStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtStart.addFocusListener(new phiPlotSymbolFrameFocusAdapter(this));

    // .. Endind Depth Panel

    pnlEnd.setBorder(titledBorderEnd);
    pnlEnd.setLayout(new BorderLayout());

    txtEnd.setText("0.0");
    txtEnd.setEditable(false);
    txtEnd.setHorizontalAlignment(SwingConstants.TRAILING);
    txtEnd.addFocusListener(new phiPlotSymbolFrameFocusAdapter(this));

    // -- Porosity Panel

    pnlPHI.setLayout(new BorderLayout());
    pnlPHI.setBorder(titledBorderPhi);

    txtPHI.setText("0.0");
    txtPHI.setHorizontalAlignment(SwingConstants.TRAILING);
    txtPHI.addFocusListener(new phiPlotSymbolFrameFocusAdapter(this));

    // Add Fossils to List Panel

    pnlAdd.setLayout(new GridLayout(2,1));

    btnAdd.setFont(new java.awt.Font("Dialog", 1, 11));
    btnAdd.setText("Add");
    btnAdd.addActionListener(this);

    pnlClear.setLayout(new GridLayout());

    btnClear.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClear.setText("Clear");
    btnClear.addActionListener(this);

    btnCancel.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnCancel.setEnabled(false);
    btnCancel.setText("Cancel");
    btnCancel.addActionListener(this);

    // Create Porosity Plot Panel

    plot = new phiPlotSymbolPanel( this, notifier, stSymbols );

    // Attach Widgets to Frame

    this.getContentPane().add(pnlWest, BorderLayout.WEST);
    pnlWest.add(pnlData,               BorderLayout.NORTH);

    pnlData.add( pnlDepthRange,        BorderLayout.NORTH );
    pnlDepthRange.add( pnlStart,       null );
    pnlStart.add( txtStart,            BorderLayout.CENTER );

    pnlDepthRange.add( pnlEnd,         null );
    pnlEnd.add( txtEnd,                BorderLayout.CENTER );

    pnlData.add( pnlPHI,               BorderLayout.CENTER );
    pnlPHI.add( txtPHI,                BorderLayout.CENTER );

    pnlData.add(pnlAdd,                BorderLayout.SOUTH);
    pnlAdd.add(btnAdd,                 null);

    pnlAdd.add(pnlClear,               null);
//    pnlClear.add(btnCancel,            null);
    pnlClear.add(btnClear,             null);

    this.getContentPane().add(plot,    BorderLayout.CENTER);

    // Display the Frame

    this.setSize(new Dimension(500, 550)); //350));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();

    int iXPos = (d.width  - this.getSize().width) / 2;
    if (iWidth > 0) iXPos = iXStart + iWidth;
    this.setLocation(iXPos, (d.height - this.getSize().height) / 2);

//    this.setResizable(false);
    this.setVisible(true);
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getData()
   * <p> This method will get the array of porosity selected
   * @return st = porosity data Structure
   */

  public phiStruct getData()
  {
    if (stPHI == null)
    {
      stPHI = new phiStruct();
      sKEY  = new String(cmn.cmnString.UniqueName());
    }

    if (plot != null) stPHI = plot.getData();

    stPHI.sKEY       = new String( sKEY );
    stPHI.depthStart = dStart;
    stPHI.depthEnd   = dEnd;

    if (dEnd == 0.0) dEnd = dStart;

    stPHI.dPHI       = dPHI;

    return (stPHI);
  }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setData()
   * <p> This method will set the array of porosity selected
   * @param st = porosity data Structure
   */

  public void setData(phiStruct st)
  {
    this.stPHI = st;

    if (st != null)
    {
      sKEY        = new String( st.sKEY );

      this.dStart = st.depthStart;
      this.dEnd   = st.depthEnd;

      txtStart.setText( "" + this.dStart );
      txtEnd.setText( "" + this.dEnd );

      this.dPHI   = st.dPHI;

      txtPHI.setText( "" + st.dPHI );
    }

    if (plot != null) plot.setData( st );

    btnAdd.setText("Modify");
  }

  /** Method setDepthRange(double depthStart, double depthEnd)
   * <p> This method will set the depth range of the data
   * @param depthStart = the starting depth
   * @param depthEnd   = the ending depth
   */

  public void setDepthRange(double depthStart, double depthEnd)
  {
    this.dStart = depthStart;
    txtStart.setText( "" + this.dStart );

    this.dEnd = depthEnd;
    txtEnd.setText( "" + this.dEnd );
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method add()
   * <p> This method will add the selected data to the fossil list structure
   */

  public void add()
  {
	if (notifier != null)
      notifier.notifyObservers(new String("ADD or MODIFY Porosity Data"));
  }

  /** Method cancel()
   * <p> This method will cancel the edit
   */

  public void cancel()
  {
	if (notifier != null)
      notifier.notifyObservers(new String("Cancel Edit"));
  }

  /** Method clear()
   * <p> This method will clear the textfields
   */

  public void clear()
  {
    dPHI = 0.0;            // Porosity (%)

    txtPHI.setText("0.0"); // Porosity (%)

    if (stPHI   != null)
      stPHI.delete();
    stPHI = null;

    if (plot != null) plot.setData( stPHI );
  }

  /** METHOD phiPlotSymbolFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class phiPlotSymbolFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
  	  if (notifier != null)
        notifier.notifyObservers(new String("Cancel Edit"));
    }
  }

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    notifier  = null; // Observable Object
    stSymbols = null;

    if (plot != null)
      plot.close();
    plot = null;

    stPHI      = null; // porosity data structure
    sKEY       = null;

    // Global Frame Widgets

    // -- Depth Range Textfields

    txtStart   = null; // Starting Depth
    txtEnd     = null; // Ending Depth

    // -- Porosity Value

    txtPHI     = null; // Porosity (%)

    // -- Global Frame Buttons

    btnAdd     = null; // Add Fossils to List
    btnClear   = null; // Clear Fossils data
    btnCancel  = null; // Cancel Modify

    mExit      = null; // Exit Profile Plot

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if ( event.getSource() == btnAdd  )   { add(); }
    if ( event.getSource() == btnClear )  { clear(); }
    if ( event.getSource() == btnCancel ) { cancel(); }

    if ( event.getSource() == mExit )     { close(); }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtStart)
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Start Depth Value is a Numeric Field");
    }

    if (e.getSource() == txtEnd)
    {
      iNumeric = 1;
      sTemp = txtEnd.getText();
      sMessage = new String("End Depth Value is a Numeric Field");
    }

    if (e.getSource() == txtPHI)
    {
      iNumeric = 1;
      sTemp = txtPHI.getText();
      sMessage = new String("Porosity Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
        if (e.getSource() == txtStart)
          dStart = cmn.cmnString.stringToDouble(txtStart.getText());

        if (e.getSource() == txtEnd)
          dEnd = cmn.cmnString.stringToDouble(txtEnd.getText());

        if (e.getSource() == txtPHI)
          dPHI = cmn.cmnString.stringToDouble(txtPHI.getText());
      }
    }
  }
}

/** CLASS phiPlotSymbolFrameFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class phiPlotSymbolFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  phiPlotSymbolFrame adaptee;

  phiPlotSymbolFrameFocusAdapter(phiPlotSymbolFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 09/29/2008
 *  @author  John Victorine
 */
