/*
 * @rockCSVFile.java Version 1.1 1/16/2008
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.io;

import java.awt.*;
import java.io.*;
import java.lang.*;
import java.util.*;

import util.utilFileIO;
import cmn.cmnString;
import rock.rockDataListStruct;
import rock.rockDataStruct;

/** Class rockCSVFile
 *  <p> This Class is designed to open an ascii text rock data CSV file and read
 *      the contents and parse the data into the rock Structure.
 *
 *  @version 1.1 1/16/2008
 *  @author  John R. Victorine
 */

public class rockCSVFile
{
  private utilFileIO         pFILE  = null;

  private String             sLine1 = "";
  private String             sLine2 = "";
  private rockDataListStruct stList = null;

  private int         iRows    = 0;
  private int         iColumns = 0;

  private int         iError   = util.utilFileIO.NOERROR;
  private String      sError   = new String("");

  /** Construct rockCSVFile()
   *  <p> This is the Constructor for this class.
   */

  public rockCSVFile() { }

  /* ====================================================================== *
   * ---------------------------- FILE METHODS ---------------------------- *
   * ====================================================================== */

  /** Method Open()
    * <p> This is the Constructor for this class.
    * @param iFileID    = To access the file as URL or FILE.
    * @param sDirectory = File URL or Directory Path
    * @param sFilename  = File Name
    */

  public void Open(int iFileID, String sDirectory, String sFilename)
  {
    pFILE   = new utilFileIO();
    pFILE.Open(iFileID, util.utilFileIO.READ, sDirectory, sFilename);
    iError = pFILE.getErrorID();
    sError = pFILE.getError();

    iRows             = 0;
    iColumns          = 0;
  }

  /** Method Close()
   *  <p> This method closes the File.
   */

  public void Close() { pFILE.Close(); }

  /** Method delete()
   * <p> This method will set all the structures to null to force java to free
   *     the allocated memory
   */

  public void delete()
  {
    sLine1   = null;
    sLine2   = null;
    sError   = null;

    if (pFILE != null)
      pFILE.delete();
    pFILE = null;
  }

  /** Method get2Rows()
   * <p> This method will read and parse the Rock File and get the first 2 lines.
   */

  public void get2Rows()
  {
    int             iCount  = 0;
    String          sTemp   = "";
    StringTokenizer stToken = null;

    if (iError == util.utilFileIO.NOERROR)
    {
      while (!pFILE.Next().equals(util.utilFileIO.EOF))
      {
        iError = pFILE.getErrorID();
        sError = pFILE.getError();

        sTemp = pFILE.getString();
        if (iCount == 0)
        {
          sLine1   = new String(sTemp);
          stToken  = new StringTokenizer(sTemp, ",");
          iColumns = stToken.countTokens();
        }

        if (iCount == 1)
          sLine2 = new String(sTemp);

        iCount++;
        iRows++;
      }
    }
  }

  /** Method getData()
   * <p> This method will read and parse the Rock File into its data structures.
   * @param iStart  = the starting row number
   * @param iColumn = Array of file column mapping to the rock Data structure
   */

  public void getData(int iStart, int iColumn[])
  {
    int    i=0;
    int    iCount = 1;
    String sTemp  = "";
    rockDataStruct st = null;
    String sKEY = cmn.cmnString.UniqueName();
    String sKID = "0";
    int    iValue = 0;
    int    iBegin = 0;

    stList = new rockDataListStruct();

    if (iError == util.utilFileIO.NOERROR)
    {
      while (!pFILE.Next().equals(util.utilFileIO.EOF))
      {
        iError = pFILE.getErrorID();
        sError = pFILE.getError();

        if (iError == util.utilFileIO.NOERROR)
        {
          if (iCount >= iStart)
          {
            sTemp  = pFILE.getString();
            sKID   = new String(sKEY + iCount);
            st     = parseRow(sKID, sTemp, iColumn);

            if (st.dTOP != rock.rockDataStruct.dNULL)
              stList = rock.rockDataUtility.add(st, stList);

            sTemp = "";
            st.delete();
            st = null;
          }

          iCount++;
        }
      }
    }

    stList = rock.rockDataUtility.computeDepthRange(stList);
    stList = rock.rockDataUtility.computeBulkDensity(stList);
    stList = rock.rockDataUtility.bubbleSort(stList);
    stList = rock.rockDataUtility.addBaseDepth( stList );
    stList = rock.rockDataUtility.addPorosity( stList );
  }

  /** Method parseRow()
   * <p> This method will parse a Rock File row into the rock structure.
   * @param  sKEY    = Unique key for formation name
   * @param  sData   = Data Row String
   * @param  iColumn = The mapped columns
   * @return st      = the rock Data Structure
   */

  private rockDataStruct parseRow(String sKEY, String sData, int iColumn[])
  {
    rockDataStruct st = new rockDataStruct();
    int    iCount = 0;
    int    iFound = 0;
    int    iNext  = 0;
    String sTemp  = "";
    StringTokenizer stToken = null;
    String sTokens[] = null;

    st.sKEY  = new String(sKEY);

    sTemp    = new String(sData.replace('\t', ' '));
    sTokens  = las3.las3Parse.splitDataLine( sTemp, "," );

    iColumns = 0;
    if (sTokens != null)
      iColumns = sTokens.length;

//    stToken  = new StringTokenizer(sTemp, ",");
//    iColumns = stToken.countTokens();

    for (int i=0; i<iColumns; i++)
    {
      sTemp = sTokens[i];
//      sTemp = stToken.nextToken();
      sTemp = new String(sTemp.trim());

      iNext  = i+1;
      iFound = -1;
      for (int j=0; j<rock.rockStandardTools.TOTAL; j++)
      {
        if (iNext == iColumn[j])
          iFound = j;
      }

      if (iFound > -1)
        st = mapData(sTemp, rock.rockStandardTools.ROCK_TOOLS[iFound][0], st);
    }

    return (st);
  }

  /** Method mapData()
   * <p> This method will parse a Rock File row into the rock structure.
   * @param  sData       = Data Row String
   * @param  sIdentifier = the column idenfier
   * @param  st          = the rock Data Structure
   * @return st          = the rock Data Structure
   */

  private rockDataStruct mapData(String sData,
                                 String sIdentifier,
                                 rockDataStruct st)
  {
    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.KID_R))
    {
      st.sKID = new String(sData);
    }
    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.KEY_R))
    {
      st.sKEY = new String(sData);
    }

    // Proprietary identifier

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.PUBLIC))
    {
        st.sProprietary = new String(sData);
    }

    // Depth Top of Rock

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.TOP))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._TOP = 1;
        st.dTOP = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Depth Base of Rock

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.BASE))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BASE = 1;
        st.dBASE = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Depth Correction

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.CORRECTION))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CORR = 1;
        st.dCORR = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Stratigraphic Unit

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.STU))
    {
      stList._STU = 1;
      st.sUNIT = new String(sData);
    }

    // Stratigraphic Name

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.STN))
    {
      stList._STN = 1;
      st.sNAME = new String(sData);
    }

    // Environment

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.ENV))
    {
      stList._ENV = 1;
      st.sENV = new String(sData);
    }

    // Lithofacies

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.LITHO))
    {
      stList._LITHO = 1;
      st.sLITHO = new String(sData);
    }

    // Porosity Whole Core

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.PCORE))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PCORE = 1;
        st.dPCORE = cmn.cmnString.stringToDouble(sData);
//        st.dNPHI  = st.dPCORE;
      }
    }

    // Porosity Plug

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.PPLUG))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PPLUG = 1;
        st.dPPLUG = cmn.cmnString.stringToDouble(sData);
//        st.dNPHI  = st.dPPLUG;
      }
    }

    // Porosity Plug 800 PSI

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.P800))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._P800 = 1;
        st.dP800 = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Porosity Plut Insitu

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.PINSI))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PINSI = 1;
        st.dPINSI = cmn.cmnString.stringToDouble(sData);
//        st.dNPHI = st.dPINSI;
      }
    }

    // Effective Rock Porosity

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.PEFF))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PEFF = 1;
        st.dPEFF = cmn.cmnString.stringToDouble(sData);
//        st.dNPHI = st.dPEFF;
      }
    }

    // Permeability Whole Max

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.KMAX))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KMAX = 1;
        st.dKMAX = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Permeability Whole 90

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.K90))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._K90 = 1;
        st.dK90 = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Permeability Whole Vertical

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.KVRT))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KVRT = 1;
        st.dKVRT = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Permeability Plug Routine

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.KPLG))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KPLG = 1;
        st.dKPLG = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Permeability Plug Klinkenberg Routine

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.KKL))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KKL = 1;
        st.dKKL = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Permeability Plug Insitu

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.KINSI))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KINSI = 1;
        st.dKINSI = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Permeability Plug Klinkenberg Insitu

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.KKLIN))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KKLIN = 1;
        st.dKKLIN = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Permeability Plug Vertical

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.KPVRT))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KPVRT = 1;
        st.dKPVRT = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Saturation Oil

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.SOIL))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SOIL = 1;
        st.dSOIL = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Saturation Water

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.SW))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SW = 1;
        st.dSW = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Grain Density

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.GMCC))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._GMCC = 1;
        st.dGMCC = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Density of Rock Dry

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.RHOD))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._RHOD = 1;
        st.dRHOD = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Density of Rock Wet

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.RHOW))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._RHOW = 1;
        st.dRHOW = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Archie Cementation (M) Ambient

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.MAMB))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._MAMB = 1;
        st.dMAMB = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Archie Cementation (M) Insitu

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.MINSI))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._MINSI = 1;
        st.dMINSI = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Archie Saturation (N) Ambient

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.NAMB))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NAMB = 1;
        st.dNAMB = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Archie Saturation (N) Insitu

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.NINSI))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NINSI = 1;
        st.dNINSI = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Gamma Ray

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.GR))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._GR = 1;
        st.dGR = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Gamma Ray minus Uranium

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.CGR))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CGR = 1;
        st.dCGR = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Thorium Concentration

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.THOR))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._TH = 1;
        st.dTh = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Uranium Concentration

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.URAN))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._U = 1;
        st.dU = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Potasium Concentration

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.POTA))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._K = 1;
        st.dK = cmn.cmnString.stringToDouble(sData);
      }
    }

    // Linear Curves

    if (sIdentifier.equals("LIN_1"))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._LIN_1 = 1;
        stList.sUnknown[0][0] = new String(sIdentifier);
        st.dLIN_1 = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals("LIN_2"))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._LIN_2 = 1;
        stList.sUnknown[1][0] = new String(sIdentifier);
        st.dLIN_2 = cmn.cmnString.stringToDouble(sData);
	  }
    }
    if (sIdentifier.equals("LIN_3"))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._LIN_3 = 1;
        stList.sUnknown[2][0] = new String(sIdentifier);
        st.dLIN_3 = cmn.cmnString.stringToDouble(sData);
	  }
    }
    if (sIdentifier.equals("LIN_4"))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._LIN_4 = 1;
        stList.sUnknown[3][0] = new String(sIdentifier);
        st.dLIN_4 = cmn.cmnString.stringToDouble(sData);
	  }
    }

    // Log Curves

    if (sIdentifier.equals("LOG_1"))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._LOG_1 = 1;
        stList.sUnknown[0][1] = new String(sIdentifier);
        st.dLOG_1 = cmn.cmnString.stringToDouble(sData);
	  }
    }
    if (sIdentifier.equals("LOG_2"))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._LOG_2 = 1;
        stList.sUnknown[1][1] = new String(sIdentifier);
        st.dLOG_2 = cmn.cmnString.stringToDouble(sData);
	  }
    }
    if (sIdentifier.equals("LOG_3"))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._LOG_3 = 1;
        stList.sUnknown[2][1] = new String(sIdentifier);
        st.dLOG_3 = cmn.cmnString.stringToDouble(sData);
	  }
    }
    if (sIdentifier.equals("LOG_4"))
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._LOG_4 = 1;
        stList.sUnknown[3][1] = new String(sIdentifier);
        st.dLOG_4 = cmn.cmnString.stringToDouble(sData);
	  }
    }

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.LTHCD))    // Lithofacies Code
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._LTHCD = 1;
        st.iLITH_CD = (int) cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.FRACTURE)) // Fractures
    {
      stList._FRACTURE = 1;
      st.sFracture = new String(sData);
    }

    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.SOURCE))   // Source
    {
      st.source = new String(sData);
    }
    if (sIdentifier.equals(rock.io.ReadRockDataXMLFile.DATE))     // Date
    {
      st.sDate = new String(sData);
    }

    return (st);
  }

  /* ====================================================================== *
   * ----------------------------- GET METHODS ---------------------------- *
   * ====================================================================== */

  /** Method getLine1()
   * <p> This method will return the 1st Line of the delimited file
   * @return sLine1 = 1st line of delimited file
   */

  public String getLine1() { return (sLine1); }

  /** Method getLine2()
   * <p> This method will return the 2nd Line of the delimited file
   * @return sLine2 = 2nd line of delimited file
   */

  public String getLine2() { return (sLine2); }

  /** Method getRock()
   * <p> This method will return rock data list
   * @return stList = rock Data list structure
   */

  public rockDataListStruct getRock() { return (stList); }

  /* ====================================================================== *
   * -------------------------- GET ERROR METHODS ------------------------- *
   * ====================================================================== */

  /** METHOD getErrorID()
   *  <p> This method will return the error number.
   *  @return iError = Error Indicator
   */

  public int getErrorID() { return (iError); }

  /** METHOD getError()
   *  <p> This method will return the error string.
   *  @return sError = Error String
   */

  public String getError() { return (sError); }
}

/*
 *  @version 1.1 1/16/2008
 *  @author  John Victorine
 */
