/*
 * @ReadRockImageXMLFile.java Version 1.0 03/26/2004
 *
 * Copyright (c) 2001 John R. Victorine
 * All Rights Reserved.
 */

package rock.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import rock.rockImagesListStruct;
import rock.rockImagesStruct;

/** Class ReadRockImageXMLFile()
 * <p> This Class will parse the Core Image File XML Data Stream
 *     directly to the Core Image File List Data Structure.
 *
 *  @version 1.1 03/26/2004
 *  @author  John R. Victorine
 */

public class ReadRockImageXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  public static final String ROOTNAME     = "image_files";  // ROOT Name
  public static final String RECORDS      = "records";  // Total # of Records

  // Well Information Data

  public static final String WELL         = "well";      // ELEMENT info

  public static final String KID          = "kid";       // Well Primary key
  public static final String KEY          = "key";       // Primary Key
  public static final String TYPE         = "type";      // type of data
  public static final String API_NUMBER   = "api";       // API-Number
  public static final String NAME         = "name";      // outcrop name
  public static final String WELL_STATUS  = "status";    // Well Status
  public static final String ERROR_VAL    = "error";     // DB Error
  public static final String LATITUDE     = "latitude";  // Latitude
  public static final String LONGITUDE    = "longitude"; // Longitude
  public static final String DEPTH        = "depth";     // Depth or Height
  public static final String GROUND_LEVEL = "gl";        // Ground Level

  // Rock Image Data

  public static final String IMAGE         = "image";     // IMAGE ELEMENT

  public static final String LOCATION      = "location";  // Location of file
  public static final String DIRECTORY     = "dir";       // Directory Path
  public static final String PATH          = "path";      // URL Path
  public static final String FILENAME      = "filename";  // Preview Filename
  public static final String FILENAME_P    = "filenam_p"; // Original Filename
  public static final String FORMAT        = "format";    // File Image Format
  public static final String IMAGE_TYPE    = "type";      // CORE Image Type
  public static final String START_DEPTH   = "start";     // CORE Starting Depth
  public static final String END_DEPTH     = "end";       // CORE Ending Depth
  public static final String AUTHORS       = "authors";   // Authors
  public static final String YEAR          = "year";      // Year
  public static final String TITLE         = "title";     // Title
  public static final String SOURCE        = "source";    // Source
  public static final String DATE          = "date";      // Update date
  public static final String THUMBNAIL     = "thumbnail"; // Thumbnail created

  public static final String DESCRIPTION   = "desc";      // Description
  public static final String COMMENTS      = "comments";  // Comments

  private int iType  = FILE;
  private int iRows  = 0;
  private int iCount = -1;

  private static final int _NONE    = 0;
  private static final int _WELL    = 1;
  private static final int _DATA    = 2;

  private int iData  = _NONE;

  private int iDescription = 0; // ELEMENT Description found
  private int iComments    = 0; // ELEMETN Comments found

  private rockImagesListStruct stList = new rockImagesListStruct();

  private int    iError = 0;
  private String sError = "";

  /** Method ReadRockImageXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadRockImageXMLFile() { iType = FILE; }

  /** Method ReadRockImageXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadRockImageXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public rockImagesListStruct Process(String filename)
  {
    stList.iCount = 0;
    Read(filename);
    stList.iCount = iCount+1;

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    // If the Data ELEMENT is detected then initialize the structure.

    if (eName.equals(WELL))
    {
      iData  = _WELL;
    }

    if (eName.equals(IMAGE))
    {
      iDescription = 0; // ELEMENT Description found
      iComments    = 0; // ELEMETN Comments found

      iCount++;
      if (iCount < iRows)
      {
        stList.stItem[iCount] = new rockImagesStruct();
      }

      iData  = _DATA;
    }

    if (eName.equals(DESCRIPTION))
    {
      iDescription = 1; // ELEMENT Description found
      iComments    = 0; // ELEMETN Comments found
    }

    if (eName.equals(COMMENTS))
    {
      iDescription = 0; // ELEMENT Description found
      iComments    = 1; // ELEMETN Comments found
    }

    // Add Attributes to the Data List Structure

    if (attrs != null)
    {
      for (int i = 0; i < attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          iRows = cmn.cmnString.stringToInt(sOut.trim());

          if (iRows > 0)
          {
            stList.stItem = new rockImagesStruct[iRows];
          }
        }

        if (iData  == _WELL)
          ParseData(aName, sOut.trim());

        // Parse each attribute and add it to the structure

        if (iData  == _DATA)
        {
          if (iCount > -1)
          {
            if (iCount < iRows)
            {
              stList.stItem[iCount] = ParseData(aName,
                                                sOut.trim(),
                                                stList.stItem[iCount]);
            }
          }
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    String s = new String(buf, offset, len);

    if (iDescription == 1)
    {
      stList.stItem[iCount].sDescription = new String(
          stList.stItem[iCount].sDescription + " " + s.trim());
    }

    if (iComments == 1)
    {
      stList.stItem[iCount].sComments = new String(
          stList.stItem[iCount].sComments + " " + s.trim());
    }
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** ParseData()
   * <p> This method will parse the Data Stream for the Information
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = Image File String
   * @param  st         = The Image File Data Structure
   * @return st         = The Image File Data Structure
   */

  private rockImagesStruct ParseData(String sIdentifier,
                                            String sData,
                                            rockImagesStruct st)
  {
    if (sIdentifier.equals(KID)) { st.sKID = new String(sData); }
    if (sIdentifier.equals(KEY)) { st.sKEY = new String(sData); }

    if (sIdentifier.equals(LOCATION))    { st.sLocation = new String(sData); }
    if (sIdentifier.equals(DIRECTORY))   { st.sDirectory = new String(sData); }
    if (sIdentifier.equals(PATH))        { st.sURL = new String(sData); }
    if (sIdentifier.equals(FILENAME))    { st.sFilename = new String(sData); }
    if (sIdentifier.equals(FILENAME_P))  { st.sOriginal = new String(sData); }
    if (sIdentifier.equals(FORMAT))      { st.sFormat = new String(sData); }
    if (sIdentifier.equals(IMAGE_TYPE))  { st.sType = new String(sData); }

    if (sIdentifier.equals(START_DEPTH))
    {
      st.depthStart = 0.0;
      if (cmn.cmnString.isNumeric(sData))
        st.depthStart = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(END_DEPTH))
    {
      st.depthEnd = 0.0;
      if (cmn.cmnString.isNumeric(sData))
        st.depthEnd = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(AUTHORS))     { st.sAuthors = new String(sData); }
    if (sIdentifier.equals(YEAR))        { st.sYear = new String(sData); }
    if (sIdentifier.equals(TITLE))       { st.sTitle = new String(sData); }
    if (sIdentifier.equals(SOURCE))
    {
	  st.source = new String(sData);
	  stList.sIMAGE_SRC = new String(sData);
	}
    if (sIdentifier.equals(DATE))
    {
	  st.sDate = new String(sData);
	  stList.sIMAGE_DATE = new String(sData);
	}
    if (sIdentifier.equals(THUMBNAIL))   { st.sThumbnail = new String(sData); }

    return (st);
  }

  /** ParseData()
   * <p> This method will parse the Data Stream for the Individual Well
   *     Header Information
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = Well Header Information String
   */

  private void ParseData(String sIdentifier, String sData)
  {
    // Identification Information

    if (sIdentifier.equals(KID)) { stList.sKID = new String(sData); }
    if (sIdentifier.equals(KEY)) { stList.sKEY = new String(sData); }
    if (sIdentifier.equals(TYPE))
    {
      if (sData.equals("WELL"))    stList.iType = iqstrat.iqstratHeadersStruct.WELL;
      if (sData.equals("OUTCROP")) stList.iType = iqstrat.iqstratHeadersStruct.OUTCROP;
    }
    if (sIdentifier.equals(API_NUMBER))  { stList.sAPI = new String(sData); }
    if (sIdentifier.equals(NAME))        { stList.sName = new String(sData);  }
    if (sIdentifier.equals(WELL_STATUS)) { stList.status = new String(sData); }
    if (sIdentifier.equals(ERROR_VAL))
    {
      iError    = -1;
      sError    = new String(sData);
    }

    // XY Position Information

    if (sIdentifier.equals(LATITUDE))
    {
      stList.dLatitude = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(LONGITUDE))
    {
      stList.dLongitude = cmn.cmnString.stringToDouble(sData);
    }

    // Z Position Information

    if (sIdentifier.equals(DEPTH))
    {
      stList.depth = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(GROUND_LEVEL))
    {
      stList.dGL = cmn.cmnString.stringToDouble(sData);
    }
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}
