/*
 * @ReadRockDataXMLFile.java Version 1.0 11/06/2007
 *
 * Copyright (c) 2007 John R. Victorine
 * All Rights Reserved.
 */

package rock.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import rock.rockDataListStruct;
import rock.rockDataStruct;
import rock.rockColumnsStruct;

/** Class ReadRockDataXMLFile()
 * <p> This Class will parse the Measured Rock XML Data Stream
 *     directly to the Measured Rock List Data Structure.
 *
 *  @version 1.1 11/06/2007
 *  @author  John R. Victorine
 */

public class ReadRockDataXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  private int              iType  = FILE;
  private int              iRows  = 0;
  private int              iCount = -1;

  private int              iComments = 0;

  private static final int _NONE    = 0;
  private static final int _WELL    = 1;
  private static final int _COLUMNS = 2;
  private static final int _DATA    = 3;

  public static final String ROOTNAME = "rock_data"; // ROOT NAME
  public static final String RECORDS  = "records";   // Attribute

  // Well Information Data

  public static final String WELL        = "well";      // ELEMENT info

  public static final String KID         = "kid";       // Well Primary key
  public static final String KEY         = "key";       // Primary Key
  public static final String TYPE        = "type";      // type of data
  public static final String API_NUMBER  = "api";       // API-Number
  public static final String WELL_STATUS = "status";    // Well Status
  public static final String ERROR_VAL   = "error";     // DB Error
  public static final String LATITUDE    = "latitude";  // Latitude
  public static final String LONGITUDE   = "longitude"; // Longitude
  public static final String DEPTH       = "depth";     // Depth or Height
  public static final String GROUND_LEVEL = "gl";       // Ground Level

  // Rock Column Data

  public static final String COLUMNS  = "columns";   // ELEMENT
  public static final String COLUMN   = "column";    // ELEMENT

  public static final String MNEMONIC = "mnemonic";  // Attribute
  public static final String NAME     = "name";      // Attribute
  public static final String UNIT     = "unit";      // Attribute

  // Rock Data

  public static final String DATA     = "data";      // ELEMENT
  public static final String ROW      = "row";       // ELEMENT

  public static final String KID_R      = "KID";    // Well Primary key
  public static final String KEY_R      = "KEY";    // Primary Key
  public static final String PUBLIC     = "PUBLIC"; // Proprietary identifier

  public static final String TOP        = "TOP";  // Depth Top of Rock
  public static final String BASE       = "BASE"; // Depth Base of Rock
  public static final String CORRECTION = "CORR"; // Depth Correction

  public static final String STU      = "STU";    // Stratigraphic Unit
  public static final String STN      = "STN";    // Stratigraphic Name
  public static final String ENV      = "ENV";    // Environment
  public static final String LITHO    = "LITHO";  // Lithofacies

  public static final String PCORE    = "PCORE";  // Porosity Whole Core
  public static final String PPLUG    = "PPLUG";  // Porosity Plug
  public static final String P800     = "P800";   // Porosity Plug 800 PSI
  public static final String PINSI    = "PINSI";  // Porosity Plut Insitu
  public static final String PEFF     = "PEFF";   // Effective Rock Porosity

  public static final String KMAX     = "KMAX";   // Permeability Whole Max
  public static final String K90      = "K90";    // Permeability Whole 90
  public static final String KVRT     = "KVRT";   // Permeability Whole Vertical
  public static final String KPLG     = "KPLG";   // Permeability Plug Routine
  public static final String KKL      = "KKL";    // Permeability Plug Klinkenberg Routine
  public static final String KINSI    = "KINSI";  // Permeability Plug Insitu
  public static final String KKLIN    = "KKLIN";  // Permeability Plug Klinkenberg Insitu
  public static final String KPVRT    = "KPVRT";  // Permeability Plug Vertical

  public static final String SOIL     = "SOIL";   // Saturation Oil
  public static final String SW       = "SW";     // Saturation Water

  public static final String GMCC     = "GMCC";   // Grain Density
  public static final String RHOD     = "RHOD";   // Density of Rock Dry
  public static final String RHOW     = "RHOW";   // Density of Rock Wet

  public static final String MAMB     = "MAMB";   // Archie Cementation (M) Ambient
  public static final String MINSI    = "MINSI";  // Archie Cementation (M) Insitu
  public static final String NAMB     = "NAMB";   // Archie Saturation (N) Ambient
  public static final String NINSI    = "NINSI";  // Archie Saturation (N) Insitu

  public static final String GR       = "GR";     // Gamma Ray
  public static final String CGR      = "CGR";    // Gamma Ray minus Uranium

  public static final String THOR     = "THOR";   // Thorium Concentration
  public static final String URAN     = "URAN";   // Uranium Concentration
  public static final String POTA     = "POTA";   // Potasium Concentration

  public static final String LTHCD    = "LTHCD";    // Lithofacies Code
  public static final String FRACTURE = "FRACTURE"; // Fractures

  public static final String SOURCE   = "source";   // Lithofacies Code
  public static final String DATE     = "date";     // Lithofacies Code
  public static final String COMMENTS = "comments"; // Lithofacies Code

  private int iData  = _NONE;

  private rockColumnsStruct  stColumn = new rockColumnsStruct();
  private rockDataListStruct stList   = new rockDataListStruct();

  private int    iError = 0;
  private String sError = "";

  /** Method ReadRockDataXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadRockDataXMLFile() { iType = FILE; }

  /** Method ReadRockDataXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadRockDataXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public rockDataListStruct Process(String filename)
  {
    Read(filename);

    stList = rock.rockDataUtility.computeDepthRange(stList);
    stList = rock.rockDataUtility.computeBulkDensity(stList);
    stList = rock.rockDataUtility.bubbleSort(stList);
    stList = rock.rockDataUtility.addBaseDepth( stList );
    stList = rock.rockDataUtility.addPorosity( stList );
/*
    if (stList != null)
    {
      for (int i=0; i<stList.iCount; i++)
      {
        stList.stItem[i].print();
      }
    }
*/

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println("ReadRockDataXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println("ReadRockDataXMLFile.Read() " + e);
          e.printStackTrace();
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    // If the ELEMENT Well is detected then initialize the structure.

    if (eName.equals(WELL))
    {
      iData  = _WELL;

      stList._TOP      = -1;
      stList._BASE     = -1;
      stList._CORR     = -1;
      stList._STU      = -1;
      stList._STN      = -1;
      stList._ENV      = -1;
      stList._LITHO    = -1;
      stList._PCORE    = -1;
      stList._PPLUG    = -1;
      stList._P800     = -1;
      stList._PINSI    = -1;
      stList._PEFF     = -1;
      stList._KMAX     = -1;
      stList._K90      = -1;
      stList._KVRT     = -1;
      stList._KPLG     = -1;
      stList._KKL      = -1;
      stList._KINSI    = -1;
      stList._KKLIN    = -1;
      stList._KPVRT    = -1;
      stList._SOIL     = -1;
      stList._SW       = -1;
      stList._GMCC     = -1;
      stList._RHOD     = -1;
      stList._RHOW     = -1;
      stList._MAMB     = -1;
      stList._MINSI    = -1;
      stList._NAMB     = -1;
      stList._NINSI    = -1;
      stList._LTHCD    = -1;
      stList._FRACTURE = -1;
      stList._GR       = -1;
      stList._CGR      = -1;
      stList._TH       = -1;
      stList._U        = -1;
      stList._K        = -1;
    }

    if (eName.equals(COLUMNS))
    {
      iRows  = 0;
      iCount = -1;
      iData  = _COLUMNS;
    }

    if (eName.equals(COLUMN))
    {
      iCount++;
      if (iCount < iRows)
      {
        stList.stColumns[iCount] = new rockColumnsStruct();
      }
    }

    if (eName.equals(DATA))
    {
      iComments = 0;

      iRows  = 0;
      iCount = -1;
      iData  = _DATA;
    }

    if (eName.equals(ROW))
    {
      iCount++;
      if (iCount < iRows)
      {
        stList.stItem[iCount] = new rockDataStruct();
      }
    }

    if (eName.equals(COMMENTS)) { iComments = 1; }

    // Add Attributes to the Well Header Data List Structure

    if (attrs != null)
    {
      for (int i=0; i < attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          switch (iData)
          {
            case _NONE:
              break;

            case _COLUMNS:
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stColumns = new rockColumnsStruct[iRows];
              stList.iColumns  = iRows;
              break;

            case _DATA:
              iRows = cmn.cmnString.stringToInt(sOut.trim());
              stList.stItem = new rockDataStruct[iRows];
              stList.iCount = iRows;
              break;
          }
        }

        // Parse each attribute and add it to the structure

        if (iCount > -1)
        {
          switch (iData)
          {
            case _NONE:
              break;

            case _WELL:
              ParseData(aName, sOut.trim());
              break;

            case _COLUMNS:
              if (iCount < iRows)
              {
                stList.stColumns[iCount] = ParseColumns(aName,
                                                     sOut.trim(),
                                                     stList.stColumns[iCount]);
              }
              break;

            case _DATA:
              if (iCount < iRows)
              {
                stList.stItem[iCount] = ParseData(aName,
                                                  sOut.trim(),
                                                  stList.stItem[iCount]);
              }
              break;
          }
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    String s = new String(buf, offset, len);
    if (iComments == 1)
    {
      stList.stItem[iCount].sComments = new String(
          stList.stItem[iCount].sComments + " " + s.trim());
    }
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** ParseColumns()
   * <p> This method will parse the Data Stream
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = Measured Core Data String
   * @param  st         = The Measured Core Data Data Structure
   * @return st         = The Measured Core Data Data Structure
   */

  private rockColumnsStruct ParseColumns( String sIdentifier,
                                          String sData,
                                          rockColumnsStruct st)
  {
    if (sIdentifier.equals(MNEMONIC))
    {
      st.sMnemonic = new String(sData);
    }

    if (sIdentifier.equals(NAME))
    {
      st.sName = new String(sData);
    }

    if (sIdentifier.equals(UNIT))
    {
      st.sUnit = new String(sData);
    }

    if (sIdentifier.equals(TYPE))
    {
      st.sKEY = new String(sData);
    }

    return (st);
  }

  /** ParseData()
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData      = Measured Core Data String
   * @param  st         = Rock Data Structure
   * @return st         = Rock Data Structure
   */

  private rockDataStruct ParseData(String sIdentifier,
                                   String sData,
                                   rockDataStruct st)
  {
    if (sIdentifier.equals(KID_R)) { st.sKID = new String(sData); }
    if (sIdentifier.equals(KEY_R)) { st.sKEY = new String(sData); }
    if (sIdentifier.equals(PUBLIC)) // Proprietary identifier
    {
        st.sProprietary = new String(sData);
    }

    if (sIdentifier.equals(TOP))  // Depth Top of Rock
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._TOP = 1;
        st.dTOP     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(BASE)) // Depth Base of Rock
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._BASE = 1;
        st.dBASE     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(CORRECTION)) // Depth Correction
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CORR = 1;
        st.dCORR     = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(STU))    // Stratigraphic Unit
    {
      stList._STU = 1;
      st.sUNIT    = new String(sData);
    }
    if (sIdentifier.equals(STN))    // Stratigraphic Name
    {
      stList._STN = 1;
      st.sNAME    = new String(sData);
    }
    if (sIdentifier.equals(ENV))    // Environment
    {
      stList._ENV = 1;
      st.sENV     = new String(sData);
    }
    if (sIdentifier.equals(LITHO))  // Lithofacies
    {
      stList._LITHO = 1;
      st.sLITHO     = new String(sData);
    }

    if (sIdentifier.equals(PCORE))  // Porosity Whole Core
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PCORE = 1;
        st.dPCORE     = cmn.cmnString.stringToDouble(sData);
//        st.dNPHI      = st.dPCORE;
      }
    }
    if (sIdentifier.equals(PPLUG))  // Porosity Plug
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PPLUG = 1;
        st.dPPLUG     = cmn.cmnString.stringToDouble(sData);
//        st.dNPHI      = st.dPPLUG;
      }
    }
    if (sIdentifier.equals(P800))   // Porosity Plug 800 PSI
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._P800 = 1;
        st.dP800     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(PINSI)) // Porosity Plut Insitu
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PINSI = 1;
        st.dPINSI     = cmn.cmnString.stringToDouble(sData);
//        st.dNPHI      = st.dPINSI;
      }
    }
    if (sIdentifier.equals(PEFF))   // Effective Rock Porosity
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._PEFF = 1;
        st.dPEFF     = cmn.cmnString.stringToDouble(sData);
//        st.dNPHI     = st.dPEFF;
      }
    }

    if (sIdentifier.equals(KMAX))   // Permeability Whole Max
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KMAX = 1;
        st.dKMAX     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(K90))    // Permeability Whole 90
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._K90 = 1;
        st.dK90     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(KVRT))   // Permeability Whole Vertical
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KVRT = 1;
        st.dKVRT     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(KPLG))   // Permeability Plug Routine
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KPLG = 1;
        st.dKPLG     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(KKL))    // Permeability Plug Klinkenberg Routine
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KKL = 1;
        st.dKKL     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(KINSI))  // Permeability Plug Insitu
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KINSI = 1;
        st.dKINSI     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(KKLIN))  // Permeability Plug Klinkenberg Insitu
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KKLIN = 1;
        st.dKKLIN     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(KPVRT))  // Permeability Plug Vertical
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._KPVRT = 1;
        st.dKPVRT     = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(SOIL))   // Saturation Oil
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SOIL = 1;
        st.dSOIL     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(SW))     // Saturation Water
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._SW = 1;
        st.dSW     = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(GMCC))   // Grain Density
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._GMCC = 1;
        st.dGMCC     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(RHOD))   // Density of Rock Dry
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._RHOD = 1;
        st.dRHOD     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(RHOW))   // Density of Rock Wet
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._RHOW = 1;
        st.dRHOW     = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(MAMB))   // Archie Cementation (M) Ambient
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._MAMB = 1;
        st.dMAMB     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(MINSI))  // Archie Cementation (M) Insitu
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._MINSI = 1;
        st.dMINSI     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(NAMB))   // Archie Saturation (N) Ambient
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NAMB = 1;
        st.dNAMB     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(NINSI))  // Archie Saturation (N) Insitu
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._NINSI = 1;
        st.dNINSI     = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(GR))     // Gamma Ray
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._GR = 1;
        st.dGR     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(CGR))     // Gamma Ray minus Uranium
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._CGR = 1;
        st.dCGR     = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(THOR))   // Thorium Concentration
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._TH = 1;
        st.dTh     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(URAN))   // Uranium Concentration
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._U = 1;
        st.dU     = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(POTA))   // Potasium Concentration
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._K = 1;
        st.dK     = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(LTHCD))    // Lithofacies Code
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        stList._LTHCD = 1;
        st.iLITH_CD   = cmn.cmnString.stringToInt(sData);
      }
    }
    if (sIdentifier.equals(FRACTURE)) // Fractures
    {
      stList._FRACTURE = 1;
      st.sFracture     = new String(sData);
    }

    if (sIdentifier.equals(SOURCE))   // Lithofacies Code
    {
      st.source = new String(sData);
    }
    if (sIdentifier.equals(DATE))     // Lithofacies Code
    {
      st.sDate = new String(sData);
    }

    return (st);
  }

  /** ParseData()
   * <p> This method will parse the Data Stream for the Individual Well
   *     Header Information
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = Well Header Information String
   */

  private void ParseData(String sIdentifier, String sData)
  {
    // Identification Information

    if (sIdentifier.equals(KID)) { stList.sKID = new String(sData); }
    if (sIdentifier.equals(KEY)) { stList.sKEY = new String(sData); }
    if (sIdentifier.equals(TYPE))
    {
      if (sData.equals("WELL"))    stList.iType = iqstrat.iqstratHeadersStruct.WELL;
      if (sData.equals("OUTCROP")) stList.iType = iqstrat.iqstratHeadersStruct.OUTCROP;
    }
    if (sIdentifier.equals(API_NUMBER))  { stList.sAPI = new String(sData); }
    if (sIdentifier.equals(NAME))
    {
      stList.sName = new String(sData.replaceAll("%20", " "));
    }
    if (sIdentifier.equals(WELL_STATUS))
    {
      stList.status = new String(sData.replaceAll("%20", " "));
    }
    if (sIdentifier.equals(ERROR_VAL))
    {
      iError    = -1;
      sError    = new String(sData);
    }

    // XY Position Information

    if (sIdentifier.equals(LATITUDE))
    {
      stList.dLatitude = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(LONGITUDE))
    {
      stList.dLongitude = cmn.cmnString.stringToDouble(sData);
    }

    // Z Position Information

    if (sIdentifier.equals(DEPTH))
    {
      stList.depth = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(GROUND_LEVEL))
    {
      stList.dGL = cmn.cmnString.stringToDouble(sData);
    }
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}
