/*
 * @rockObsEntryPlotPanel.java Version 1.1 04/11/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.gui;

import java.awt.*;
import java.awt.image.*;
import java.util.Observable;
import javax.swing.*;

import iqstrat.iqstratStruct;

import lith.rock.rockColumnListStruct;     // Rock Column List Data Structure
import lith.rock.rockColumnStruct;         // Rock Column Data Structure

import lith.lithology.lithologyListStruct;
import lith.texture.textureListStruct;

import rock.color.rockColorListStruct;
import rock.fossil.fossilListStruct;
import rock.sedimentary.sedimentaryListStruct;
import rock.phi.phiListStruct;
import rock.plot.rockObsEntryPlot;

/** Class rockObsEntryPlotPanel
 *  <p> This Class will display the sedimentary structures selected
 *
 *  @version 1.1 04/11/2011
 *  @author  John R. Victorine
 */

public class rockObsEntryPlotPanel extends JPanel
{
  // Global Variables

  private int              iWidth  = 0;
  private int              iHeight = 0;
  private rockObsEntryPlot plot    = null;

  // Global Panel Widgets

  private ScrollPane  scroll  = new ScrollPane(); // Scroll Pane
  private JPanel      panel   = new JPanel();     // Panel containing plot

  /** CONSTRUCTOR rockObsEntryPlotPanel()
   *  <p> This is the constructor for this class.
   *  @param stStruct   = Global IQSTRAT Data Structure
   *  @param depthStart = The starting depth of plot
   *  @param depthEnd   = The ending depth of plot
   *  @param iScale     = The scale
   *  @param iSelected  = The selected tracks for plotting
   */

  public rockObsEntryPlotPanel( iqstratStruct stStruct,
                                double depthStart,
                                double depthEnd,
                                int iScale,
                                int iSelected[] )
  {
    try
    {
      jbInit(stStruct, depthStart, depthEnd, iScale, iSelected);
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    scroll    = null;
    panel     = null;

    if (plot != null)
      plot.close();
    plot    = null;
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @param stStruct   = Global IQSTRAT Data Structure
   *  @param depthStart = The starting depth of plot
   *  @param depthEnd   = The ending depth of plot
   *  @param iScale     = The scale
   *  @param iSelected  = The selected tracks for plotting
   *  @throws Exception
   */

  private void jbInit( iqstratStruct stStruct,
                       double depthStart,
                       double depthEnd,
                       int iScale,
                       int iSelected[] ) throws Exception
  {
    Panel pnlCenter = new Panel();

    this.setLayout(new BorderLayout());
    pnlCenter.setLayout(new BorderLayout());

    plot    = new rockObsEntryPlot(
		        stStruct, depthStart, depthEnd, iScale, iSelected);
    iWidth  = 50 + plot.getPlotWidth();
    iHeight = plot.getPlotHeight();

    panel.setLayout( new BorderLayout() );
    panel.setPreferredSize( new Dimension( iWidth, iHeight ) );

    this.add(pnlCenter,   BorderLayout.CENTER);
    pnlCenter.add(scroll, BorderLayout.CENTER);
    scroll.add(panel,     null);
    panel.add(plot,       BorderLayout.CENTER);
  }

  /* ======================================================================= *
   * ----------------------------- SET METHODS ----------------------------- *
   * ======================================================================= */

  /** Method setScale( int iScale, double dMaximum, double dMinimum )
   * <p> This method will set the Plot scale in the
   *     Cross Plot Data List Structure
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setScale( int iScale, double dMaximum, double dMinimum )
  {
    if (plot != null)
    {
      plot.setScale( iScale, dMaximum, dMinimum );
      iWidth  = 50 + plot.getPlotWidth();
      iHeight = 40 + plot.getPlotHeight();
      panel.setPreferredSize( new Dimension( iWidth, iHeight ) );
    }

    scroll.doLayout();
    panel.doLayout();
  }


  /** Method modifyWidths( int iBio, int iSed )
   * <p> This method will modify the plot track width depending on the
   *     maximum number of fossils or sedimentary structures per depth.
   * @param iBio = The total number of fossils per depth per well
   * @param iSed = The total number of structures per depth per well
   */

  public void modifyWidths( int iBio, int iSed )
  {
    if (plot != null)
    {
      plot.modifyWidths( iBio, iSed );
      iWidth  = 50 + plot.getPlotWidth();
      iHeight = plot.getPlotHeight();
      panel.setPreferredSize( new Dimension( iWidth, iHeight ) );
    }

    scroll.doLayout();
    panel.doLayout();
  }

  /* ----------------------------------------------------------------------- *
   * ----------------- Available Tracks Selected for view ------------------ *
   * ----------------------------------------------------------------------- */

  /** Method setSelected( int iSelected[] )
   * <p> This method will change the selected tracks
   *  @param iSelected  = The Tracks that will be displayed
   */

  public void setSelected( int iSelected[] )
  {
    if (plot != null) plot.setSelected( iSelected );
  }

  /* ----------------------------------------------------------------------- *
   * --------------------------- Set Plot Data ----------------------------- *
   * ----------------------------------------------------------------------- */

  /** Method setPHI( phiListStruct st )
   * <p> This method will set the Porosity Data
   * @param st = The Porosity Data List Structure
   */

  public void setPHI( phiListStruct st )
  {
    if (plot != null) plot.setPHI( st );
  }

  /** Method setRockLithology( lithologyListStruct st )
   * <p> This method will set the rock column that will be shown
   * @param st = Rock Column determined by user from rock data.
   *

  public void setRockLithology( lithologyListStruct st )
  {
    if (plot != null)  plot.setRockLithology( st );
  }
*/
  /** Method setTexture( textureListStruct st )
   * <p> This method will set the rock texture that will be shown
   * @param st = Rock Texture determined by user from rock data.
   *

  public void setTexture( textureListStruct st )
  {
    if (plot != null) plot.setRockTexture( st );
  }
*/

  /** Method setRockColumn( rockColumnListStruct st)
   *  <p> This method will set the rock column lithology
   *  @param st = Rock Column List Data Structure
   */

  public void setRockColumn( rockColumnListStruct st )
  {
	if (plot != null) plot.setRockColumn( st );
  }

  /** Method setStructures( sedimentaryListStruct st)
   * <p> This method will set the sedimentary structures Data
   * @param st = The sedimentary structures Data List Structure
   */

  public void setStructures( sedimentaryListStruct st )
  {
    if (plot != null)  plot.setStructures( st );
  }

  /** Method setFossils( fossilListStruct st )
   * <p> This method will set the Fossil Data
   * @param st = The Fossil Data List Structure
   */

  public void setFossils( fossilListStruct st )
  {
    if (plot != null) plot.setFossils( st );
  }

  /** Method setRockColorData( rockColorListStruct st )
   * <p> This method will set the Rock Color Data.
   * @param st = the Rock Color List Data Structure
   */

  public void setRockColorData( rockColorListStruct st )
  {
    if (plot != null) plot.setRockColorData( st );
  }
}

/*
 *  @version 1.1 04/11/2011
 *  @author  John Victorine
 */
