/*
 * @rockObsEntryPanel.java Version 1.1 04/13/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.gui;

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.util.Observable;
import java.util.Observer;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import iqstrat.iqstratStruct;

import lith.rock.rockColumnListStruct;     // Rock Column List Data Structure
import lith.rock.rockColumnStruct;         // Rock Column Data Structure

import lith.lithology.lithologyListStruct;
import lith.lithology.lithologyStruct;
import lith.lithology.plot.lithologyPlotData;

import lith.texture.textureListStruct;
import lith.texture.textureStruct;

import lith.gui.lithologyEntryFrame;

import rock.phi.phiListStruct;
import rock.phi.phiStruct;
import rock.phi.gui.phiPlotDataEntryPanel;
import rock.phi.gui.phiPlotSymbolFrame;

import rock.sedimentary.sedimentaryListStruct;
import rock.sedimentary.sedimentaryStruct;
import rock.sedimentary.gui.sedimentaryPlotDataEntryPanel;
import rock.sedimentary.gui.sedimentaryPlotSymbolFrame;

import rock.fossil.fossilListStruct;
import rock.fossil.fossilStruct;
import rock.fossil.gui.fossilPlotDataEntryPanel;
import rock.fossil.gui.fossilPlotSymbolFrame;

import rock.color.rockColorListStruct;
import rock.color.rockColorStruct;
import rock.color.gui.rockColorChartPlotFrame;

/** Class rockObsEntryPanel
 *  <p> This Class will display the sedimentary structures selected
 *
 *  @version 1.1 04/13/2011
 *  @author  John R. Victorine
 */

public class rockObsEntryPanel extends Panel implements ActionListener, Observer
{
  // Input Variables

  private Observable    pNotifier = null; // Input Observable
  private iqstratStruct stStruct  = null; // Global IQSTRAT Data Structure

  // Global Variables

  private Observable notifier = null;     // Panel Observable

  private lithologyPlotData             plotComp   = null; // Composition
  private phiPlotDataEntryPanel         plotPhi    = null; // Porosity plot panel
  private sedimentaryPlotDataEntryPanel plotSed    = null; // Sed Struct plot panel
  private fossilPlotDataEntryPanel      plotFossil = null; // Fossil plot panel

  // Lookup Dialogs

  private lithologyEntryFrame           pLithology = null; // Lithology/Texture Lookup
  private phiPlotSymbolFrame            pPHI       = null; // Porosity Types Lookup Frame
  private sedimentaryPlotSymbolFrame    pStruct    = null; // Sed Structure Lookup Frame
  private fossilPlotSymbolFrame         pFossils   = null; // Fossils Lookup Frame
  private rockColorChartPlotFrame       pColors    = null; // Rock & Soil Color Chart Frame

  // -- Rock Data

  private rockColumnListStruct stColumn    = null; // Rock Column Data List

  private lithologyStruct      stLithology = null; // Lithology Data Structure
  private textureStruct        stTexture   = null; // Texture Data Structure
  private phiStruct            stPhi       = null; // Porosity Data Structure
  private sedimentaryStruct    stSedStruct = null; // Sedimentary Data Structure
  private fossilStruct         stFossil    = null; // Fossil Data Structure
  private rockColorStruct      stColor     = null; // Rock Color Data Structure

  // -- Porsity Value

  private static final int _NONE      = 0;
  private static final int _TRACE     = 1;
  private static final int _POOR      = 2;
  private static final int _FAIR      = 3;
  private static final int _GOOD      = 4;
  private static final int _EXCELLENT = 5;

  private int iPHI = _NONE;

  private static final String PHI[] = {
    "No Visible Porosity",
    "Trace Porosity",
    "Poor Porosity",
    "Fair Porosity",
    "Good Porosity",
    "Excellent Porosity"
  };

  // Global Widgets

  private JLabel       lblPhi              = new JLabel();

  private JTextArea    txtTexture          = new JTextArea();    // Description

  // -- Primary Rock Color Widgets

  private JButton      btnPrimary          = new JButton();      // View Color
  private JTextField   txtPrimaryMunsell   = new JTextField();   // Munsell Code
  private JTextField   txtPrimary          = new JTextField();   // ISCC Name

  // -- Secondary Rock Color Widgets

  private JButton      btnSecondary        = new JButton();      // View Color
  private JTextField   txtSecondaryMunsell = new JTextField();   // Munsell Code
  private JTextField   txtSecondary        = new JTextField();   // ISCC Name

  // -- Streaks Color Widgets

  private JButton      btnStreaks          = new JButton();      // View Color
  private JTextField   txtStreaksMunsell   = new JTextField();   // Munsell Code
  private JTextField   txtStreaks          = new JTextField();// ISCC Name

  // -- Edit Buttons

  private JButton      btnEditLithology    = new JButton();      // View Lithology/Texture
  private JButton      btnEditPhi          = new JButton();      // View Porsity Types
  private JButton      btnEditSedStruct    = new JButton();      // View Sed Struct
  private JButton      btnEditFossil       = new JButton();      // View Fossil
  private JButton      btnEditColor        = new JButton();      // View Color

  /** Constructor rockObsEntryPanel()
   *  <p> This is the Constructor for this class.
   *  @param pNotifier = Observable
   *  @param stStruct = Global IQSTRAT Data Structure
   */

  public rockObsEntryPanel( Observable pNotifier, iqstratStruct stStruct )
  {
    try
    {
      this.pNotifier = pNotifier;
      this.stStruct  = stStruct;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlCenter           = new JPanel();
    JPanel pnlData2            = new JPanel();
    JPanel pnlLithology        = new JPanel();
    JPanel pnlLithTop          = new JPanel();
    JPanel pnlPHITypes         = new JPanel();
    JPanel pnlPHIBase          = new JPanel();
    JPanel pnlTexture          = new JPanel();
    JPanel pnlData             = new JPanel();
    JPanel pnlSedStructs       = new JPanel();
    JPanel pnlFossils          = new JPanel();
    JPanel pnlBlock            = new JPanel();
    JPanel pnlColor            = new JPanel();
    JPanel pnlPrimary          = new JPanel();
    JPanel pnlStreaks          = new JPanel();
    JPanel pnlSecondary        = new JPanel();
    JPanel pnlPrimaryBtn       = new JPanel();
    JPanel pnlPrimaryMunsell   = new JPanel();
    JPanel pnlPrimaryColor     = new JPanel();
    JPanel pnlSecondaryBtn     = new JPanel();
    JPanel pnlSecondaryMunsell = new JPanel();
    JPanel pnlSecondaryColor   = new JPanel();
    JPanel pnlStreaksBtn       = new JPanel();
    JPanel pnlStreaksMunsell   = new JPanel();
    JPanel pnlStreaksColor     = new JPanel();
    JPanel pnlColorEdit        = new JPanel();

    JScrollPane scrollTxt      = new JScrollPane();

    TitledBorder titledBorderLithology = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Lithology/Texture:");
    titledBorderLithology.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderPHI = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Porosity Types:");
    titledBorderPHI.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderSed = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Sedimentary Structures:");
    titledBorderSed.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderFossil = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Fossils:");
    titledBorderFossil.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderColor = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Rock Color:");
    titledBorderColor.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderPrimary = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Primary:");
    titledBorderPrimary.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderSecondary = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Secondary:");
    titledBorderSecondary.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderStreaks = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Streaks:");
    titledBorderStreaks.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Create an Observable

    notifier = new rockObsEntryPanelObservable();

    // Add this Dialog to the notifier as observer

    notifier.addObserver(this);

    // Build Panel

    this.setLayout(new BorderLayout());
    pnlCenter.setLayout(new BorderLayout());

    // Data 2 Panel

    pnlData2.setLayout(new GridLayout(2,1));

    // -- Lithology/Texture

    pnlLithology.setLayout(new BorderLayout());
    pnlLithology.setBorder(titledBorderLithology);

    pnlLithTop.setLayout(new BorderLayout());
    pnlLithTop.setPreferredSize(new Dimension(10, 30));

    plotComp = new lithologyPlotData( stStruct.stSymbols );

    btnEditLithology.setFont(new java.awt.Font("Dialog", 1, 11));
    btnEditLithology.setText("Edit");
    btnEditLithology.addActionListener(this);

    pnlTexture.setLayout(new BorderLayout());

    txtTexture.setText("");
    txtTexture.setFont(new java.awt.Font("Dialog", 0, 11));
    txtTexture.setLineWrap(true);
    txtTexture.setWrapStyleWord(true);

    // -- Plot Selected Porosity Types

    pnlPHITypes.setLayout(new BorderLayout());
    pnlPHITypes.setBorder(titledBorderPHI);

    plotPhi = new phiPlotDataEntryPanel( stStruct.stPorosity );

    pnlPHIBase.setLayout(new BorderLayout());

    lblPhi.setFont(new java.awt.Font("Dialog", 1, 11));
    lblPhi.setHorizontalAlignment(SwingConstants.CENTER);
    lblPhi.setText(PHI[iPHI]);

    btnEditPhi.setFont(new java.awt.Font("Dialog", 1, 11));
    btnEditPhi.setText("Edit");
    btnEditPhi.addActionListener(this);

    // Data Panel

    pnlData.setLayout(new GridLayout(2,1));
    pnlData.setPreferredSize(new Dimension(300, 10));

    // -- Plot Select Sedimentary Structures Panel

    pnlSedStructs.setLayout(new BorderLayout());
    pnlSedStructs.setBorder(titledBorderSed);

    plotSed = new sedimentaryPlotDataEntryPanel( stStruct.stSedStruct );

    btnEditSedStruct.setFont(new java.awt.Font("Dialog", 1, 11));
    btnEditSedStruct.setText("Edit Sedimenary Structure");
    btnEditSedStruct.addActionListener(this);

    // -- Plot Selected Fossils Panel

    pnlFossils.setLayout(new BorderLayout());
    pnlFossils.setBorder(titledBorderFossil);

    plotFossil = new fossilPlotDataEntryPanel( stStruct.stFossils );

    btnEditFossil.setFont(new java.awt.Font("Dialog", 1, 11));
    btnEditFossil.setText("Edit Fossil");
    btnEditFossil.addActionListener(this);

    // Rock Color Panel

    pnlColor.setLayout(new BorderLayout());
    pnlColor.setBorder(titledBorderColor);

    // -- Rock Color Panel

    pnlBlock.setLayout(new GridLayout(3,1));

    // .. Primary Rock Color Panel

    pnlPrimary.setBorder(titledBorderPrimary);
    pnlPrimary.setLayout(new GridLayout(2,1));

    pnlPrimaryBtn.setLayout(new BorderLayout());

    btnPrimary.setBackground(Color.white);
    btnPrimary.setText("");

    pnlPrimaryMunsell.setLayout(new BorderLayout());

    txtPrimaryMunsell.setEditable(false);
    txtPrimaryMunsell.setText("");
    txtPrimaryMunsell.setColumns(8);

    pnlPrimaryColor.setLayout(new BorderLayout());

    txtPrimary.setEditable(false);
    txtPrimary.setText("");

    // .. Secondary Rock Color Panel

    pnlSecondary.setBorder(titledBorderSecondary);
    pnlSecondary.setLayout(new GridLayout(2,1));

    pnlSecondaryBtn.setLayout(new BorderLayout());

    btnSecondary.setBackground(Color.white);
    btnSecondary.setText("");

    pnlSecondaryMunsell.setLayout(new BorderLayout());

    txtSecondaryMunsell.setEditable(false);
    txtSecondaryMunsell.setText("");
    txtSecondaryMunsell.setColumns(8);

    pnlSecondaryColor.setLayout(new BorderLayout());

    txtSecondary.setEditable(false);
    txtSecondary.setText("");

    // .. Streak Color Panel

    pnlStreaks.setBorder(titledBorderStreaks);
    pnlStreaks.setLayout(new GridLayout(2,1));

    pnlStreaksBtn.setLayout(new BorderLayout());

    btnStreaks.setBackground(Color.white);
    btnStreaks.setText("");

    pnlStreaksMunsell.setLayout(new BorderLayout());

    txtStreaksMunsell.setEditable(false);
    txtStreaksMunsell.setText("");
    txtStreaksMunsell.setColumns(8);

    pnlStreaksColor.setLayout(new BorderLayout());

    txtStreaks.setEditable(false);
    txtStreaks.setText("");

    // -- Edit Color Panel

    pnlColorEdit.setLayout(new BorderLayout());

    btnEditColor.setFont(new java.awt.Font("Dialog", 1, 11));
    btnEditColor.setText("Edit Color");
    btnEditColor.addActionListener(this);

    // Attach Widgets to Panel

    this.add(pnlCenter,                          BorderLayout.CENTER);
    pnlCenter.add(pnlData2,                      BorderLayout.CENTER);
    pnlData2.add(pnlLithology,                   null);
    pnlLithology.add(pnlLithTop,                 BorderLayout.NORTH);
    pnlLithTop.add(plotComp,                     BorderLayout.CENTER);
    pnlLithTop.add(btnEditLithology,             BorderLayout.EAST);

    pnlLithology.add(pnlTexture,                 BorderLayout.CENTER);
    pnlTexture.add(scrollTxt,                    BorderLayout.CENTER);
    scrollTxt.getViewport().add(txtTexture,      null);

    pnlData2.add(pnlPHITypes,                    null);
    pnlPHITypes.add(plotPhi,                     BorderLayout.CENTER);
    pnlPHITypes.add(pnlPHIBase,                  BorderLayout.SOUTH);
    pnlPHIBase.add(lblPhi,	     				 BorderLayout.WEST);
    pnlPHIBase.add(btnEditPhi,					 BorderLayout.EAST);

    pnlCenter.add(pnlData,                       BorderLayout.EAST);
    pnlData.add(pnlSedStructs,                   null);
    pnlSedStructs.add(plotSed,                   BorderLayout.CENTER);
    pnlSedStructs.add(btnEditSedStruct,          BorderLayout.SOUTH);

    pnlData.add(pnlFossils,                      null);
    pnlFossils.add(plotFossil,                   BorderLayout.CENTER);
    pnlFossils.add(btnEditFossil,                BorderLayout.SOUTH);

    this.add(pnlColor,                           BorderLayout.EAST);
    pnlColor.add(pnlBlock,                       BorderLayout.CENTER);
    pnlBlock.add(pnlPrimary,                     null);
    pnlPrimary.add(pnlPrimaryBtn,                null);
    pnlPrimaryBtn.add(btnPrimary,                BorderLayout.CENTER);
    pnlPrimaryBtn.add(pnlPrimaryMunsell,         BorderLayout.EAST);
    pnlPrimaryMunsell.add(txtPrimaryMunsell,     BorderLayout.CENTER);
    pnlPrimary.add(pnlPrimaryColor,              null);
    pnlPrimaryColor.add(txtPrimary,              BorderLayout.CENTER);

    pnlBlock.add(pnlSecondary,                   null);
    pnlSecondary.add(pnlSecondaryBtn,            null);
    pnlSecondaryBtn.add(btnSecondary,            BorderLayout.CENTER);
    pnlSecondaryBtn.add(pnlSecondaryMunsell,     BorderLayout.EAST);
    pnlSecondaryMunsell.add(txtSecondaryMunsell, BorderLayout.CENTER);
    pnlSecondary.add(pnlSecondaryColor,          null);
    pnlSecondaryColor.add(txtSecondary,          BorderLayout.CENTER);

    pnlBlock.add(pnlStreaks,                     null);
    pnlStreaks.add(pnlStreaksBtn,                null);
    pnlStreaksBtn.add(btnStreaks,                BorderLayout.CENTER);
    pnlStreaksBtn.add(pnlStreaksMunsell,         BorderLayout.EAST);
    pnlStreaksMunsell.add(txtStreaksMunsell,     BorderLayout.CENTER);
    pnlStreaks.add(pnlStreaksColor,              null);
    pnlStreaksColor.add(txtStreaks,              BorderLayout.CENTER);

    pnlColor.add(pnlColorEdit,                   BorderLayout.SOUTH);
    pnlColorEdit.add(btnEditColor,               BorderLayout.CENTER);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getData()
   *  <p> This method will reduce the different plot track data to text that
   *      will be inserted into the Comments/Remarks/Notes Textfield.
   */

  public String getData()
  {
	int    i,j;
	String sComments = "";
	String sPhi      = "";

    // ADD Lithology/Texture to Description

    if (txtTexture.getText().length() > 0)
    {
	  sComments = new String( txtTexture.getText() + "; " );
	}

	// ADD Colors to Description

    if (stColor != null)
    {
	  for (i=0; i<3; i++)
	  {
        if ((i== 0) && (stColor.sColor[i].length() > 0))
        {
		  sComments = new String( sComments + "1st: " +
		                          stColor.sColor[i] +
		                          " (" +stColor.sMunsell[i]+")");
	    }

		if ((i== 1) && (stColor.sColor[i].length() > 0))
		{
		  sComments = new String( sComments + ", 2nd: " +
		                          stColor.sColor[i] +
		                          " (" +stColor.sMunsell[i]+")");
		}

		if ((i== 2) && (stColor.sColor[i].length() > 0))
		{
		  sComments = new String( sComments + ", other: " +
		                          stColor.sColor[i] +
		                          " (" +stColor.sMunsell[i]+")");
		}
	  }

	  if (stColor.sColor[0].length() > 0)
	    sComments = new String( sComments + "; " );
	}

    // ADD Fossil List

    if (stFossil != null)
    {
	  if ((stFossil.sEnvironment.length() > 0) &&
	      (!stFossil.sEnvironment.equals("None")))
	    sComments = new String( sComments +  stFossil.sEnvironment + ", " );

      for (j=0; j<stFossil.iTotal; j++)
      {
        if (j>0) sComments = new String( sComments + ", " );

        if (stFossil.sMod[j].length() > 0)
        {
          sComments = new String( sComments + stFossil.sMod[j] + " " +
                                              stFossil.sText[j] );
        }
        else
        {
          sComments = new String( sComments + stFossil.sText[j] );
        }
      }

      sComments = new String( sComments + "; " );
	}

    // ADD Sedimentary Structure List

	if (stSedStruct != null)
    {
      for (j=0; j<stSedStruct.iTotal; j++)
      {
        if (j>0) sComments = new String( sComments + ", " );

        sComments = new String( sComments + stSedStruct.sText[j] );
	  }

      sComments = new String( sComments + "; " );
	}

    // ADD Porosity Type List

	if (stPhi != null)
    {
	  sPhi = new String( "No Visible Porosity" );
	  if ((this.stPhi.dPHI > 0.0) &&
	      (this.stPhi.dPHI < 1.0))  { sPhi = new String( "Trace Porosity" ); }
      if ((this.stPhi.dPHI >= 1.0) &&
          (this.stPhi.dPHI < 5.0))  { sPhi = new String( "Poor Porosity" ); }
      if ((this.stPhi.dPHI >= 5.0) &&
          (this.stPhi.dPHI < 10.0)) { sPhi = new String( "Fair Porosity" ); }
      if ((this.stPhi.dPHI >= 10.0) &&
          (this.stPhi.dPHI < 15.0)) { sPhi = new String( "Good Porosity" ); }
      if (this.stPhi.dPHI >= 15.0)  { sPhi = new String( "Excellent Porosity" ); }

      sComments = new String( sComments + sPhi + " (" + this.stPhi.dPHI + "%) " );

      for (j=0; j<stPhi.iTotal; j++)
      {
//        if (j>0)
        sComments = new String( sComments + ", " );
        sComments = new String( sComments + stPhi.sText[j] + " porosity" );
	  }

      sComments = new String( sComments + "; " );
    }

	return (sComments);
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setRockColumn( rockColumnListStruct st)
   *  <p> This method will set the rock column lithology
   *  @param st = Rock Column List Data Structure
   */

  public void setData( rockColumnListStruct st )
  {
	this.stColumn = st;
//	if (plot != null) plot.setRockColumn( st );
  }

  /** Method setData()
   * <p> This method will set the array of porosity data type structures selected
   * @param st = Porosity Type data List Structure
   */

  public void setData(lithologyListStruct st)
  {
	if (st != null)
	{
	  if (st.iCount > 0)
	  {
		this.stLithology = st.stItem[0];

		if (plotComp != null)
		{
		  plotComp.setData(stLithology);
		}
      }
    }
  }

  /** Method setData()
   * <p> This method will set the array of porosity data type structures selected
   * @param sRemarks = The remarks text.
   * @param st       = Porosity Type data List Structure
   */

  public void setData(String sRemarks, textureListStruct st)
  {
	String sComments = "";

	if ((st != null) && (sRemarks != null))
	{
      sComments = parse.parseTextureUtility.parse(
	                  sRemarks,
                      stStruct.stSymbols,
                      stStruct.stParseTexture );

	  if (st.iCount > 0)
	  {
		this.stTexture = st.stItem[0];
		sComments = new String( sComments +
		                        " ( Energy: " + this.stTexture.iEnergy + " )");
      }

  	  txtTexture.setText( sComments );
    }
  }

  /** Method setData()
   * <p> This method will set the array of porosity data type structures selected
   * @param st = Porosity Type data List Structure
   */

  public void setData(phiListStruct st)
  {
	if (st != null)
	{
	  if (st.iCount > 0)
	  {
		this.stPhi = st.stItem[0];

        if (plotPhi != null)
        {
		  iPHI = _NONE;
		  if ((this.stPhi.dPHI > 0.0) && (this.stPhi.dPHI < 1.0)) { iPHI = _TRACE; }
    	  if ((this.stPhi.dPHI >= 1.0) && (this.stPhi.dPHI < 5.0)) { iPHI = _POOR; }
    	  if ((this.stPhi.dPHI >= 5.0) && (this.stPhi.dPHI < 10.0)) { iPHI = _FAIR; }
    	  if ((this.stPhi.dPHI >= 10.0) && (this.stPhi.dPHI < 15.0)) { iPHI = _GOOD; }
    	  if (this.stPhi.dPHI >= 15.0) { iPHI = _EXCELLENT; }
          lblPhi.setText(PHI[iPHI]);

          plotPhi.setData(stPhi);
	    }
      }
    }
  }

  /** Method setData()
   * <p> This method will set the array of sedimentary structures selected
   * @param st = sedimentary data List Structure
   */

  public void setData(sedimentaryListStruct st)
  {
	if (st != null)
	{
	  if (st.iCount > 0)
	  {
		this.stSedStruct = st.stItem[0];
        if (plotSed != null)
        {
          plotSed.setData(stSedStruct);
	    }
      }
    }
  }

  /** Method setData()
   * <p> This method will set the array of fossils selected
   * @param st = fossil data List Structure
   */

  public void setData(fossilListStruct st)
  {
	if (st != null)
	{
	  if (st.iCount > 0)
	  {
		this.stFossil = st.stItem[0];
        if (plotFossil != null)
        {
		  plotFossil.setData(stFossil);
		}
      }
    }
  }

  /** Method setData()
   * <p> This method will set the Color Data Structure to be modified
   * @param stColor = the color data list structure
    */

  public void setData(rockColorListStruct st)
  {
	if (st != null)
	{
	  if (st.iCount > 0)
	  {
        this.stColor = rock.color.rockColorUtility.copy( st.stItem[0] );

        // Primary Rock Color Panel

        btnPrimary.setBackground(Color.white);
        if (stColor.iRGB[0][0] > -1)
          btnPrimary.setBackground(
			  new Color(stColor.iRGB[0][0], stColor.iRGB[0][1], stColor.iRGB[0][2]));

        txtPrimaryMunsell.setText(stColor.sMunsell[0]);
        txtPrimary.setText(stColor.sColor[0]);

        // Secondary Rock Color Panel

        btnSecondary.setBackground(Color.white);
        if (stColor.iRGB[1][0] > -1)
          btnSecondary.setBackground(
			  new Color(stColor.iRGB[1][0], stColor.iRGB[1][1], stColor.iRGB[1][2]));

        txtSecondaryMunsell.setText(stColor.sMunsell[1]);
        txtSecondary.setText(stColor.sColor[1]);

        // Streak Color Panel

        btnStreaks.setBackground(Color.white);
        if (stColor.iRGB[2][0] > -1)
          btnStreaks.setBackground(
			  new Color(stColor.iRGB[2][0], stColor.iRGB[2][1], stColor.iRGB[2][2]));

        txtStreaksMunsell.setText(stColor.sMunsell[2]);
        txtStreaks.setText(stColor.sColor[2]);
      }
    }
  }

  /** Method clear()
   * <p> This method will clear the selected values
   */

  public void clear()
  {
    closeData();

    iPHI = _NONE;
    lblPhi.setText(PHI[iPHI]);

    txtTexture.setText("");

	if (plotComp   != null) { plotComp.setData(stLithology); }
    if (plotPhi    != null) { plotPhi.setData(stPhi); }
    if (plotSed    != null) { plotSed.setData(stSedStruct); }
    if (plotFossil != null) { plotFossil.setData(stFossil); }

    // Primary Rock Color Panel

    btnPrimary.setBackground(Color.white);
    txtPrimaryMunsell.setText("");
    txtPrimary.setText("");

    // Secondary Rock Color Panel

    btnSecondary.setBackground(Color.white);
    txtSecondaryMunsell.setText("");
    txtSecondary.setText("");

    // Streak Color Panel

    btnStreaks.setBackground(Color.white);
    txtStreaksMunsell.setText("");
    txtStreaks.setText("");
  }

  /* ===================================================================== *
   * ---------------------- DISPLAY Lookup Dialogs ----------------------- *
   * ===================================================================== */

  /** Method displayLithology()
   * <p> This method will display the Lithology/Texture frame
   */

  public void displayLithology()
  {
    closeLith();
    pLithology = new lithologyEntryFrame( notifier, stStruct );
  }

  /** Method displayPHI()
   * <p> This method will display the available porosity frame
   */

  public void displayPHI()
  {
    closePHI();
    pPHI = new phiPlotSymbolFrame( notifier, stStruct.stPorosity );

    if (stPhi != null)
    {
	  pPHI.setData( stPhi );
	}
  }

  /** Method displaySedStructs()
   * <p> This method will display the available Sedimentary Structure frame
   */

  public void displaySedStructs()
  {
    closeSedStruct();
    pStruct = new sedimentaryPlotSymbolFrame( notifier, stStruct.stSedStruct );

    if (stSedStruct != null)
    {
	  pStruct.setData( stSedStruct );
	}
  }

  /** Method displayFossils()
   * <p> This method will display the available fossils frame
   */

  public void displayFossils()
  {
    closeFossils();
    pFossils = new fossilPlotSymbolFrame( notifier, stStruct.stFossils );

    if (stFossil != null)
    {
	  pFossils.setData( stFossil );
	}
  }

  /** Method displayColors()
   * <p> This method will display the available colors frame
   */

  public void displayColors()
  {
    closeColors();
    pColors = new rockColorChartPlotFrame( notifier, stStruct );

    if (stColor != null)
    {
	  pColors.setData( stColor );
	}
  }

  /* ===================================================================== *
   * ----------------------------- CLOSE --------------------------------- *
   * ===================================================================== */

  /** Method closeLith()
   * <p> This method will dispose of the lithology/texture frame
   */

  public void closeLith()
  {
    if (pLithology != null)
      pLithology.close();
    pLithology = null;
  }

  /** Method closePHI()
   * <p> This method will dispose of the available porosity frame
   */

  public void closePHI()
  {
    if (pPHI != null)
      pPHI.close();
    pPHI = null;
  }

  /** Method closeSedStruct()
   * <p> This method will dispose of the available Sedimentary Structure frame
   */

  public void closeSedStruct()
  {
    if (pStruct != null)
      pStruct.close();
    pStruct = null;
  }

  /** Method closeFossils()
   * <p> This method will dispose of the available fossils frame
   */

  public void closeFossils()
  {
    if (pFossils != null)
      pFossils.close();
    pFossils = null;
  }

  /** Method closeColors()
   * <p> This method will dispose of the available fossils frame
   */

  public void closeColors()
  {
    if (pColors != null)
      pColors.close();
    pColors = null;
  }

  /** Method closeData()
   * <p> This method will close the rock data.
   */

  public void closeData()
  {
	if (stLithology != null)
	  stLithology.delete();
	stLithology = null; // Lithology Data Structure

    if (stTexture != null)
      stTexture.delete();
    stTexture   = null; // Texture Data Structure

	if (stPhi != null)
	  stPhi.delete();
	stPhi       = null; // Porosity Data Structure

	if (stSedStruct != null)
	  stSedStruct.delete();
    stSedStruct = null; // Sedimentary Data Structure

    if (stFossil != null)
      stFossil.delete();
    stFossil    = null; // Fossil Data Structure

    if (stColor != null)
      stColor.delete();
    stColor     = null; // Rock Color Data Structure
  }

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    // Input Variables

    pNotifier = null; // Input Observable
    stStruct  = null; // Global IQSTRAT Data Structure

    // Global Variables

    notifier = null;  // Panel Observable

    // Data

	closeData();

    // Lookup Dialogs

    closeLith();
    closePHI();
    closeSedStruct();
	closeFossils();
    closeColors();

    // Plot Panels

    if (plotComp != null)
      plotComp.close();
    plotComp   = null; // Composition

    if (plotPhi != null)
      plotPhi.close();
    plotPhi    = null; // Porosity plot panel

    if (plotSed != null)
      plotSed.close();
    plotSed    = null; // Sed Struct plot panel

    if (plotFossil != null)
      plotFossil.close();
    plotFossil = null; // Fossil plot panel

    // Global Widgets

    lblPhi              = null;
    txtTexture          = null; // Description

    // -- Primary Rock Color Widgets

    btnPrimary          = null; // View Color
    txtPrimaryMunsell   = null; // Munsell Code
    txtPrimary          = null; // ISCC Name

    // -- Secondary Rock Color Widgets

    btnSecondary        = null; // View Color
    txtSecondaryMunsell = null; // Munsell Code
    txtSecondary        = null; // ISCC Name

    // -- Streaks Color Widgets

    btnStreaks          = null; // View Color
    txtStreaksMunsell   = null; // Munsell Code
    txtStreaks          = null; // ISCC Name

    // -- Edit Buttons

    btnEditLithology    = null; // View Lithology/Texture
    btnEditPhi          = null; // View Porsity Types
    btnEditSedStruct    = null; // View Sed Struct
    btnEditFossil       = null; // View Fossil
    btnEditColor        = null; // View Color
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
	if (event.getSource() == btnEditLithology) { displayLithology(); }
	if (event.getSource() == btnEditPhi)       { displayPHI(); }
	if (event.getSource() == btnEditSedStruct) { displaySedStructs(); }
	if (event.getSource() == btnEditFossil)    { displayFossils(); }
	if (event.getSource() == btnEditColor)     { displayColors(); }
  }

  /* ===================================================================== *
   * ----------------------- OBSERVABLE ACTIONS -------------------------- *
   * ===================================================================== */

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String sArg      = new String((String) arg);

    // Porosity Types

    if (sArg.equals("ADD or MODIFY Porosity Data"))
    {
      phiListStruct stPs = new phiListStruct();
      phiStruct     stP  = null;

      stP = rock.phi.phiUtility.copy( pPHI.getData() );

      if (stP !=null)
      {
        stPs = rock.phi.phiUtility.add(stP, stPs);
	  }

      closePHI();

	  setData( stPs );

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("Rock Description"));
	}

    // Lithology/Texture Data Structure

    if (sArg.equals("ADD or MODIFY Lithology Data"))
    {
      lithologyListStruct stLs = new lithologyListStruct();
      lithologyStruct     stL  = null;

      stL = lith.lithology.lithologyUtility.copy( pLithology.getData() );
      txtTexture.setText( pLithology.getText() );

      if (stL != null)
      {
        stLs = lith.lithology.lithologyUtility.add( stL, stLs );
      }

      closeLith();

      setData( stLs );

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("Rock Description"));
	}

    // Sedimentary Structure

    if (sArg.equals("ADD or MODIFY Sed Struct Data"))
    {
      sedimentaryListStruct stSs = new sedimentaryListStruct();
      sedimentaryStruct     stS  = null;

      stS = rock.sedimentary.sedimentaryUtility.copy( pStruct.getData() );

      if (stS != null)
      {
        stSs = rock.sedimentary.sedimentaryUtility.add(stS, stSs);
	  }

      closeSedStruct();

	  setData( stSs );

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("Rock Description"));
    }

    // Fossils

    if (sArg.equals("ADD or MODIFY Fossil Data"))
    {
	  fossilListStruct stFs  = new fossilListStruct();
	  fossilStruct     stF   = null;

	  stF = rock.fossil.fossilUtility.copy( pFossils.getData() );

	  if (stF != null)
	  {
        stFs = rock.fossil.fossilUtility.add(stF, stFs);
	  }

  	  closeFossils();

	  setData( stFs );

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("Rock Description"));
	}

    // Rock Color

    if (sArg.equals("ADD or MODIFY Rock Color Data"))
    {
	  rockColorListStruct stCs = new rockColorListStruct();
	  rockColorStruct     stC  = null;

      stC = rock.color.rockColorUtility.copy( pColors.getData() );

      if (stC != null)
      {
        stCs = rock.color.rockColorUtility.add(stC, stCs);
      }

      closeColors();

      setData( stCs );

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("Rock Description"));
    }
  }
}

/** CLASS rockObsEntryPanelObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class rockObsEntryPanelObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 04/13/2011
 *  @author  John Victorine
 */
