/*
 * @rockObsEntryFrame.java Version 1.1 04/08/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observer;
import java.util.Observable;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import iqstrat.iqstratStruct;              // Global IQSTRAT Data Structure

import iqstrat.iqstratRemarkListStruct;    // Remarks List Structure
import iqstrat.iqstratRemarkStruct;        // Remarks Data Structure

import lith.rock.rockColumnListStruct;     // Rock Column List Data Structure
import lith.rock.rockColumnStruct;         // Rock Column Data Structure

import lith.lithology.lithologyListStruct; // Rock Lithology List Data Structure
import lith.lithology.lithologyStruct;     // Rock Lithology Data Structure

import lith.texture.textureListStruct;     // Rock Texture List Data Structure
import lith.texture.textureStruct;         // Rock Texture Data Structure

import rock.phi.phiListStruct;             // Porosity List Data Strurture
import rock.phi.phiStruct;                 // Porosity Data Strurture

import rock.sedimentary.sedimentaryListStruct; // Sed Structure List Data Structure
import rock.sedimentary.sedimentaryStruct; // Sed Structure Data Structure

import rock.fossil.fossilListStruct;       // Fossil List Data Structure
import rock.fossil.fossilStruct;           // Fossil Data Structure

import rock.color.rockColorListStruct;     // Rock Color List Data Structure
import rock.color.rockColorStruct;         // Rock Color Data Structure

import rock.gui.rockObsEntryPlotPanel;     // Image Track Plot Panel
import rock.gui.rockObsEntryPanel;         // Rock Data Entry Panel

/** Class rockObsEntryFrame
 *  <p> This Class will allow the user to type in the description of
 *      core data by depth range or outcrop/roadcut beds and to parse
 *      and convert that data and draw image plot tracks.
 *
 *  @version 1.1 04/08/2011
 *  @author  John R. Victorine
 */

public class rockObsEntryFrame extends JFrame implements ActionListener, Observer
{
  // Input Variables

  private Observable    pNotifier = null; // Input Observable
  private iqstratStruct stStruct  = null; // Global IQSTRAT Data Structure

  // Global Variables

  private Observable notifier = null;  // Frame Observable

  // Remarks Dialog Action

  public static final int _ADD        = 0;
  public static final int _MODIFY     = 1;
  public static final int _REMOVE     = 2;
  public static final int _REMOVE_ALL = 3;

  private int iAction = _ADD;

  private String sKEY = "0";

  // -- Depth range variables

  private double     dStart     =  0.0; // Starting Depth
  private double     dEnd       = 20.0; // Ending Depth
  private double     dThickness = 20.0; // Thickness of Bed
  private int        iScale     = iqstrat.iqstratTracksStruct.SCALE_20;

  // -- Data Entry List Table

  private rockEnterDataTable pTable = null; // Rock Data Entry Table

  // -- Plot Panel

  private rockObsEntryPlotPanel plot = null; // Rock Plot Bed Panel

  private int iSelected[] = {                    // Rock Image Plot Tracks
    iqstrat.iqstratTracksStruct._ROCK_COLOR,     // .. Rock Color
    iqstrat.iqstratTracksStruct._ROCK_MEASURED_SECTION, // .. Rock Coumn
//    iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY, // .. Rock Lithology
//    iqstrat.iqstratTracksStruct._ROCK_TEXTURE,   // .. Rock Texture
    iqstrat.iqstratTracksStruct._SED_STRUCTURE,  // .. Sedimentary Structure
    iqstrat.iqstratTracksStruct._ROCK_FOSSILS,   // .. Fossils
    iqstrat.iqstratTracksStruct._ROCK_THIN_PHI,  // .. Porosity Thin
    iqstrat.iqstratTracksStruct._ROCK_POROSITY,  // .. Porosity Type
  };

  // -- Input Data Panel

  private rockObsEntryPanel panel = null; // Rock Data Entry Panel

  // -- Rock Data Structures

  private iqstratRemarkListStruct stRemarks     = null; // Remarks List Data Struct
  private iqstratRemarkStruct     stRemark      = null; // Remarks Data Struct

  private iqstratRemarkListStruct stTemp        = null; // Temporary Remarks List Data Struct
  private phiListStruct           stPHI         = null; // Porosity type struct
  private rockColumnListStruct    stColumn      = null; // Rock Column Data List
  private lithologyListStruct     stLithology   = null; // Rock Column
  private textureListStruct       stTexture     = null; // Rock Texture
  private sedimentaryListStruct   stSedimentary = null; // Sed Struct Data
  private fossilListStruct        stFossil      = null; // Fossil List struct
  private rockColorListStruct     stRockColor   = null; // Rock Color Data

  private lithologyListStruct     stPlotLithology = null; // Rock Column (Plot)
  private textureListStruct       stPlotTexture   = null; // Rock Texture (Plot)

  // Type of data

  public static final int _HEADERS     = 0;  // Headers Data

  // -- LAS File & Geological Data

  public static final int _LAS3        = 1;  // LAS File Data
  public static final int _LAS         = 2;  // LAS File Data
  public static final int _ROCK        = 3;  // Rock Data
  public static final int _ROCK_IMAGES = 4;  // Rock Images

  // -- Horizon Data

  public static final int _TOPS        = 5;  // Tops Data
  public static final int _SEQ_STRAT   = 6;  // Sequence Stratigraphy
  public static final int _PERFORATION = 7;  // Perforations

  // -- Rock Observational & Computed Data

  public static final int _ROCK_COLUMN = 20; // (*) Rock Column Data

  public static final int _LITHOLOGY   = 8;  // (*) Rock Lithology
  public static final int _TEXTURE     = 9;  // (*) Rock Texture
  public static final int _ROCK_PHI    = 10; // (*) Rock Porosity
  public static final int _SED_STRUCT  = 11; // (*) Sedimentary Structure
  public static final int _FOSSILS     = 12; // (*) Fossils
  public static final int _ROCK_COLOR  = 13; // (*) Rock Color
  public static final int _REMARKS     = 14; // Remarks/Comments

  // -- Misc Computation Type Tracks

  public static final int _PFEFFER     = 15;  // Perforations
  public static final int _SHALE       = 16;  // Shale Gamma Ray Levels

  // -- Data Entry Panel

  private static final int _BY_PLOT    = 0;   // Enter & Plot Description
  private static final int _BY_TRACK   = 1;   // Enter Description by Track

  private static final String BUTTONS[] = {
    "Parse & Plot Description", "Enter Description by Image Tracks" };

  private static final String PANEL[] = {
    "Plot Description", "Enter Description by Image Track" };

  private int iPanel = _BY_PLOT;

  // Global Frame Widgets

  // -- Rock Description

  private JTextArea  txtRemarks   = new JTextArea();  // Rock Description

  // -- Depth Range Textfields

  private JTextField txtStart     = new JTextField(); // Starting Depth
  private JTextField txtEnd       = new JTextField(); // Ending Depth
  private JTextField txtThickness = new JTextField(); // Thickness of Bed

  // -- Global Frame Buttons

  private JButton    btnAdd       = new JButton();    // Add data to List
  private JButton    btnClear     = new JButton();    // Clear data
  private JButton    btnCancel    = new JButton();    // Cancel Modify

  private JButton    btnModify    = new JButton();    // Modify Row
  private JButton    btnRemove    = new JButton();    // Remove Row
  private JButton    btnRemoveAll = new JButton();    // Delete List

  // -- Diplay Descriptive Data Panel

  private JPanel       pnlData           = new JPanel();
  private CardLayout   cardLayout        = new CardLayout();
  private TitledBorder titledBorderEntry = null;

  // -- Rock Data Entry Buttons

  private JRadioButton rbByPlot   = new JRadioButton();
  private JRadioButton rbByData   = new JRadioButton();

  // -- Menus

  // .... File Button Menu

  private JMenuItem mHelp  = null;  // Help Menu Option
  private JMenuItem mExit  = null;  // Exit Dialog Menu Option

  /** Constructor rockObsEntryFrame()
   *  <p> This is the Constructor for this class.
   *  @param pNotifier = Observable
   *  @param stStruct = Global IQSTRAT Data Structure
   */

  public rockObsEntryFrame( Observable pNotifier, iqstratStruct stStruct )
  {
    try
    {
      this.pNotifier = pNotifier;
      this.stStruct  = stStruct;

      jbInit();
      addWindowListener( new rockObsEntryFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
	JPanel pnlTop         = new JPanel();
	JPanel pnlDisplayBy   = new JPanel();
	JPanel pnlCenter      = new JPanel();
	JPanel pnlText        = new JPanel();
	JPanel pnlBase        = new JPanel();
    JPanel pnlDepth       = new JPanel();
    JPanel pnlStart       = new JPanel();
    JPanel pnlEnd         = new JPanel();
    JPanel pnlThickness   = new JPanel();
    JPanel pnlAdd         = new JPanel();
    JPanel pnlClear       = new JPanel();
	JPanel pnlList        = new JPanel();
	JPanel pnlButtons     = new JPanel();
    JPanel pnlTemp        = new JPanel();

    JScrollPane scroll    = new JScrollPane();
    JScrollPane scrollTxt = new JScrollPane();

    ButtonGroup group     = new ButtonGroup();

    TitledBorder titledBorderRemarks = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Comments/Remarks/Notes:");
    titledBorderRemarks.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderStart = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Start Depth:");
    titledBorderStart.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderEnd = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "End Depth:");
    titledBorderEnd.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderThickness = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Thickness:");
    titledBorderThickness.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    titledBorderEntry = new TitledBorder(
        new EtchedBorder( EtchedBorder.RAISED,
                          Color.white,
                          new Color(165, 163, 151) ),
        "Plot Description");
    titledBorderEntry.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Create Menu Options

    JMenuBar menuBar   = new JMenuBar();
    JMenu    menuFile  = new JMenu("Close Dialog");
    JMenu    menuHelp  = new JMenu("Help");

    mExit = new JMenuItem("Close");
    mExit.addActionListener(this);

    menuFile.add(mExit);

    mHelp = new JMenuItem("Help");
    mHelp.addActionListener(this);

    menuHelp.add(mHelp);

    menuBar.add(menuFile);
    menuBar.add(menuHelp);

    this.setJMenuBar(menuBar);

    // Create an Observable

    notifier = new rockObsEntryFrameObservable();

    // Add this Dialog to the notifier as observer

    notifier.addObserver(this);

    // Create Frame

    this.setTitle("Enter Core / Rock / Bed Description");
    this.getContentPane().setLayout(new BorderLayout());

    // Top Panel

    pnlTop.setPreferredSize(new Dimension(10, 250));
    pnlTop.setLayout(new BorderLayout());

    // -- Display By Buttons

    pnlDisplayBy.setLayout(new GridLayout());

    // .. Rock Lithology

    rbByPlot.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbByPlot.setHorizontalAlignment(SwingConstants.CENTER);
    rbByPlot.setSelected(true);
    rbByPlot.setText(BUTTONS[_BY_PLOT]);
    rbByPlot.addActionListener(this);

    // .. Rock Lithology

    rbByData.setFont(new java.awt.Font("Monospaced", 1, 11));
    rbByData.setHorizontalAlignment(SwingConstants.CENTER);
    rbByData.setText(BUTTONS[_BY_TRACK]);
    rbByData.addActionListener(this);

    // -- Rock Data Panels

    pnlData.setBorder(titledBorderEntry);
    pnlData.setLayout(cardLayout);

    // .. Plot Image Tracks

    plot = new rockObsEntryPlotPanel(
		     stStruct, dStart, dEnd, iScale, iSelected );

    // .. Data Entry Panel

    panel = new rockObsEntryPanel( notifier, stStruct );

    // Center Panel

    pnlCenter.setLayout(new BorderLayout());

    // -- Remarks/Comments/Notes Panel

    pnlText.setBorder(titledBorderRemarks);
//    pnlText.setPreferredSize(new Dimension(10, 200));
    pnlText.setLayout(new BorderLayout());

    txtRemarks.setText("");
    txtRemarks.setRows(7);
    txtRemarks.setFont(new java.awt.Font("Dialog", 0, 14));
    txtRemarks.setLineWrap(true);
    txtRemarks.setWrapStyleWord(true);
    txtRemarks.addFocusListener(new rockObsEntryFrameFocusAdapter(this));

    // -- Base Panel

    pnlBase.setPreferredSize(new Dimension(10, 60));
    pnlBase.setLayout(new BorderLayout());

    // .. Depth Range Panel

    pnlDepth.setLayout(new GridLayout());

    // .... Starting Depth Panel

    pnlStart.setBorder(titledBorderStart);
    pnlStart.setLayout(new BorderLayout());

    txtStart.setText("0.0");
    txtStart.setColumns(8);
    txtStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtStart.addFocusListener(new rockObsEntryFrameFocusAdapter(this));

    // .... Endind Depth Panel

    pnlEnd.setBorder(titledBorderEnd);
    pnlEnd.setLayout(new BorderLayout());

    txtEnd.setText("20.0");
    txtEnd.setColumns(8);
    txtEnd.setHorizontalAlignment(SwingConstants.TRAILING);
    txtEnd.addFocusListener(new rockObsEntryFrameFocusAdapter(this));

    // .... Thickness Panel

    pnlThickness.setBorder(titledBorderThickness);
    pnlThickness.setLayout(new BorderLayout());

    txtThickness.setText("20.0");
    txtThickness.setColumns(8);
    txtThickness.setHorizontalAlignment(SwingConstants.TRAILING);
    txtThickness.addFocusListener(new rockObsEntryFrameFocusAdapter(this));

    // .. Add Buttons to Panel

    pnlAdd.setLayout(new GridLayout(2,1));

    btnAdd.setFont(new java.awt.Font("Dialog", 1, 11));
    btnAdd.setText("Add");
    btnAdd.addActionListener(this);

    pnlClear.setLayout(new GridLayout());

    btnClear.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClear.setText("Clear");
    btnClear.addActionListener(this);

    btnCancel.setFont(new java.awt.Font("Dialog", 1, 11));
    btnCancel.setText("Cancel");
    btnCancel.addActionListener(this);

    // .. Modify Existing Data

    pnlButtons.setLayout(new GridLayout());

    btnModify.setFont(new java.awt.Font("Dialog", 1, 11));
    btnModify.setPreferredSize(new Dimension(98, 25));
    btnModify.setText("Modify");
    btnModify.addActionListener(this);

    btnRemove.setFont(new java.awt.Font("Dialog", 1, 11));
    btnRemove.setPreferredSize(new Dimension(98, 25));
    btnRemove.setText("Remove");
    btnRemove.addActionListener(this);

    btnRemoveAll.setFont(new java.awt.Font("Dialog", 1, 11));
    btnRemoveAll.setPreferredSize(new Dimension(98, 25));
    btnRemoveAll.setText("Remove All");
    btnRemoveAll.addActionListener(this);

    // Rock Data List Table Panel

//    pnlList.setBorder(titledBorderData);
    pnlList.setLayout(new BorderLayout());
    pnlList.setPreferredSize(new Dimension(10, 200));

    // .. Build Data Entry Table

    pTable = new rockEnterDataTable( stStruct.stSymbols );
    scroll = pTable.getScrollPane();

    setButtons();

    // Attach Widgets to Frame

    this.getContentPane().add( pnlTop,      BorderLayout.CENTER );
    pnlTop.add( pnlDisplayBy,               BorderLayout.NORTH );
    pnlDisplayBy.add( rbByPlot,             null );
    pnlDisplayBy.add( rbByData,             null );

    group.add( rbByPlot );
    group.add( rbByData );

    pnlTop.add( pnlData,                    BorderLayout.CENTER );
    pnlData.add( plot,                     "BY_PLOT" );
    pnlData.add( panel,                    "BY_TRACK" );
//    pnlTop.add( plot,                       BorderLayout.CENTER);

    this.getContentPane().add( pnlCenter,   BorderLayout.NORTH );
    pnlCenter.add( pnlText,                 BorderLayout.CENTER );
    pnlText.add(scrollTxt,                  BorderLayout.CENTER);
    scrollTxt.getViewport().add(txtRemarks, null);

    pnlCenter.add( pnlBase,                 BorderLayout.NORTH );
    pnlBase.add( pnlDepth,                  BorderLayout.WEST );
    pnlDepth.add( pnlStart,                 null );
    pnlStart.add( txtStart,                 BorderLayout.CENTER );

    pnlDepth.add( pnlEnd,                   null );
    pnlEnd.add( txtEnd,                     BorderLayout.CENTER );

    pnlDepth.add( pnlThickness,             null );
    pnlThickness.add( txtThickness,         BorderLayout.CENTER );

    pnlBase.add(pnlAdd,                     BorderLayout.EAST);
    pnlAdd.add(btnAdd,                      null);

    pnlAdd.add(pnlClear,                    null);
    pnlClear.add(btnCancel,                 null);
    pnlClear.add(btnClear,                  null);

    this.getContentPane().add( pnlList,     BorderLayout.SOUTH );
    pnlList.add( scroll,                    BorderLayout.CENTER );

    pnlList.add( pnlButtons,                BorderLayout.SOUTH );
    pnlButtons.add(btnModify,               null);
    pnlButtons.add(btnRemove,               null);
    pnlButtons.add(btnRemoveAll,            null);

    // Display the Frame

    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setSize(new Dimension(770, 780));
//    this.setSize(new Dimension(550, d.height-25));
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

//    this.setResizable(false);
    this.setVisible(true);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getRemarks()
   *  <p> This method will retrieve the final list of the Remarks List Data
   *  @return stRemarks = Remarks List Data Structure
   */

  public iqstratRemarkListStruct getRemarks() { return (stRemarks); }

  /** Method getData()
   * <p> This method will retrieve data
   * @param iData = Data Type being retrieved
   */

  private void getData( int iData)
  {
    switch (iData)
    {
      case _HEADERS:     // Headers Data
      case _LAS3:        // LAS 3 File Data
      case _LAS:         // LAS File Data
      case _SHALE:       // Shale gamma ray limits
      case _ROCK:        // Rock Data
      case _ROCK_IMAGES: // Rock Images
      case _TOPS:        // Tops Data
      case _SEQ_STRAT:   // Sequence Stratigraphy
      case _PERFORATION: // Perforations
      case _REMARKS:     // Remarks/Comments
        break;

      case _ROCK_PHI:    // Rock Porosity
        stPHI = rock.phi.phiUtility.copyList(
			      parse.parsePHIUtility.parse(
                    iqstrat.iqstratRemarkUtility.copyList( stTemp ),
                    stStruct.stParsePorosity) );

        break;

      case _ROCK_COLUMN: // Rock Column Data
        stColumn = lith.rock.rockColumnUtility.copyList(
                     parse.parseRockColumnSymbolsUtility.parse(
                       iqstrat.iqstratRemarkUtility.copyList( stTemp ),
                       stStruct.stParseRock ) );
        break;

      case _LITHOLOGY:   // Rock Lithology
        stLithology = parse.parseMineralsUtility.parse(
                        iqstrat.iqstratRemarkUtility.copyList( stTemp ),
                        stStruct.stSymbols,
                        stStruct.stParseTexture,
                        stStruct.stMinerals );

        stPlotLithology = iqstrat.iqstratControlPlotData.getPlotLithology(
                            stLithology, stPlotLithology );
        break;

      case _TEXTURE:     // Rock Texture
        stTexture = parse.parseTextureUtility.parse(
                      iqstrat.iqstratRemarkUtility.copyList( stTemp ),
                      stStruct.stSymbols,
                      stStruct.stParseTexture );

        stPlotTexture = iqstrat.iqstratControlPlotData.getPlotTexture(
                          stTexture, stPlotTexture );
        break;

      case _SED_STRUCT:  // Sedimentary Structure
        stSedimentary = rock.sedimentary.sedimentaryUtility.copyList(
			              parse.parseSedStructUtility.parse(
                            iqstrat.iqstratRemarkUtility.copyList( stTemp ),
                            stStruct.stParseSedStruct ));

        modifyWidths();
        break;

      case _FOSSILS:     // Fossils
        stFossil = rock.fossil.fossilUtility.copyList(
                     parse.parseFossilUtility.parse(
                       iqstrat.iqstratRemarkUtility.copyList( stTemp ),
                       stStruct.stParseFossils,
                       stStruct.stBio,
                       stStruct.stKSBio ));

        modifyWidths();
        break;

      case _ROCK_COLOR:  // Rock Color
        stRockColor = rock.color.rockColorUtility.copyList(
                        parse.parseColorUtility.parse(
                          iqstrat.iqstratRemarkUtility.copyList( stTemp ),
                          stStruct ) );

        break;
    }
  }

  /** Method getGeoReportData()
   * <p> This method will compute the Geologist Report Data tracks if the
   *     tracks are empty
   */

  private void getGeoReportData()
  {
    getData( _ROCK_COLUMN );
    getData( _ROCK_PHI );
    getData( _LITHOLOGY );
    getData( _TEXTURE );
    getData( _ROCK_COLOR );
    getData( _FOSSILS );
    getData( _SED_STRUCT );

    plotData();
  }

  /** Method getRemark()
   * <p> This method will retrieve the Remarks to be added to the list
   * @return st = Remarks Data Structure
   */

  private iqstratRemarkStruct getRemark()
  {
    iqstratRemarkStruct st = new iqstratRemarkStruct();

//    if (iAction == _ADD)
//      st.sKEY = new String(cmn.cmnString.UniqueName());
//    else
//      st.sKEY = new String(stRemark.sKEY);

    st.depthStart = dStart;
    st.depthEnd   = dEnd;
    st.sText      = txtRemarks.getText();

    if ((st.sNote.length() == 0) && (st.sText.length() > 0))
    {
      if (st.sText.length() < 41)
        st.sNote = new String(st.sText);
      else
        st.sNote = new String(st.sText.substring(0, 40));
    }

    return (st);
  }

  /** Method getTextArea()
   *  <p> This method will retrieve the Text Area and build list for plotting
   */

  private void getTextArea()
  {
	closeData();

    iqstratRemarkStruct st = getRemark();
    st.depthStart =  0.0;
    st.depthEnd   = 20.0;

    stTemp = iqstrat.iqstratRemarkUtility.add( st, stTemp );

    getGeoReportData();
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setButtons()
   * <p> This method will enable or disable buttons depending on data
   */

  private void setButtons()
  {
	btnCancel.setEnabled(false);

	if (iAction == _MODIFY)
	{
  	  btnCancel.setEnabled(true);
	}

    btnModify.setEnabled(false);
    btnRemove.setEnabled(false);
    btnRemoveAll.setEnabled(false);

    if ((pTable.getTotalRows() > 0) && (iAction == _ADD))
    {
      btnModify.setEnabled(true);
      btnRemove.setEnabled(true);
      btnRemoveAll.setEnabled(true);
    }
  }

  /** Method setBorders()
   * <p> This method will set the borders of the List and Data Entry Panel
   *  @param iPanel = The type of panel to display
   */

  private void setBorders(int iPanel)
  {
    this.iPanel = iPanel;

	switch (iPanel)
	{
	  case _BY_PLOT:
        getTextArea();
        cardLayout.show(pnlData, "BY_PLOT");
        break;
      case _BY_TRACK:
        cardLayout.show(pnlData, "BY_TRACK");
        break;
    }

    titledBorderEntry.setTitle( PANEL[iPanel] );
    pnlData.updateUI();
  }

  /** Method setRemarksData()
   * <p> This method will load the comments data list structure into this frame
   * @param stRemarks = the comments data list structure
   */

  public void setRemarksData( iqstratRemarkListStruct stRemarks )
  {
    this.stRemarks = stRemarks;
    if (pTable != null) { pTable.setRemarksData(stRemarks); }
    setButtons();
  }

  /** Method setDepth()
   *  <p> This method will set the depth Range and scale.
   */

  private void setDepth()
  {
	int    istrt = (int) ( dStart / 10.0 );
	double dS    = dStart; //10.0 * istrt;
	double dE    = 0.0;

    setScale();

    switch(iScale)
    {
      case iqstrat.iqstratTracksStruct.SCALE_100:
        dE = dS + 100.0;
  	    break;
      case iqstrat.iqstratTracksStruct.SCALE_50:
        dE = dS + 50.0;
  	    break;
      case iqstrat.iqstratTracksStruct.SCALE_20:
        dE = dS + 20.0;
  	    break;
      case iqstrat.iqstratTracksStruct.SCALE_10:
        dE = dS + 10.0;
  	    break;
      case iqstrat.iqstratTracksStruct.SCALE_5:
        dE = dS + 5.0;
  	    break;
	  case iqstrat.iqstratTracksStruct.SCALE_2:
        dE = dS + 4.0;
  	    break;
	  case iqstrat.iqstratTracksStruct.SCALE_1:
        dE = dS + 2.0;
  	    break;
	}

//    if (plot != null) { plot.setScale( iScale, dE, dS ); }
    if (plot != null)
    {
	  plot.setScale( iqstrat.iqstratTracksStruct.SCALE_20, 20.0, 0.0 );
	}
  }

  /** Method setScale()
   *  <p> This method will set the Scale ( ft/inch ) based on bed thickness.
   */

  private void setScale()
  {
    if (dThickness <= 100.0)
      iScale = iqstrat.iqstratTracksStruct.SCALE_100;

    if (dThickness <= 50.0)
      iScale = iqstrat.iqstratTracksStruct.SCALE_50;

    if (dThickness <= 20.0)
      iScale = iqstrat.iqstratTracksStruct.SCALE_20;

    if (dThickness <= 10.0)
      iScale = iqstrat.iqstratTracksStruct.SCALE_10;

    if (dThickness <= 5.0)
      iScale = iqstrat.iqstratTracksStruct.SCALE_5;

    if (dThickness <= 2.0)
	  iScale = iqstrat.iqstratTracksStruct.SCALE_2;

    if (dThickness <= 1.0)
	  iScale = iqstrat.iqstratTracksStruct.SCALE_1;
  }

  /** Method modifyWidths()
   * <p> This method will modify the plot track width depending on the
   *     maximum number of fossils or sedimentary structures per depth.
   */

  private void modifyWidths()
  {
    int iBio = 0;
    int iSed = 0;

    iSed = rock.sedimentary.sedimentaryUtility.computeWidth( stSedimentary );
    iBio = rock.fossil.fossilUtility.computeWidth( stFossil );

    if (plot != null) { plot.modifyWidths( iBio, iSed ); }
  }

  /** Method clear()
   * <p> This method will clear the selected values
   */

  public void clear()
  {
//    iAction  = _ADD;
//    btnAdd.setText("Add");

    sKEY = new String("0");

    txtRemarks.setText("");

    dStart     = 0.0;
    dEnd       = 20.0;
    dThickness = 20.0;

    txtStart.setText( "" + dStart );
    txtEnd.setText( "" + dEnd );
    txtThickness.setText( "" + dThickness );

    setDepth();

    closeData();

    if (plot != null)
    {
	  plot.setPHI( stPHI );
      plot.setRockColumn( stColumn );
//      plot.setRockLithology( stPlotLithology );
//      plot.setTexture( stPlotTexture );
      plot.setStructures( stSedimentary );
      plot.setFossils( stFossil );
      plot.setRockColorData( stRockColor );
    }

    modifyWidths();

    if (panel != null) { panel.clear(); }
  }

  /** Method cancel()
   * <p> This method will clear the edit
   */

  public void cancel()
  {
    iAction  = _ADD;
    btnAdd.setText("Add");

    txtRemarks.setText("");

    dStart     = 0.0;
    dEnd       = 20.0;
    dThickness = 20.0;

    txtStart.setText( "" + dStart );
    txtEnd.setText( "" + dEnd );
    txtThickness.setText( "" + dThickness );

    setDepth();

    closeData();

    if (plot != null)
    {
	  plot.setPHI( stPHI );
      plot.setRockColumn( stColumn );
//      plot.setRockLithology( stPlotLithology );
//      plot.setTexture( stPlotTexture );
      plot.setStructures( stSedimentary );
      plot.setFossils( stFossil );
      plot.setRockColorData( stRockColor );
    }

    modifyWidths();

    if (panel != null) { panel.clear(); }
  }

  /** Method plotData()
   *  <p> This method will load the plot & panel for the selected data or
   *      changes to the plot.
   */

  private void plotData()
  {
    if (plot != null)
    {
	  plot.setPHI( stPHI );
      plot.setRockColumn( stColumn );
//      plot.setRockLithology( stPlotLithology );
//      plot.setTexture( stPlotTexture );
      plot.setStructures( stSedimentary );
      plot.setFossils( stFossil );
      plot.setRockColorData( stRockColor );
    }

    modifyWidths();

    if (panel != null)
    {
  	  panel.setData( stPHI );
  	  panel.setData( stColumn );
  	  panel.setData( stLithology );
  	  panel.setData( txtRemarks.getText(), stTexture );
  	  panel.setData( stSedimentary );
  	  panel.setData( stFossil );
  	  panel.setData( stRockColor );
  	}
  }

  /* ===================================================================== *
   * ------------------------ Button Actions ----------------------------- *
   * ===================================================================== */

  /** Method add()
   * <p> This method will add/modify the data to the Remarks List and then
   *     loads data to the plot.
   */

  private void add()
  {
    if (stRemark == null)
    {
      stRemark      = new iqstratRemarkStruct();
      stRemark.sKEY = new String( cmn.cmnString.UniqueName() );
    }

    stRemark.depthStart = dStart;
    stRemark.depthEnd   = dEnd;
    stRemark.dThickness = dThickness;
    stRemark.sText      = new String( txtRemarks.getText() );

    if (stRemark.sText.length() < 41)
      stRemark.sNote = new String( stRemark.sText );
    else
      stRemark.sNote = new String( stRemark.sText.substring(0, 40) );

    switch (iAction)
    {
      case _ADD:
        stRemarks = iqstrat.iqstratRemarkUtility.add( stRemark, stRemarks );
        break;

      case _MODIFY:
        stRemarks = iqstrat.iqstratRemarkUtility.modify(
			          stRemark.sKEY, stRemark, stRemarks );
        break;
    }

    stRemarks = iqstrat.iqstratRemarkUtility.bubbleSort( stRemarks );

    setRemarksData( stRemarks );

//iqstrat.iqstratRemarkUtility.print(stRemarks);

    clear();

	if (stRemark != null)
	  stRemark.delete();
    stRemark  = null; // Remarks Data Struct

	if (pNotifier != null) {
      pNotifier.notifyObservers(new String("ADD or MODIFY Remarks Data")); }
  }

  /** Method modify()
   * <p> This method will load the selected data type into the corresponding
   *     panel for modification
   */

  private void modify()
  {
	int iTemp = 0;

	if (stRemark != null)
	  stRemark.delete();
    stRemark  = null; // Remarks Data Struct

	closeData();

    iAction  = _MODIFY;
    btnAdd.setText("Modify");

    stRemark = iqstrat.iqstratRemarkUtility.copy( pTable.getRemarkData() );
    sKEY     = new String(stRemark.sKEY);

    txtRemarks.setText( stRemark.sText );

    dStart     = stRemark.depthStart;
    dEnd       = stRemark.depthEnd;
    iTemp      = (int) ( 10.0 * Math.abs(dEnd - dStart) );
    dThickness = (double) iTemp / 10.0;

    txtStart.setText( "" + dStart );
    txtEnd.setText( "" + dEnd );
    txtThickness.setText( "" + dThickness );

    setDepth();
    getTextArea();
  }

  /** Method removeRow()
   * <p> This method will remove a record from the selected data type list
   */

  private void removeRow()
  {
    iqstratRemarkStruct st = pTable.getRemarkData();

    stRemarks = iqstrat.iqstratRemarkUtility.remove( st.sKEY, stRemarks );

    setRemarksData( stRemarks );

	if (pNotifier != null) {
      pNotifier.notifyObservers(new String("REMOVE Remarks Data")); }
  }

  /** Method removeAllRows()
   * <p> This method will remove all records from the selected data type list
   */

  private void removeAllRows()
  {
	if (stRemarks != null)
	  stRemarks.delete();
	stRemarks = null;

    setRemarksData( stRemarks );

    clear();

	if (pNotifier != null) {
      pNotifier.notifyObservers(new String("REMOVE All Remarks Data")); }
  }

  /* ===================================================================== *
   * ------------------------- Close Methods ----------------------------- *
   * ===================================================================== */

  /** Method closeData()
   * <p> This method will clear the data structures
   */

  private void closeData()
  {
	if (stTemp != null)
	  stTemp.delete();
	stTemp = null;          // Temporary Remarks List Data Struct

    if (stPHI != null)
      stPHI.delete();
    stPHI = null;           // Porosity Track data

    if (stColumn != null)
      stColumn.delete();
    stColumn      = null;   // Rock Column Data List

    if (stLithology != null)
      stLithology.delete();
    stLithology = null;     // Rock Column

    if (stTexture != null)
      stTexture.delete();
    stTexture    = null;    // Rock Texture

    if (stSedimentary != null)
      stSedimentary.delete();
    stSedimentary = null;   // Sedimentary Structure

    if (stFossil != null)
      stFossil.delete();
    stFossil = null;        // Fossils

    if (stRockColor != null)
      stRockColor.delete();
    stRockColor = null;     // Rock Color Data

    if (stPlotLithology != null)
      stPlotLithology.delete();
    stPlotLithology = null;

    if (stPlotTexture != null)
      stPlotTexture.delete();
    stPlotTexture = null;
  }

  /** Method close()
   * <p> This method will cancel the Rock Color Dialog
   */

  public void close()
  {
    // Input Variables

    pNotifier = null; // Input Observable
    stStruct  = null; // Global IQSTRAT Data Structure

    // Global Variables

    sKEY     = null;

    notifier = null;  // Frame Observable

    // -- Data Entry List Table

    if (pTable != null)
      pTable.close();
    pTable = null; // Rock Data Entry Table

    // -- Plot Panel

	if (plot != null)
	  plot.close();
    plot = null; // Rock Plot Bed Panel

    if (panel != null)
      panel.close();
    panel = null; // Rock Data Entry Panel

    // -- Rock Data Structures

    stRemarks = null; // Remarks List Data Struct

    closeData();

    // Global Frame Widgets

    // -- Rock Description

    txtRemarks   = null;  // Rock Description

    // -- Depth Range Textfields

    txtStart     = null; // Starting Depth
    txtEnd       = null; // Ending Depth
    txtThickness = null; // Thickness of Bed

    // -- Global Frame Buttons

    btnAdd       = null;    // Add data to List
    btnClear     = null;    // Clear data
    btnCancel    = null;    // Cancel Modify

    btnModify    = null;    // Modify Row
    btnRemove    = null;    // Remove Row
    btnRemoveAll = null;    // Delete List

    // -- Diplay Descriptive Data Panel

    pnlData           = null;
    cardLayout        = null;
    titledBorderEntry = null;

    // -- Rock Data Entry Buttons

    rbByPlot     = null;
    rbByData     = null;

    // -- Menus

    // .... File Button Menu

    mHelp        = null;  // Help Menu Option
    mExit        = null;  // Exit Dialog Menu Option

    dispose();
  }

  /** METHOD rockObsEntryFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class rockObsEntryFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
      close();
    }
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
	if (event.getSource() == rbByPlot)     { setBorders(_BY_PLOT); }
	if (event.getSource() == rbByData)     { setBorders(_BY_TRACK); }

    if (event.getSource() == btnAdd)       { add(); }
    if (event.getSource() == btnClear)     { clear(); }
    if (event.getSource() == btnCancel)    { cancel(); }

    if (event.getSource() == btnModify)    { modify(); }
    if (event.getSource() == btnRemove)    { removeRow(); }
    if (event.getSource() == btnRemoveAll) { removeAllRows(); }

    setButtons();

    if (event.getSource() == mExit)        { close(); }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    int    iTemp    = 0;
    String sMessage = new String("");
    String sTemp    = new String("");
    double depthS   = 0.0;

    if (e.getSource() == txtStart)
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Start Depth Value is a Numeric Field");
    }

    if (e.getSource() == txtEnd)
    {
      iNumeric = 1;
      sTemp = txtEnd.getText();
      sMessage = new String("End Depth Value is a Numeric Field");
    }

    if (e.getSource() == txtThickness)
    {
      iNumeric = 1;
      sTemp = txtThickness.getText();
      sMessage = new String("Bed Thickness Value is a Numeric Field");
	}

    if (e.getSource() == txtRemarks) { getTextArea(); }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
		depthS = cmn.cmnString.stringToDouble(txtStart.getText());

        if (e.getSource() == txtStart)
        {
          dStart = cmn.cmnString.stringToDouble(txtStart.getText());

          if (dThickness > 0.0)
          {
            iTemp      = (int) ( 10.0 * (depthS + dThickness) );
            dEnd       = (double) iTemp / 10.0;
            txtEnd.setText( "" + dEnd );
	      }
        }

        if (e.getSource() == txtEnd)
        {
          dEnd       = cmn.cmnString.stringToDouble(txtEnd.getText());

          if (dEnd == dStart)
          {
            dEnd = dStart + 10.0;
            txtEnd.setText( "" + dEnd );
		  }

          iTemp      = (int) ( 10.0 * Math.abs(dEnd - depthS) );
          dThickness = (double) iTemp / 10.0;
          txtThickness.setText( "" + dThickness );
	    }

        if (e.getSource() == txtThickness)
        {
          dThickness = cmn.cmnString.stringToDouble(txtThickness.getText());

          if (dThickness <= 0.0)
          {
            dThickness = 10.0;
            txtThickness.setText( "" + dThickness );
	      }

          iTemp      = (int) ( 10.0 * (depthS + dThickness) );
          dEnd       = (double) iTemp / 10.0;
          txtEnd.setText( "" + dEnd );
	    }

        setDepth();
  	    getTextArea();
      }
    }
  }

  /* ===================================================================== *
   * ----------------------- OBSERVABLE ACTIONS -------------------------- *
   * ===================================================================== */

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String sArg      = new String((String) arg);

    if (sArg.equals("Rock Description"))
    {
      txtRemarks.setText( "" + dStart + "; " +
                               dEnd   + "; " +
                               panel.getData() );
	}
  }
}


/** CLASS rockObsEntryFrameFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class rockObsEntryFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  rockObsEntryFrame adaptee;

  rockObsEntryFrameFocusAdapter(rockObsEntryFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/** CLASS rockObsEntryFrameObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class rockObsEntryFrameObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 04/08/2011
 *  @author  John R. Victorine
 */

