/*
 * @rockImagePanel.java Version 1.1 11/08/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.gui;

import java.net.*;
import java.lang.*;
import java.awt.*;
import java.awt.event.*;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;

import cmn.cmnStruct;
import rock.rockImagesListStruct;
import rock.rockImagesStruct;

/** Class rockImagePanel
 *  <p> This class will allow the user to select core Images.
 *
 *  @version 1.1 11/08/2007
 *  @author  John R. Victorine
 */

public class rockImagePanel extends Panel implements ActionListener
{
  // Input Variables

  private Observable           notifier = null;
  private cmnStruct            stCMN    = null;
  private rockImagesListStruct stImages = null;

  // Global Variables

  private rockImagesListStruct     stCoreList = null;
  private rockImagePanelPlotCanvas pCoreImage = null;

  private JPanel  pCenter  = new JPanel();
  private JPanel  pImages  = new JPanel();

  private int     iWidth     = 0;
  private int     iDisplay[] = { cmn.cmnStruct.OFF,
                                 cmn.cmnStruct.OFF,
                                 cmn.cmnStruct.OFF };
  private String  sDisplay[] = { "IMAGE: CORE",
                                 "IMAGE: SLAB",
                                 "IMAGE: THIN SECTION" };
//  private int     iState     = cmn.cmnStruct.SELECT;
  private int     iSelect    = 0;
  private JPanel  pDisplay   = new JPanel();

  private int     iTotal = 0;  // Total Number of Core Images
  private int     iButtons[];  // Yes/No Indicator for row
  private String  sKID[];      // Array of KID's for the Core Data
  private JPanel  pRow[];      // Row to place Core Image Data in
  private JPanel  pRWest[];    // Panel Holding the View and Yes/No Buttons
  private JPanel  pRCenter[];  // Panel Holding the Depth Range
  private JPanel  pREast[];    // Panel Holding the Core Data Type
  private JButton btnYesNo[];  // Yes/No Indicator Button
  private JButton btnView[];   // View Core Image Button
  private JLabel  lblStart[];  // Begin Image Depth Label
  private JLabel  lblEnd[];    // End Image Depth Label
  private JLabel  lblType[];   // Image Type Label

  /** Constructor rockImagePanel()
   *  <p> This is the Constructor for this class.
   *  @param stCMN    = The Common Global Data Structure
   *  @param notifier = Observable
   *  @param stImages = Rock Image Data List Structure
   */

  public rockImagePanel(cmnStruct stCMN,
                        Observable notifier,
                        rockImagesListStruct stImages)
  {
    try
    {
      this.stCMN    = stCMN;
      this.notifier = notifier;
      this.stImages = stImages;
      if (this.stImages != null)
      {
        jbInit();
      }
    }
    catch(Exception ex)
    {
      ex.printStackTrace();
    }
  }

  /** Method jbInit()
   *  <p> This method will create this Frame.
   * @throws Exception
   */

  void jbInit() throws Exception
  {
    JScrollPane jScrollPane = new JScrollPane();

    this.setLayout(new BorderLayout());
    pCenter.setLayout(new BorderLayout());

    this.add(pCenter, BorderLayout.CENTER);
    pCenter.add(jScrollPane, BorderLayout.CENTER);
    jScrollPane.getViewport().add(pImages, null);

    BuildTable(stImages);

//    setSelection(cmn.cmnStruct.ALL);
  }

  /* ===================================================================== *
   * ---------------------- BUILD DIALOG METHODS ------------------------- *
   * ===================================================================== */

  /** Method BuildTable()
   * <p> This method will build the Image table
   * @param stList = The Core Image File List Data Structure
   */

  private void BuildTable(rockImagesListStruct stList)
  {
    pCenter.setPreferredSize(new Dimension(450, 200));
    pImages.setLayout(new GridLayout(8,1));

    iTotal = 0;
    if (stList != null)
      iTotal = stList.iCount;

    if (iTotal > 7)
    {
      pCenter.setPreferredSize(new Dimension(450, (iTotal+1)*25));
      pImages.setLayout(new GridLayout(iTotal+1, 1));
    }

    JPanel pHDR = BuildHeaderRow();
    pImages.add(pHDR, null);

    InitializeRow(iTotal);

    for (int i=0; i<iTotal; i++)
    {
      BuildRow(i, stList.stItem[i]);
      pImages.add(pRow[i], null);
    }
  }

  /** Method BuildHeaderRow()
   * <p> This method will build the Header Row for Image List Table
   * @return pHDR = The Header Row Panel
   */

  private JPanel BuildHeaderRow()
  {
    JPanel  pHDR        = new JPanel();
    JPanel  pHDRWest    = new JPanel();
    JPanel  pHDRCenter  = new JPanel();
    JPanel  pHDREast    = new JPanel();
    JButton btnHDRYesNo = new JButton();
    JButton btnHDRView  = new JButton();
    JButton btnHDRStart = new JButton();
    JButton btnHDREnd   = new JButton();
    JButton btnHDRType  = new JButton();

    pHDR.setLayout(new BorderLayout());
    pHDRWest.setPreferredSize(new Dimension(120, 10));
    pHDRWest.setLayout(new GridLayout(1,2));
    pHDREast.setPreferredSize(new Dimension(150, 10));
    pHDREast.setLayout(new BorderLayout());
    pHDRCenter.setLayout(new GridLayout(1,2));

    btnHDRYesNo.setFont(new java.awt.Font("Dialog", 1, 11));
    btnHDRYesNo.setBorder(BorderFactory.createRaisedBevelBorder());
    btnHDRYesNo.setText("Yes/No");

    btnHDRView.setFont(new java.awt.Font("Dialog", 1, 11));
    btnHDRView.setBorder(BorderFactory.createRaisedBevelBorder());
    btnHDRView.setText("View");

    btnHDRStart.setFont(new java.awt.Font("Dialog", 1, 11));
    btnHDRStart.setBorder(BorderFactory.createRaisedBevelBorder());
    btnHDRStart.setText("Start");

    btnHDREnd.setFont(new java.awt.Font("Dialog", 1, 11));
    btnHDREnd.setBorder(BorderFactory.createRaisedBevelBorder());
    btnHDREnd.setText("End");

    btnHDRType.setFont(new java.awt.Font("Dialog", 1, 11));
    btnHDRType.setBorder(BorderFactory.createRaisedBevelBorder());
    btnHDRType.setText("Core Image Type");

    // Add Widgets to Panel

    pHDR.add(pHDRWest, BorderLayout.WEST);
    pHDRWest.add(btnHDRYesNo, null);
    pHDRWest.add(btnHDRView, null);
    pHDR.add(pHDRCenter, BorderLayout.CENTER);
    pHDRCenter.add(btnHDRStart, null);
    pHDRCenter.add(btnHDREnd, null);
    pHDR.add(pHDREast,  BorderLayout.EAST);
    pHDREast.add(btnHDRType, BorderLayout.CENTER);

    return (pHDR);
  }

  /** Method InitializeRow()
   * <p> This method will allocate space for the Image Data
   * @param iRows = The total rows to initialize
   */

  private void InitializeRow(int iRows)
  {
    iButtons = new int[iRows];
    sKID     = new String[iRows];

    pRow     = new JPanel[iRows];
    pRWest   = new JPanel[iRows];
    pRCenter = new JPanel[iRows];
    pREast   = new JPanel[iRows];
    btnYesNo = new JButton[iRows];
    btnView  = new JButton[iRows];
    lblStart = new JLabel[iRows];
    lblEnd   = new JLabel[iRows];
    lblType  = new JLabel[iRows];
  }

  /** Method BuildRow()
   * <p> This method will build the Row for Image List Table
   * @param iRow        = The Row ID for the image row
   * @param st          = The Core Image File Data Structure
   * @return pRow[iRow] = The Row holding the Core Information data
   */

  private JPanel BuildRow(int iRow, rockImagesStruct st)
  {
    pRow[iRow] = new JPanel();
    pRow[iRow].setLayout(new BorderLayout());

    pRWest[iRow] = new JPanel();
    pRWest[iRow].setPreferredSize(new Dimension(120, 10));
    pRWest[iRow].setLayout(new GridLayout(1,2));

    pREast[iRow] = new JPanel();
    pREast[iRow].setPreferredSize(new Dimension(150, 10));
    pREast[iRow].setLayout(new BorderLayout());

    pRCenter[iRow] = new JPanel();
    pRCenter[iRow].setLayout(new GridLayout(1,2));

    iButtons[iRow] = cmn.cmnStruct.OFF;
    sKID[iRow]     = new String(st.sKID);

    btnYesNo[iRow] = new JButton();
    btnYesNo[iRow].setBackground(Color.darkGray);
    btnYesNo[iRow].setFont(new java.awt.Font("Dialog", 1, 11));
    btnYesNo[iRow].setForeground(Color.yellow);
    btnYesNo[iRow].setBorder(BorderFactory.createRaisedBevelBorder());
    btnYesNo[iRow].setText("No");
    btnYesNo[iRow].addActionListener(this);

    btnView[iRow] = new JButton();
    btnView[iRow].setFont(new java.awt.Font("Dialog", 1, 11));
    btnView[iRow].setBorder(BorderFactory.createRaisedBevelBorder());
    btnView[iRow].setText("View");
    btnView[iRow].addActionListener(this);

    lblStart[iRow] = new JLabel();
    lblStart[iRow].setFont(new java.awt.Font("Dialog", 1, 11));
    lblStart[iRow].setBorder(BorderFactory.createEtchedBorder());
    lblStart[iRow].setHorizontalAlignment(SwingConstants.CENTER);
    lblStart[iRow].setText(""+st.depthStart);

    lblEnd[iRow] = new JLabel();
    lblEnd[iRow].setFont(new java.awt.Font("Dialog", 1, 11));
    lblEnd[iRow].setBorder(BorderFactory.createEtchedBorder());
    lblEnd[iRow].setHorizontalAlignment(SwingConstants.CENTER);
    lblEnd[iRow].setText(""+st.depthEnd);

    lblType[iRow] = new JLabel();
    lblType[iRow].setFont(new java.awt.Font("Dialog", 1, 11));
    lblType[iRow].setBorder(BorderFactory.createEtchedBorder());
    lblType[iRow].setHorizontalAlignment(SwingConstants.CENTER);
    lblType[iRow].setText(st.sType);

    // Attach Widgets to Row

    pRow[iRow].add(pRWest[iRow],       BorderLayout.WEST);
    pRow[iRow].add(pRCenter[iRow],     BorderLayout.CENTER);
    pRow[iRow].add(pREast[iRow],       BorderLayout.EAST);
    pRWest[iRow].add(btnYesNo[iRow],   null);
    pRWest[iRow].add(btnView[iRow],    null);
    pRCenter[iRow].add(lblStart[iRow], null);
    pRCenter[iRow].add(lblEnd[iRow],   null);
    pREast[iRow].add(lblType[iRow],    BorderLayout.CENTER);

    return (pRow[iRow]);
  }

  /* ===================================================================== *
   * ------------------- DISPLAY CORE IMAGE METHODS ---------------------- *
   * ===================================================================== */

  /** Method DisplayCoreImage()
   *  <p> This Panel will display the Core Image
   *  @param sKEY   = The Primary Key for the Core Image
   */

  private void DisplayCoreImage(String sKEY)
  {
    JScrollPane psImage      = new JScrollPane();
    String      sFilename    = "";
    URL         urlCore      = null; //getURL(sFilename);
    int         iImageWidth  = iWidth;
    int         iImageHeight = iWidth;
    int         iRow         = -1;

    for (int i=0; i<stImages.iCount; i++)
    {
      if (sKEY.equals(stImages.stItem[i].sKID))
      {
        iRow = i;
      }
    }

    if (iRow > -1)
    {
      sFilename = new String(stImages.stItem[iRow].sURL + "/" +
                             stImages.stItem[iRow].sFilename);
      urlCore = getURL(sFilename);

      if (stImages.stItem[iRow].sType.equals("IMAGE: CORE"))  // Core Box
      {
        iImageWidth =  iWidth / 2;
      }

      Image image = Toolkit.getDefaultToolkit().getImage(urlCore);
      if (pCoreImage == null)
      {
        pCoreImage = new rockImagePanelPlotCanvas(image,
                                                  iImageWidth,
                                                  iImageHeight);
        pCoreImage.setBackground(Color.white);
        psImage.getViewport().add(pCoreImage, null);

        if (pDisplay != null)
          pDisplay.add(psImage, null);
      }
      else
        pCoreImage.setImage(image, iImageWidth, iImageHeight);
    }
  }

  /** METHOD getURL()
   *  <p> This method will locate the URL for the specific file.
   * @param  filename = The Filename to be converted to URL
   * @return url      = The URL of the Filename.
   */

  protected URL getURL(String filename)
  {
     URL url         = null;
     URL urlCodeBase = null;

     if (urlCodeBase == null)
     {
       urlCodeBase = stCMN.urlDirCodeBase;
     }

     try
     {
       url = new URL(urlCodeBase, filename);
     }
     catch (java.net.MalformedURLException e)
     {
       String Message = new String("rockImagesPanel.getURL(): \n" +
                                  e.getMessage() + "\n" +
                                  "Couldn't create image: " + filename);
       System.out.println(e.getMessage());
       JOptionPane.showMessageDialog((Component) null,
                                     Message,
                                     "ERROR",
                                     JOptionPane.ERROR_MESSAGE);
       return null;
     }

     return url;
  }

  /* ===================================================================== *
   * ------------------------ DOWNLOAD METHODS --------------------------- *
   * ===================================================================== */

  /** Method getData()
   * <p> This method will retrieve the selected data.
   * @return stCoreList = The Selected Core Image List Data.
   */

  public rockImagesListStruct getData()
  {
    int i      = 0;
    int j      = 0;
    int iCount = 0;
    int iCoreType = -1;

    stCoreList        = new rockImagesListStruct();
    stCoreList.iCount = 0;

    if (stImages != null)
    {
      stCoreList.stItem = new rockImagesStruct[stImages.iCount];

      for (i=0; i<stImages.iCount; i++)
      {
        if (iButtons[i] == cmn.cmnStruct.ON)
        {
          stCoreList.stItem[iCount] =
              rock.rockImagesUtility.copy(stImages.stItem[i]);
          iCount++;
        }
      }
    }

    stCoreList.iCount = iCount;

    return (stCoreList);
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setImageDisplayPanel()
   * <p> This method will set the Image Display Panel for showing the images
   * @param pDisplay = The Panel that will hold the core image.
   * @param iWidth   = The Image Maximum Width
   */

  public void setImageDisplayPanel(JPanel pDisplay, int iWidth)
  {
    this.pDisplay = pDisplay;
    this.iWidth   = iWidth;
  }

  /** Method setColor()
   * <p> This method will set the button color of the On/Off Button
   * @param  iButton = The Button Number
   * @param  iStatus = The Status of the Button
   */

  private void setColor(int iButton, int iStatus)
  {
    switch (iStatus)
    {
      case cmn.cmnStruct.OFF:
        iButtons[iButton] = cmn.cmnStruct.OFF;
        btnYesNo[iButton].setText("No");
        btnYesNo[iButton].setBackground(Color.darkGray);
        btnYesNo[iButton].setForeground(Color.yellow);
        break;

      case cmn.cmnStruct.ON:
        iButtons[iButton] = cmn.cmnStruct.ON;
        btnYesNo[iButton].setText("Yes");
        btnYesNo[iButton].setBackground(Color.yellow);
        btnYesNo[iButton].setForeground(Color.darkGray);
        break;
    }
  }

  /** Method setGroup()
   *  <p> This method will set the LAS File Selection Buttons by Type.
   *  @param iDisplay - The Gropu Type of Button selected.
   */

  public void setGroup(int iDisplay[])
  {
    int i         = 0;
    int j         = 0;

    for (j=0; j<3; j++)
    {
      for (i=0; i<stImages.iCount; i++)
      {
        if (stImages.stItem[i].sType.equals(sDisplay[j]))
          setColor(i, iDisplay[j]);
      }
    }
  }

  /** Method setFilterType()
   *  <p> This method will set the Core Image Buttons by Type.
   *  @param iDisplay - The Type of Core Image selected.
   */

  public void setFilterType(int iDisplay[])
  {
    this.iDisplay = iDisplay;
    setGroup(iDisplay);
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will close this dialog and set objects to null to force
   *     Java to free memory
   */

  public void close()
  {
    stCMN      = null;
    stImages   = null;
    stCoreList = null;
    pCoreImage = null;

    pCenter    = null;
    pImages    = null;

    iDisplay   = null;
    sDisplay   = null;
    pDisplay   = null;

    iButtons   = null;  // Yes/No Indicator for row
    sKID       = null;  // Array of KID's for the Core Data
    pRow       = null;  // Row to place Core Image Data in
    pRWest     = null;  // Panel Holding the View and Yes/No Buttons
    pRCenter   = null;  // Panel Holding the Depth Range
    pREast     = null;  // Panel Holding the Core Data Type
    btnYesNo   = null;  // Yes/No Indicator Button
    btnView    = null;  // View Core Image Button
    lblStart   = null;  // Begin Image Depth Label
    lblEnd     = null;  // End Image Depth Label
    lblType    = null;  // Image Type Label
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Well Curve
   *      Selection Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    int    i = 0;
    String sEventArgument = event.getActionCommand();

    for (i=0; i<iTotal; i++)
    {
      if (event.getSource() == btnYesNo[i])
      {
        if (iButtons[i] == cmn.cmnStruct.OFF)
        {
          setColor(i, cmn.cmnStruct.ON);
        }
        else
        {
          setColor(i, cmn.cmnStruct.OFF);
        }
      }
    }

    for (i=0; i<iTotal; i++)
    {
      if (event.getSource() == btnView[i])
      {
        DisplayCoreImage(sKID[i]);
      }
    }
  }
}

/** Class rockImagePanelPlotCanvas
 *  <p> This class will create a Canvas to draw the Core Image.
 */

class rockImagePanelPlotCanvas extends Canvas
{
  Image saved_i;
  int   iX;
  int   iY;

  public rockImagePanelPlotCanvas(Image i, int ix, int iy)
  {
    saved_i = i;
    iX = ix;
    iY = iy;
    this.setSize(iX, iY);
  }

  public void setImage(Image i, int ix, int iy)
  {
    saved_i = i;
    iX = ix;
    iY = iy;
    this.setSize(iX, iY);

    repaint();
  }

  public void paint(Graphics g)
  {
    g.drawImage(saved_i, 0, 0, iX, iY, this);
  }
}
