/*
 * @rockEnterDataTable.java Version 1.1 05/30/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.gui;

import java.awt.*;
import java.lang.*;
import javax.swing.*;
import javax.swing.table.*;

import gui.guiTable;

import lith.lithology.lithologySymbolsListStruct;
import lith.lithology.lithologyListStruct;
import lith.lithology.lithologyStruct;

import lith.texture.textureListStruct;
import lith.texture.textureStruct;

import rock.color.rockColorListStruct;
import rock.color.rockColorStruct;

import rock.phi.phiStruct;
import rock.phi.phiListStruct;

import rock.fossil.fossilStruct;
import rock.fossil.fossilListStruct;

import rock.sedimentary.sedimentaryListStruct;
import rock.sedimentary.sedimentaryStruct;

import iqstrat.iqstratRemarkListStruct;
import iqstrat.iqstratRemarkStruct;

/** Class rockEnterDataTable
 *  <p> This Class will display the Rock Data.
 *
 *  @version 1.1 05/30/2008
 *  @author  John R. Victorine
 */

public class rockEnterDataTable
{
  private int      iRows     = 0;
  private int      iColumns  = 3;
  private int      iChange   = 0;
  private String   sColumn[] = {"Start", "End", "Text"};
  private Object   oData[][] = null;
  private guiTable pTable    = null;

  private int    iTotal   = 0;
  private String sKEY[]   = null;
  private double dStart[] = null;
  private double dEnd[]   = null;
  private String sText[]  = null;

  private lithologySymbolsListStruct stSymbols   = null;
  private lithologyListStruct        stLithology = null;
  private textureListStruct          stTexture   = null;
  private rockColorListStruct        stColor     = null;
  private phiListStruct              stPHI       = null;
  private fossilListStruct           stFossils   = null;
  private sedimentaryListStruct      stSedStruct = null;
  private iqstratRemarkListStruct    stRemarks   = null;

  /** Constructor rockEnterDataTable()
   *  <p> This is the Constructor for this class.
   * @param stSymbols = Lithology Symbols Data List Structure
   */

  public rockEnterDataTable( lithologySymbolsListStruct stSymbols )
  {
    this.stSymbols = stSymbols;

    pTable = new guiTable(0, iColumns, sColumn);

    pTable.resizeColumn(0, 10);
    pTable.resizeColumn(1, 10);
  }

  /** Method close()
   * <p> This method will cancel the Rock Color Dialog
   */

  public void close()
  {
    sColumn     = null;
    oData       = null;

    stSymbols   = null;
    stLithology = null;
    stTexture   = null;
    stColor     = null;
    stPHI       = null;
    stFossils   = null;
    stSedStruct = null;
    stRemarks   = null;

    sKEY        = null;
    dStart      = null;
    dEnd        = null;
    sText       = null;

    pTable      = null;
  }

  /* ===================================================================== *
   * ---------------------- POPULATE LIST METHOD ------------------------- *
   * ===================================================================== */

  /** Method populateList()
   *  <p> This method will populate the Region List data.
   */

  private void populateList()
  {
    int    i     = 0;
    String sTemp = "";
    int    len   = 0;

    iChange = 0;
    iRows = 1;
    oData = new String[1][iColumns];

    for (i=0; i<iColumns; i++)
    {
      oData[0][i] = new String("");
    }

    if (iTotal > 0)
    {
      iChange = 1;

      oData = new String[iTotal][iColumns];

      for (i=0; i<iTotal; i++)
      {
        sTemp = new String(""+dStart[i]);
        len   = sTemp.length();
        if (len > 9) sTemp = new String(sTemp.substring(0,9));

        oData[i][0] = new String(sTemp);

        sTemp = new String(""+dEnd[i]);
        len   = sTemp.length();
        if (len > 9) sTemp = new String(sTemp.substring(0,9));

        oData[i][1] = new String(sTemp);

        oData[i][2] = new String(sText[i]);
      }

      iRows = iTotal;
    }
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setLithologyData(lithologyListStruct st)
   * <p> This method will set the Lithology List Data Structure and
   *     display the data to the table
   * @param st = the Lithology List Data Structure
   */

  public void setLithologyData(lithologyListStruct st)
  {
    int iLith   = -1;
    int iFound  = -1;
    int iPer    = 0;
    int iSymbol = -1;

    this.stLithology = st;

    iTotal = 0;
    sKEY   = null;
    dStart = null;
    dEnd   = null;
    sText  = null;

    if (st != null)
    {
      iTotal = st.iCount;
      sKEY   = new String[iTotal];
      dStart = new double[iTotal];
      dEnd   = new double[iTotal];
      sText  = new String[iTotal];

      for (int i=0; i<iTotal; i++)
      {
        sKEY[i]   = new String(""+i);
        dStart[i] = st.stItem[i].depthStart;
        dEnd[i]   = st.stItem[i].depthEnd;
        sText[i]  = new String("");

        iLith = st.stItem[i].iLithology[0];
        iPer  = 0;

        for (int j=0; j<10; j++)
        {
          if ((iLith != st.stItem[i].iLithology[j]) || (j==9))
          {
            if (j==9) iPer = iPer + 10;

            iSymbol = -1;
            if (stSymbols != null)
            {
              for (int n=0; n<stSymbols.iCount; n++)
              {
                if (iLith == stSymbols.stItem[n].id)
                  iSymbol = n;
              }
            }

            if (iSymbol == -1) iSymbol = 0;

            if (stSymbols != null)
            {
              sText[i] = new String( sText[i] +
                                     stSymbols.stItem[iSymbol].sName +
                                     " (" + iPer + " %) " );
            }

            iLith = st.stItem[i].iLithology[j];
            iPer = 0;
          }

          iPer = iPer + 10;
        }
      }
    }

    populateList();

    pTable.reload(iRows, oData);

    if (iChange == 0)
      iRows = 0;

    if (iRows > 0)
      pTable.setSelectedRow(0);
  }

  /** Method setTextureData(textureListStruct st)
   * <p> This method will set the Lithology List Data Structure and
   *     display the data to the table
   * @param st = the Lithology List Data Structure
   */

  public void setTextureData(textureListStruct st)
  {
    this.stTexture = st;

    iTotal = 0;
    sKEY   = null;
    dStart = null;
    dEnd   = null;
    sText  = null;

    if (st != null)
    {
      iTotal = st.iCount;
      sKEY   = new String[iTotal];
      dStart = new double[iTotal];
      dEnd   = new double[iTotal];
      sText  = new String[iTotal];

      for (int i=0; i<iTotal; i++)
      {
        if (st.stItem[i] != null)
        {
          sKEY[i]   = new String(""+i);
          dStart[i] = st.stItem[i].depthStart;
          dEnd[i]   = st.stItem[i].depthEnd;
          sText[i]  = new String(st.stItem[i].sTexture + "  (" +
                                 st.stItem[i].iEnergy + ")");
        }
      }
    }

    populateList();

    pTable.reload(iRows, oData);

    if (iChange == 0)
      iRows = 0;

    if (iRows > 0)
      pTable.setSelectedRow(0);
  }

  /** Method setColorData(rockColorListStruct st)
   * <p> This method will set the Rock Color List Data Structure and
   *     display the data to the table
   * @param st = the Rock Color List Data Structure
   */

  public void setColorData(rockColorListStruct st)
  {
    this.stColor = st;

    iTotal = 0;
    sKEY   = null;
    dStart = null;
    dEnd   = null;
    sText  = null;

    if (st != null)
    {
      iTotal = st.iCount;
      sKEY   = new String[iTotal];
      dStart = new double[iTotal];
      dEnd   = new double[iTotal];
      sText  = new String[iTotal];

      for (int i=0; i<iTotal; i++)
      {
        sKEY[i]   = new String(st.stItem[i].sKEY);
        dStart[i] = st.stItem[i].depthStart;
        dEnd[i]   = st.stItem[i].depthEnd;
        sText[i]  = new String(st.stItem[i].sColor[0] + " (" +
                               st.stItem[i].sMunsell[0] + ")");
      }
    }

    populateList();

    pTable.reload(iRows, oData);

    if (iChange == 0)
      iRows = 0;

    if (iRows > 0)
      pTable.setSelectedRow(0);
  }

  /** Method setRemarksData(iqstratRemarkListStruct st)
   * <p> This method will set the Remarks List Data Structure and
   *     display the data to the table
   * @param st = the Remarks List Data Structure
   */

  public void setRemarksData(iqstratRemarkListStruct st)
  {
    this.stRemarks = st;

    iTotal = 0;
    sKEY   = null;
    dStart = null;
    dEnd   = null;
    sText  = null;

    if (st != null)
    {
      iTotal = st.iCount;
      sKEY   = new String[iTotal];
      dStart = new double[iTotal];
      dEnd   = new double[iTotal];
      sText  = new String[iTotal];

      for (int i=0; i<iTotal; i++)
      {
        sKEY[i]   = new String(st.stItem[i].sKEY);
        dStart[i] = st.stItem[i].depthStart;
        dEnd[i]   = st.stItem[i].depthEnd;
        sText[i]  = new String(st.stItem[i].sText); //sNote);
      }
    }

    populateList();

    pTable.reload(iRows, oData);

    if (iChange == 0)
      iRows = 0;

    if (iRows > 0)
      pTable.setSelectedRow(0);
  }

  /** Method setPHIData(phiListStruct st)
   * <p> This method will set the Porosity List Data Structure and
   *     display the data to the table
   * @param st = the Porosity List Data Structure
   */

  public void setPHIData(phiListStruct st)
  {
    String sTemp = "";

    this.stPHI = st;

    iTotal = 0;
    sKEY   = null;
    dStart = null;
    dEnd   = null;
    sText  = null;

    if (st != null)
    {
      iTotal = st.iCount;
      sKEY   = new String[iTotal];
      dStart = new double[iTotal];
      dEnd   = new double[iTotal];
      sText  = new String[iTotal];

      for (int i=0; i<iTotal; i++)
      {
        sKEY[i]   = new String(st.stItem[i].sKEY);
        dStart[i] = st.stItem[i].depthStart;
        dEnd[i]   = st.stItem[i].depthEnd;
        sTemp = "Porosity: " + st.stItem[i].dPHI + " %";
        for (int j=0; j<st.stItem[i].iTotal; j++)
          sTemp = sTemp + " " + st.stItem[i].sText[j];
        sText[i]  = new String(sTemp);
      }
    }

    populateList();

    pTable.reload(iRows, oData);

    if (iChange == 0)
      iRows = 0;

    if (iRows > 0)
      pTable.setSelectedRow(0);
  }

  /** Method setFossilData(fossilListStruct st)
   * <p> This method will set the Fossil List Data Structure and
   *     display the data to the table
   * @param st = the Fossil List Data Structure
   */

  public void setFossilData(fossilListStruct st)
  {
    String sTemp = "";

    this.stFossils = st;

    iTotal = 0;
    sKEY   = null;
    dStart = null;
    dEnd   = null;
    sText  = null;

    if (st != null)
    {
      iTotal = st.iCount;
      sKEY   = new String[iTotal];
      dStart = new double[iTotal];
      dEnd   = new double[iTotal];
      sText  = new String[iTotal];

      for (int i=0; i<iTotal; i++)
      {
        sKEY[i]   = new String(st.stItem[i].sKEY);
        dStart[i] = st.stItem[i].depthStart;
        dEnd[i]   = st.stItem[i].depthEnd;
        sTemp = "";
        for (int j=0; j<st.stItem[i].iTotal; j++)
          sTemp = sTemp + " " + st.stItem[i].sText[j];
        sText[i]  = new String(sTemp);
      }
    }

    populateList();

    pTable.reload(iRows, oData);

    if (iChange == 0)
      iRows = 0;

    if (iRows > 0)
      pTable.setSelectedRow(0);
  }

  /** Method setSedStructData(fossilListStruct st)
   * <p> This method will set the Sedimentary Structure List Data Structure and
   *     display the data to the table
   * @param st = the Sedimentary Structure List Data Structure
   */

  public void setSedStructData(sedimentaryListStruct st)
  {
    String sTemp = "";

    this.stSedStruct = st;

    iTotal = 0;
    sKEY   = null;
    dStart = null;
    dEnd   = null;
    sText  = null;

    if (st != null)
    {
      iTotal = st.iCount;
      sKEY   = new String[iTotal];
      dStart = new double[iTotal];
      dEnd   = new double[iTotal];
      sText  = new String[iTotal];

      for (int i=0; i<iTotal; i++)
      {
        sKEY[i]   = new String(st.stItem[i].sKEY);
        dStart[i] = st.stItem[i].depthStart;
        dEnd[i]   = st.stItem[i].depthEnd;
        sTemp = "";
        for (int j=0; j<st.stItem[i].iTotal; j++)
          sTemp = sTemp + " " + st.stItem[i].sText[j];
        sText[i]  = new String(sTemp);
      }
    }

    populateList();

    pTable.reload(iRows, oData);

    if (iChange == 0)
      iRows = 0;

    if (iRows > 0)
      pTable.setSelectedRow(0);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /* --------------------------------------------------------------------- *
   * ------------------ Rock Lithology Get Methods ----------------------- *
   * --------------------------------------------------------------------- */

  /** Method getLithologyData()
   *  <p> This method will return the Lithology Data Structure of the
   *      selected row.
   *  @return stData = Lithology Data Structure.
   */

  public lithologyStruct getLithologyData()
  {
    lithologyStruct stData = null;

    int iSelectedRow = pTable.getSelectedRow();
    stData = lith.lithology.lithologyUtility.copy(stLithology.stItem[iSelectedRow]);

    return (stData);
  }

  /** Method getLithologyList()
   * <p> This method will return the Lithology List Data Structure
   * @return stLithology = the Lithology List Data Structure
   */

  public lithologyListStruct getLithologyList() { return (stLithology); }

  /* --------------------------------------------------------------------- *
   * ------------------- Rock Texture Get Methods ------------------------ *
   * --------------------------------------------------------------------- */

  /** Method getTextureData()
   *  <p> This method will return the Texture Data Structure of the
   *      selected row.
   *  @return stData = Texture Data Structure.
   */

  public textureStruct getTextureData()
  {
    textureStruct stData = null;

    int iSelectedRow = pTable.getSelectedRow();
    stData = lith.texture.textureUtility.copy(stTexture.stItem[iSelectedRow]);

    return (stData);
  }

  /** Method getTextureList()
   * <p> This method will return the Texture List Data Structure
   * @return stTexture = the Texture List Data Structure
   */

  public textureListStruct getTextureList() { return (stTexture); }

  /* --------------------------------------------------------------------- *
   * -------------------- Rock Color Get Methods ------------------------- *
   * --------------------------------------------------------------------- */

  /** Method getColorData()
   *  <p> This method will return the Rock Color Data Structure of the
   *      selected row.
   *  @return stData = Rock Color Data Structure.
   */

  public rockColorStruct getColorData()
  {
    rockColorStruct stData = null;

    int iSelectedRow = pTable.getSelectedRow();
    stData = rock.color.rockColorUtility.copy(stColor.stItem[iSelectedRow]);

    return (stData);
  }

  /** Method getColorDataList()
   *  <p> This method will returns the Rock Color List Data Structure.
   *  @return st = Rock Color List Data Structure.
   */

  public rockColorListStruct getColorDataList() { return (stColor); }

  /* --------------------------------------------------------------------- *
   * -------------- Sedimentary Structures Get Methods ------------------- *
   * --------------------------------------------------------------------- */

  /** Method getSedStructData()
   *  <p> This method will return the Sedimentary Structure Data Structure of
   *      the selected row.
   *  @return stData = Sedimentary Structure Data Structure.
   */

  public sedimentaryStruct getSedStructData()
  {
    sedimentaryStruct stData = null;

    int iSelectedRow = pTable.getSelectedRow();
    stData = rock.sedimentary.sedimentaryUtility.copy(
        stSedStruct.stItem[iSelectedRow]);

    return (stData);
  }

  /** Method getSedStructList()
   *  <p> This method will returns the Sedimentary Structure List Data Structure.
   *  @return stSedStruct = Sedimentary Structure List Data Structure.
   */

  public sedimentaryListStruct getSedStructList() { return (stSedStruct); }

  /* --------------------------------------------------------------------- *
   * --------------------- Porosity Get Methods -------------------------- *
   * --------------------------------------------------------------------- */

  /** Method getPHIData()
   *  <p> This method will return the Porosity Data Structure of the
   *      selected row.
   *  @return stData = Porosity Data Structure.
   */

  public phiStruct getPHIData()
  {
    phiStruct stData = null;

    int iSelectedRow = pTable.getSelectedRow();
    stData = rock.phi.phiUtility.copy(stPHI.stItem[iSelectedRow]);

    return (stData);
  }

  /** Method getPHIList()
   *  <p> This method will returns the Porosity List Data Structure.
   *  @return stPHI = Porosity List Data Structure.
   */

  public phiListStruct getPHIList() { return (stPHI); }

  /* --------------------------------------------------------------------- *
   * --------------------- Fossils Get Methods --------------------------- *
   * --------------------------------------------------------------------- */

  /** Method getFossilData()
   *  <p> This method will return the Fossils Data Structure of the
   *      selected row.
   *  @return stData = Fossils Data Structure.
   */

  public fossilStruct getFossilData()
  {
    fossilStruct stData = null;

    int iSelectedRow = pTable.getSelectedRow();
    stData = rock.fossil.fossilUtility.copy(stFossils.stItem[iSelectedRow]);

    return (stData);
  }

  /** Method getFossilsList()
   *  <p> This method will returns the Fossils List Data Structure.
   *  @return stFossils = Fossils List Data Structure.
   */

  public fossilListStruct getFossilsList() { return (stFossils); }

  /* --------------------------------------------------------------------- *
   * --------------------- Remarks Get Methods --------------------------- *
   * --------------------------------------------------------------------- */

  /** Method getRemarkData()
   *  <p> This method will return the Remarks Data Structure of the
   *      selected row.
   *  @return stData = Remarks Data Structure.
   */

  public iqstratRemarkStruct getRemarkData()
  {
    iqstratRemarkStruct stData = null;

    int iSelectedRow = pTable.getSelectedRow();
    stData = iqstrat.iqstratRemarkUtility.copy(stRemarks.stItem[iSelectedRow]);

    return (stData);
  }

  /** Method getRemarksList()
   *  <p> This method will returns the Remarks List Data Structure.
   *  @return stRemarks = Remarks List Data Structure.
   */

  public iqstratRemarkListStruct getRemarksList() { return (stRemarks); }

  /* --------------------------------------------------------------------- *
   * ---------------------- General Get Methods -------------------------- *
   * --------------------------------------------------------------------- */

  /** Method getTotalRows()
   *  <p> This method will return the total rows in the table.
   *  @return Total Number of Rows.
   */

  public int getTotalRows() { return (iRows); }

  /** Method getScrollPane()
   *  <p> This method will return the scroll pane.
   *  @return jScrollPane - Scroll Pane containing the Table.
   */

  public JScrollPane getScrollPane() { return (pTable.getScrollPane()); }
}