/*
 * @fossilPlotTrack.java Version 1.1 09/24/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.fossil.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;
import java.net.*;
import javax.swing.*;

import mask.maskSymbolsListStruct;
import mask.plot.maskPlotSymbol;
import rock.fossil.fossilListStruct;

/** CLASS fossilPlotTrack
 *  <p> This Class will create the Fossil Track.
 *
 *  @version 1.1 09/24/2007
 *  @author  John R. Victorine
 */

public class fossilPlotTrack extends Canvas
{
  // Input Variables

  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables

  private fossilListStruct      stFossil  = null; // Fossil List Data Structure
  private maskSymbolsListStruct stSymbols = null; // Fossil Symbols Data
  private maskPlotSymbol        plot      = null; // Plot Mask Symbol

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR fossilPlotTrack( int iDataType,
   *                               double depthStart,
   *                               double depthEnd,
   *                               int iScale,
   *                               maskSymbolsListStruct stSymbols )
   *  <p> This is the constructor for this class.
   *  @param iDataType  = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart = The starting depth of plot
   *  @param depthEnd   = The ending depth of plot
   *  @param iScale     = The scale
   *  @param stSymbols  = Fossil Plot Symbols List Data Structure
   */

  public fossilPlotTrack( int iDataType,
                          double depthStart,
                          double depthEnd,
                          int iScale,
                          maskSymbolsListStruct stSymbols )
  {
    this.iDataType = iDataType;
    this.stSymbols = stSymbols;

    plot = new maskPlotSymbol( stSymbols );

    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
    stFossil  = null;
    stSymbols = null;

    if (plot != null)
      plot.close();
    plot = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getFossilData()
   * <p> This method will get an existing Fossil List Data Structure
   * @return stFossil = Fossil List Data Structure
   */

  public fossilListStruct getFossilData() { return (stFossil); }

  /** Method getNextDepth( int iRow, double diff )
   * <p> This method will retrieve the next depth data in the list in such
   *     a way that the text does not overwrite
   * @param  iRow  = the present row
   * @param  diff  = the difference between text
   * @return iNext = The next Row
   */

  private int getNextDepth( int iRow, double diff )
  {
    int iNext = -1;
    int iValue = (int) diff;

    diff = (double) iValue;

    if (iRow > -1)
    {
      if (stFossil != null)
      {
        for (int i=iRow; i<stFossil.iCount; i++)
        {
          if ((iNext == -1) &&
              (stFossil.stItem[i].depthStart >=
                 stFossil.stItem[iRow].depthEnd + diff))
          {
            iNext = i;
            break;
          }
        }
      }
    }

    return (iNext);
  }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setFossilData( fossilListStruct stFossil )
   * <p> This method will set an existing Fossil List Data Structure
   * @param stFossil = Fossil List Data Structure
   */

  public void setFossilData( fossilListStruct stFossil )
  {
    this.stFossil = stFossil;
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int         iStart    = 0;
    int         i         = 0;
    int         j         = 0;
    int         jXinc     = 0;
    int         iY1       = 0;
    int         iY2       = 0;
    int         iEndTrack = iStartTrack+iWidthTrack;
    Font        fsb       = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm      = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    iStart = iWidthTrack/10;

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES)  && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.black);

    plot.drawSymbol(
          g, stSymbols.stItem[0].iRows, 0,
          stSymbols.stItem[0].symbol,
          iStartTrack+5*(iStart-1)+1, LABELSTART+4 );

    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Fossils", LABELSTART+20, -1*(iStartTrack+5*(iStart-1)+3));
    g2.rotate( -1.0 * (Math.PI / 2.0));
  }

  /** Method drawFossils( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the fossil images that were selected
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawFossils( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iStart   = iStartTrack;
    int    iEnd     = iStartTrack+iWidthTrack;
    int    iFound   = -1;
    int    iDepth1  = 0;
    int    iDepth2  = 0;
    int    iDepth   = 0;
    int    iGroup   = 0;
    int    iX       = 0;
    int    iColor   = -1;
    int    iRGB[]   = { 255, 255, 255 };
    int    iSymbol  = -1;
    double dRatio   = 0.0;
    double depth1   = 0.0;
    double depth2   = 0.0;
    double depth    = 0.0;
    int    iTemp    = -1;
    double diff     = iqstrat.iqstratTracksStruct.SCALE[iScale] / 10.0;

    if (stFossil != null)
    {
      for (int i=0; i<stFossil.iCount; i++)
      {
        depth1 = stFossil.stItem[i].depthStart;
        depth2 = stFossil.stItem[i].depthEnd;

        dRatio = (double) iLogHeight*(depthStart-depth1)/(depthStart-depthEnd);
        iDepth1 = PLOT_TITLES + (int) dRatio;

        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth1 = iHeight - (int) dRatio;

        dRatio = (double) iLogHeight*(depthStart-depth2)/(depthStart-depthEnd);
        iDepth2 = PLOT_TITLES + (int) dRatio;

        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth2 = iHeight - (int) dRatio;

        if (iDepth1 == iDepth2)
        {
          iDepth1 = iDepth1 - 5;
          iDepth2 = iDepth2 - 5;
        }

        iDepth = Math.abs(iDepth2 - iDepth1);
        iGroup = iDepth / 10;
        if (iGroup == 0)
          iGroup = 1;

        iColor = mask.maskSymbolsUtility.getColor(
            stFossil.stItem[i].sEnvironment, stSymbols.stBackground );
        iRGB = mask.maskSymbolsUtility.getRGB(iColor, stSymbols.stBackground );

        if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
            (iDepth1 < iHeight) && (iDepth2 <= iHeight))
        {
          for (int n=0; n<iGroup; n++)
          {
            iX = iStartTrack;
            iDepth = iDepth1 + n * 10;

            for (int k=0; k<stFossil.stItem[i].iTotal; k++)
            {
              iFound = mask.maskSymbolsUtility.getSymbolID(
                  stFossil.stItem[i].sID[k],
                  mask.maskSymbolsListStruct._SYMBOLS,
                  stSymbols );

              if (iFound > -1)
              {
                if (stFossil.stItem[i].sMod[k].equals("Rare"))
                {
                  if (iColor > -1)
                  {
                    plot.drawBackground( g, iRGB[0], iRGB[1], iRGB[2],
                                         iX, iDepth, 3, 10);
                  }

                  iSymbol = mask.maskSymbolsUtility.getSymbolIDByName(
                   "few left", mask.maskSymbolsListStruct._MODIFIERS, stSymbols);

                  if (iSymbol > -1)
                    plot.drawSymbol( g, stSymbols.stModifiers[iSymbol].iRows, 0,
                                     stSymbols.stModifiers[iSymbol].symbol,
                                     iX, iDepth );
                  iX = iX + 3;
                }

                if (iColor > -1)
                {
                  plot.drawBackground( g, iRGB[0], iRGB[1], iRGB[2],
                                       iX, iDepth, 10, 10);
                }

                plot.drawSymbol( g, stSymbols.stItem[iFound].iRows, 0,
                                 stSymbols.stItem[iFound].symbol,iX, iDepth );

                if (stFossil.stItem[i].sMod[k].equals("Broken"))
                {
                  iSymbol = mask.maskSymbolsUtility.getSymbolIDByName(
                    "broken", mask.maskSymbolsListStruct._MODIFIERS, stSymbols);

                  if (iSymbol > -1)
                    plot.drawSymbol( g, stSymbols.stModifiers[iSymbol].iRows, 0,
                                     stSymbols.stModifiers[iSymbol].symbol,
                                     iX, iDepth );
                }

                if (stFossil.stItem[i].sMod[k].equals("Abundant"))
                {
                  iSymbol = mask.maskSymbolsUtility.getSymbolIDByName(
                      "rich", mask.maskSymbolsListStruct._MODIFIERS, stSymbols);

                  if (iSymbol > -1)
                    plot.drawSymbol( g, stSymbols.stModifiers[iSymbol].iRows, 0,
                                     stSymbols.stModifiers[iSymbol].symbol,
                                     iX, iDepth );
                }

                iX = iX + 10;

                if (stFossil.stItem[i].sMod[k].equals("Rare"))
                {
                  if (iColor > -1)
                  {
                    plot.drawBackground( g, iRGB[0], iRGB[1], iRGB[2],
                                         iX, iDepth, 3, 10);
                  }

                  iSymbol = mask.maskSymbolsUtility.getSymbolIDByName(
                   "few right", mask.maskSymbolsListStruct._MODIFIERS, stSymbols);

                  if (iSymbol > -1)
                    plot.drawSymbol( g, stSymbols.stModifiers[iSymbol].iRows, 0,
                                     stSymbols.stModifiers[iSymbol].symbol,
                                     iX, iDepth );
                  iX = iX + 3;
                }

                iX = iX + 2;
              }
            }
          }
        }

        iTemp = getNextDepth(i, diff);
        if (iTemp > 0)
        {
          if (iTemp-1 > i) i = iTemp-1;
        }
      }
    }

//    g.setColor(Color.black);
//    g.drawLine(iStart, LABELSTART, iStart, PLOT_TITLES+iHeight);
//    g.drawLine(iEnd,   LABELSTART, iEnd,   PLOT_TITLES+iHeight);
  }

  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Porosity Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw( Graphics g,
                    int iSelected,
                    int iStartTrack,
                    int iWidthTrack )
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;

    if (iSelected == iqstrat.iqstratTracksStruct._ROCK_FOSSILS)
    {
      g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2] ) );
      g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

      drawGrid( g, iStartTrack, iWidthTrack );
      drawFossils( g, iStartTrack, iWidthTrack );
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}

/*
 *  @version 1.1 09/24/2007
 *  @author  John Victorine
 */
