/*
 * @fossilPlotSymbol.java Version 1.1 09/04/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.fossil.plot;

import java.awt.*;
import java.awt.geom.*;
import java.util.Observable;
import java.awt.event.*;

import mask.maskSymbolsListStruct;
import mask.plot.maskPlotSymbol;
import rock.fossil.fossilStruct;

/** Class fossilPlotSymbol
 *  <p> This method will create the different symbol types for plotting data
 *      values on a plot.
 *
 *  @version 1.1 09/04/2008
 *  @author  John R. Victorine
 */

public class fossilPlotSymbol extends Canvas implements MouseListener
{
  private Observable            notifier  = null; // Observable Object
  private maskSymbolsListStruct stSymbols = null; // Fossil Symbols Data
  private maskPlotSymbol        plot      = null; // Plot Mask Symbol

  private static final int NO  = 0;
  private static final int YES = 1;

  private static final int _GENERAL = 0;
  private static final int _RARE    = 1;
  private static final int _MANY    = 2;
  private static final int _BROKEN  = 3;

  private static final String MOD[] = { "G", "R", "A", "B" };

  private fossilStruct st          = null;
  private int          iSelected[] = null;
  private int          iMod[][]    = null;
  private int iWidth  = 300;
  private int iHeight = 950;

  private static final int _START       = 10;
  private static final int _SELECTED    = 10;
  private static final int _SELECT_GEN  = 22;
  private static final int _SELECT_RARE = 34;
  private static final int _SELECT_MANY = 46;
  private static final int _SELECT_BROK = 58;
  private static final int _IMAGE_START = 90;
  private static final int _DESC_START  = 105;
  private static final int _DESCRIPTION = 115;
  private static final int _SEP         = 15;

  /** CONSTRUCTOR fossilPlotSymbol()
   *  <p> This is the constructor for this class.
   *  @param notifier = Observable
   *  @param stSymbols  = Fossil Plot Symbols List Data Structure
   */

  public fossilPlotSymbol(Observable notifier, maskSymbolsListStruct stSymbols )
  {
    this.notifier  = notifier;
    this.stSymbols = stSymbols;

    plot = new maskPlotSymbol( stSymbols );

    this.setBackground(Color.white);
    addMouseListener(this);

    iSelected = new int[stSymbols.iCount];
    iMod      = new int[stSymbols.iCount][4];

    for (int i=0; i<stSymbols.iCount; i++)
    {
      iSelected[i] = NO;
      for (int j=0; j<4; j++)
        iMod[i][j] = NO;
    }
  }

  /** Method close()
   * <p> This method will force Java to deallocate memory
   */

  public void close()
  {
    notifier  = null;
    stSymbols = null;

    if (plot != null)
      plot.close();
    plot = null;

    st        = null;
    iSelected = null;
    iMod      = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Width of the plot panel.
   * @return WIDTH = the width of the plot panel
   */

  public int getPlotWidth() { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Width of the plot panel.
   * @return the height of the plot panel
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getData()
   * <p> This method will get the array of fossils selected
   * @return st = fossil data Structure
   */

  public fossilStruct getData() { return (st); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setData()
   * <p> This method will set the array of fossils selected
   * @param st = fossil data Structure
   */

  public void setData(fossilStruct st)
  {
    int iValue = NO;

    this.st = st;

    for (int i=0; i<stSymbols.iCount; i++)
    {
      iSelected[i] = NO;
      if (isSelected(i))
        iSelected[i] = YES;

      for (int j=0; j<4; j++)
        iMod[i][j] = NO;

      if (iSelected[i] == YES)
      {
        iValue = getModification(i);

        if (iValue > -1)
          iMod[i][iValue] = YES;
      }
    }

    repaint();
  }

  /* =============================================================== *
   * ------------------------ MISC METHODS ------------------------- *
   * =============================================================== */

  /** Method isSelected()
   * <p> This method will test the ID's to see if they have been selected
   *     and return true if they have and false if not.
   * @param  iValue = the id to check
   * @return bValue = true/false if the Fossil ID's selected or not
   */

  private boolean isSelected(int iValue)
  {
    boolean bValue = false;

    if (st != null)
    {
      if (st.iTotal > 0)
      {
        for (int i=0; i<st.iTotal; i++)
        {
          if (st.sID[i].equals(stSymbols.stItem[iValue].sID))
            bValue = true;
        }
      }
    }

    return (bValue);
  }

  /** Method getModification()
   * <p> This method will get the modification to the fossil
   * @param  iValue = the id to check
   * @return iMod   = the modification parameter
   */

  private int getModification(int iValue)
  {
    int iMod = -1;

    if (st != null)
    {
      if (st.iTotal > 0)
      {
        for (int i=0; i<st.iTotal; i++)
        {
          if (st.sID[i].equals(stSymbols.stItem[iValue].sID))
          {
            iMod = _GENERAL;

            if (st.sMod[i].equals("Broken"))
              iMod = _BROKEN;

            if (st.sMod[i].equals("Abundant"))
              iMod = _MANY;

            if (st.sMod[i].equals("Rare"))
              iMod = _RARE;
          }
        }
      }
    }

    return (iMod);
  }

  /** Method toggleFossil()
   * <p> This method will toggle a fossil from one state to the other.
   * @param iFossil = the fossil number to toggle.
   */

  private void toggleFossil(int iFossil)
  {
    if (iSelected[iFossil] == NO)
      iSelected[iFossil] = YES;
    else
      iSelected[iFossil] = NO;
  }

  /** Method modifyFossilList()
   * <p> This method will check add or remove fossils from the fossil list
   *     depending if it is selected
   */

  private void modifyFossilList()
  {
    int i      = 0;
    int iCount = 0;

    if (st == null)
    {
      st = new fossilStruct();
      st.sKEY = new String(cmn.cmnString.UniqueName());
    }

    for (i=0; i<stSymbols.iCount; i++)
    {
      if (iSelected[i] == YES)
        iCount++;
    }

    if (iCount > 0)
    {
      st.iTotal = iCount;
      st.sID    = new String[iCount];
      st.sMod   = new String[iCount];
      st.sText  = new String[iCount];

      iCount = 0;
      for (i=0; i<stSymbols.iCount; i++)
      {
        if (iSelected[i] == YES)
        {
          if (iCount < st.iTotal)
          {
            st.sID[iCount]  = new String(stSymbols.stItem[i].sID);
            st.sMod[iCount] = new String("");

            for (int j=0; j<4; j++)
            {
              if (iMod[i][j] == YES)
              {
                switch (j)
                {
                  case _GENERAL:
                    st.sMod[iCount] = new String("");
                    break;
                  case _RARE:
                    st.sMod[iCount] = new String("Rare");
                    break;
                  case _MANY:
                    st.sMod[iCount] = new String("Abundant");
                    break;
                  case _BROKEN:
                    st.sMod[iCount] = new String("Broken");
                    break;
                }
              }
            }

            st.sText[iCount] = new String(stSymbols.stItem[i].sName);
            iCount++;
          }
        }
      }
    }
    else
    {
      st.iTotal = 0;
      st.sID    = null;
      st.sMod   = null;
      st.sText  = null;
    }
  }

  /* =============================================================== *
   * -------------------- MouseListener Methods -------------------- *
   * =============================================================== */

  public void mouseMoved(MouseEvent event) { }
  public void mouseEntered(MouseEvent event) { }
  public void mouseExited(MouseEvent event) { }
  public void mousePressed(MouseEvent event) { }
  public void mouseDragged(MouseEvent event) {}
  public void mouseReleased(MouseEvent event) {}

  /** Method mouseClicked()
   *  <p> Mouse Clicked Method retrieve the location of the mouse pointer
   *      location.
   * @param event - Mouse Clicked Event
   */

  public void mouseClicked(MouseEvent event)
  {
    int    iXm     = event.getX();
    int    iYm     = event.getY();
    int    iValue[] = { NO, NO, NO, NO };

    for (int i=0; i<stSymbols.iCount; i++)
    {
      if ((iYm >= _START + i*_SEP) && (iYm < _START + (i+1)*_SEP))
      {
        toggleFossil(i);

        for (int j=0; j<4; j++)
          iMod[i][j] = NO;

        if (!isSelected(i))
        {
          if ((iXm >= _SELECTED) && (iXm < _SELECT_RARE))
            iMod[i][_GENERAL] = YES;
          if ((iXm >= _SELECT_RARE) && (iXm < _SELECT_MANY))
            iMod[i][_RARE] = YES;
          else if ((iXm >= _SELECT_MANY) && (iXm < _SELECT_BROK))
            iMod[i][_MANY] = YES;
          else if ((iXm >= _SELECT_BROK) && (iXm < _IMAGE_START))
            iMod[i][_BROKEN] = YES;
        }

        modifyFossilList();
        repaint();

        notifier.notifyObservers(new String("Fossils Changed"));
      }
    }
  }

  /* ======================================================================== *
   * ------------------------------- PAINT ---------------------------------- *
   * ======================================================================== */

  /** Method draw()
   * <p> This method will draw the fossil image panels
   * @param g - Graphics Parameter.
   */

  public void draw(Graphics g)
  {
    Font        fsb   = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm  = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (int i=0; i<stSymbols.iCount; i++)
    {
      if (notifier != null)
      {
        if (isSelected(i))
        {
          g.setColor(Color.red);
          g.drawLine(_SELECTED, _START + i * _SEP,
                     _SELECTED + 10, _START + i * _SEP + 10);
          g.drawLine(_SELECTED + 10, _START + i * _SEP,
                     _SELECTED, _START + i * _SEP + 10);
        }

        for (int j=0; j<4; j++)
        {
          if (iMod[i][j] == NO)
          {
            g.setColor(Color.lightGray);
            g.drawRect(_SELECTED+(j+1)*12, _START + i * _SEP, 10, 10);
          }
          else
          {
            g.setColor(Color.red);
            g.drawString(MOD[j], _SELECTED+(j+1)*12, _START + i*_SEP + 8);
          }
        }

        g.setColor(Color.black);
        g.drawRect(_SELECTED, _START + i*_SEP, 10, 10);
      }

      g.setColor(Color.black);
      g.drawLine(_DESC_START, _START+i*_SEP+6, _DESC_START+5, _START+i*_SEP+6);
      g.drawString( stSymbols.stItem[i].sName, _DESCRIPTION, _START+i*_SEP+8 );
//          rock.fossil.fossilLookupStruct.FOSSILS[i][1],

      plot.drawSymbol( g, stSymbols.stItem[i].iRows, 0,
                       stSymbols.stItem[i].symbol,
                       _IMAGE_START, _START + i*_SEP );

//      plot.plotMaskSymbol.drawSymbol(
//          g, 10, rock.fossil.plot.fossilPlotSymbolStruct.getSymbol(i),
//          _IMAGE_START, _START + i*_SEP );
    }
  }

  /** Method paint(Graphics g)
   * <p> This method will paint Porosity & Permeability Plot
   * @param g - Graphics Parameter.
   */

  public void paint(Graphics g)
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);

    draw(g);
  }
}

/*
 *  @version 1.1 09/04/2008
 *  @author  John Victorine
 */
