/*
 * @fossilPlotLegend.java Version 1.1 03/09/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.fossil.plot;

import java.awt.*;
import java.awt.geom.*;

import mask.maskSymbolsListStruct;
import mask.plot.maskPlotSymbol;
import rock.fossil.fossilStruct;

/** Class fossilPlotLegend
 *  <p> This method will create the different symbol types for plotting data
 *      values on a plot.
 *
 *  @version 1.1 03/09/2009
 *  @author  John R. Victorine
 */

public class fossilPlotLegend extends Canvas
{
  private int                   iSelected[] = null; // Selected Fossils
  private maskSymbolsListStruct stFossils   = null; // Fossil List Data Structure
  private maskPlotSymbol        plot        = null; // Plot Mask Symbol

  private int iWidth  = 250; //150;
  private int iHeight = 950;

  private static final int _START       = 10;
  private static final int _IMAGE_START = 10; //90;
  private static final int _DESC_START  = 25; //105;
  private static final int _DESCRIPTION = 35; //115;
  private static final int _SEP         = 20; //15;

  /** CONSTRUCTOR fossilPlotLegend()
   *  <p> This is the constructor for this class.
   *  @param iSelected = Array of indicators for plotting fossils
   *  @param stFossils = Fossil Plot Symbols List Data Structure
   */

  public fossilPlotLegend( int iSelected[], maskSymbolsListStruct stFossils )
  {
    int iCount = 0;

    this.iSelected = iSelected;
    if (iSelected != null)
    {
      for (int i=0; i<iSelected.length; i++)
      {
        if (iSelected[i] == 0)
          iCount++;
      }
    }

    this.stFossils = stFossils;
    plot = new maskPlotSymbol(stFossils);

    iHeight = 100 + _SEP * iCount;
  }

  /** Method close()
   * <p> This method will force Java to deallocate memory
   */

  public void close()
  {
    iSelected = null;
    stFossils = null;

    if (plot != null)
      plot.close();
    plot = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Width of the plot panel.
   * @return WIDTH = the width of the plot panel
   */

  public int getPlotWidth() { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Width of the plot panel.
   * @return the height of the plot panel
   */

  public int getPlotHeight() { return (iHeight); }

  /* ======================================================================== *
   * ------------------------------- PAINT ---------------------------------- *
   * ======================================================================== */

  /** Method drawLegend( Graphics g, int iStart )
   * <p> This method will draw the depth scale
   * @param g      = Graphics Parameter.
   * @param iStart = Starting point in the Y direction
   * @return the end of the labels
   */

  public int drawLegend( Graphics g, int iStart  )
  {
    int    iColor   = -1;
    int    iRGB[]   = { 255, 255, 255 };
    int    iStartTrack = _START;
    int    iWidthTrack = iWidth; //150;
    int    iIncrement  = iWidthTrack/5;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iSymbol     = -1;

    Font        fsb  = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    // Fresh Water Label

    iColor = mask.maskSymbolsUtility.getColor( "Fresh Water",
                                               stFossils.stBackground );
    iRGB = mask.maskSymbolsUtility.getRGB(iColor, stFossils.stBackground );
    plot.drawBackground( g, iRGB[0], iRGB[1], iRGB[2],
                         iStartTrack+5, iStart, 10, 10);

    g.setColor(Color.black);
    g.drawString("Fresh Water", iStartTrack+20, iStart+8);

    // Brackish Water Label

    iColor = mask.maskSymbolsUtility.getColor( "Brackish Water",
                                               stFossils.stBackground );
    iRGB = mask.maskSymbolsUtility.getRGB(iColor, stFossils.stBackground );
    plot.drawBackground( g, iRGB[0], iRGB[1], iRGB[2],
                         iStartTrack+5, iStart+15, 10, 10);

    g.setColor(Color.black);
    g.drawString("Brackish Water", iStartTrack+20, iStart+23);

    // Marine Label

    iColor = mask.maskSymbolsUtility.getColor( "Marine",
                                               stFossils.stBackground );
    iRGB = mask.maskSymbolsUtility.getRGB(iColor, stFossils.stBackground );
    plot.drawBackground( g, iRGB[0], iRGB[1], iRGB[2],
                         iStartTrack+5, iStart+30, 10, 10);

    g.setColor(Color.black);
    g.drawString("Marine", iStartTrack+20, iStart+38);

    // Few Fossils Label

    iSymbol = mask.maskSymbolsUtility.getSymbolIDByName(
        "few left", mask.maskSymbolsListStruct._MODIFIERS, stFossils );

    if (iSymbol > -1)
      plot.drawSymbol(
          g, stFossils.stModifiers[iSymbol].iRows, 0,
          stFossils.stModifiers[iSymbol].symbol,
          iStartTrack+iWidthTrack/2+17, iStart );

    plot.drawSymbol(
          g, stFossils.stItem[0].iRows, 0,
          stFossils.stItem[0].symbol,
          iStartTrack+iWidthTrack/2+20, iStart );

    iSymbol = mask.maskSymbolsUtility.getSymbolIDByName(
        "few right", mask.maskSymbolsListStruct._MODIFIERS, stFossils );

    if (iSymbol > -1)
      plot.drawSymbol(
          g, stFossils.stModifiers[iSymbol].iRows, 0,
          stFossils.stModifiers[iSymbol].symbol,
          iStartTrack+iWidthTrack/2+30, iStart );

    g.setColor(Color.black);
    g.drawString("Few", iStartTrack+iWidthTrack/2+35, iStart+8);

    // Rich Fossils Label

    plot.drawSymbol(
          g, stFossils.stItem[0].iRows, 0,
          stFossils.stItem[0].symbol,
          iStartTrack+iWidthTrack/2+20, iStart+15 );

    iSymbol = mask.maskSymbolsUtility.getSymbolIDByName(
        "rich", mask.maskSymbolsListStruct._MODIFIERS, stFossils);

    if (iSymbol > -1)
      plot.drawSymbol(
          g, stFossils.stModifiers[iSymbol].iRows, 0,
          stFossils.stModifiers[iSymbol].symbol,
          iStartTrack+iWidthTrack/2+20, iStart+15 );

    g.setColor(Color.black);
    g.drawString("Many", iStartTrack+iWidthTrack/2+35, iStart+23);

    // Broken Fossils Label

    plot.drawSymbol(
          g, stFossils.stItem[0].iRows, 0,
          stFossils.stItem[0].symbol,
          iStartTrack+iWidthTrack/2+20, iStart+30 );

    iSymbol = mask.maskSymbolsUtility.getSymbolIDByName(
        "broken", mask.maskSymbolsListStruct._MODIFIERS, stFossils);

    if (iSymbol > -1)
      plot.drawSymbol(
          g, stFossils.stModifiers[iSymbol].iRows, 0,
          stFossils.stModifiers[iSymbol].symbol,
          iStartTrack+iWidthTrack/2+20, iStart+30 );

    g.setColor(Color.black);
    g.drawString("Broken", iStartTrack+iWidthTrack/2+35, iStart+38);

    return (iStart+50);
  }

  /** Method draw( Graphics g )
   * <p> This method will draw the fossil image panels
   * @param g = Graphics Parameter.
   */

  public void draw( Graphics g )
  {
    int         iStart = _START;
    int         iLen   = 0;
    int         iCount = 0;
    String      sText  = "";
    Font        fsb  = new Font("Serif", Font.BOLD, 14);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    sText = new String( "Fossils" );
    iLen  = sText.length();

    g.setColor(Color.black);
    g.drawString( sText, iWidth/2 - 4*iLen, iStart);

    iStart = drawLegend( g, iStart + _SEP  );

    fsb  = new Font("Serif", Font.BOLD, 12);
    fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor( Color.black );

    if (stFossils != null)
    {
      for (int i=0; i<stFossils.iCount; i++)
      {
        if (iSelected != null)
        {
          if (iSelected[i] == 0)
          {
            g.drawLine( _DESC_START, iStart + iCount*_SEP + 6, _DESC_START + 5,
                        iStart + iCount*_SEP + 6 );

            g.drawString( stFossils.stItem[i].sName.trim(),
                          _DESCRIPTION, iStart + iCount*_SEP + 8 );

            plot.drawSymbol( g, stFossils.stItem[i].iRows, 0,
                             stFossils.stItem[i].symbol,
                             _IMAGE_START, iStart + iCount*_SEP );

            iCount++;
          }
        }
      }
    }
  }

  /** Method paint(Graphics g)
   * <p> This method will paint Porosity & Permeability Plot
   * @param g - Graphics Parameter.
   */

  public void paint(Graphics g)
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);

    draw(g);
  }
}

/*
 *  @version 1.1 03/09/2009
 *  @author  John Victorine
 */
