/*
 * @fossilPlotDataEntry.java Version 1.1 09/05/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.fossil.plot;

import java.awt.*;
import java.awt.geom.*;

import mask.maskSymbolsListStruct;
import mask.plot.maskPlotSymbol;
import rock.fossil.fossilStruct;

/** Class fossilPlotDataEntry
 *  <p> This method will display the fossil selected
 *
 *  @version 1.1 09/05/2008
 *  @author  John R. Victorine
 */

public class fossilPlotDataEntry extends Canvas
{
  private static final int NO  = 0;
  private static final int YES = 1;

  private fossilStruct          st        = null; // Fossil List Data Structure
  private maskSymbolsListStruct stSymbols = null; // Fossil Symbols List
  private maskPlotSymbol        plot      = null; // Plot Mask Symbol

  private int iWidth  = 200;
  private int iHeight = 250;

  private static final int _START       = 10;
  private static final int _IMAGE_START = 20;
  private static final int _DESC_START  = 55;
  private static final int _DESCRIPTION = 65;
  private static final int _SEP         = 15;

  /** CONSTRUCTOR fossilPlotDataEntry()
   *  <p> This is the constructor for this class.
   *  @param stSymbols = Fossil Plot Symbols List Data Structure
   */

  public fossilPlotDataEntry( maskSymbolsListStruct stSymbols )
  {
    this.stSymbols = stSymbols;
    plot = new maskPlotSymbol(stSymbols);
  }

  /** Method close()
   * <p> This method will force Java to deallocate memory
   */

  public void close()
  {
    if (st != null)
      st.delete();
    st = null;

    stSymbols = null;

    if (plot != null)
      plot.close();
    plot = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Width of the plot panel.
   * @return WIDTH = the width of the plot panel
   */

  public int getPlotWidth() { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Width of the plot panel.
   * @return the height of the plot panel
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getData()
   * <p> This method will get the array of fossils selected
   * @return st = fossil data Structure
   */

  public fossilStruct getData() { return (st); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setData()
   * <p> This method will set the array of fossils selected
   * @param st = fossil data Structure
   */

  public void setData(fossilStruct st)
  {
	int    iFound = -1;
	int    len    = 0;
	String str    = "";

    this.st = st;

    iWidth  = 200;
    iHeight = 250;

    if (st != null)
    {
      iWidth  = 200;
      iHeight = 250;

	  if (st.iTotal > 0)
	  {
	    iHeight = _START + st.iTotal*_SEP;

        for (int i=0; i<st.iTotal; i++)
        {
          iFound = mask.maskSymbolsUtility.getSymbolID(
                   st.sID[i], mask.maskSymbolsListStruct._SYMBOLS, stSymbols );
          len    = 0;
          str    = "";

          if (iFound > -1)
          {
		    str = new String(st.sMod[i] + " " + stSymbols.stItem[iFound].sName.trim());
            len = _DESCRIPTION + 6 * str.length();
            if (len > iWidth)
              iWidth = len;
  		  }
        }
	  }
	}

    repaint();
  }

  /* ======================================================================== *
   * ------------------------------- PAINT ---------------------------------- *
   * ======================================================================== */

  /** Method draw()
   * <p> This method will draw the fossil image panels
   * @param g - Graphics Parameter.
   */

  public void draw(Graphics g)
  {
    int         iFound  = -1;
    int         iColor  = -1;
    int         iRGB[]  = { 255, 255, 255 };
    int         iSymbol = -1;
    Font        fsb     = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm    = g.getFontMetrics(fsb);

    g.setFont( fsb );

    if (st != null)
    {
      iColor = mask.maskSymbolsUtility.getColor(
          st.sEnvironment, stSymbols.stBackground );
      iRGB = mask.maskSymbolsUtility.getRGB(iColor, stSymbols.stBackground );

      g.setColor(Color.white);
      if (iColor > -1)
      {
        g.setColor(new Color(iRGB[0], iRGB[1], iRGB[2]));
      }

      g.fillRect(0, 0, iWidth, iHeight);

      for (int i=0; i<st.iTotal; i++)
      {
        iFound = mask.maskSymbolsUtility.getSymbolID(
                   st.sID[i], mask.maskSymbolsListStruct._SYMBOLS, stSymbols );

        if (iFound > -1)
        {
          g.setColor(Color.black);
          g.drawLine(_DESC_START,      _START + i*_SEP + 6,
                     _DESC_START + 5,  _START + i*_SEP + 6);

          g.drawString( st.sMod[i] + " " +
                        stSymbols.stItem[iFound].sName.trim(),
                        _DESCRIPTION, _START + i*_SEP + 8 );

          if (st.sMod[i].equals("Rare"))
          {
            iSymbol = mask.maskSymbolsUtility.getSymbolIDByName(
             "few left", mask.maskSymbolsListStruct._MODIFIERS, stSymbols);

            if (iSymbol > -1)
              plot.drawSymbol( g, stSymbols.stModifiers[iSymbol].iRows, 0,
                               stSymbols.stModifiers[iSymbol].symbol,
                               _IMAGE_START-3, _START + i*_SEP );
          }

          plot.drawSymbol( g, stSymbols.stItem[iFound].iRows, 0,
                           stSymbols.stItem[iFound].symbol,
                           _IMAGE_START, _START + i*_SEP );

          if (st.sMod[i].equals("Broken"))
          {
            iSymbol = mask.maskSymbolsUtility.getSymbolIDByName(
              "broken", mask.maskSymbolsListStruct._MODIFIERS, stSymbols);

            if (iSymbol > -1)
              plot.drawSymbol( g, stSymbols.stModifiers[iSymbol].iRows, 0,
                               stSymbols.stModifiers[iSymbol].symbol,
                               _IMAGE_START, _START + i*_SEP );
          }

          if (st.sMod[i].equals("Abundant"))
          {
            iSymbol = mask.maskSymbolsUtility.getSymbolIDByName(
                "rich", mask.maskSymbolsListStruct._MODIFIERS, stSymbols);

            if (iSymbol > -1)
              plot.drawSymbol( g, stSymbols.stModifiers[iSymbol].iRows, 0,
                               stSymbols.stModifiers[iSymbol].symbol,
                               _IMAGE_START, _START + i*_SEP );
          }

          if (st.sMod[i].equals("Rare"))
          {
            iSymbol = mask.maskSymbolsUtility.getSymbolIDByName(
             "few right", mask.maskSymbolsListStruct._MODIFIERS, stSymbols);

            if (iSymbol > -1)
              plot.drawSymbol( g, stSymbols.stModifiers[iSymbol].iRows, 0,
                               stSymbols.stModifiers[iSymbol].symbol,
                               _IMAGE_START+10, _START + i*_SEP );
          }
        }
      }
    }
  }

  /** Method paint(Graphics g)
   * <p> This method will paint Porosity & Permeability Plot
   * @param g - Graphics Parameter.
   */

  public void paint(Graphics g)
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);

    draw(g);
  }
}