/*
 * @fossilPlotSymbolFrame.java Version 1.1 09/04/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.fossil.gui;

import java.util.Observable;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

import mask.maskSymbolsListStruct;
import rock.fossil.gui.fossilPlotSymbolPanel;
import rock.fossil.fossilStruct;

import util.utilImageIO;

/** Class fossilPlotSymbolFrame
 *  <p> This Class will display the fossil images available
 *
 *  @version 1.1 09/04/2008
 *  @author  John R. Victorine
 */

public class fossilPlotSymbolFrame extends JFrame implements ActionListener
{
  private Observable            notifier  = null; // Observable Object
  private maskSymbolsListStruct stSymbols = null; // Fossil Symbols Data
  private fossilPlotSymbolPanel plot      = null;
  private fossilStruct          stFossil  = null; // fossil data structure
  private String                sKEY      = "0";

  // Depth range variables

  private double dStart = 0.0;
  private double dEnd   = 0.0;

  // Environment

  private int iEnvironment = rock.fossil.fossilLookupStruct._GENERAL;

  // Global Frame Widgets

  // Depth Range Textfields

  private JTextField   txtStart   = new JTextField();    // Starting Depth
  private JTextField   txtEnd     = new JTextField();    // Ending Depth

  // -- Global Frame Buttons

  private JButton      btnAdd     = new JButton();       // Add Fossils to List
  private JButton      btnClear   = new JButton();       // Clear Fossils data
  private JButton      btnCancel  = new JButton();       // Cancel Modify

  // -- Fossil Fluid Environment

  private JRadioButton rbGeneral  = new JRadioButton();  // General Fossils
  private JRadioButton rbFresh    = new JRadioButton();  // Fresh Water Fossils
  private JRadioButton rbBrackish = new JRadioButton();  // Brackish Water Fossils
  private JRadioButton rbMarine   = new JRadioButton();  // Marine Fossils

  private JMenuItem    mExit      = null;                // Exit Profile Plot

  // Dialog Position & Size

  private int iXStart = 0;
  private int iYStart = 0;
  private int iWidth  = 0;
  private int iHeight = 0;

  /** Constructor fossilPlotSymbolFrame()
   *  <p> This is the Constructor for this class.
   *  @param stSymbols  = Fossil Plot Symbols List Data Structure
   */

  public fossilPlotSymbolFrame( maskSymbolsListStruct stSymbols )
  {
    try
    {
      this.stSymbols = stSymbols;

      jbInit();
      addWindowListener( new fossilPlotSymbolFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Constructor fossilPlotSymbolFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier  = Observable
   *  @param stSymbols = Fossil Plot Symbols List Data Structure
   */

  public fossilPlotSymbolFrame( Observable notifier,
                                maskSymbolsListStruct stSymbols  )
  {
    try
    {
      this.notifier  = notifier;
      this.stSymbols = stSymbols;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Constructor fossilPlotSymbolFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier  = Observable
   *  @param iXStart   = the x-coordinate of Data entry dialog
   *  @param iYStart   = the y-coordinate of Data entry dialog
   *  @param iWidth    = the width of Data entry dialog
   *  @param iHeight   = the height of Data entry dialog
   *  @param stSymbols = Fossil Plot Symbols List Data Structure
   */

  public fossilPlotSymbolFrame( Observable notifier,
                                int iXStart, int iYStart,
                                int iWidth, int iHeight,
                                maskSymbolsListStruct stSymbols  )
  {
    try
    {
      this.notifier  = notifier;
      this.iXStart   = iXStart;
      this.iYStart   = iYStart;
      this.iWidth    = iWidth;
      this.iHeight   = iHeight;
      this.stSymbols = stSymbols;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlWest       = new JPanel();
    JPanel pnlData       = new JPanel();
    JPanel pnlDepthRange = new JPanel();
    JPanel pnlStart      = new JPanel();
    JPanel pnlEnd        = new JPanel();
    JPanel pnlFossils    = new JPanel();
    JPanel pnlAdd        = new JPanel();
    JPanel pnlClear      = new JPanel();

    ButtonGroup groupFossils = new ButtonGroup();

    TitledBorder titledBorderStart = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Starting Depth:");

    TitledBorder titledBorderEnd = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Ending Depth:");

    TitledBorder titledBorderFossils = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Fossils");

    titledBorderStart.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderEnd.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderFossils.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Build Menu

    JMenuBar    menuBar  = new JMenuBar();
    JMenu       menuFile = new JMenu("Close Dialog");

    mExit = new JMenuItem("Close");
    mExit.addActionListener(this);

    menuFile.add( mExit );

    menuBar.add(menuFile);

    this.setJMenuBar(menuBar);

    // Build Dialog

    this.setTitle("Fossils");
    this.getContentPane().setLayout(new BorderLayout());

    // Data Entry Panel

    pnlWest.setLayout(new BorderLayout());

    // Enter Fossil Data Panel

    pnlData.setLayout(new BorderLayout());

    // -- Depth Range Panel

    pnlDepthRange.setPreferredSize(new Dimension(100, 120));
    pnlDepthRange.setLayout(new GridLayout(2,1));

    // .. Starting Depth Panel

    pnlStart.setBorder(titledBorderStart);
    pnlStart.setLayout(new BorderLayout());

    txtStart.setText("0.0");
    txtStart.setEditable(false);
    txtStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtStart.addFocusListener(new fossilPlotSymbolFrameFocusAdapter(this));

    // .. Endind Depth Panel

    pnlEnd.setBorder(titledBorderEnd);
    pnlEnd.setLayout(new BorderLayout());

    txtEnd.setText("0.0");
    txtEnd.setEditable(false);
    txtEnd.setHorizontalAlignment(SwingConstants.TRAILING);
    txtEnd.addFocusListener(new fossilPlotSymbolFrameFocusAdapter(this));

    // Fossils Panel

    pnlFossils.setBorder(titledBorderFossils);
    pnlFossils.setLayout(new GridLayout(4,1));

    rbGeneral.setFont(new java.awt.Font("Dialog", 1, 11));
    rbGeneral.setSelected(true);
    rbGeneral.setText("General");
    rbGeneral.addActionListener(this);

    rbFresh.setFont(new java.awt.Font("Dialog", 1, 11));
    rbFresh.setText("Fresh Water");
    rbFresh.addActionListener(this);

    rbBrackish.setFont(new java.awt.Font("Dialog", 1, 11));
    rbBrackish.setText("Brackish Water");
    rbBrackish.addActionListener(this);

    rbMarine.setFont(new java.awt.Font("Dialog", 1, 11));
    rbMarine.setText("Marine");
    rbMarine.addActionListener(this);

    // Add Fossils to List Panel

    pnlAdd.setLayout(new GridLayout(2,1));

    btnAdd.setFont(new java.awt.Font("Dialog", 1, 11));
    btnAdd.setText("Add");
    btnAdd.addActionListener(this);

    pnlClear.setLayout(new GridLayout());

    btnClear.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClear.setText("Clear");
    btnClear.addActionListener(this);

    btnCancel.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnCancel.setEnabled(false);
    btnCancel.setText("Cancel");
    btnCancel.addActionListener(this);

    // Create Fossil Plot Panel

    plot = new fossilPlotSymbolPanel( this, notifier, stSymbols );

    // Attach Widgets to Frame

    this.getContentPane().add(pnlWest, BorderLayout.WEST);
    pnlWest.add(pnlData,               BorderLayout.NORTH);

    pnlData.add( pnlDepthRange,        BorderLayout.NORTH );
    pnlDepthRange.add( pnlStart,       null );
    pnlStart.add( txtStart,            BorderLayout.CENTER );

    pnlDepthRange.add( pnlEnd,         null );
    pnlEnd.add( txtEnd,                BorderLayout.CENTER );

    pnlData.add(pnlFossils,            BorderLayout.CENTER);
    pnlFossils.add(rbGeneral,          null);
    pnlFossils.add(rbFresh,            null);
    pnlFossils.add(rbBrackish,         null);
    pnlFossils.add(rbMarine,           null);

    groupFossils.add(rbGeneral);
    groupFossils.add(rbFresh);
    groupFossils.add(rbBrackish);
    groupFossils.add(rbMarine);

    pnlData.add(pnlAdd,                BorderLayout.SOUTH);
    pnlAdd.add(btnAdd,                 null);

    pnlAdd.add(pnlClear,               null);
//    pnlClear.add(btnCancel,            null);
    pnlClear.add(btnClear,             null);

    this.getContentPane().add(plot,    BorderLayout.CENTER);

    // Display the Frame

    this.setSize(new Dimension(470, 750));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();

    int iXPos = (d.width  - this.getSize().width) / 2;
    if (iWidth > 0) iXPos = iXStart + iWidth;
    this.setLocation(iXPos, (d.height - this.getSize().height) / 2);

//    this.setResizable(false);
    this.setVisible(true);
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getData()
   * <p> This method will get the array of fossils selected
   * @return st = fossil data Structure
   */

  public fossilStruct getData()
  {
    if (stFossil == null)
    {
      stFossil = new fossilStruct();
      sKEY = new String(cmn.cmnString.UniqueName());
    }

    if (plot != null) { stFossil = plot.getData(); }

    if (dEnd == 0.0) dEnd = dStart;

    stFossil.sKEY       = new String( sKEY );
    stFossil.depthStart = dStart;
    stFossil.depthEnd   = dEnd;

    stFossil.sEnvironment = new String(
        stSymbols.stBackground.sColors[iEnvironment] );

    return (stFossil);
  }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setData()
   * <p> This method will set the array of fossils selected
   * @param st = fossil data Structure
   */

  public void setData(fossilStruct st)
  {
    this.stFossil = st;

    if (st != null) { sKEY = new String( st.sKEY ); }

    if (plot != null) plot.setData(st);

    this.dStart = st.depthStart;
    this.dEnd   = st.depthEnd;

    txtStart.setText( "" + this.dStart );
    txtEnd.setText( "" + this.dEnd );

    for (int i=0; i<stSymbols.stBackground.iColors; i++)
    {
      if (st.sEnvironment.equals(stSymbols.stBackground.sColors[i]))
        this.iEnvironment = i;
    }

    setButtons();

//    btnCancel.setEnabled(true);
    btnAdd.setText("Modify");
  }

  /** Method setDepthRange(double depthStart, double depthEnd)
   * <p> This method will set the depth range of the data
   * @param depthStart = the starting depth
   * @param depthEnd   = the ending depth
   */

  public void setDepthRange(double depthStart, double depthEnd)
  {
    this.dStart = depthStart;
    txtStart.setText( "" + this.dStart );

    this.dEnd = depthEnd;
    txtEnd.setText( "" + this.dEnd );
  }

  /** Method setButtons()
   * <p> This method will set the buttons depending on data
   */

  private void setButtons()
  {
    rbGeneral.setSelected(false);   // General Fossils
    rbFresh.setSelected(false);     // Fresh Water Fossils
    rbBrackish.setSelected(false);  // Brackish Water Fossils
    rbMarine.setSelected(false);    // Marine Fossils

    switch (iEnvironment)
    {
      case rock.fossil.fossilLookupStruct._GENERAL:
        rbGeneral.setSelected(true);   // General Fossils
        break;
      case rock.fossil.fossilLookupStruct._FRESH:
        rbFresh.setSelected(true);     // Fresh Water Fossils
        break;
      case rock.fossil.fossilLookupStruct._BRACKISH:
        rbBrackish.setSelected(true);  // Brackish Water Fossils
        break;
      case rock.fossil.fossilLookupStruct._MARINE:
        rbMarine.setSelected(true);    // Marine Fossils
        break;
    }
  }

  /** Method clear()
   * <p> This method will clear the fossil edit fields
   */

  public void clear()
  {
    fossilStruct st = new fossilStruct();

    setData( st );

    setButtons();
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method add()
   * <p> This method will add the selected data to the fossil list structure
   */

  public void add()
  {
	if (notifier != null)
      notifier.notifyObservers(new String("ADD or MODIFY Fossil Data"));
  }

  /** Method cancel()
   * <p> This method will cancel the edit
   */

  public void cancel()
  {
	if (notifier != null)
      notifier.notifyObservers(new String("Cancel Edit"));
  }

  /** METHOD fossilPlotSymbolFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class fossilPlotSymbolFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
  	  if (notifier != null)
        notifier.notifyObservers(new String("Cancel Edit"));
    }
  }

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    notifier  = null; // Observable Object
    stSymbols = null; // Fossil Symbols Data
    stFossil  = null; // fossil data structure
    sKEY      = null; // Unique KEY for Data

    if (plot != null)
      plot.close();
    plot = null;

    txtStart   = null; // Starting Depth
    txtEnd     = null; // Ending Depth

    btnAdd     = null; // Add Fossils to List
    btnClear   = null; // Clear Fossils data
    btnCancel  = null; // Cancel Fossil edit

    rbGeneral  = null; // General Fossils
    rbFresh    = null; // Fresh Water Fossils
    rbBrackish = null; // Brackish Water Fossils
    rbMarine   = null; // Marine Fossils

    mExit      = null; // Exit Profile Plot

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if ( event.getSource() == btnAdd  )   { add(); }
    if ( event.getSource() == btnClear )  { clear(); }
    if ( event.getSource() == btnCancel ) { cancel(); }

    if ( event.getSource() == rbGeneral )
    {
      iEnvironment = rock.fossil.fossilLookupStruct._GENERAL;
    }

    if ( event.getSource() == rbFresh )
    {
      iEnvironment = rock.fossil.fossilLookupStruct._FRESH;
    }

    if ( event.getSource() == rbBrackish )
    {
      iEnvironment = rock.fossil.fossilLookupStruct._BRACKISH;
    }

    if ( event.getSource() == rbMarine )
    {
      iEnvironment = rock.fossil.fossilLookupStruct._MARINE;
    }

    if ( event.getSource() == mExit )  { close(); }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtStart)
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Start Depth Value is a Numeric Field");
    }

    if (e.getSource() == txtEnd)
    {
      iNumeric = 1;
      sTemp = txtEnd.getText();
      sMessage = new String("End Depth Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
        if (e.getSource() == txtStart)
          dStart = cmn.cmnString.stringToDouble(txtStart.getText());

        if (e.getSource() == txtEnd)
          dEnd = cmn.cmnString.stringToDouble(txtEnd.getText());
      }
    }
  }
}


/** CLASS fossilPlotSymbolFrameFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class fossilPlotSymbolFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  fossilPlotSymbolFrame adaptee;

  fossilPlotSymbolFrameFocusAdapter(fossilPlotSymbolFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 09/04/2008
 *  @author  John Victorine
 */
