/*
 * @fossilUtility.java Version 1.1 09/14/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.fossil;

import iqstrat.iqstratHeadersStruct;
import mask.maskSymbolsListStruct;
import kgs.kgsMeasSectListStruct;

import rock.fossil.fossilListStruct;
import rock.fossil.fossilStruct;

/** Class fossilUtility
 *  <p> This Class will provide basic utilities for the Rock Fossil
 *      data structures.
 *
 *  @version 1.1 09/14/2007
 *  @author  John R. Victorine
 */

public class fossilUtility
{
  /** Method addHeaderData()
   * <p> This method will add the header data to the Fossil data List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Rock Fossil List Data Structure
   * @return st       = The Rock Fossil List Data Structure
   */

  public static fossilListStruct addHeaderData(iqstratHeadersStruct stHeader,
                                               fossilListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method computeDepthRange()
   * <p> This method will compute the depth range of the core data
   * @param   st = the Rock Fossil data list structure
   * @return  st = the Rock Fossil data list structure
   */

  public static fossilListStruct computeDepthRange(fossilListStruct st)
  {
    double depthStart = 0.0;
    double depthEnd   = 0.0;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        depthStart = st.stItem[i].depthStart;
        depthEnd   = st.stItem[i].depthEnd;

        if (i==0)
        {
          st.depthStart = depthStart;
          st.depthEnd   = depthEnd;
        }

        if (st.depthStart > depthStart)
          st.depthStart = depthStart;

        if (st.depthEnd < depthEnd)
          st.depthEnd   = depthEnd;
      }
    }

    return (st);
  }

  /** Method computeWidth( fossilListStruct st )
   * <p> This method will compute the plot track width from the total number of
   *     objects in the observation data
   * @param  st     = the fossil data list structure
   * @return iWidth = the plot track width
   */

  public static int computeWidth( fossilListStruct st )
  {
    int iWidth = 10;
    int iCol   = 12;
    int iTemp  = 0;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        for (int i=0; i<st.iCount; i++)
        {
          iTemp = iCol*st.stItem[i].iTotal;
          for (int k=0; k<st.stItem[i].iTotal; k++)
          {
            if (st.stItem[i].sMod[k].equals("Rare"))
              iTemp = iTemp + 12;
          }

          if (iWidth < iTemp)
            iWidth = iTemp;
        }
      }
    }

    return (iWidth);
  }

  /** Method addFossil()
   * <p> This method will add a fossil to a specific depth this allows for
   *     the ability of inserting multiple fossils to a specific depth without
   *     having to know how many fossils are in the depth.
   * @param  sID     = The ID representing a specific fossil
   * @param  sMod    = modification to the fossil name, i.e. Broken, Rare, etc.
   * @param  sName   = the fossil name
   * @param  st      = The Old Fossils Data Structure
   * @return st      = The new Fossils Data Structure
   */

  public static fossilStruct addFossil(String sID,
                                       String sMod,
                                       String sName,
                                       fossilStruct st)
  {
    int    i           = 0;
    String sTempID[]   = null;
    String sTempMod[]  = null;
    String sTempName[] = null;
    int    iCount      = 0;
    int    iFound      = 0;

    if (st != null)
    {
      sTempID   = new String[st.iTotal+1];
      sTempMod  = new String[st.iTotal+1];
      sTempName = new String[st.iTotal+1];

      for (i=0; i<st.iTotal; i++)
      {
        if ((sName.equals(st.sText[i]) || (sID.equals(st.sID[i])))) iFound = 1;

        sTempID[iCount]   = new String(st.sID[i]);
        sTempMod[iCount]  = new String(st.sMod[i]);
        sTempName[iCount] = new String(st.sText[i]);
        iCount++;
      }

      if (iFound == 0)
      {
        sTempID[iCount]   = new String(sID);
        sTempMod[iCount]  = new String(sMod);
        sTempName[iCount] = new String(sName);
        iCount++;
      }

      st.iTotal = iCount;
      st.sID    = new String[iCount];
      st.sMod   = new String[iCount];
      st.sText  = new String[iCount];

      for (i=0; i<iCount; i++)
      {
        st.sID[i]   = new String(sTempID[i]);
        st.sMod[i]  = new String(sTempMod[i]);
        st.sText[i] = new String(sTempName[i]);
      }
    }

    return (st);
  }

  /** Method add()
   * <p> This method will add the Fossils to an existing list
   * @param  stBlock = The Fossils Block List Data Structure
   * @param  st      = The Old Fossils List Data Structure
   * @return st      = The new Fossils List.
   */

  public static fossilListStruct add( fossilStruct stBlock, fossilListStruct st)
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    fossilListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new fossilListStruct();
    stTemp.stItem = new fossilStruct[iRecords];

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new fossilStruct();
    stTemp.stItem[iCount] = copy(stBlock);
    iCount++;

    stTemp.iCount = iCount;

    st        = new fossilListStruct();
    st.stItem = new fossilStruct[iCount];
    st.iCount = iCount;

    st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
    st.source   = new String( source );    // Source of Data
    st.sCreated = new String( sCreated );  // Created Date

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Rock fossil in an existing list
   * @param  sKEY    = The Primary Key of the data to be removed
   * @param  stBlock = The fossil structure that will be replaced
   * @param  st      = The Old Fossils List Data Structure
   * @return st      = The new Fossils List.
   */

  public static fossilListStruct modify( String sKEY,
                                         fossilStruct stBlock,
                                         fossilListStruct st)
  {
    int i=0;
    fossilListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st != null)
    {
      stTemp        = new fossilListStruct();
      stTemp.stItem = new fossilStruct[st.iCount];
      stTemp.iCount = st.iCount;

      if (st.iCount > 0)
      {
        sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
        source   = new String( st.source );    // Source of Data
        sCreated = new String( st.sCreated );  // Created Date

        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            stTemp.stItem[i] = copy(stBlock);
          }
          else
          {
            stTemp.stItem[i] = copy(st.stItem[i]);
          }
        }
      }

      st.delete();

      st        = new fossilListStruct();
      st.stItem = new fossilStruct[stTemp.iCount];
      st.iCount = stTemp.iCount;

      st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
      st.source   = new String( source );    // Source of Data
      st.sCreated = new String( sCreated );  // Created Date

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Rock fossil from an existing list
   * @param  sKEY   = The Primary Key of the data to be removed
   * @param  st     = The Old Fossils List Data Structure
   * @return st     = The new Fossils List.
   */

  public static fossilListStruct remove( String sKEY, fossilListStruct st )
  {
    int i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iFound   = -1;
    fossilListStruct stTemp = null;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    if (st.iCount == 1)
    {
      st.delete();
      st = null;
    }
    else
    {
      if (st != null)
      {
        sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
        source   = new String( st.source );    // Source of Data
        sCreated = new String( st.sCreated );  // Created Date

        iRecords = st.iCount-1;

        stTemp        = new fossilListStruct();
        stTemp.stItem = new fossilStruct[iRecords];

        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            iFound = i;
          }
        }
      }

      if (st != null)
      {
        if (st.iCount > 0)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (iCount < iRecords)
            {
              if (i != iFound)
              {
                stTemp.stItem[iCount] = copy(st.stItem[i]);
                iCount++;
              }
            }
          }
        }

        st.delete();
      }

      stTemp.iCount = iCount;

      st        = new fossilListStruct();
      st.stItem = new fossilStruct[iCount];
      st.iCount = iCount;

      st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
      st.source   = new String( source );    // Source of Data
      st.sCreated = new String( sCreated );  // Created Date

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method copyList()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static fossilListStruct copyList(fossilListStruct stOld)
  {
    fossilListStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new fossilListStruct();

      stNew.depthStart = stOld.depthStart; // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth

      // LAS 3 Dependent Variable

      stNew.sKGS       = new String( stOld.sKGS ); // KGS Saved Data Indicator
      stNew.source     = new String( stOld.source );    // Source of Data
      stNew.sCreated   = new String( stOld.sCreated );  // Created Date

      stNew.iCount     = stOld.iCount;
      stNew.stItem     = new fossilStruct[stOld.iCount];

      for (int i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy(stOld.stItem[i]);
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static fossilListStruct transfer(fossilListStruct stOld)
  {
    fossilListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy data from one structure to another
   * @param  stOld = The original Fossil Data Structure
   * @return stNew = The Fossil Data Structure
   */

  public static fossilStruct copy(fossilStruct stOld)
  {
    fossilStruct stNew = new fossilStruct();

    if (stOld != null)
    {
      stNew.sKEY       = new String(stOld.sKEY); // Primary Key for data set
      stNew.depthStart = stOld.depthStart; // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth

      stNew.sEnvironment = stOld.sEnvironment; // Envrironment

      stNew.iTotal     = stOld.iTotal;
      stNew.sID        = new String[stNew.iTotal];
      stNew.sMod       = new String[stNew.iTotal];
      stNew.sText      = new String[stNew.iTotal];

      for (int i=0; i<stNew.iTotal; i++)
      {
        stNew.sID[i]   = new String(stOld.sID[i]);
        stNew.sMod[i]  = new String(stOld.sMod[i]);
        stNew.sText[i] = new String(stOld.sText[i]);
      }
    }

    return (stNew);
  }

  /** Method parse()
   * <p> This method will parse the Measured Section into rock texture list
   *     data structure
   * @param  stList     = KGS Measured Section List Data Structure
   * @param  stSymbols  = fossil Symbols List Data Structure
   * @return stTexture  = Rock texture Data List Structure
   */

  public static fossilListStruct parse( kgsMeasSectListStruct stList,
                                        maskSymbolsListStruct stSymbols )
  {
    fossilListStruct  stFossils  = null;
    double            depth      = 0.0;
    double            depth0     = 0.0;
    String            tokens[]   = null;
    String            sDelimiter = new String("[,-; ]+");
    fossilStruct      stFossil   = null;

    if (stList != null)
    {
      for (int i=0; i<stList.iRows; i++)
      {
        if (stList.stData[i].sUnitNo.length() > 0)
        {
          depth = depth + stList.stData[i].dThickness;

          tokens = stList.stData[i].sDescription.split(sDelimiter);
          stFossil = parseFossils( depth0, depth, tokens, stSymbols );
          if (stFossil != null)
            stFossils = add(stFossil, stFossils);

          depth0 = depth;
        }
      }

      stList.delete();
      stList = null;
    }

    stFossils = computeDepthRange( stFossils );

    return (stFossils);
  }

  /** Method parseFossils()
   * <p> This method will parse the text string for fossils.
   * @param  depthStart = the starting depth
   * @param  depthEnd   = the ending depth
   * @param  sData      = the fossil string phrase or word
   * @param  stFossils  = fossil Symbols List Data Structure
   * @return st         = array of fossils with the section of rock
   */

  public static  fossilStruct parseFossils( double depthStart,
                                            double depthEnd,
                                            String sData[],
                                            maskSymbolsListStruct stFossils )
  {
    fossilStruct st      = null;
    int          iFossil = -1;
    int          iCount  = 0;
    String       sID     = "";
    String       sMod    = "";
    String       sText   = "";

    for (int j=0; j<sData.length; j++)
    {
      iFossil = mask.maskSymbolsUtility.getMask(
          sData[j], mask.maskSymbolsListStruct._SYMBOLS, stFossils );

      if (iFossil > -1)
      {
         if (iCount == 0)
         {
           st            = new fossilStruct();
           st.sKEY       = new String( cmn.cmnString.UniqueName() +
                                       ((int)depthStart));
           st.depthStart = depthStart;
           st.depthEnd   = depthEnd;
           iCount++;
         }

         sID   = new String(stFossils.stItem[iFossil].sID);
         sMod  = new String("");
         sText = new String(stFossils.stItem[iFossil].sName);

         st = addFossil(sID, sMod, sText, st);
      }
    }

    return (st);
  }

  /** Method setLegend( int iFossils[], fossilListStruct st )
   * <p> This method will set the legend variables
   * @param  iFossils = Array of indicators to plot fossil symbol or not
   * @param  st       = Fossils List Data Structure
   * @param  stList   = fossil Symbols List Data Structure
   * @return iLith    = Array of indicators to plot fossils symbol or not
   */

  public static int[] setLegend( int iFossils[], fossilListStruct st,
                                 maskSymbolsListStruct stList )
  {
    if ((st != null) && (stList != null))
    {
      for (int i=0; i<st.iCount; i++)
      {
        for (int k=0; k<st.stItem[i].iTotal; k++)
        {
          for (int j=0; j<stList.iCount; j++)
          {
            if (st.stItem[i].sID[k].equals(stList.stItem[j].sID))
              iFossils[j] = 0;
          }
        }
      }
    }

    return (iFossils);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = data list structure
   * @return st = sorted data list structure
   */

  public static fossilListStruct bubbleSort(fossilListStruct st)
  {
    boolean swappedOnPrevRun = true;
    fossilStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].depthStart > st.stItem[i+1].depthStart)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  public static void print(fossilListStruct st)
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        System.out.println(st.stItem[i].sKEY+" "+st.stItem[i].sEnvironment+" "+
                           st.stItem[i].depthStart+" "+st.stItem[i].depthEnd);
        for (int j=0; j<st.stItem[i].iTotal; j++)
        {
          System.out.println("  "+j+" "+st.stItem[i].sID[j]+" "+
                             st.stItem[i].sMod[j]+" "+st.stItem[i].sText[j]);
        }
        System.out.println(" ----------------------------------------------- ");
      }
    }
  }
}