/*
 * @rockMunsellColorStruct.java Version 1.1 05/21/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.color;

/** Class rockMunsellColorStruct
 *  <p> This Class holds the Munsell Standard Color Data
 *
 *  Color Codes from:
 *    Universal Color Language, Level 3 Color Names
 *    (http://www.december.com/html/spec/colorucl.html) &
 *    (http://www.december.com/html/spec/colorucl2.html)
 *
 *    The ISCC-NBS Method identifies 267 blocks of colors. Each block is given
 *    a color name devised using a set of adjectives and suffixes. Each color
 *    block defines a range of colors--not a single color--that have the same
 *    name. This range of color in each block is an acknowledged disadvantage,
 *    but it is pointed out (COLOR, p. 4) that a set of 267 color names is
 *    analogous to calendar dates for chronological events. This method is
 *    suitable for "a variety of scientific and industrial applications"
 *    (COLOR, p. A-13). The UCL at levels above 3 defines finer divisions of
 *    colors with no names, rising to approximately five million divisions in
 *    UCL Level 6. COLOR also lists color names from various sources for each
 *    UCL Level 3 color block (pp. 37-82) and a Dictionary of Color Names which
 *    lists thousands of color names and their corresponding UCL Level 3
 *    block(s) (pp. 85-158).
 *
 *    The color blocks of the UCL Level 3 are defined using the Munsell System
 *    of color which describes the hue, value, and chroma of colors. I prepared
 *    this page using Munsell Renotation Data from the Munsell Color Science
 *    Laboratory, Rochester Institute of Technology. This data was converted to
 *    hexadecimal using formulas given by Easy RGB, and the colors were named
 *    according to COLOR, pp. 16-31. Up to 6 swatches per color name were chosen
 *    to represent the range of colors in each block. Where fewer than 6 colors
 *    are shown, fewer than 6 samples of color were available in that block
 *    range in the data.
 *
 *    The block number is given for each of the 267 UCL Level 3 color names.
 *    Color swatches are defined by using the hexadecimal code for the color.
 *    The swatch shows the Munsell notation for the color in the form Hue
 *    Value/Chroma above the hexadecimal code. I was not able to find samples
 *    of color for Munsell chroma values under 2, so these color blocks have
 *    "No samples available." Remember: each of these 267 color names do not
 *    each identify a single color, but a range of colors.
 *
 *  @version 1.1 05/21/2008
 *  @author  John R. Victorine
 */

public class rockMunsellColorStruct
{
  // Color Numbers

  public static final int    _R  = 0;
  public static final int    _YR = 1;
  public static final int    _Y  = 2;
  public static final int    _GY = 3;
  public static final int    _G  = 4;
  public static final int    _BG = 5;
  public static final int    _B  = 6;
  public static final int    _PB = 7;
  public static final int    _P  = 8;
  public static final int    _RP = 9;
  public static final int    _N  = 10;

  public static final String HUE[] = { "R", "YR", "Y", "GY", "G",
                                       "BG", "B", "PB", "P", "RP", "N" };

  public static final int    _2  = 0;
  public static final int    _5  = 1;
  public static final int    _7  = 2;
  public static final int    _10 = 3;

  // Column Numbers

  public static final int    _ISCC_NO        = 0;
  public static final int    _ISCC_COLOR     = 1;
  public static final int    _MUNSELL_NO     = 2;
  public static final int    _MUNSELL_HUE    = 3;
  public static final int    _MUNSELL_VALUE  = 4;
  public static final int    _MUNSELL_CHROMA = 5;
  public static final int    _MUNSELL_CODE   = 6;
  public static final int    _RGB_HEX        = 7;

  // Color Matrix

  public static final String COLORS[][] = {
  // ISCC-NBS Color Names                Hue   Hue Value         Munsell       RGB HEX
  //                                      #   Color     Chroma     Code         Value

  // N

//  { "268", "white",                       "0", "N",   "10", "0", "N10",       "#FFFFFF" },
  { "267", "black",                       "0", "N",   "1", "0",  "N1",        "#1D1D1D" },
  { "",    "grayish black",               "0", "N",   "2", "0",  "N2",        "#313131" },
  { "266", "dark gray",                   "0", "N",   "3", "0",  "N3",        "#484848" },
  { "",    "medium dark gray",            "0", "N",   "4", "0",  "N4",        "#616161" },
  { "265", "medium gray",                 "0", "N",   "5", "0",  "N5",        "#7B7B7B" },
  { "",    "medium light gray",           "0", "N",   "6", "0",  "N6",        "#959595" },
  { "264", "light gray",                  "0", "N",   "7", "0",  "N7",        "#AFAFAF" },
  { "",    "very light gray",             "0", "N",   "8", "0",  "N8",        "#CACACA" },
  { "263", "white",                       "0", "N",   "9", "0",  "N9",        "#E4E4E4" },
//  { "269", "black",                       "0", "N",   "0", "0",  "N0",        "#000000" },

  // Red

  { "17", "very dark red",               "2.5", "R", "1","4",    "2.5R 1/4",   "#360D20" },
  { "17", "very dark red",               "2.5", "R", "1","6",    "2.5R 1/6",   "#3E0222" },
  { "14", "very deep red",               "2.5", "R", "1","8",    "2.5R 1/8",   "#460024" },
  { "14", "very deep red",               "2.5", "R", "1","10",   "2.5R 1/10",  "#4D0026" },
  { "20", "dark grayish red",            "2.5", "R", "3","2",    "2.5R 3/2",   "#5C4147" },
  { "16", "dark red",                    "2.5", "R", "3","4",    "2.5R 3/4",   "#6A3A43" },
  { "16", "dark red",                    "2.5", "R", "3","6",    "2.5R 3/6",   "#763240" },
  { "16", "dark red",                    "2.5", "R", "3","8",    "2.5R 3/8",   "#82263C" },
  { "13", "deep red",                    "2.5", "R", "3","10",   "2.5R 3/10",  "#8D153A" },
  { "12", "strong red",                  "2.5", "R", "4","12",   "2.5R 4/12",  "#B32B4D" },
  { "12", "strong red",                  "2.5", "R", "5","12",   "2.5R 5/12",  "#D34A64" },
  { "18", "light grayish red",           "2.5", "R", "6","2",    "2.5R 6/2",   "#A88E96" },
  { "18", "light grayish red",           "2.5", "R", "6","4",    "2.5R 6/4",   "#B88891" },
  { "6", "dark pink",                    "2.5", "R", "6","6",    "2.5R 6/6",   "#C7818B" },
  { "3", "deep pink",                    "2.5", "R", "6","10",   "2.5R 6/10",  "#E27281" },
  { "8", "grayish pink",                 "2.5", "R", "7","2",    "2.5R 7/2",   "#C2A8B2" },
  { "5", "moderate pink",                "2.5", "R", "7","4",    "2.5R 7/4",   "#D3A2AC" },
  { "5", "moderate pink",                "2.5", "R", "7","6",    "2.5R 7/6",   "#E39CA6" },
  { "1", "vivid pink",                   "2.5", "R", "7","12",   "2.5R 7/12",  "#FF8395" },
  { "1", "vivid pink",                   "2.5", "R", "7","14",   "2.5R 7/14",  "#FF7890" },
  { "1", "vivid pink",                   "2.5", "R", "7","16",   "2.5R 7/16",  "#FF6B8B" },
  { "7", "pale pink",                    "2.5", "R", "9","2",    "2.5R 9/2",   "#F6DFEB" },
  { "4", "light pink",                   "2.5", "R", "9","4",    "2.5R 9/4",   "#FFD7E3" },
  { "4", "light pink",                   "2.5", "R", "9","6",    "2.5R 9/6",   "#FFCFDB" },

  { "17", "very dark red",               "5", "R", "1","4",      "5R 1/4",     "#370D1C" },
  { "17", "very dark red",               "5", "R", "1","6",      "5R 1/6",     "#40011C" },
  { "14", "very deep red",               "5", "R", "1","8",      "5R 1/8",     "#48001D" },
  { "14", "very deep red",               "5", "R", "1","10",     "5R 1/10",    "#4F001E" },
  { "20", "dark grayish red",            "5", "R", "3","2",      "5R 3/2",     "#5D4145" },
  { "16", "dark red",                    "5", "R", "3","4",      "5R 3/4",     "#6B3A3E" },
  { "16", "dark red",                    "5", "R", "3","6",      "5R 3/6",     "#763238" },
  { "16", "dark red",                    "5", "R", "3","8",      "5R 3/8",     "#822832" },
  { "13", "deep red",                    "5", "R", "3","10",     "5R 3/10",    "#8D172D" },
  { "15", "moderate red",                "5", "R", "4","8",      "5R 4/8",     "#9E4449" },
  { "15", "moderate red",                "5", "R", "4","10",     "5R 4/10",    "#A93A43" },
  { "12", "strong red",                  "5", "R", "4","12",     "5R 4/12",    "#B32E3D" },
  { "19", "grayish red",                 "5", "R", "5","4",      "5R 5/4",     "#9F6E72" },
  { "19", "grayish red",                 "5", "R", "5","6",      "5R 5/6",     "#AD676A" },
  { "15", "moderate red",                "5", "R", "5","8",      "5R 5/8",     "#BB5F63" },
  { "15", "moderate red",                "5", "R", "5","10",     "5R 5/10",    "#C7565C" },
  { "12", "strong red",                  "5", "R", "5","12",     "5R 5/12",    "#D44B54" },
  { "11", "vivid red",                   "5", "R", "5","18",     "5R 5/18",    "#F40141" },
  { "18", "light grayish red",           "5", "R", "6","2",      "5R 6/2",     "#A88E95" },
  { "18", "light grayish red",           "5", "R", "6","4",      "5R 6/4",     "#B9888D" },
  { "6", "dark pink",                    "5", "R", "6","6",      "5R 6/6",     "#C98185" },
  { "3", "deep pink",                    "5", "R", "6","8",      "5R 6/8",     "#D67B7E" },
  { "27", "deep yellowish pink",         "5", "R", "6","12",     "5R 6/12",    "#F0696F" },
  { "27", "deep yellowish pink",         "5", "R", "6","14",     "5R 6/14",    "#FC5F68" },
  { "11", "vivid red",                   "5", "R", "6","18",     "5R 6/18",    "#FF4159" },
  { "8", "grayish pink",                 "5", "R", "7","2",      "5R 7/2",     "#C3A8B0" },
  { "5", "moderate pink",                "5", "R", "7","4",      "5R 7/4",     "#D4A2A8" },
  { "5", "moderate pink",                "5", "R", "7","6",      "5R 7/6",     "#E49CA0" },
  { "26", "strong yellowish pink",       "5", "R", "7","10",     "5R 7/10",    "#FF8D90" },
  { "25", "vivid yellowish pink",        "5", "R", "7","14",     "5R 7/14",    "#FF7980" },
  { "26", "strong yellowish pink",       "5", "R", "8","10",     "5R 8/10",    "#FFA6A8" },
  { "7", "pale pink",                    "5", "R", "9","2",      "5R 9/2",     "#F8DEE9" },
  { "4", "light pink",                   "5", "R", "9","4",      "5R 9/4",     "#FFD7DE" },
  { "4", "light pink",                   "5", "R", "9","6",      "5R 9/6",     "#FFD0D4" },

  { "44", "dark reddish brown",          "7.5", "R", "1","2",    "7.5R 1/2",   "#2E151A" },
  { "14", "very deep red",               "7.5", "R", "1","8",    "7.5R 1/8",   "#490016" },
  { "14", "very deep red",               "7.5", "R", "1","10",   "7.5R 1/10",  "#500016" },
  { "47", "dark grayish reddish brown",  "7.5", "R", "2","2",    "7.5R 2/2",   "#432B2E" },
  { "44", "dark reddish brown",          "7.5", "R", "2","4",    "7.5R 2/4",   "#4F2428" },
  { "11", "vivid red",                   "7.5", "R", "2","14",   "7.5R 2/14",  "#810014" },
  { "46", "grayish reddish brown",       "7.5", "R", "3","2",    "7.5R 3/2",   "#5D4142" },
  { "43", "moderate reddish brown",      "7.5", "R", "3","4",    "7.5R 3/4",   "#6B3A3A" },
  { "43", "moderate reddish brown",      "7.5", "R", "3","6",    "7.5R 3/6",   "#763332" },
  { "13", "deep red",                    "7.5", "R", "3","10",   "7.5R 3/10",  "#8C1B20" },
  { "11", "vivid red",                   "7.5", "R", "3","14",   "7.5R 3/14",  "#A00011" },
  { "19", "grayish red",                 "7.5", "R", "4","2",    "7.5R 4/2",   "#755A5C" },
  { "19", "grayish red",                 "7.5", "R", "4","4",    "7.5R 4/4",   "#855353" },
  { "19", "grayish red",                 "7.5", "R", "4","6",    "7.5R 4/6",   "#924D4A" },
  { "38", "dark reddish orange",         "7.5", "R", "4","8",    "7.5R 4/8",   "#9E4540" },
  { "38", "dark reddish orange",         "7.5", "R", "4","10",   "7.5R 4/10",  "#A93C37" },
  { "36", "deep reddish orange",         "7.5", "R", "4","12",   "7.5R 4/12",  "#B3302E" },
  { "11", "vivid red",                   "7.5", "R", "4","14",   "7.5R 4/14",  "#BD1F24" },
  { "11", "vivid red",                   "7.5", "R", "4","20",   "7.5R 4/20",  "#D90000" },
  { "19", "grayish red",                 "7.5", "R", "5","2",    "7.5R 5/2",   "#8E7478" },
  { "35", "strong reddish orange",       "7.5", "R", "5","12",   "7.5R 5/12",  "#D34D44" },
  { "34", "vivid reddish orange",        "7.5", "R", "5","14",   "7.5R 5/14",  "#DD423A" },
  { "34", "vivid reddish orange",        "7.5", "R", "5","20",   "7.5R 5/20",  "#FA0014" },
  { "18", "light grayish red",           "7.5", "R", "6","2",    "7.5R 6/2",   "#A98E92" },
  { "30", "dark yellowish pink",         "7.5", "R", "6","6",    "7.5R 6/6",   "#C9827E" },
  { "35", "strong reddish orange",       "7.5", "R", "6","12",   "7.5R 6/12",  "#F06B5E" },
  { "34", "vivid reddish orange",        "7.5", "R", "6","16",   "7.5R 6/16",  "#FF5645" },
  { "32", "grayish yellowish pink",      "7.5", "R", "7","2",    "7.5R 7/2",   "#C4A8AE" },
  { "26", "strong yellowish pink",       "7.5", "R", "7","8",    "7.5R 7/8",   "#F5958D" },
  { "25", "vivid yellowish pink",        "7.5", "R", "7","12",   "7.5R 7/12",  "#FF8576" },
  { "26", "strong yellowish pink",       "7.5", "R", "8","8",    "7.5R 8/8",   "#FFAFA5" },
  { "31", "pale yellowish pink",         "7.5", "R", "9","2",    "7.5R 9/2",   "#F9DEE6" },
  { "28", "light yellowish pink",        "7.5", "R", "9","4",    "7.5R 9/4",   "#FFD7D8" },
  { "28", "light yellowish pink",        "7.5", "R", "9","6",    "7.5R 9/6",   "#FFD0CA" },

  { "44", "dark reddish brown",          "10", "R", "1","4",     "10R 1/4",    "#380D13" },
  { "41", "deep reddish brown",          "10", "R", "1","6",     "10R 1/6",    "#410110" },
  { "41", "deep reddish brown",          "10", "R", "1","10",    "10R 1/10",   "#51000B" },
  { "47", "dark grayish reddish brown",  "10", "R", "2","2",     "10R 2/2",    "#422B2B" },
  { "41", "deep reddish brown",          "10", "R", "2","6",     "10R 2/6",    "#581D1C" },
  { "41", "deep reddish brown",          "10", "R", "2","10",    "10R 2/10",   "#6C000C" },
  { "41", "deep reddish brown",          "10", "R", "2","14",    "10R 2/14",   "#800000" },
  { "46", "grayish reddish brown",       "10", "R", "3","2",     "10R 3/2",    "#5C4140" },
  { "40", "strong reddish brown",        "10", "R", "3","8",     "10R 3/8",    "#802C20" },
  { "40", "strong reddish brown",        "10", "R", "3","10",    "10R 3/10",   "#8A2013" },
  { "40", "strong reddish brown",        "10", "R", "3","12",    "10R 3/12",   "#930C00" },
  { "40", "strong reddish brown",        "10", "R", "3","14",    "10R 3/14",   "#9C0000" },
  { "46", "grayish reddish brown",       "10", "R", "4","2",     "10R 4/2",    "#765A5A" },
  { "43", "moderate reddish brown",      "10", "R", "4","6",     "10R 4/6",    "#914E42" },
  { "38", "dark reddish orange",         "10", "R", "4","8",     "10R 4/8",    "#9D4735" },
  { "38", "dark reddish orange",         "10", "R", "4","10",    "10R 4/10",   "#A73F27" },
  { "36", "deep reddish orange",         "10", "R", "4","12",    "10R 4/12",   "#AF3615" },
  { "42", "light reddish brown",         "10", "R", "5","4",     "10R 5/4",    "#9F6E68" },
  { "39", "grayish reddish orange",      "10", "R", "5","6",     "10R 5/6",    "#AD685B" },
  { "37", "moderate reddish orange",     "10", "R", "5","8",     "10R 5/8",    "#BA624D" },
  { "37", "moderate reddish orange",     "10", "R", "5","10",    "10R 5/10",   "#C55A3D" },
  { "35", "strong reddish orange",       "10", "R", "5","12",    "10R 5/12",   "#D0522C" },
  { "34", "vivid reddish orange",        "10", "R", "5","16",    "10R 5/16",   "#E04200" },
  { "18", "light grayish red",           "10", "R", "6","2",     "10R 6/2",    "#AA8E8F" },
  { "42", "light reddish brown",         "10", "R", "6","4",     "10R 6/4",    "#BB8882" },
  { "39", "grayish reddish orange",      "10", "R", "6","6",     "10R 6/6",    "#C88375" },
  { "37", "moderate reddish orange",     "10", "R", "6","8",     "10R 6/8",    "#D67D68" },
  { "37", "moderate reddish orange",     "10", "R", "6","10",    "10R 6/10",   "#E27658" },
  { "35", "strong reddish orange",       "10", "R", "6","12",    "10R 6/12",   "#EE6E47" },
  { "34", "vivid reddish orange",        "10", "R", "6","18",    "10R 6/18",   "#FF5600" },
  { "32", "grayish yellowish pink",      "10", "R", "7","2",     "10R 7/2",    "#C4A8AB" },
  { "26", "strong yellowish pink",       "10", "R", "7","10",    "10R 7/10",   "#FF9072" },
  { "25", "vivid yellowish pink",        "10", "R", "7","14",    "10R 7/14",   "#FF814E" },
  { "25", "vivid yellowish pink",        "10", "R", "7","16",    "10R 7/16",   "#FF7937" },
  { "26", "strong yellowish pink",       "10", "R", "8","10",    "10R 8/10",   "#FFAA8A" },
  { "31", "pale yellowish pink",         "10", "R", "9","2",     "10R 9/2",    "#FADEE4" },
  { "28", "light yellowish pink",        "10", "R", "9","4",     "10R 9/4",    "#FFD7D2" },
  { "28", "light yellowish pink",        "10", "R", "9","6",     "10R 9/6",    "#FFD1C1" },

  // Yellow Red

  { "44", "dark reddish brown",           "2.5", "YR", "1","4",  "2.5YR 1/4",  "#380E0B" },
  { "56", "deep brown",                   "2.5", "YR", "1","6",  "2.5YR 1/6",  "#400304" },
  { "56", "deep brown",                   "2.5", "YR", "1","8",  "2.5YR 1/8",  "#490000" },
  { "47", "dark grayish reddish brown",   "2.5", "YR", "2","2",  "2.5YR 2/2",  "#422B29" },
  { "56", "deep brown",                   "2.5", "YR", "2","6",  "2.5YR 2/6",  "#571F13" },
  { "56", "deep brown",                   "2.5", "YR", "2","8",  "2.5YR 2/8",  "#601601" },
  { "46", "grayish reddish brown",        "2.5", "YR", "3","2",  "2.5YR 3/2",  "#5C423E" },
  { "46", "grayish reddish brown",        "2.5", "YR", "4","2",  "2.5YR 4/2",  "#755A57" },
  { "43", "moderate reddish brown",       "2.5", "YR", "4","4",  "2.5YR 4/4",  "#845548" },
  { "45", "light grayish reddish brown",  "2.5", "YR", "5","2",  "2.5YR 5/2",  "#8F7472" },
  { "42", "light reddish brown",          "2.5", "YR", "5","4",  "2.5YR 5/4",  "#9E6F63" },
  { "39", "grayish reddish orange",       "2.5", "YR", "5","6",  "2.5YR 5/6",  "#AB6A53" },
  { "54", "brownish orange",              "2.5", "YR", "5","8",  "2.5YR 5/8",  "#B76441" },
  { "51", "deep orange",                  "2.5", "YR", "5","12", "2.5YR 5/12", "#C85A15" },
  { "45", "light grayish reddish brown",  "2.5", "YR", "6","2",  "2.5YR 6/2",  "#AA8E8C" },
  { "42", "light reddish brown",          "2.5", "YR", "6","4",  "2.5YR 6/4",  "#B9897D" },
  { "39", "grayish reddish orange",       "2.5", "YR", "6","6",  "2.5YR 6/6",  "#C7846D" },
  { "53", "moderate orange",              "2.5", "YR", "6","8",  "2.5YR 6/8",  "#D37F5D" },
  { "50", "strong orange",                "2.5", "YR", "6","12", "2.5YR 6/12", "#E87434" },
  { "48", "vivid orange",                 "2.5", "YR", "6","16", "2.5YR 6/16", "#F56B00" },
  { "48", "vivid orange",                 "2.5", "YR", "6","18", "2.5YR 6/18", "#FA6700" },
  { "32", "grayish yellowish pink",       "2.5", "YR", "7","2",  "2.5YR 7/2",  "#C4A8A8" },
  { "29", "moderate yellowish pink",      "2.5", "YR", "7","4",  "2.5YR 7/4",  "#D5A498" },
  { "53", "moderate orange",              "2.5", "YR", "7","8",  "2.5YR 7/8",  "#F19976" },
  { "50", "strong orange",                "2.5", "YR", "7","12", "2.5YR 7/12", "#FF8E4E" },
  { "48", "vivid orange",                 "2.5", "YR", "7","16", "2.5YR 7/16", "#FF8403" },
  { "48", "vivid orange",                 "2.5", "YR", "7","18", "2.5YR 7/18", "#FF8000" },
  { "48", "vivid orange",                 "2.5", "YR", "7","20", "2.5YR 7/20", "#FF7D00" },
  { "52", "light orange",                 "2.5", "YR", "8","8",  "2.5YR 8/8",  "#FFB38D" },
  { "49", "brilliant orange",             "2.5", "YR", "8","12", "2.5YR 8/12", "#FFA865" },
  { "31", "pale yellowish pink",          "2.5", "YR", "9","2",  "2.5YR 9/2",  "#FBDEE0" },
  { "28", "light yellowish pink",         "2.5", "YR", "9","4",  "2.5YR 9/4",  "#FFD8CB" },

  { "59", "dark brown",                   "5", "YR", "1","2",    "5YR 1/2",    "#2C1711" },
  { "59", "dark brown",                   "5", "YR", "1","4",    "5YR 1/4",    "#371001" },
  { "62", "dark grayish brown",           "5", "YR", "2","2",    "5YR 2/2",    "#412C27" },
  { "59", "dark brown",                   "5", "YR", "2","4",    "5YR 2/4",    "#4B2819" },
  { "56", "deep brown",                   "5", "YR", "2","6",    "5YR 2/6",    "#542304" },
  { "58", "moderate brown",               "5", "YR", "3","4",    "5YR 3/4",    "#653E2D" },
  { "58", "moderate brown",               "5", "YR", "4","4",    "5YR 4/4",    "#825743" },
  { "55", "strong brown",                 "5", "YR", "4","6",    "5YR 4/6",    "#8B5332" },
  { "55", "strong brown",                 "5", "YR", "4","8",    "5YR 4/8",    "#934F1F" },
  { "55", "strong brown",                 "5", "YR", "4","10",   "5YR 4/10",   "#9A4B00" },
  { "55", "strong brown",                 "5", "YR", "4","12",   "5YR 4/12",   "#9F4800" },
  { "57", "light brown",                  "5", "YR", "5","4",    "5YR 5/4",    "#9C715D" },
  { "54", "brownish orange",              "5", "YR", "5","8",    "5YR 5/8",    "#B26836" },
  { "51", "deep orange",                  "5", "YR", "5","12",   "5YR 5/12",   "#BF6100" },
  { "57", "light brown",                  "5", "YR", "6","4",    "5YR 6/4",    "#B78B78" },
  { "53", "moderate orange",              "5", "YR", "6","8",    "5YR 6/8",    "#CF8250" },
  { "50", "strong orange",                "5", "YR", "6","12",   "5YR 6/12",   "#DF7B1D" },
  { "29", "moderate yellowish pink",      "5", "YR", "7","4",    "5YR 7/4",    "#D3A592" },
  { "53", "moderate orange",              "5", "YR", "7","8",    "5YR 7/8",    "#EB9D6B" },
  { "50", "strong orange",                "5", "YR", "7","12",   "5YR 7/12",   "#FF943A" },
  { "52", "light orange",                 "5", "YR", "8","8",    "5YR 8/8",    "#FFB781" },
  { "49", "brilliant orange",             "5", "YR", "8","12",   "5YR 8/12",   "#FFAF54" },
  { "28", "light yellowish pink",         "5", "YR", "9","4",    "5YR 9/4",    "#FFDAC4" },

  { "59", "dark brown",                   "7.5", "YR", "1","2",  "7.5YR 1/2",  "#2B180E" },
  { "62", "dark grayish brown",           "7.5", "YR", "2","2",  "7.5YR 2/2",  "#3F2D25" },
  { "59", "dark brown",                   "7.5", "YR", "2","4",  "7.5YR 2/4",  "#482A13" },
  { "56", "deep brown",                   "7.5", "YR", "2","6",  "7.5YR 2/6",  "#502600" },
  { "61", "grayish brown",                "7.5", "YR", "3","2",  "7.5YR 3/2",  "#59433A" },
  { "58", "moderate brown",               "7.5", "YR", "3","4",  "7.5YR 3/4",  "#634029" },
  { "55", "strong brown",                 "7.5", "YR", "3","6",  "7.5YR 3/6",  "#6A3D14" },
  { "55", "strong brown",                 "7.5", "YR", "3","8",  "7.5YR 3/8",  "#703A00" },
  { "61", "grayish brown",                "7.5", "YR", "4","2",  "7.5YR 4/2",  "#735C52" },
  { "58", "moderate brown",               "7.5", "YR", "4","4",  "7.5YR 4/4",  "#7F583E" },
  { "60", "light grayish brown",          "7.5", "YR", "5","2",  "7.5YR 5/2",  "#8D766D" },
  { "57", "light brown",                  "7.5", "YR", "5","4",  "7.5YR 5/4",  "#9A7259" },
  { "60", "light grayish brown",          "7.5", "YR", "6","2",  "7.5YR 6/2",  "#A88F87" },
  { "57", "light brown",                  "7.5", "YR", "6","4",  "7.5YR 6/4",  "#B58C73" },
  { "72", "dark orange yellow",           "7.5", "YR", "6","8",  "7.5YR 6/8",  "#C98646" },
  { "69", "deep orange yellow",           "7.5", "YR", "6","12", "7.5YR 6/12", "#D68000" },
  { "33", "brownish pink",                "7.5", "YR", "7","2",  "7.5YR 7/2",  "#C3AAA1" },
  { "76", "light yellowish brown",        "7.5", "YR", "7","4",  "7.5YR 7/4",  "#D1A68D" },
  { "71", "moderate orange yellow",       "7.5", "YR", "7","8",  "7.5YR 7/8",  "#E6A061" },
  { "68", "strong orange yellow",         "7.5", "YR", "7","12", "7.5YR 7/12", "#F59A26" },
  { "66", "vivid orange yellow",          "7.5", "YR", "7","16", "7.5YR 7/16", "#FE9700" },
  { "66", "vivid orange yellow",          "7.5", "YR", "7","18", "7.5YR 7/18", "#FF9500" },
  { "73", "pale orange yellow",           "7.5", "YR", "8","4",  "7.5YR 8/4",  "#ECC1A6" },
  { "66", "vivid orange yellow",          "7.5", "YR", "8","16", "7.5YR 8/16", "#FFB000" },
  { "66", "vivid orange yellow",          "7.5", "YR", "8","18", "7.5YR 8/18", "#FFAF00" },
  { "66", "vivid orange yellow",          "7.5", "YR", "8","20", "7.5YR 8/20", "#FFAE00" },
  { "31", "pale yellowish pink",          "7.5", "YR", "9","2",  "7.5YR 9/2",  "#FBDFD7" },
  { "73", "pale orange yellow",           "7.5", "YR", "9","4",  "7.5YR 9/4",  "#FFDCBE" },
  { "70", "light orange yellow",          "7.5", "YR", "9","8",  "7.5YR 9/8",  "#FFD68F" },

  { "78", "dark yellowish brown",         "10", "YR", "1","2",   "10YR 1/2",   "#29190A" },
  { "81", "dark grayish yellowish brown", "10", "YR", "2","2",   "10YR 2/2",   "#3D2E23" },
  { "78", "dark yellowish brown",         "10", "YR", "2","4",   "10YR 2/4",   "#462B0E" },
  { "81", "dark grayish yellowish brown", "10", "YR", "3","2",   "10YR 3/2",   "#564438" },
  { "78", "dark yellowish brown",         "10", "YR", "3","4",   "10YR 3/4",   "#5F4225" },
  { "75", "deep yellowish brown",         "10", "YR", "3","6",   "10YR 3/6",   "#65400B" },
  { "75", "deep yellowish brown",         "10", "YR", "3","8",   "10YR 3/8",   "#6A3E00" },
  { "80", "grayish yellowish brown",      "10", "YR", "4","2",   "10YR 4/2",   "#715D4F" },
  { "77", "moderate yellowish brown",     "10", "YR", "4","4",   "10YR 4/4",   "#7B5A3A" },
  { "74", "strong yellowish brown",       "10", "YR", "4","8",   "10YR 4/8",   "#875602" },
  { "80", "grayish yellowish brown",      "10", "YR", "5","2",   "10YR 5/2",   "#8B766A" },
  { "77", "moderate yellowish brown",     "10", "YR", "5","4",   "10YR 5/4",   "#967454" },
  { "74", "strong yellowish brown",       "10", "YR", "5","6",   "10YR 5/6",   "#9F713B" },
  { "74", "strong yellowish brown",       "10", "YR", "5","8",   "10YR 5/8",   "#A56F20" },
  { "74", "strong yellowish brown",       "10", "YR", "5","10",  "10YR 5/10",  "#AA6E00" },
  { "74", "strong yellowish brown",       "10", "YR", "5","12",  "10YR 5/12",  "#AC6D00" },
  { "76", "light yellowish brown",        "10", "YR", "6","4",   "10YR 6/4",   "#B18E6D" },
  { "72", "dark orange yellow",           "10", "YR", "6","8",   "10YR 6/8",   "#C2893B" },
  { "69", "deep orange yellow",           "10", "YR", "6","12",  "10YR 6/12",  "#CC8600" },
  { "76", "light yellowish brown",        "10", "YR", "7","4",   "10YR 7/4",   "#CDA887" },
  { "71", "moderate orange yellow",       "10", "YR", "7","8",   "10YR 7/8",   "#DFA456" },
  { "68", "strong orange yellow",         "10", "YR", "7","12",  "10YR 7/12",  "#EBA004" },
  { "73", "pale orange yellow",           "10", "YR", "8","4",   "10YR 8/4",   "#E8C3A1" },
  { "73", "pale orange yellow",           "10", "YR", "9","4",   "10YR 9/4",   "#FFDEB9" },
  { "70", "light orange yellow",          "10", "YR", "9","8",   "10YR 9/8",   "#FFDA86" },

  // Yellow

  { "96", "dark olive brown",             "2.5", "Y", "1","2",   "2.5Y 1/2",   "#261A08" },
  { "96", "dark olive brown",             "2.5", "Y", "2","2",   "2.5Y 2/2",   "#3B2F22" },
  { "96", "dark olive brown",             "2.5", "Y", "2","4",   "2.5Y 2/4",   "#422D09" },
  { "95", "moderate olive brown",         "2.5", "Y", "3","4",   "2.5Y 3/4",   "#5B4422" },
  { "95", "moderate olive brown",         "2.5", "Y", "4","2",   "2.5Y 4/2",   "#6E5E4E" },
  { "95", "moderate olive brown",         "2.5", "Y", "4","6",   "2.5Y 4/6",   "#7C5B1E" },
  { "95", "moderate olive brown",         "2.5", "Y", "4","10",  "2.5Y 4/10",  "#835900" },
  { "94", "light olive brown",            "2.5", "Y", "5","2",   "2.5Y 5/2",   "#897868" },
  { "94", "light olive brown",            "2.5", "Y", "5","4",   "2.5Y 5/4",   "#917650" },
  { "94", "light olive brown",            "2.5", "Y", "5","6",   "2.5Y 5/6",   "#997534" },
  { "94", "light olive brown",            "2.5", "Y", "5","8",   "2.5Y 5/8",   "#9E7315" },
  { "94", "light olive brown",            "2.5", "Y", "5","10",  "2.5Y 5/10",  "#A17200" },
  { "94", "light olive brown",            "2.5", "Y", "5","12",  "2.5Y 5/12",  "#A37200" },
  { "91", "dark grayish yellow",          "2.5", "Y", "6","4",   "2.5Y 6/4",   "#AC9069" },
  { "88", "dark yellow",                  "2.5", "Y", "6","6",   "2.5Y 6/6",   "#B48F4F" },
  { "85", "deep yellow",                  "2.5", "Y", "6","10",  "2.5Y 6/10",  "#BF8C00" },
  { "90", "grayish yellow",               "2.5", "Y", "7","4",   "2.5Y 7/4",   "#C8AA83" },
  { "87", "moderate yellow",              "2.5", "Y", "7","6",   "2.5Y 7/6",   "#D0A969" },
  { "84", "strong yellow",                "2.5", "Y", "7","10",  "2.5Y 7/10",  "#DCA62B" },
  { "89", "pale yellow",                  "2.5", "Y", "9","4",   "2.5Y 9/4",   "#FFE1B5" },
  { "86", "light yellow",                 "2.5", "Y", "9","6",   "2.5Y 9/6",   "#FFDF9A" },
  { "83", "brilliant yellow",             "2.5", "Y", "9","10",  "2.5Y 9/10",  "#FFDC62" },
  { "82", "vivid yellow",                 "2.5", "Y", "9","12",  "2.5Y 9/12",  "#FFDB3F" },

  { "108", "dark olive",                  "5", "Y", "1","2",     "5Y 1/2",     "#241B08" },
  { "111", "dark grayish olive",          "5", "Y", "2","2",     "5Y 2/2",     "#393022" },
  { "108", "dark olive",                  "5", "Y", "2","4",     "5Y 2/4",     "#3E2F05" },
  { "110", "grayish olive",               "5", "Y", "3","2",     "5Y 3/2",     "#514737" },
  { "110", "grayish olive",               "5", "Y", "4","2",     "5Y 4/2",     "#6B5F4D" },
  { "91", "dark grayish yellow",          "5", "Y", "6","4",     "5Y 6/4",     "#A79266" },
  { "88", "dark yellow",                  "5", "Y", "6","6",     "5Y 6/6",     "#AD924A" },
  { "85", "deep yellow",                  "5", "Y", "6","10",    "5Y 6/10",    "#B59100" },
  { "82", "vivid yellow",                 "5", "Y", "6","14",    "5Y 6/14",    "#B89000" },
  { "90", "grayish yellow",               "5", "Y", "7","4",     "5Y 7/4",     "#C2AD80" },
  { "87", "moderate yellow",              "5", "Y", "7","6",     "5Y 7/6",     "#C8AC64" },
  { "84", "strong yellow",                "5", "Y", "7","10",    "5Y 7/10",    "#D1AB1C" },
  { "82", "vivid yellow",                 "5", "Y", "7","14",    "5Y 7/14",    "#D6AA00" },
  { "82", "vivid yellow",                 "5", "Y", "8","12",    "5Y 8/12",    "#F0C600" },
  { "82", "vivid yellow",                 "5", "Y", "8","16",    "5Y 8/16",    "#F4C500" },
  { "89", "pale yellow",                  "5", "Y", "9","4",     "5Y 9/4",     "#F9E3B2" },
  { "86", "light yellow",                 "5", "Y", "9","6",     "5Y 9/6",     "#FFE395" },
  { "83", "brilliant yellow",             "5", "Y", "9","10",    "5Y 9/10",    "#FFE259" },
  { "82", "vivid yellow",                 "5", "Y", "9","18",    "5Y 9/18",    "#FFE100" },

  { "108", "dark olive",                  "7.5", "Y", "1","2",   "7.5Y 1/2",   "#221C0A" },
  { "111", "dark grayish olive",          "7.5", "Y", "2","2",   "7.5Y 2/2",   "#363122" },
  { "108", "dark olive",                  "7.5", "Y", "2","4",   "7.5Y 2/4",   "#3A3004" },
  { "110", "grayish olive",               "7.5", "Y", "3","2",   "7.5Y 3/2",   "#4F4737" },
  { "110", "grayish olive",               "7.5", "Y", "4","2",   "7.5Y 4/2",   "#68604D" },
  { "103", "dark greenish yellow",        "7.5", "Y", "6","6",   "7.5Y 6/6",   "#A69448" },
  { "100", "deep greenish yellow",        "7.5", "Y", "6","10",  "7.5Y 6/10",  "#AB9400" },
  { "105", "grayish greenish yellow",     "7.5", "Y", "7","4",   "7.5Y 7/4",   "#BDAE7F" },
  { "102", "moderate greenish yellow",    "7.5", "Y", "7","6",   "7.5Y 7/6",   "#C1AF63" },
  { "99", "strong greenish yellow",       "7.5", "Y", "7","10",  "7.5Y 7/10",  "#C7AF13" },
  { "97", "vivid greenish yellow",        "7.5", "Y", "7","12",  "7.5Y 7/12",  "#C9AF00" },
  { "97", "vivid greenish yellow",        "7.5", "Y", "8","16",  "7.5Y 8/16",  "#E8CA00" },
  { "104", "pale greenish yellow",        "7.5", "Y", "9","4",   "7.5Y 9/4",   "#F5E4B1" },
  { "101", "light greenish yellow",       "7.5", "Y", "9","6",   "7.5Y 9/6",   "#F9E594" },
  { "98", "brilliant greenish yellow",    "7.5", "Y", "9","10",  "7.5Y 9/10",  "#FFE555" },
  { "97", "vivid greenish yellow",        "7.5", "Y", "9","18",  "7.5Y 9/18",  "#FFE600" },

  { "108", "dark olive",                  "10", "Y", "1","2",    "10Y 1/2",    "#1F1D0D" },
  { "111", "dark grayish olive",          "10", "Y", "2","2",    "10Y 2/2",    "#343123" },
  { "108", "dark olive",                  "10", "Y", "2","4",    "10Y 2/4",    "#363207" },
  { "110", "grayish olive",               "10", "Y", "3","2",    "10Y 3/2",    "#4C4837" },
  { "107", "moderate olive",              "10", "Y", "3","4",    "10Y 3/4",    "#4E4920" },
  { "107", "moderate olive",              "10", "Y", "3","6",    "10Y 3/6",    "#4F4900" },
  { "110", "grayish olive",               "10", "Y", "4","2",    "10Y 4/2",    "#64614E" },
  { "107", "moderate olive",              "10", "Y", "4","4",    "10Y 4/4",    "#686234" },
  { "107", "moderate olive",              "10", "Y", "4","6",    "10Y 4/6",    "#696216" },
  { "107", "moderate olive",              "10", "Y", "4","8",    "10Y 4/8",    "#6A6300" },
  { "109", "light grayish olive",         "10", "Y", "5","2",    "10Y 5/2",    "#807A67" },
  { "106", "light olive",                 "10", "Y", "5","4",    "10Y 5/4",    "#827C4C" },
  { "106", "light olive",                 "10", "Y", "5","6",    "10Y 5/6",    "#847C2F" },
  { "106", "light olive",                 "10", "Y", "5","8",    "10Y 5/8",    "#857D00" },
  { "106", "light olive",                 "10", "Y", "5","10",   "10Y 5/10",   "#867D00" },
  { "106", "light olive",                 "10", "Y", "5","12",   "10Y 5/12",   "#867D00" },
  { "109", "light grayish olive",         "10", "Y", "6","2",    "10Y 6/2",    "#9B9480" },
  { "103", "dark greenish yellow",        "10", "Y", "6","6",    "10Y 6/6",    "#9F9649" },
  { "100", "deep greenish yellow",        "10", "Y", "6","10",   "10Y 6/10",   "#A29700" },
  { "122", "grayish yellow green",        "10", "Y", "7","2",    "10Y 7/2",    "#B6AF9A" },
  { "105", "grayish greenish yellow",     "10", "Y", "7","4",    "10Y 7/4",    "#B9B07F" },
  { "102", "moderate greenish yellow",    "10", "Y", "7","6",    "10Y 7/6",    "#BBB163" },
  { "99", "strong greenish yellow",       "10", "Y", "7","10",   "10Y 7/10",   "#BEB20D" },
  { "97", "vivid greenish yellow",        "10", "Y", "7","16",   "10Y 7/16",   "#BFB300" },
  { "121", "pale yellow green",           "10", "Y", "8","2",    "10Y 8/2",    "#D2C9B3" },
  { "121", "pale yellow green",           "10", "Y", "9","2",    "10Y 9/2",    "#EEE4CD" },
  { "104", "pale greenish yellow",        "10", "Y", "9","4",    "10Y 9/4",    "#F1E6B0" },
  { "101", "light greenish yellow",       "10", "Y", "9","6",    "10Y 9/6",    "#F3E794" },
  { "98", "brilliant greenish yellow",    "10", "Y", "9","10",   "10Y 9/10",   "#F6E853" },
  { "97", "vivid greenish yellow",        "10", "Y", "9","12",   "10Y 9/12",   "#F8E91F" },

  // Green Yellow

  { "128", "dark grayish olive green",    "2.5", "GY", "2","2",  "2.5GY 2/2",  "#313225" },
  { "126", "dark olive green",            "2.5", "GY", "2","4",  "2.5GY 2/4",  "#30330E" },
  { "127", "grayish olive green",         "2.5", "GY", "3","2",  "2.5GY 3/2",  "#494939" },
  { "125", "moderate olive green",        "2.5", "GY", "3","5",  "2.5GY 3/6",  "#474C02" },
  { "127", "grayish olive green",         "2.5", "GY", "4","2",  "2.5GY 4/2",  "#626250" },
  { "125", "moderate olive green",        "2.5", "GY", "4","6",  "2.5GY 4/6",  "#60651C" },
  { "118", "deep yellow green",           "2.5", "GY", "4","8",  "2.5GY 4/8",  "#5F6600" },
  { "122", "grayish yellow green",        "2.5", "GY", "6","2",  "2.5GY 6/2",  "#979581" },
  { "115", "vivid yellow green",          "2.5", "GY", "7","14", "2.5GY 7/14", "#AFB700" },
  { "119", "light yellow green",          "2.5", "GY", "8","6",  "2.5GY 8/6",  "#CECE7F" },
  { "116", "brilliant yellow green",      "2.5", "GY", "8","10", "2.5GY 8/10", "#CCD13E" },
  { "115", "vivid yellow green",          "2.5", "GY", "8","18", "2.5GY 8/18", "#CAD400" },
  { "119", "light yellow green",          "2.5", "GY", "9","6",  "2.5GY 9/6",  "#EAEA96" },
  { "116", "brilliant yellow green",      "2.5", "GY", "9","10", "2.5GY 9/10", "#E8ED57" },
  { "115", "vivid yellow green",          "2.5", "GY", "9","14", "2.5GY 9/14", "#E7EF00" },

  { "126", "dark olive green",            "5", "GY", "1","2",    "5GY 1/2",    "#1A1E13" },
  { "126", "dark olive green",            "5", "GY", "1","4",    "5GY 1/4",    "#112100" },
  { "128", "dark grayish olive green",    "5", "GY", "2","2",    "5GY 2/2",    "#2F3227" },
  { "127", "grayish olive green",         "5", "GY", "3","2",    "5GY 3/2",    "#464A3B" },
  { "125", "moderate olive green",        "5", "GY", "3","4",    "5GY 3/4",    "#424C27" },
  { "123", "strong olive green",          "5", "GY", "3","8",    "5GY 3/8",    "#385000" },
  { "127", "grayish olive green",         "5", "GY", "4","2",    "5GY 4/2",    "#5F6252" },
  { "125", "moderate olive green",        "5", "GY", "4","4",    "5GY 4/4",    "#5B653C" },
  { "118", "deep yellow green",           "5", "GY", "4","8",    "5GY 4/8",    "#536900" },
  { "118", "deep yellow green",           "5", "GY", "4","10",   "5GY 4/10",   "#4F6A00" },
  { "122", "grayish yellow green",        "5", "GY", "5","2",    "5GY 5/2",    "#7A7C6B" },
  { "122", "grayish yellow green",        "5", "GY", "7","2",    "5GY 7/2",    "#B0B09D" },
  { "121", "pale yellow green",          "5", "GY", "8","2",    "5GY 8/2",    "#CBCBB6" },
  { "119", "light yellow green",          "5", "GY", "8","4",    "5GY 8/4",    "#C9CE9E" },
  { "116", "brilliant yellow green",      "5", "GY", "8","8",    "5GY 8/8",    "#C1D36B" },
  { "115", "vivid yellow green",          "5", "GY", "8","12",   "5GY 8/12",   "#B9D718" },
  { "121", "pale yellow green",          "5", "GY", "9","2",    "5GY 9/2",    "#E7E6CF" },
  { "119", "light yellow green",          "5", "GY", "9","4",    "5GY 9/4",    "#E5E9B4" },
  { "116", "brilliant yellow green",      "5", "GY", "9","8",    "5GY 9/8",    "#DDEE7E" },

  { "128", "dark grayish olive green",    "7.5", "GY", "2","2",  "7.5GY 2/2",  "#2C332A" },
  { "126", "dark olive green",            "7.5", "GY", "2","4",  "7.5GY 2/4",  "#24361C" },
  { "126", "dark olive green",            "7.5", "GY", "2","6",  "7.5GY 2/6",  "#18380A" },
  { "124", "deep olive green",            "7.5", "GY", "2","8",  "7.5GY 2/8",  "#013B00" },
  { "127", "grayish olive green",         "7.5", "GY", "3","2",  "7.5GY 3/2",  "#424A3F" },
  { "125", "moderate olive green",        "7.5", "GY", "3","6",  "7.5GY 3/6",  "#30501D" },
  { "123", "strong olive green",          "7.5", "GY", "3","8",  "7.5GY 3/8",  "#245201" },
  { "123", "strong olive green",          "7.5", "GY", "3","10", "7.5GY 3/10", "#105500" },
  { "127", "grayish olive green",         "7.5", "GY", "4","2",  "7.5GY 4/2",  "#5B6356" },
  { "125", "moderate olive green",        "7.5", "GY", "4","6",  "7.5GY 4/6",  "#496A31" },
  { "118", "deep yellow green",           "7.5", "GY", "4","8",  "7.5GY 4/8",  "#3E6C18" },
  { "118", "deep yellow green",           "7.5", "GY", "4","10", "7.5GY 4/10", "#316E00" },
  { "120", "moderate yellow green",       "7.5", "GY", "5","4",  "7.5GY 5/4",  "#6D805D" },
  { "120", "moderate yellow green",       "7.5", "GY", "5","6",  "7.5GY 5/6",  "#64844A" },
  { "117", "strong yellow green",         "7.5", "GY", "5","8",  "7.5GY 5/8",  "#5A8634" },
  { "117", "strong yellow green",         "7.5", "GY", "5","10", "7.5GY 5/10", "#4F8913" },
  { "122", "grayish yellow green",        "7.5", "GY", "6","2",  "7.5GY 6/2",  "#8F9787" },
  { "120", "moderate yellow green",       "7.5", "GY", "6","4",  "7.5GY 6/4",  "#879B75" },
  { "120", "moderate yellow green",       "7.5", "GY", "6","6",  "7.5GY 6/6",  "#7F9E62" },
  { "117", "strong yellow green",         "7.5", "GY", "6","8",  "7.5GY 6/8",  "#75A14B" },
  { "117", "strong yellow green",         "7.5", "GY", "6","10", "7.5GY 6/10", "#6BA431" },
  { "115", "vivid yellow green",          "7.5", "GY", "6","12", "7.5GY 6/12", "#61A603" },
  { "120", "moderate yellow green",       "7.5", "GY", "7","4",  "7.5GY 7/4",  "#A2B58E" },
  { "120", "moderate yellow green",       "7.5", "GY", "7","6",  "7.5GY 7/6",  "#9AB97B" },
  { "117", "strong yellow green",         "7.5", "GY", "7","8",  "7.5GY 7/8",  "#91BC66" },
  { "117", "strong yellow green",         "7.5", "GY", "7","10", "7.5GY 7/10", "#87BF4D" },
  { "119", "light yellow green",          "7.5", "GY", "8","6",  "7.5GY 8/6",  "#B4D492" },
  { "116", "brilliant yellow green",      "7.5", "GY", "8","10", "7.5GY 8/10", "#A3DA65" },
  { "119", "light yellow green",          "7.5", "GY", "9","6",  "7.5GY 9/6",  "#CFF0A7" },
  { "116", "brilliant yellow green",      "7.5", "GY", "9","10", "7.5GY 9/10", "#BDF677" },
  { "115", "vivid yellow green",          "7.5", "GY", "9","18", "7.5GY 9/18", "#97FF00" },

  { "138", "very dark yellowish green",   "10", "GY", "1","4",   "10GY 1/4",   "#00220D" },
  { "138", "very dark yellowish green",   "10", "GY", "1","6",   "10GY 1/6",   "#002501" },
  { "138", "very dark yellowish green",   "10", "GY", "2","4",   "10GY 2/4",   "#1E3622" },
  { "138", "very dark yellowish green",   "10", "GY", "2","6",   "10GY 2/6",   "#0A3917" },
  { "151", "dark grayish green",          "10", "GY", "3","2",   "10GY 3/2",   "#404B41" },
  { "137", "dark yellowish green",        "10", "GY", "3","4",   "10GY 3/4",   "#334E35" },
  { "137", "dark yellowish green",        "10", "GY", "3","6",   "10GY 3/6",   "#245129" },
  { "137", "dark yellowish green",        "10", "GY", "4","4",   "10GY 4/4",   "#4C684B" },
  { "137", "dark yellowish green",        "10", "GY", "4","6",   "10GY 4/6",   "#3C6B3C" },
  { "132", "deep yellowish green",        "10", "GY", "4","14",  "10GY 4/14",  "#007600" },
  { "136", "moderate yellowish green",    "10", "GY", "5","4",   "10GY 5/4",   "#668265" },
  { "136", "moderate yellowish green",    "10", "GY", "5","6",   "10GY 5/6",   "#588556" },
  { "131", "strong yellowish green",      "10", "GY", "5","8",   "10GY 5/8",   "#488946" },
  { "131", "strong yellowish green",      "10", "GY", "5","10",  "10GY 5/10",  "#338C34" },
  { "136", "moderate yellowish green",    "10", "GY", "6","4",   "10GY 6/4",   "#7F9C7D" },
  { "136", "moderate yellowish green",    "10", "GY", "6","6",   "10GY 6/6",   "#72A06E" },
  { "131", "strong yellowish green",      "10", "GY", "6","8",   "10GY 6/8",   "#63A35D" },
  { "131", "strong yellowish green",      "10", "GY", "6","10",  "10GY 6/10",  "#52A74D" },
  { "135", "light yellowish green",       "10", "GY", "7","4",   "10GY 7/4",   "#99B796" },
  { "135", "light yellowish green",       "10", "GY", "7","6",   "10GY 7/6",   "#8DBB87" },
  { "130", "brilliant yellowish green",   "10", "GY", "7","8",   "10GY 7/8",   "#7FBE77" },
  { "130", "brilliant yellowish green",   "10", "GY", "7","10",  "10GY 7/10",  "#6FC266" },
  { "129", "vivid yellowish green",       "10", "GY", "7","16",  "10GY 7/16",  "#14CB22" },
  { "135", "light yellowish green",       "10", "GY", "8","4",   "10GY 8/4",   "#B4D2AF" },
  { "135", "light yellowish green",       "10", "GY", "8","6",   "10GY 8/6",   "#A6D69E" },
  { "130", "brilliant yellowish green",   "10", "GY", "8","8",   "10GY 8/8",   "#99DA8F" },
  { "130", "brilliant yellowish green",   "10", "GY", "8","10",  "10GY 8/10",  "#89DD7D" },
  { "129", "vivid yellowish green",       "10", "GY", "8","24",  "10GY 8/24",  "#00F000" },
  { "134", "very light yellowish green",  "10", "GY", "9","4",   "10GY 9/4",   "#CEEEC8" },
  { "134", "very light yellowish green",  "10", "GY", "9","6",   "10GY 9/6",   "#BEF3B5" },
  { "130", "brilliant yellowish green",   "10", "GY", "9","8",   "10GY 9/8",   "#B1F7A5" },
  { "130", "brilliant yellowish green",   "10", "GY", "9","10",  "10GY 9/10",  "#A1FA91" },

  // Green

  { "133", "very deep yellowish green",   "2.5", "G", "2","8",   "2.5G 2/8",   "#003C1D" },
  { "133", "very deep yellowish green",   "2.5", "G", "2","10",  "2.5G 2/10",  "#003F17" },
  { "133", "very deep yellowish green",   "2.5", "G", "2","12",  "2.5G 2/12",  "#004012" },
  { "133", "very deep yellowish green",   "2.5", "G", "2","14",  "2.5G 2/14",  "#00420F" },
  { "132", "deep yellowish green",        "2.5", "G", "3","12",  "2.5G 3/12",  "#005A24" },
  { "132", "deep yellowish green",        "2.5", "G", "3","18",  "2.5G 3/18",  "#005E18" },
  { "132", "deep yellowish green",        "2.5", "G", "4","10",  "2.5G 4/10",  "#00733C" },
  { "132", "deep yellowish green",        "2.5", "G", "4","22",  "2.5G 4/22",  "#007E1E" },
  { "150", "grayish green",               "2.5", "G", "5","2",   "2.5G 5/2",   "#6E7E76" },
  { "129", "vivid yellowish green",       "2.5", "G", "5","24",  "2.5G 5/24",  "#009D25" },
  { "129", "vivid yellowish green",       "2.5", "G", "6","20",  "2.5G 6/20",  "#00B740" },
  { "149", "pale green",                  "2.5", "G", "7","2",   "2.5G 7/2",   "#A1B3AA" },
  { "129", "vivid yellowish green",       "2.5", "G", "8","14",  "2.5G 8/14",  "#00E784" },
  { "148", "very pale green",             "2.5", "G", "9","2",   "2.5G 9/2",   "#D6EADF" },
  { "134", "very light yellowish green",  "2.5", "G", "9","4",   "2.5G 9/4",   "#C4EFD3" },
  { "134", "very light yellowish green",  "2.5", "G", "9","6",   "2.5G 9/6",   "#AFF5C5" },

  { "147", "very dark green",             "5", "G", "1","4",     "5G 1/4",     "#002216" },
  { "147", "very dark green",             "5", "G", "1","6",     "5G 1/6",     "#002613" },
  { "146", "dark green",                  "5", "G", "3","4",     "5G 3/4",     "#284F41" },
  { "146", "dark green",                  "5", "G", "3","6",     "5G 3/6",     "#00533D" },
  { "139", "vivid green",                 "5", "G", "3","22",    "5G 3/22",    "#00612F" },
  { "150", "grayish green",               "5", "G", "4","2",     "5G 4/2",     "#54645F" },
  { "139", "vivid green",                 "5", "G", "5","28",    "5G 5/28",    "#00A052" },
  { "149", "pale green",                  "5", "G", "6","2",     "5G 6/2",     "#859993" },
  { "140", "brilliant green",             "5", "G", "6","10",    "5G 6/10",    "#00A97C" },
  { "139", "vivid green",                 "5", "G", "7","26",    "5G 7/26",    "#00DB7B" },
  { "148", "very pale green",             "5", "G", "8","2",     "5G 8/2",     "#B9CEC9" },
  { "140", "brilliant green",             "5", "G", "8","8",     "5G 8/8",     "#75DDB3" },
  { "140", "brilliant green",             "5", "G", "9","10",    "5G 9/10",    "#69FFC6" },

  { "147", "very dark green",             "7.5", "G", "1","4",   "7.5G 1/4",   "#00221A" },
  { "147", "very dark green",             "7.5", "G", "1","6",   "7.5G 1/6",   "#002618" },
  { "142", "deep green",                  "7.5", "G", "2","8",   "7.5G 2/8",   "#003D2B" },
  { "142", "deep green",                  "7.5", "G", "2","10",  "7.5G 2/10",  "#003F2A" },
  { "146", "dark green",                  "7.5", "G", "3","4",   "7.5G 3/4",   "#245044" },
  { "146", "dark green",                  "7.5", "G", "3","6",   "7.5G 3/6",   "#005342" },
  { "142", "deep green",                  "7.5", "G", "3","8",   "7.5G 3/8",   "#005641" },
  { "142", "deep green",                  "7.5", "G", "3","10",  "7.5G 3/10",  "#00583F" },
  { "145", "moderate green",              "7.5", "G", "4","4",   "7.5G 4/4",   "#3C695D" },
  { "145", "moderate green",              "7.5", "G", "4","6",   "7.5G 4/6",   "#166D5A" },
  { "141", "strong green",                "7.5", "G", "4","8",   "7.5G 4/8",   "#007157" },
  { "141", "strong green",                "7.5", "G", "4","10",  "7.5G 4/10",  "#007455" },
  { "139", "vivid green",                 "7.5", "G", "4","26",  "7.5G 4/26",  "#008150" },
  { "145", "moderate green",              "7.5", "G", "5","4",   "7.5G 5/4",   "#568377" },
  { "145", "moderate green",              "7.5", "G", "5","6",   "7.5G 5/6",   "#3C8773" },
  { "141", "strong green",                "7.5", "G", "5","8",   "7.5G 5/8",   "#008B70" },
  { "141", "strong green",                "7.5", "G", "5","10",  "7.5G 5/10",  "#008F6E" },
  { "144", "light green",                 "7.5", "G", "6","4",   "7.5G 6/4",   "#6E9E90" },
  { "144", "light green",                 "7.5", "G", "6","6",   "7.5G 6/6",   "#56A28D" },
  { "139", "vivid green",                 "7.5", "G", "6","26",  "7.5G 6/26",  "#00BE79" },
  { "144", "light green",                 "7.5", "G", "7","4",   "7.5G 7/4",   "#89B8AB" },
  { "144", "light green",                 "7.5", "G", "7","6",   "7.5G 7/6",   "#73BDA7" },
  { "140", "brilliant green",             "7.5", "G", "7","8",   "7.5G 7/8",   "#55C1A3" },
  { "143", "very light green",            "7.5", "G", "8","4",   "7.5G 8/4",   "#A3D4C6" },
  { "143", "very light green",            "7.5", "G", "8","6",   "7.5G 8/6",   "#8AD9C1" },
  { "140", "brilliant green",             "7.5", "G", "8","10",  "7.5G 8/10",  "#46E1B9" },
  { "143", "very light green",            "7.5", "G", "9","4",   "7.5G 9/4",   "#BBF0E1" },
  { "143", "very light green",            "7.5", "G", "9","6",   "7.5G 9/6",   "#9FF6DB" },

  { "166", "very dark bluish green",      "10", "G", "1","4",    "10G 1/4",    "#00231D" },
  { "166", "very dark bluish green",      "10", "G", "1","6",    "10G 1/6",    "#00261D" },
  { "161", "deep bluish green",           "10", "G", "2","10",   "10G 2/10",   "#003F31" },
  { "165", "dark bluish green",           "10", "G", "3","4",    "10G 3/4",    "#205048" },
  { "165", "dark bluish green",           "10", "G", "3","6",    "10G 3/6",    "#005348" },
  { "158", "vivid bluish green",          "10", "G", "4","22",   "10G 4/22",   "#007E5F" },
  { "150", "grayish green",               "10", "G", "5","2",    "10G 5/2",    "#6A7E7D" },
  { "164", "moderate bluish green",       "10", "G", "5","4",    "10G 5/4",    "#54837B" },
  { "160", "strong bluish green",         "10", "G", "5","8",    "10G 5/8",    "#008B78" },
  { "149", "pale green",                  "10", "G", "7","2",    "10G 7/2",    "#9DB3B3" },
  { "163", "light bluish green",          "10", "G", "7","4",    "10G 7/4",    "#87B8B0" },
  { "159", "brilliant bluish green",      "10", "G", "7","8",    "10G 7/8",    "#4DC1AB" },
  { "159", "brilliant bluish green",      "10", "G", "8","10",   "10G 8/10",   "#37E1C3" },
  { "148", "very pale green",             "10", "G", "9","2",    "10G 9/2",    "#D1EAEA" },
  { "162", "very light bluish green",     "10", "G", "9","4",    "10G 9/4",    "#B9F0E6" },

  // Blue Green

  { "161", "deep bluish green",           "2.5", "BG", "2","10", "2.5BG 2/10", "#003F3A" },
  { "151", "dark grayish green",          "2.5", "BG", "3","2",  "2.5BG 3/2",  "#394B4C" },
  { "158", "vivid bluish green",          "2.5", "BG", "3","20", "2.5BG 3/20", "#006057" },
  { "164", "moderate bluish green",       "2.5", "BG", "5","4",  "2.5BG 5/4",  "#52837F" },
  { "160", "strong bluish green",         "2.5", "BG", "5","8",  "2.5BG 5/8",  "#008B80" },
  { "158", "vivid bluish green",          "2.5", "BG", "5","22", "2.5BG 5/22", "#009D89" },
  { "158", "vivid bluish green",          "2.5", "BG", "6","20", "2.5BG 6/20", "#00B89F" },
  { "163", "light bluish green",          "2.5", "BG", "7","4",  "2.5BG 7/4",  "#86B8B4" },
  { "158", "vivid bluish green",          "2.5", "BG", "7","20", "2.5BG 7/20", "#00D5B7" },
  { "162", "very light bluish green",     "2.5", "BG", "9","4",  "2.5BG 9/4",  "#B7F0EB" },

  { "166", "very dark bluish green",      "5", "BG", "1","4",    "5BG 1/4",    "#002226" },
  { "166", "very dark bluish green",      "5", "BG", "1","6",    "5BG 1/6",    "#00252B" },
  { "161", "deep bluish green",           "5", "BG", "2","8",    "5BG 2/8",    "#003D42" },
  { "151", "dark grayish green",          "5", "BG", "3","2",    "5BG 3/2",    "#384B4E" },
  { "165", "dark bluish green",           "5", "BG", "3","4",    "5BG 3/4",    "#185052" },
  { "165", "dark bluish green",           "5", "BG", "3","6",    "5BG 3/6",    "#005356" },
  { "161", "deep bluish green",           "5", "BG", "3","10",   "5BG 3/10",   "#00585D" },
  { "164", "moderate bluish green",       "5", "BG", "4","6",    "5BG 4/6",    "#006D6F" },
  { "160", "strong bluish green",         "5", "BG", "4","10",   "5BG 4/10",   "#007376" },
  { "150", "grayish green",               "5", "BG", "5","2",    "5BG 5/2",    "#697E82" },
  { "163", "light bluish green",          "5", "BG", "6","6",    "5BG 6/6",    "#48A2A2" },
  { "159", "brilliant bluish green",      "5", "BG", "6","10",   "5BG 6/10",   "#00AAA7" },
  { "149", "pale green",                  "5", "BG", "7","2",    "5BG 7/2",    "#9CB3B8" },
  { "162", "very light bluish green",     "5", "BG", "8","6",    "5BG 8/6",    "#81D8D6" },
  { "159", "brilliant bluish green",      "5", "BG", "8","8",    "5BG 8/8",    "#59DDD8" },
  { "148", "very pale green",             "5", "BG", "9","2",    "5BG 9/2",    "#D0EAEF" },
  { "159", "brilliant bluish green",      "5", "BG", "9","10",   "5BG 9/10",   "#25FFF5" },

  { "151", "dark grayish green",          "7.5", "BG", "3","2",  "7.5BG 3/2",  "#374B51" },
  { "161", "deep bluish green",           "7.5", "BG", "3","8",  "7.5BG 3/8",  "#005561" },
  { "150", "grayish green",               "7.5", "BG", "4","2",  "7.5BG 4/2",  "#4F646A" },
  { "164", "moderate bluish green",       "7.5", "BG", "4","4",  "7.5BG 4/4",  "#336970" },
  { "160", "strong bluish green",         "7.5", "BG", "4","8",  "7.5BG 4/8",  "#00707B" },
  { "164", "moderate bluish green",       "7.5", "BG", "5","6",  "7.5BG 5/6",  "#28878E" },
  { "160", "strong bluish green",         "7.5", "BG", "5","10", "7.5BG 5/10", "#008E9A" },
  { "149", "pale green",                  "7.5", "BG", "6","2",  "7.5BG 6/2",  "#8298A0" },
  { "163", "light bluish green",          "7.5", "BG", "6","4",  "7.5BG 6/4",  "#679DA4" },
  { "163", "light bluish green",          "7.5", "BG", "7","6",  "7.5BG 7/6",  "#67BCC3" },
  { "148", "very pale green",             "7.5", "BG", "8","2",  "7.5BG 8/2",  "#B7CDD6" },
  { "162", "very light bluish green",     "7.5", "BG", "8","4",  "7.5BG 8/4",  "#9ED3DA" },
  { "162", "very light bluish green",     "7.5", "BG", "9","6",  "7.5BG 9/6",  "#92F5FB" },

  // Blue

  { "175", "very dark greenish blue",     "2.5", "B", "1","4",   "2.5B 1/4",   "#002131" },
  { "175", "very dark greenish blue",     "2.5", "B", "1","6",   "2.5B 1/6",   "#00243B" },
  { "170", "deep greenish blue",          "2.5", "B", "2","8",   "2.5B 2/8",   "#003B59" },
  { "174", "dark greenish blue",          "2.5", "B", "3","4",   "2.5B 3/4",   "#154E60" },
  { "174", "dark greenish blue",          "2.5", "B", "3","6",   "2.5B 3/6",   "#005169" },
  { "170", "deep greenish blue",          "2.5", "B", "3","8",   "2.5B 3/8",   "#005473" },
  { "173", "moderate greenish blue",      "2.5", "B", "4","6",   "2.5B 4/6",   "#006A82" },
  { "169", "strong greenish blue",        "2.5", "B", "4","10",  "2.5B 4/10",  "#007197" },
  { "186", "grayish blue",                "2.5", "B", "5","2",   "2.5B 5/2",   "#6A7D89" },
  { "173", "moderate greenish blue",      "2.5", "B", "5","6",   "2.5B 5/6",   "#2B859C" },
  { "169", "strong greenish blue",        "2.5", "B", "5","10",  "2.5B 5/10",  "#008BB0" },
  { "172", "light greenish blue",         "2.5", "B", "6","6",   "2.5B 6/6",   "#49A0B7" },
  { "185", "pale blue",                   "2.5", "B", "7","2",   "2.5B 7/2",   "#9FB2BF" },
  { "172", "light greenish blue",         "2.5", "B", "7","6",   "2.5B 7/6",   "#69BAD1" },
  { "171", "very light greenish blue",    "2.5", "B", "8","6",   "2.5B 8/6",   "#80D6ED" },
  { "184", "very pale blue",              "2.5", "B", "9","2",   "2.5B 9/2",   "#D5E8F7" },

  { "175", "very dark greenish blue",     "5", "B", "1","4",     "5B 1/4",     "#002034" },
  { "175", "very dark greenish blue",     "5", "B", "1","6",     "5B 1/6",     "#00223E" },
  { "170", "deep greenish blue",          "5", "B", "2","10",    "5B 2/10",    "#003C6D" },
  { "174", "dark greenish blue",          "5", "B", "3","4",     "5B 3/4",     "#1A4D63" },
  { "174", "dark greenish blue",          "5", "B", "3","6",     "5B 3/6",     "#00506E" },
  { "170", "deep greenish blue",          "5", "B", "3","10",    "5B 3/10",    "#005486" },
  { "186", "grayish blue",                "5", "B", "4","2",     "5B 4/2",     "#516370" },
  { "173", "moderate greenish blue",      "5", "B", "4","4",     "5B 4/4",     "#39667C" },
  { "169", "strong greenish blue",        "5", "B", "4","8",     "5B 4/8",     "#006C94" },
  { "173", "moderate greenish blue",      "5", "B", "5","4",     "5B 5/4",     "#558096" },
  { "169", "strong greenish blue",        "5", "B", "5","8",     "5B 5/8",     "#0086AE" },
  { "167", "vivid greenish blue",         "5", "B", "5","12",    "5B 5/12",    "#008BC7" },
  { "167", "vivid greenish blue",         "5", "B", "5","16",    "5B 5/16",    "#0091E0" },
  { "185", "pale blue",                   "5", "B", "6","2",     "5B 6/2",     "#8697A5" },
  { "172", "light greenish blue",         "5", "B", "6","4",     "5B 6/4",     "#709AB0" },
  { "167", "vivid greenish blue",         "5", "B", "6","14",    "5B 6/14",    "#00A9EF" },
  { "172", "light greenish blue",         "5", "B", "7","4",     "5B 7/4",     "#8AB5CC" },
  { "167", "vivid greenish blue",         "5", "B", "7","12",    "5B 7/12",    "#00C2FF" },
  { "184", "very pale blue",              "5", "B", "8","2",     "5B 8/2",     "#BDCBDC" },
  { "171", "very light greenish blue",    "5", "B", "8","4",     "5B 8/4",     "#A4D0E8" },
  { "171", "very light greenish blue",    "5", "B", "9","4",     "5B 9/4",     "#BEECFF" },

  { "175", "very dark greenish blue",     "7.5", "B", "1","4",   "7.5B 1/4",   "#001F36" },
  { "175", "very dark greenish blue",     "7.5", "B", "1","6",   "7.5B 1/6",   "#002141" },
  { "170", "deep greenish blue",          "7.5", "B", "2","8",   "7.5B 2/8",   "#003865" },
  { "174", "dark greenish blue",          "7.5", "B", "3","4",   "7.5B 3/4",   "#214C66" },
  { "174", "dark greenish blue",          "7.5", "B", "3","6",   "7.5B 3/6",   "#004E72" },
  { "170", "deep greenish blue",          "7.5", "B", "3","8",   "7.5B 3/8",   "#00507F" },
  { "173", "moderate greenish blue",      "7.5", "B", "4","6",   "7.5B 4/6",   "#20678B" },
  { "169", "strong greenish blue",        "7.5", "B", "4","10",  "7.5B 4/10",  "#006BA5" },
  { "167", "vivid greenish blue",         "7.5", "B", "4","12",  "7.5B 4/12",  "#006DB2" },
  { "186", "grayish blue",                "7.5", "B", "5","2",   "7.5B 5/2",   "#6E7C8C" },
  { "173", "moderate greenish blue",      "7.5", "B", "5","6",   "7.5B 5/6",   "#4181A5" },
  { "169", "strong greenish blue",        "7.5", "B", "5","10",  "7.5B 5/10",  "#0086C0" },
  { "172", "light greenish blue",         "7.5", "B", "6","6",   "7.5B 6/6",   "#5D9CC0" },
  { "168", "brilliant greenish blue",     "7.5", "B", "6","8",   "7.5B 6/8",   "#399ECE" },
  { "168", "brilliant greenish blue",     "7.5", "B", "6","10",  "7.5B 6/10",  "#00A0DB" },
  { "185", "pale blue",                   "7.5", "B", "7","2",   "7.5B 7/2",   "#A4B0C1" },
  { "172", "light greenish blue",         "7.5", "B", "7","6",   "7.5B 7/6",   "#78B7DB" },
  { "168", "brilliant greenish blue",     "7.5", "B", "7","8",   "7.5B 7/8",   "#58B9E9" },
  { "168", "brilliant greenish blue",     "7.5", "B", "7","10",  "7.5B 7/10",  "#00BCF9" },
  { "171", "very light greenish blue",    "7.5", "B", "8","6",   "7.5B 8/6",   "#8FD2F9" },
  { "168", "brilliant greenish blue",     "7.5", "B", "8","8",   "7.5B 8/8",   "#71D5FF" },
  { "184", "very pale blue",              "7.5", "B", "9","2",   "7.5B 9/2",   "#DBE6F9" },

  { "183", "dark blue",                   "10", "B", "1","6",    "10B 1/6",    "#002042" },
  { "183", "dark blue",                   "10", "B", "2","6",    "10B 2/6",    "#00355A" },
  { "179", "deep blue",                   "10", "B", "2","10",   "10B 2/10",   "#003875" },
  { "186", "grayish blue",                "10", "B", "4","2",    "10B 4/2",    "#566272" },
  { "182", "moderate blue",               "10", "B", "4","6",    "10B 4/6",    "#2F658D" },
  { "182", "moderate blue",               "10", "B", "4","8",    "10B 4/8",    "#00679B" },
  { "178", "strong blue",                 "10", "B", "4","12",   "10B 4/12",   "#006AB6" },
  { "182", "moderate blue",               "10", "B", "5","6",    "10B 5/6",    "#4D7FA8" },
  { "182", "moderate blue",               "10", "B", "5","8",    "10B 5/8",    "#2C81B5" },
  { "178", "strong blue",                 "10", "B", "5","12",   "10B 5/12",   "#0085D2" },
  { "176", "vivid blue",                  "10", "B", "5","14",   "10B 5/14",   "#0087E0" },
  { "185", "pale blue",                   "10", "B", "6","2",    "10B 6/2",    "#8B95A7" },
  { "181", "light blue",                  "10", "B", "6","6",    "10B 6/6",    "#689AC3" },
  { "181", "light blue",                  "10", "B", "6","8",    "10B 6/8",    "#4E9BD1" },
  { "177", "brilliant blue",              "10", "B", "6","12",   "10B 6/12",   "#009FEF" },
  { "176", "vivid blue",                  "10", "B", "6","16",   "10B 6/16",   "#00A3FF" },
  { "181", "light blue",                  "10", "B", "7","6",    "10B 7/6",    "#83B4DE" },
  { "181", "light blue",                  "10", "B", "7","8",    "10B 7/8",    "#6AB6ED" },
  { "177", "brilliant blue",              "10", "B", "7","12",   "10B 7/12",   "#00BAFF" },
  { "184", "very pale blue",              "10", "B", "8","2",    "10B 8/2",    "#C2CADD" },
  { "180", "very light blue",             "10", "B", "8","6",    "10B 8/6",    "#9BCFFC" },
  { "180", "very light blue",             "10", "B", "8","8",    "10B 8/8",    "#88D2FF" },
  { "184", "very pale blue",              "10", "B", "10","4",   "10B 9/4",    "#CDE8FF" },

  // Purple Blue

  { "183", "dark blue",                   "2.5", "PB", "1","6",  "2.5PB 1/6",  "#001D44" },
  { "183", "dark blue",                   "2.5", "PB", "2","6",  "2.5PB 2/6",  "#00335C" },
  { "179", "deep blue",                   "2.5", "PB", "2","10", "2.5PB 2/10", "#003477" },
  { "178", "strong blue",                 "2.5", "PB", "4","12", "2.5PB 4/12", "#0065B8" },
  { "176", "vivid blue",                  "2.5", "PB", "4","18", "2.5PB 4/18", "#0068E4" },
  { "186", "grayish blue",                "2.5", "PB", "5","4",  "2.5PB 5/4",  "#687C9B" },
  { "178", "strong blue",                 "2.5", "PB", "5","12", "2.5PB 5/12", "#0080D5" },
  { "177", "brilliant blue",              "2.5", "PB", "6","12", "2.5PB 6/12", "#149AF3" },
  { "185", "pale blue",                   "2.5", "PB", "7","4",  "2.5PB 7/4",  "#9DB0D1" },
  { "180", "very light blue",             "2.5", "PB", "8","6",  "2.5PB 8/6",  "#A7CCFF" },

  { "183", "dark blue",                   "5", "PB", "1","4",    "5PB 1/4",    "#0C1C38" },
  { "179", "deep blue",                   "5", "PB", "1","8",    "5PB 1/8",    "#001A51" },
  { "179", "deep blue",                   "5", "PB", "1","10",   "5PB 1/10",   "#00195E" },
  { "183", "dark blue",                   "5", "PB", "2","4",    "5PB 2/4",    "#23304F" },
  { "179", "deep blue",                   "5", "PB", "2","8",    "5PB 2/8",    "#00306A" },
  { "176", "vivid blue",                  "5", "PB", "3","14",   "5PB 3/14",   "#0047AD" },
  { "178", "strong blue",                 "5", "PB", "4","10",   "5PB 4/10",   "#2661AB" },
  { "176", "vivid blue",                  "5", "PB", "4","14",   "5PB 4/14",   "#0061C7" },
  { "178", "strong blue",                 "5", "PB", "5","10",   "5PB 5/10",   "#487BC7" },
  { "176", "vivid blue",                  "5", "PB", "5","18",   "5PB 5/18",   "#007BFF" },
  { "177", "brilliant blue",              "5", "PB", "6","10",   "5PB 6/10",   "#6595E3" },
  { "177", "brilliant blue",              "5", "PB", "7","10",   "5PB 7/10",   "#7DB0FF" },

  { "201", "dark purplish blue",          "7.5", "PB", "1","4",  "7.5PB 1/4",  "#191938" },
  { "201", "dark purplish blue",          "7.5", "PB", "1","6",  "7.5PB 1/6",  "#171643" },
  { "197", "deep purplish blue",          "7.5", "PB", "1","8",  "7.5PB 1/8",  "#17134E" },
  { "197", "deep purplish blue",          "7.5", "PB", "1","10", "7.5PB 1/10", "#190D57" },
  { "194", "vivid purplish blue",         "7.5", "PB", "1","28", "7.5PB 1/28", "#3C00A6" },
  { "197", "deep purplish blue",          "7.5", "PB", "2","8",  "7.5PB 2/8",  "#262A68" },
  { "197", "deep purplish blue",          "7.5", "PB", "2","10", "7.5PB 2/10", "#252774" },
  { "194", "vivid purplish blue",         "7.5", "PB", "2","18", "7.5PB 2/18", "#2C079C" },
  { "194", "vivid purplish blue",         "7.5", "PB", "2","36", "7.5PB 2/36", "#4F00EE" },
  { "204", "grayish purplish blue",       "7.5", "PB", "3","4",  "7.5PB 3/4",  "#43456A" },
  { "194", "vivid purplish blue",         "7.5", "PB", "3","28", "7.5PB 3/28", "#4000F0" },
  { "204", "grayish purplish blue",       "7.5", "PB", "4","4",  "7.5PB 4/4",  "#5D5E82" },
  { "200", "moderate purplish blue",      "7.5", "PB", "4","6",  "7.5PB 4/6",  "#595D8F" },
  { "200", "moderate purplish blue",      "7.5", "PB", "4","8",  "7.5PB 4/8",  "#545B9D" },
  { "196", "strong purplish blue",        "7.5", "PB", "4","10", "7.5PB 4/10", "#4F5AAA" },
  { "196", "strong purplish blue",        "7.5", "PB", "4","12", "7.5PB 4/12", "#4A58B7" },
  { "194", "vivid purplish blue",         "7.5", "PB", "4","24", "7.5PB 4/24", "#3F3EFF" },
  { "203", "pale purplish blue",          "7.5", "PB", "5","4",  "7.5PB 5/4",  "#76789B" },
  { "199", "light purplish blue",         "7.5", "PB", "5","6",  "7.5PB 5/6",  "#7277AA" },
  { "199", "light purplish blue",         "7.5", "PB", "5","8",  "7.5PB 5/8",  "#6E75B8" },
  { "195", "brilliant purplish blue",     "7.5", "PB", "5","10", "7.5PB 5/10", "#6974C5" },
  { "196", "strong purplish blue",        "7.5", "PB", "5","12", "7.5PB 5/12", "#6373D4" },
  { "203", "pale purplish blue",          "7.5", "PB", "6","4",  "7.5PB 6/4",  "#9192B6" },
  { "199", "light purplish blue",         "7.5", "PB", "6","6",  "7.5PB 6/6",  "#8D91C5" },
  { "199", "light purplish blue",         "7.5", "PB", "6","8",  "7.5PB 6/8",  "#8890D3" },
  { "195", "brilliant purplish blue",     "7.5", "PB", "6","10", "7.5PB 6/10", "#838FE1" },
  { "195", "brilliant purplish blue",     "7.5", "PB", "6","12", "7.5PB 6/12", "#7C8DF1" },
  { "203", "pale purplish blue",          "7.5", "PB", "7","4",  "7.5PB 7/4",  "#ABACD1" },
  { "199", "light purplish blue",         "7.5", "PB", "7","6",  "7.5PB 7/6",  "#A7ABE1" },
  { "199", "light purplish blue",         "7.5", "PB", "7","8",  "7.5PB 7/8",  "#A1AAF1" },
  { "195", "brilliant purplish blue",     "7.5", "PB", "7","10", "7.5PB 7/10", "#9BA9FF" },
  { "202", "very pale purplish blue",     "7.5", "PB", "8","4",  "7.5PB 8/4",  "#C6C6EE" },
  { "198", "very light purplish blue",    "7.5", "PB", "8","6",  "7.5PB 8/6",  "#C0C6FF" },

  { "212", "dark violet",                 "10", "PB", "1","4",   "10PB 1/4",   "#211637" },
  { "212", "dark violet",                 "10", "PB", "1","6",   "10PB 1/6",   "#231341" },
  { "208", "deep violet",                 "10", "PB", "1","8",   "10PB 1/8",   "#250E49" },
  { "208", "deep violet",                 "10", "PB", "1","10",  "10PB 1/10",  "#280651" },
  { "208", "deep violet",                 "10", "PB", "1","12",  "10PB 1/12",  "#2C005A" },
  { "205", "vivid violet",                "10", "PB", "1","30",  "10PB 1/30",  "#5800AD" },
  { "212", "dark violet",                 "10", "PB", "2","4",   "10PB 2/4",   "#352C4D" },
  { "212", "dark violet",                 "10", "PB", "2","6",   "10PB 2/6",   "#362958" },
  { "208", "deep violet",                 "10", "PB", "2","8",   "10PB 2/8",   "#382563" },
  { "208", "deep violet",                 "10", "PB", "2","10",  "10PB 2/10",  "#3A206E" },
  { "208", "deep violet",                 "10", "PB", "2","12",  "10PB 2/12",  "#3D1A77" },
  { "205", "vivid violet",                "10", "PB", "2","28",  "10PB 2/28",  "#6000C1" },
  { "215", "grayish violet",              "10", "PB", "3","4",   "10PB 3/4",   "#4B4268" },
  { "211", "moderate violet",             "10", "PB", "3","6",   "10PB 3/6",   "#4C4074" },
  { "211", "moderate violet",             "10", "PB", "3","8",   "10PB 3/8",   "#4E3D80" },
  { "207", "strong violet",               "10", "PB", "3","10",  "10PB 3/10",  "#4F398C" },
  { "207", "strong violet",               "10", "PB", "3","12",  "10PB 3/12",  "#513596" },
  { "205", "vivid violet",                "10", "PB", "3","24",  "10PB 3/24",  "#6700D2" },
  { "215", "grayish violet",              "10", "PB", "4","4",   "10PB 4/4",   "#645C81" },
  { "211", "moderate violet",             "10", "PB", "4","6",   "10PB 4/6",   "#64598D" },
  { "211", "moderate violet",             "10", "PB", "4","8",   "10PB 4/8",   "#65579A" },
  { "207", "strong violet",               "10", "PB", "4","10",  "10PB 4/10",  "#6654A6" },
  { "207", "strong violet",               "10", "PB", "4","12",  "10PB 4/12",  "#6750B2" },
  { "205", "vivid violet",                "10", "PB", "4","18",  "10PB 4/18",  "#6E42D3" },
  { "214", "pale violet",                 "10", "PB", "5","4",   "10PB 5/4",   "#7D769A" },
  { "210", "light violet",                "10", "PB", "5","6",   "10PB 5/6",   "#7E73A8" },
  { "210", "light violet",                "10", "PB", "5","8",   "10PB 5/8",   "#7E71B5" },
  { "206", "brilliant violet",            "10", "PB", "5","10",  "10PB 5/10",  "#7E6FC1" },
  { "206", "brilliant violet",            "10", "PB", "5","12",  "10PB 5/12",  "#7F6CCE" },
  { "214", "pale violet",                 "10", "PB", "6","4",   "10PB 6/4",   "#9790B5" },
  { "210", "light violet",                "10", "PB", "6","6",   "10PB 6/6",   "#978EC3" },
  { "210", "light violet",                "10", "PB", "6","8",   "10PB 6/8",   "#978CD0" },
  { "206", "brilliant violet",            "10", "PB", "6","10",  "10PB 6/10",  "#9789DD" },
  { "206", "brilliant violet",            "10", "PB", "6","12",  "10PB 6/12",  "#9787EC" },
  { "227", "pale purple",                 "10", "PB", "7","2",   "10PB 7/2",   "#B2ACC2" },
  { "214", "pale violet",                 "10", "PB", "7","4",   "10PB 7/4",   "#B2AAD1" },
  { "210", "light violet",                "10", "PB", "7","6",   "10PB 7/6",   "#B1A8DF" },
  { "210", "light violet",                "10", "PB", "7","8",   "10PB 7/8",   "#B1A6EE" },
  { "206", "brilliant violet",            "10", "PB", "7","10",  "10PB 7/10",  "#B0A4FE" },
  { "206", "brilliant violet",            "10", "PB", "7","12",  "10PB 7/12",  "#B5A0FF" },
  { "226", "very pale purple",            "10", "PB", "8","2",   "10PB 8/2",   "#CEC6DC" },
  { "213", "very pale violet",            "10", "PB", "8","4",   "10PB 8/4",   "#CCC5EC" },
  { "209", "very light violet",           "10", "PB", "8","6",   "10PB 8/6",   "#CBC3FD" },
  { "209", "very light violet",           "10", "PB", "8","8",   "10PB 8/8",   "#D2BFFF" },
  { "226", "very pale purple",            "10", "PB", "9","2",   "10PB 9/2",   "#EAE1F8" },
  { "213", "very pale violet",            "10", "PB", "9","4",   "10PB 9/4",   "#EDDFFF" },

  // Purple

  { "215", "grayish violet",              "2.5", "P", "3","4",   "2.5P 3/4",   "#524166" },
  { "215", "grayish violet",              "2.5", "P", "4","4",   "2.5P 4/4",   "#6A5A7F" },
  { "214", "pale violet",                 "2.5", "P", "5","4",   "2.5P 5/4",   "#837499" },
  { "205", "vivid violet",                "2.5", "P", "5","16",  "2.5P 5/16",  "#9A5BDC" },
  { "214", "pale violet",                 "2.5", "P", "6","4",   "2.5P 6/4",   "#9D8EB3" },
  { "227", "pale purple",                 "2.5", "P", "7","2",   "2.5P 7/2",   "#B5ABC1" },
  { "214", "pale violet",                 "2.5", "P", "7","4",   "2.5P 7/4",   "#B7A8CF" },
  { "213", "very pale violet",            "2.5", "P", "8","4",   "2.5P 8/4",   "#D2C3EB" },
  { "209", "very light violet",           "2.5", "P", "8","6",   "2.5P 8/6",   "#D5C0F9" },
  { "209", "very light violet",           "2.5", "P", "8","8",   "2.5P 8/8",   "#DABDFF" },
  { "213", "very pale violet",            "2.5", "P", "9","4",   "2.5P 9/4",   "#F1DEFF" },

  { "225", "very dark purple",            "5", "P", "1","4",     "5P 1/4",     "#291333" },
  { "225", "very dark purple",            "5", "P", "1","6",     "5P 1/6",     "#2E0D3A" },
  { "220", "very deep purple",            "5", "P", "1","8",     "5P 1/8",     "#330441" },
  { "220", "very deep purple",            "5", "P", "1","10",    "5P 1/10",    "#380048" },
  { "220", "very deep purple",            "5", "P", "1","12",    "5P 1/12",    "#3D004F" },
  { "216", "vivid purple",                "5", "P", "2","20",    "5P 2/20",    "#6A008A" },
  { "224", "dark purple",                 "5", "P", "3","4",     "5P 3/4",     "#573F63" },
  { "224", "dark purple",                 "5", "P", "3","6",     "5P 3/6",     "#5C3B6D" },
  { "219", "deep purple",                 "5", "P", "3","8",     "5P 3/8",     "#623576" },
  { "219", "deep purple",                 "5", "P", "3","10",    "5P 3/10",    "#682F80" },
  { "219", "deep purple",                 "5", "P", "3","12",    "5P 3/12",    "#6D2689" },
  { "216", "vivid purple",                "5", "P", "3","22",    "5P 3/22",    "#8A00B2" },
  { "228", "grayish purple",              "5", "P", "4","4",     "5P 4/4",     "#6F587C" },
  { "227", "pale purple",                 "5", "P", "6","2",     "5P 6/2",     "#9D90A6" },
  { "216", "vivid purple",                "5", "P", "6","14",    "5P 6/14",    "#BE77E7" },
  { "226", "very pale purple",            "5", "P", "8","2",     "5P 8/2",     "#D1C5DB" },
  { "221", "very light purple",           "5", "P", "8","6",     "5P 8/6",     "#DCBEF5" },
  { "221", "very light purple",           "5", "P", "8","8",     "5P 8/8",     "#E2BAFF" },
  { "226", "very pale purple",            "5", "P", "9","2",     "5P 9/2",     "#EDE0F7" },

  { "225", "very dark purple",            "7.5", "P", "1","4",   "7.5P 1/4",   "#2C1231" },
  { "225", "very dark purple",            "7.5", "P", "1","6",   "7.5P 1/6",   "#310B37" },
  { "220", "very deep purple",            "7.5", "P", "1","8",   "7.5P 1/8",   "#36023E" },
  { "220", "very deep purple",            "7.5", "P", "1","10",  "7.5P 1/10",  "#3C0045" },
  { "220", "very deep purple",            "7.5", "P", "1","12",  "7.5P 1/12",  "#41004C" },
  { "224", "dark purple",                 "7.5", "P", "3","4",   "7.5P 3/4",   "#5B3E60" },
  { "224", "dark purple",                 "7.5", "P", "3","6",   "7.5P 3/6",   "#623868" },
  { "219", "deep purple",                 "7.5", "P", "3","8",   "7.5P 3/8",   "#693271" },
  { "219", "deep purple",                 "7.5", "P", "3","10",  "7.5P 3/10",  "#702A79" },
  { "219", "deep purple",                 "7.5", "P", "3","12",  "7.5P 3/12",  "#771F82" },
  { "223", "moderate purple",             "7.5", "P", "4","6",   "7.5P 4/6",   "#7B5281" },
  { "223", "moderate purple",             "7.5", "P", "4","8",   "7.5P 4/8",   "#824D89" },
  { "218", "strong purple",               "7.5", "P", "4","10",  "7.5P 4/10",  "#894791" },
  { "218", "strong purple",               "7.5", "P", "4","12",  "7.5P 4/12",  "#8F4099" },
  { "216", "vivid purple",                "7.5", "P", "4","18",  "7.5P 4/18",  "#A518B3" },
  { "223", "moderate purple",             "7.5", "P", "5","6",   "7.5P 5/6",   "#956D9B" },
  { "223", "moderate purple",             "7.5", "P", "5","8",   "7.5P 5/8",   "#9D67A4" },
  { "218", "strong purple",               "7.5", "P", "5","10",  "7.5P 5/10",  "#A462AC" },
  { "218", "strong purple",               "7.5", "P", "5","12",  "7.5P 5/12",  "#AB5CB5" },
  { "216", "vivid purple",                "7.5", "P", "5","14",  "7.5P 5/14",  "#B354BD" },
  { "222", "light purple",                "7.5", "P", "6","6",   "7.5P 6/6",   "#B087B5" },
  { "222", "light purple",                "7.5", "P", "6","8",   "7.5P 6/8",   "#B882BE" },
  { "217", "brilliant purple",            "7.5", "P", "6","10",  "7.5P 6/10",  "#C07CC7" },
  { "217", "brilliant purple",            "7.5", "P", "6","12",  "7.5P 6/12",  "#C777D0" },
  { "222", "light purple",                "7.5", "P", "7","6",   "7.5P 7/6",   "#CBA1D0" },
  { "222", "light purple",                "7.5", "P", "7","8",   "7.5P 7/8",   "#D49CD9" },
  { "217", "brilliant purple",            "7.5", "P", "7","10",  "7.5P 7/10",  "#DC96E3" },
  { "217", "brilliant purple",            "7.5", "P", "7","12",  "7.5P 7/12",  "#E491EC" },
  { "226", "very pale purple",            "7.5", "P", "8","4",   "7.5P 8/4",   "#DEC0E2" },
  { "221", "very light purple",           "7.5", "P", "8","6",   "7.5P 8/6",   "#E6BCEB" },
  { "221", "very light purple",           "7.5", "P", "8","8",   "7.5P 8/8",   "#F1B6F7" },
  { "217", "brilliant purple",            "7.5", "P", "8","10",  "7.5P 8/10",  "#FAB0FF" },
  { "226", "very pale purple",            "7.5", "P", "9","4",   "7.5P 9/4",   "#FCDAFF" },
  { "221", "very light purple",           "7.5", "P", "9","6",   "7.5P 9/6",   "#FFD5FF" },

  { "243", "very dark reddish purple",    "10", "P", "1","4",    "10P 1/4",    "#2E112F" },
  { "243", "very dark reddish purple",    "10", "P", "1","6",    "10P 1/6",    "#330A35" },
  { "239", "very deep reddish purple",    "10", "P", "1","8",    "10P 1/8",    "#39003C" },
  { "239", "very deep reddish purple",    "10", "P", "1","10",   "10P 1/10",   "#3F0042" },
  { "239", "very deep reddish purple",    "10", "P", "1","12",   "10P 1/12",   "#440048" },
  { "242", "dark reddish purple",         "10", "P", "3","4",    "10P 3/4",    "#5F3C5C" },
  { "242", "dark reddish purple",         "10", "P", "3","6",    "10P 3/6",    "#673763" },
  { "238", "deep reddish purple",         "10", "P", "3","8",    "10P 3/8",    "#6F2F6B" },
  { "238", "deep reddish purple",         "10", "P", "3","10",   "10P 3/10",   "#772672" },
  { "238", "deep reddish purple",         "10", "P", "3","12",   "10P 3/12",   "#7F177A" },
  { "236", "vivid reddish purple",        "10", "P", "3","22",   "10P 3/22",   "#A1009B" },
  { "241", "moderate reddish purple",     "10", "P", "4","6",    "10P 4/6",    "#81507B" },
  { "241", "moderate reddish purple",     "10", "P", "4","8",    "10P 4/8",    "#894A82" },
  { "237", "strong reddish purple",       "10", "P", "4","10",   "10P 4/10",   "#914389" },
  { "237", "strong reddish purple",       "10", "P", "4","12",   "10P 4/12",   "#993B90" },
  { "236", "vivid reddish purple",        "10", "P", "4","22",   "10P 4/22",   "#C000B3" },
  { "241", "moderate reddish purple",     "10", "P", "5","6",    "10P 5/6",    "#9B6B95" },
  { "241", "moderate reddish purple",     "10", "P", "5","8",    "10P 5/8",    "#A5649C" },
  { "237", "strong reddish purple",       "10", "P", "5","10",   "10P 5/10",   "#AD5EA3" },
  { "237", "strong reddish purple",       "10", "P", "5","12",   "10P 5/12",   "#B657AA" },
  { "236", "vivid reddish purple",        "10", "P", "5","18",   "10P 5/18",   "#CF36BE" },
  { "227", "pale purple",                 "10", "P", "6","2",    "10P 6/2",    "#A28FA1" },
  { "244", "pale reddish purple",         "10", "P", "6","4",    "10P 6/4",    "#AB8BA8" },
  { "240", "light reddish purple",        "10", "P", "6","6",    "10P 6/6",    "#B685AF" },
  { "240", "light reddish purple",        "10", "P", "6","8",    "10P 6/8",    "#BF80B6" },
  { "248", "deep purplish pink",          "10", "P", "6","10",   "10P 6/10",   "#C979BD" },
  { "248", "deep purplish pink",          "10", "P", "6","12",   "10P 6/12",   "#D273C4" },
  { "248", "deep purplish pink",          "10", "P", "6","14",   "10P 6/14",   "#DC6ACC" },
  { "236", "vivid reddish purple",        "10", "P", "6","16",   "10P 6/16",   "#E462D3" },
  { "253", "grayish purplish pink",       "10", "P", "7","4",    "10P 7/4",    "#C7A5C3" },
  { "250", "moderate purplish pink",      "10", "P", "7","6",    "10P 7/6",    "#D19FCA" },
  { "250", "moderate purplish pink",      "10", "P", "7","8",    "10P 7/8",    "#DC9AD1" },
  { "247", "strong purplish pink",        "10", "P", "7","10",   "10P 7/10",   "#E693D9" },
  { "247", "strong purplish pink",        "10", "P", "7","12",   "10P 7/12",   "#EF8DE0" },
  { "247", "strong purplish pink",        "10", "P", "7","14",   "10P 7/14",   "#F985E7" },
  { "247", "strong purplish pink",        "10", "P", "7","16",   "10P 7/16",   "#FF7CEF" },
  { "252", "pale purplish pink",          "10", "P", "8","4",    "10P 8/4",    "#E2BFDE" },
  { "249", "light purplish pink",         "10", "P", "8","6",    "10P 8/6",    "#EDBAE5" },
  { "249", "light purplish pink",         "10", "P", "8","8",    "10P 8/8",    "#F9B3EE" },
  { "246", "brilliant purplish pink",     "10", "P", "8","10",   "10P 8/10",   "#FFADF6" },
  { "246", "brilliant purplish pink",     "10", "P", "8","12",   "10P 8/12",   "#FFA6FD" },
  { "246", "brilliant purplish pink",     "10", "P", "8","14",   "10P 8/14",   "#FF9DFF" },
  { "252", "pale purplish pink",          "10", "P", "9","4",    "10P 9/4",    "#FFDAFB" },
  { "249", "light purplish pink",         "10", "P", "9","6",    "10P 9/6",    "#FFD4FF" },

  // Red Purple

  { "243", "very dark reddish purple",    "2.5", "RP", "1","4",  "2.5RP 1/4",  "#30102D" },
  { "243", "very dark reddish purple",    "2.5", "RP", "1","6",  "2.5RP 1/6",  "#360833" },
  { "239", "very deep reddish purple",    "2.5", "RP", "1","8",  "2.5RP 1/8",  "#3C0039" },
  { "239", "very deep reddish purple",    "2.5", "RP", "1","10", "2.5RP 1/10", "#42003E" },
  { "239", "very deep reddish purple",    "2.5", "RP", "1","12", "2.5RP 1/12", "#480044" },
  { "236", "vivid reddish purple",        "2.5", "RP", "2","20", "2.5RP 2/20", "#85006E" },
  { "229", "dark grayish purple",         "2.5", "RP", "3","2",  "2.5RP 3/2",  "#584152" },
  { "242", "dark reddish purple",         "2.5", "RP", "3","4",  "2.5RP 3/4",  "#633B57" },
  { "242", "dark reddish purple",         "2.5", "RP", "3","6",  "2.5RP 3/6",  "#6C355C" },
  { "238", "deep reddish purple",         "2.5", "RP", "3","8",  "2.5RP 3/8",  "#752C61" },
  { "238", "deep reddish purple",         "2.5", "RP", "3","10", "2.5RP 3/10", "#7E2067" },
  { "238", "deep reddish purple",         "2.5", "RP", "3","12", "2.5RP 3/12", "#870B6C" },
  { "228", "grayish purple",              "2.5", "RP", "5","2",  "2.5RP 5/2",  "#897584" },
  { "244", "pale reddish purple",         "2.5", "RP", "6","4",  "2.5RP 6/4",  "#AF8AA3" },
  { "240", "light reddish purple",        "2.5", "RP", "6","6",  "2.5RP 6/6",  "#BB84A8" },
  { "240", "light reddish purple",        "2.5", "RP", "6","8",  "2.5RP 6/8",  "#C67EAD" },
  { "253", "grayish purplish pink",       "2.5", "RP", "7","4",  "2.5RP 7/4",  "#CAA4BE" },
  { "247", "strong purplish pink",        "2.5", "RP", "7","20", "2.5RP 7/20", "#FF58E8" },

  { "260", "very dark purplish red",      "5", "RP", "1","4",    "5RP 1/4",    "#320F2A" },
  { "260", "very dark purplish red",      "5", "RP", "1","6",    "5RP 1/6",    "#38062F" },
  { "257", "very deep purplish red",      "5", "RP", "1","8",    "5RP 1/8",    "#3F0034" },
  { "257", "very deep purplish red",      "5", "RP", "1","10",   "5RP 1/10",   "#460039" },
  { "229", "dark grayish purple",         "5", "RP", "3","2",    "5RP 3/2",    "#59414E" },
  { "259", "dark purplish red",           "5", "RP", "3","6",    "5RP 3/6",    "#703355" },
  { "256", "deep purplish red",           "5", "RP", "3","10",   "5RP 3/10",   "#841C5C" },
  { "256", "deep purplish red",           "5", "RP", "3","12",   "5RP 3/12",   "#8D0060" },
  { "262", "grayish purplish red",        "5", "RP", "4","6",    "5RP 4/6",    "#8B4D6C" },
  { "258", "moderate purplish red",       "5", "RP", "4","10",   "5RP 4/10",   "#9F3D71" },
  { "255", "strong purplish red",         "5", "RP", "4","12",   "5RP 4/12",   "#A93074" },
  { "262", "grayish purplish red",        "5", "RP", "5","6",    "5RP 5/6",    "#A56885" },
  { "258", "moderate purplish red",       "5", "RP", "5","10",   "5RP 5/10",   "#BD5889" },
  { "255", "strong purplish red",         "5", "RP", "5","12",   "5RP 5/12",   "#C84E8C" },
  { "254", "vivid purplish red",          "5", "RP", "5","14",   "5RP 5/14",   "#D2428E" },
  { "254", "vivid purplish red",          "5", "RP", "5","20",   "5RP 5/20",   "#F00097" },
  { "251", "dark purplish pink",          "5", "RP", "6","6",    "5RP 6/6",    "#C1829F" },
  { "251", "dark purplish pink",          "5", "RP", "6","8",    "5RP 6/8",    "#CC7CA1" },
  { "248", "deep purplish pink",          "5", "RP", "6","10",   "5RP 6/10",   "#D875A3" },
  { "248", "deep purplish pink",          "5", "RP", "6","12",   "5RP 6/12",   "#E46CA5" },
  { "248", "deep purplish pink",          "5", "RP", "6","14",   "5RP 6/14",   "#F061A7" },
  { "254", "vivid purplish red",          "5", "RP", "6","18",   "5RP 6/18",   "#FF46AD" },
  { "253", "grayish purplish pink",       "5", "RP", "7","4",    "5RP 7/4",    "#CDA3B8" },
  { "250", "moderate purplish pink",      "5", "RP", "7","6",    "5RP 7/6",    "#DC9DBA" },
  { "250", "moderate purplish pink",      "5", "RP", "7","8",    "5RP 7/8",    "#EA96BB" },
  { "252", "pale purplish pink",          "5", "RP", "8","4",    "5RP 8/4",    "#E9BED4" },
  { "249", "light purplish pink",         "5", "RP", "8","6",    "5RP 8/6",    "#F9B7D5" },
  { "249", "light purplish pink",         "5", "RP", "8","8",    "5RP 8/8",    "#FFAFD7" },
  { "246", "brilliant purplish pink",     "5", "RP", "8","10",   "5RP 8/10",   "#FFA8D8" },
  { "246", "brilliant purplish pink",     "5", "RP", "8","12",   "5RP 8/12",   "#FF9EDB" },
  { "252", "pale purplish pink",          "5", "RP", "9","4",    "5RP 9/4",    "#FFD8F0" },
  { "249", "light purplish pink",         "5", "RP", "9","6",    "5RP 9/6",    "#FFD1F1" },

  { "260", "very dark purplish red",      "7.5", "RP", "1","4",  "7.5RP 1/4",  "#330E27" },
  { "260", "very dark purplish red",      "7.5", "RP", "1","6",  "7.5RP 1/6",  "#3A042B" },
  { "257", "very deep purplish red",      "7.5", "RP", "1","8",  "7.5RP 1/8",  "#42002F" },
  { "257", "very deep purplish red",      "7.5", "RP", "1","10", "7.5RP 1/10", "#490033" },
  { "229", "dark grayish purple",         "7.5", "RP", "3","2",  "7.5RP 3/2",  "#5B414C" },
  { "259", "dark purplish red",           "7.5", "RP", "3","4",  "7.5RP 3/4",  "#683A4D" },
  { "259", "dark purplish red",           "7.5", "RP", "3","8",  "7.5RP 3/8",  "#7E274F" },
  { "256", "deep purplish red",           "7.5", "RP", "3","10", "7.5RP 3/10", "#881951" },
  { "256", "deep purplish red",           "7.5", "RP", "3","12", "7.5RP 3/12", "#920053" },
  { "254", "vivid purplish red",          "7.5", "RP", "3","16", "7.5RP 3/16", "#A50058" },
  { "228", "grayish purple",              "7.5", "RP", "4","2",  "7.5RP 4/2",  "#735A65" },
  { "262", "grayish purplish red",        "7.5", "RP", "4","6",  "7.5RP 4/6",  "#8D4D65" },
  { "258", "moderate purplish red",       "7.5", "RP", "4","8",  "7.5RP 4/8",  "#994465" },
  { "255", "strong purplish red",         "7.5", "RP", "4","12", "7.5RP 4/12", "#AE2E67" },
  { "254", "vivid purplish red",          "7.5", "RP", "4","16", "7.5RP 4/16", "#C2006A" },
  { "228", "grayish purple",              "7.5", "RP", "5","4",  "7.5RP 5/4",  "#9B6E7E" },
  { "262", "grayish purplish red",        "7.5", "RP", "5","6",  "7.5RP 5/6",  "#A8677E" },
  { "258", "moderate purplish red",       "7.5", "RP", "5","8",  "7.5RP 5/8",  "#B65F7E" },
  { "255", "strong purplish red",         "7.5", "RP", "5","12", "7.5RP 5/12", "#CD4B7F" },
  { "227", "pale purple",                 "7.5", "RP", "6","2",  "7.5RP 6/2",  "#A68E9A" },
  { "251", "dark purplish pink",          "7.5", "RP", "6","6",  "7.5RP 6/6",  "#C48298" },
  { "251", "dark purplish pink",          "7.5", "RP", "6","8",  "7.5RP 6/8",  "#CF7B98" },
  { "253", "grayish purplish pink",       "7.5", "RP", "7","4",  "7.5RP 7/4",  "#CFA3B4" },

  { "260", "very dark purplish red",      "10", "RP", "1","4",   "10RP 1/4",   "#350E24" },
  { "260", "very dark purplish red",      "10", "RP", "1","6",   "10RP 1/6",   "#3C0327" },
  { "257", "very deep purplish red",      "10", "RP", "1","8",   "10RP 1/8",   "#44002A" },
  { "257", "very deep purplish red",      "10", "RP", "1","10",  "10RP 1/10",  "#4B002E" },
  { "229", "dark grayish purple",         "10", "RP", "3","4",   "10RP 3/2",   "#5C4149" },
  { "259", "dark purplish red",           "10", "RP", "3","6",   "10RP 3/6",   "#753246" },
  { "256", "deep purplish red",           "10", "RP", "3","10",  "10RP 3/10",  "#8B1545" },
  { "256", "deep purplish red",           "10", "RP", "3","12",  "10RP 3/12",  "#960045" },
  { "262", "grayish purplish red",        "10", "RP", "4","6",   "10RP 4/6",   "#8F4C5E" },
  { "258", "moderate purplish red",       "10", "RP", "4","10",  "10RP 4/10",  "#A63A5B" },
  { "255", "strong purplish red",         "10", "RP", "4","12",  "10RP 4/12",  "#B22B5A" },
  { "228", "grayish purple",              "10", "RP", "5","2",   "10RP 5/2",   "#8D747D" },
  { "262", "grayish purplish red",        "10", "RP", "5","6",   "10RP 5/6",   "#AA6777" },
  { "258", "moderate purplish red",       "10", "RP", "5","10",  "10RP 5/10",  "#C45673" },
  { "255", "strong purplish red",         "10", "RP", "5","12",  "10RP 5/12",  "#D14A71" },
  { "6", "dark pink",                     "10", "RP", "6","6",   "10RP 6/6",   "#C68192" },
  { "3", "deep pink",                     "10", "RP", "6","10",  "10RP 6/10",  "#E0728C" },
  { "3", "deep pink",                     "10", "RP", "6","12",  "10RP 6/12",  "#EC698A" },
  { "3", "deep pink",                     "10", "RP", "6","14",  "10RP 6/14",  "#F85F88" },
  { "8", "grayish pink",                  "10", "RP", "7","2",   "10RP 7/2",   "#C1A8B4" },
  { "5", "moderate pink",                 "10", "RP", "7","4",   "10RP 7/4",   "#D1A3B0" },
  { "5", "moderate pink",                 "10", "RP", "7","6",   "10RP 7/6",   "#E19CAD" },
  { "2", "strong pink",                   "10", "RP", "7","8",   "10RP 7/8",   "#F095A9" },
  { "2", "strong pink",                   "10", "RP", "7","10",  "10RP 7/10",  "#FD8DA6" },
  { "1", "vivid pink",                    "10", "RP", "7","12",  "10RP 7/12",  "#FF83A3" },
  { "1", "vivid pink",                    "10", "RP", "7","14",  "10RP 7/14",  "#FF78A0" },
  { "1", "vivid pink",                    "10", "RP", "7","16",  "10RP 7/16",  "#FF6B9E" },
  { "2", "strong pink",                   "10", "RP", "8","8",   "10RP 8/8",   "#FFAEC3" },
  { "2", "strong pink",                   "10", "RP", "8","10",  "10RP 8/10",  "#FFA6C0" },
  { "7", "pale pink",                     "10", "RP", "9","2",   "10RP 9/2",   "#F6DFEC" },
  { "4", "light pink",                    "10", "RP", "9","4",   "10RP 9/4",   "#FFD7E7" },
  { "4", "light pink",                    "10", "RP", "9","6",   "10RP 9/6",   "#FFD0E2" },
  };

  public static final String KEY_WORDS[] =
  {
    "very", "vivid", "pale", "brilliant", "dark", "deep", "moderate",
    "strong", "light", "medium",
    "red",  "orange", "yellow", "green", "blue", "violet", "purple",
    "brown", "olive", "pink",   "white", "gray", "black",
    "reddish", "brownish", "yellowish", "greenish", "bluish", "purplish", "grayish"
  };

  /** Method isKEY()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @return bValue = true it does and falls it does not
   */

  public static boolean isKEY(String str)
  {
    int     i = 0;
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    for (i=0; i<KEY_WORDS.length; i++)
    {
      if (sWord.equals(KEY_WORDS[i])) bValue = true;
    }

    return (bValue);
  }

  /** Method isCOLOR()
   * <p> This method will compare a string or string phrase to the standard
   *     color strings or phrase to see if the string or string phrase is part
   *     of the color set
   * @param  str    = string or string phrase being compared.
   * @return bValue = true is part of set, false is not part of set
   */

  public static boolean isCOLOR(String str)
  {
    int     i = 0;
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());
    String  sColor = "";
    String  sMun1  = "";
    String  sMun2  = "";

    for (i=0; i<COLORS.length; i++)
    {
      sColor = new String(COLORS[i][_ISCC_COLOR]);
      sMun1  = new String(COLORS[i][_MUNSELL_CODE].toLowerCase());
      sMun2  = new String(cmn.cmnString.removeSpaces(sMun1));

      if ((sWord.equals(sColor)) ||
          (sWord.equals(sMun1)) || (sWord.equals(sMun2)))
      {
        bValue = true;
      }
    }

    return (bValue);
  }

  /** Method getColor()
   * <p> This method will find the color by string or by munsell code and return
   *     the color id number
   * @param  str    = string or string phrase being compared.
   * @return iColor = color id number
   */

  public static int getColor(String str)
  {
    int     i = 0;
    int     iColor = -1;
    int     iFound = -1;
    String  sWord  = new String(str.toLowerCase());
    String  sColor = "";
    String  sMun1  = "";
    String  sMun2  = "";

    for (i=0; i<COLORS.length; i++)
    {
      sColor = new String(COLORS[i][_ISCC_COLOR]);
      sMun1  = new String(COLORS[i][_MUNSELL_CODE].toLowerCase());
      sMun2  = new String(cmn.cmnString.removeSpaces(sMun1));

      if ((iFound == -1) && ((sWord.equals(sColor)) ||
          (sWord.equals(sMun1)) || (sWord.equals(sMun2))))
      {
        iFound = i;
      }
    }

    if (iFound > -1) iColor = iFound;

    return (iColor);
  }
}