/*
 * @rockColorUtility.java Version 1.1 09/10/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.color;

import iqstrat.iqstratHeadersStruct;

//import rock.color.rockColorsStruct;
import rock.color.rockColorListStruct;
import rock.color.rockColorStruct;

/** Class rockColorUtility
 *  <p> This Class will provide basic utilities for the Rock Color
 *      data structures.
 *
 *  @version 1.1 09/10/2007
 *  @author  John R. Victorine
 */

public class rockColorUtility
{
  /** Method addHeaderData()
   * <p> This method will add the header data to the Fossil data List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Rock Fossil List Data Structure
   * @return st       = The Rock Fossil List Data Structure
   */

  public static rockColorListStruct addHeaderData(iqstratHeadersStruct stHeader,
                                                  rockColorListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method add()
   * <p> This method will add a Color to an existing list
   * @param  stNew = The Rock Color Data Structure
   * @param  st    = The Old Rock Color List Data Structure
   * @return st    = The new Rock Color List.
   */

  public static rockColorListStruct add( rockColorStruct stNew,
                                         rockColorListStruct st )
  {
    int i = 0;
    int    iCount   = 0;
    rockColorListStruct stTemp = new rockColorListStruct();
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    stTemp.stItem = new rockColorStruct[1];

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      if (st.iCount > 0)
        stTemp.stItem = new rockColorStruct[st.iCount+1];

      for (i=0; i<st.iCount; i++)
      {
        stTemp.stItem[iCount] = new rockColorStruct();
        stTemp.stItem[iCount] = copy(st.stItem[i]);
        iCount++;
      }

      st.delete();
      st = null;
    }

    if (stNew != null)
    {
      stTemp.stItem[iCount] = new rockColorStruct();
      stTemp.stItem[iCount] = copy(stNew);
      iCount++;
    }

    st        = new rockColorListStruct();
    st.iCount = iCount;
    st.stItem = new rockColorStruct[st.iCount];

    st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
    st.source   = new String( source );    // Source of Data
    st.sCreated = new String( sCreated );  // Created Date

    for (i=0; i<st.iCount; i++)
    {
      st.stItem[i] = new rockColorStruct();
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();
    stTemp = null;

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Color in an existing list
   * @param  stNew = The New Rock Color Data Structure
   * @param  st   = The Old Rock Color List Data Structure
   * @return st   = The new Rock Color List.
   */

  public static rockColorListStruct modify( rockColorStruct stNew,
                                            rockColorListStruct st )
  {
    int i = 0;
    int    iCount   = 0;
    rockColorListStruct stTemp = new rockColorListStruct();
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    stTemp.stItem = new rockColorStruct[1];

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      if (st.iCount > 0)
        stTemp.stItem = new rockColorStruct[st.iCount];

      for (i=0; i<st.iCount; i++)
      {
        if (st.stItem[i].sKEY.equals(stNew.sKEY))
        {
          stTemp.stItem[iCount] = new rockColorStruct();
          stTemp.stItem[iCount] = copy(stNew);
          iCount++;
        }
        else
        {
          stTemp.stItem[iCount] = new rockColorStruct();
          stTemp.stItem[iCount] = copy(st.stItem[i]);
          iCount++;
        }
      }

      st.delete();
      st = null;

      st        = new rockColorListStruct();
      st.iCount = iCount;
      st.stItem = new rockColorStruct[st.iCount];

      st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
      st.source   = new String( source );    // Source of Data
      st.sCreated = new String( sCreated );  // Created Date

      for (i=0; i<st.iCount; i++)
      {
        st.stItem[i] = new rockColorStruct();
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
      stTemp = null;
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Color from an existing list
   * @param  sKEY = Unique Identifier for depth range
   * @param  st   = The Old Rock Color List Data Structure
   * @return st   = The new Rock Color List.
   */

  public static rockColorListStruct remove(String sKEY, rockColorListStruct st)
  {
    int i = 0;
    int    iCount   = 0;
    String sKGS     = "YES"; // KGS Saved Data Indicator
    String source   = "";    // Source of Data
    String sCreated = "";    // Created Date

    rockColorListStruct stTemp = null;

    if (st != null)
    {
      sKGS     = new String( st.sKGS );      // KGS Saved Data Indicator
      source   = new String( st.source );    // Source of Data
      sCreated = new String( st.sCreated );  // Created Date

      if (st.iCount < 2)
      {
        st.iCount = 0;
        st.delete();
        st = null;
      }
      else
      {
        stTemp        = new rockColorListStruct();
        stTemp.stItem = new rockColorStruct[st.iCount-1];

        for (i=0; i<st.iCount; i++)
        {
          if (st.stItem[i] != null)
          {
            if (!st.stItem[i].sKEY.equals(sKEY))
            {
              stTemp.stItem[iCount] = new rockColorStruct();
              stTemp.stItem[iCount] = copy(st.stItem[i]);
              iCount++;
            }
          }
        }

        st.delete();
        st = null;

        st        = new rockColorListStruct();
        st.iCount = iCount;
        st.stItem = new rockColorStruct[st.iCount];

        st.sKGS     = new String( sKGS );      // KGS Saved Data Indicator
        st.source   = new String( source );    // Source of Data
        st.sCreated = new String( sCreated );  // Created Date

        for (i=0; i<st.iCount; i++)
        {
          st.stItem[i] = new rockColorStruct();
          st.stItem[i] = copy(stTemp.stItem[i]);
        }

        if (stTemp != null)
          stTemp.delete();
        stTemp = null;
      }
    }

    return (st);
  }

  /** Method copyList()
   * <p> This method will copy one Color list structure to another
   * @param  stOld = Old Color list data structure
   * @return stNew = New Color unit list data structure
   */

  public static rockColorListStruct copyList( rockColorListStruct stOld )
  {
    rockColorListStruct stNew = null;

    if (stOld != null)
    {
      if (stOld.iCount > 0)
      {
        stNew        = new rockColorListStruct();

        // Identification Information

        stNew.sKID   = new String(stOld.sKID);   // Location Header KID
        stNew.sKEY   = new String(stOld.sKEY);   // Primary Key created on user
        stNew.iType  = stOld.iType;
        stNew.sAPI   = new String(stOld.sAPI);   // API-Number of Location
        stNew.sName  = new String(stOld.sName);  // Name
        stNew.status = new String(stOld.status); // Status

        // XY Position Information

        stNew.dLatitude  = stOld.dLatitude;  // Latitude
        stNew.dLongitude = stOld.dLongitude; // Longitude

        // Z Position - Depth, Elevation

        stNew.depth = stOld.depth; // Total Depth of Well or Height of Outcrop
        stNew.dGL   = stOld.dGL;   // Ground Level

        // LAS 3 Dependent Variable

        stNew.sKGS       = new String( stOld.sKGS ); // KGS Saved Data Indicator
        stNew.source     = new String( stOld.source );    // Source of Data
        stNew.sCreated   = new String( stOld.sCreated );  // Created Date

        stNew.iCount = stOld.iCount;
        stNew.stItem = new rockColorStruct[stOld.iCount];

        for (int i=0; i<stOld.iCount; i++)
        {
          stNew.stItem[i] = copy(stOld.stItem[i]);
        }
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static rockColorListStruct transfer(rockColorListStruct stOld)
  {
    rockColorListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copy()
   * <p> This method will add a Color Block to an existing list
   * @param  stOld = The Rock Color Data Structure
   * @return stNew = The New Rock Color Data Structure
   */

  public static rockColorStruct copy (rockColorStruct stOld)
  {
    rockColorStruct stNew = new rockColorStruct();

    stNew.sKEY       = new String(stOld.sKEY);  // Unique Identifier

    stNew.depthStart = stOld.depthStart; // Starting Depth
    stNew.depthEnd   = stOld.depthEnd;   // Ending Depth

    stNew.iTotal     = stOld.iTotal;
    for (int i=0; i<3; i++)
    {
      stNew.sColor[i]   = new String(stOld.sColor[i]);
      stNew.sMunsell[i] = new String(stOld.sMunsell[i]);
      stNew.sRGBHex[i]  = new String(stOld.sRGBHex[i]);

      for (int j=0; j<3; j++)
        stNew.iRGB[i][j] = stOld.iRGB[i][j]; // RGB Color Values
    }

    return (stNew);
  }

  /** Method initLegend( int iColors[], lithologyListStruct st )
   * <p> This method will set the legend variables
   * @return iColors = Array of indicators to rock color Types symbol or not
   */

  public static int[] initLegend()
  {
    int iColors[] = null;

    iColors = new int[rock.color.rockMunsellColorStruct.COLORS.length];

    for (int j=0; j<rock.color.rockMunsellColorStruct.COLORS.length; j++)
      iColors[j] = -1;

    return (iColors);
  }

  /** Method setLegend( int iColors[], lithologyListStruct st )
   * <p> This method will set the legend variables
   * @param  iColors = Array of indicators to rock color Types symbol or not
   * @param  st      = Lithology List Data Structure
   * @return iColors = Array of indicators to rock color Types symbol or not
   */

  public static int[] setLegend( int iColors[], rockColorListStruct st )
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        for (int k=0; k<3; k++)
        {
          for (int j=0; j<rock.color.rockMunsellColorStruct.COLORS.length; j++)
          {
            if (st.stItem[i].sRGBHex[k].equals(
                  rock.color.rockMunsellColorStruct.COLORS[j][7]))
            {
              iColors[j] = 0;
            }
          }
        }
      }
    }

    return (iColors);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = data list structure
   * @return st = sorted data list structure
   */

  public static rockColorListStruct bubbleSort(rockColorListStruct st)
  {
    boolean swappedOnPrevRun = true;
    rockColorStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].depthStart > st.stItem[i+1].depthStart)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method getColor()
   * <p> This method will retrieve the color from the RGB Hex Value Text
   *     NOTE: Format "#RRGGBB"
   * @param  iClr = The Color Requested 0 = Red, 1 = Green, 2 = Blue
   * @param  sClr = The RGB Hex String, i.e. #RRGGBB
   * @return iRGB = The Color required
   */

  public static int getColor(int iClr, String sClr)
  {
    int    iRGB = -1;
    char   c[];
    String sHex1 = "";
    String sHex0 = "";
    int    iHex1 = 0;
    int    iHex0 = 0;

    c = sClr.toCharArray();

    switch (iClr)
    {
      case 0: // Red
        sHex1 = new String("" + c[1]);
        sHex0 = new String("" + c[2]);
        break;
      case 1: // Green
        sHex1 = new String("" + c[3]);
        sHex0 = new String("" + c[4]);
        break;
      case 2: // Blue
        sHex1 = new String("" + c[5]);
        sHex0 = new String("" + c[6]);
        break;
    }

    if (sHex1.equals("A")) iHex1 = 10;
    else if (sHex1.equals("B")) iHex1 = 11;
    else if (sHex1.equals("C")) iHex1 = 12;
    else if (sHex1.equals("D")) iHex1 = 13;
    else if (sHex1.equals("E")) iHex1 = 14;
    else if (sHex1.equals("F")) iHex1 = 15;
    else iHex1 = cmn.cmnString.stringToInt(sHex1);

    if (sHex0.equals("A")) iHex0 = 10;
    else if (sHex0.equals("B")) iHex0 = 11;
    else if (sHex0.equals("C")) iHex0 = 12;
    else if (sHex0.equals("D")) iHex0 = 13;
    else if (sHex0.equals("E")) iHex0 = 14;
    else if (sHex0.equals("F")) iHex0 = 15;
    else iHex0 = cmn.cmnString.stringToInt(sHex0);

    iRGB = iHex1 * 16 + iHex0;

    return (iRGB);
  }

  public static void print(rockColorListStruct st)
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        System.out.println(st.stItem[i].depthStart+" "+
                           st.stItem[i].depthEnd+" "+
                           st.stItem[i].iTotal);

        for (int j=0; j<3; j++)
        {
          System.out.println("RGB="+st.stItem[i].iRGB[j][0]+" "+
                                    st.stItem[i].iRGB[j][1]+" "+
                                    st.stItem[i].iRGB[j][2]);
        }
      }
    }
  }
}