/*
 * @rockColorChartUtility.java Version 1.1 11/06/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.color;

import rock.color.rockColorChartStruct;
import rock.color.rockColorChartListStruct;
import rock.color.rockColorListStruct;
import rock.color.rockColorStruct;

/** Class rockColorChartUtility
 *  <p> This Class holds the utilities for manipulating the rock color chart
 *      structures.
 *
 *  @version 1.1 11/06/2009
 *  @author  John R. Victorine
 */

public class rockColorChartUtility
{
  /** Method add( rockColorChartStruct stNew, rockColorChartListStruct st )
   * <p> This method will add a Color to an existing list
   * @param  stNew = The Rock Color Data Structure
   * @param  st    = The Old Rock Color List Data Structure
   * @return st    = The new Rock Color List.
   */

  public static rockColorChartListStruct add( rockColorChartStruct stNew,
                                              rockColorChartListStruct st )
  {
    int                      i      = 0;
    int                      iCount = 0;
    rockColorChartListStruct stTemp = new rockColorChartListStruct();

    stTemp.stItem = new rockColorChartStruct[1];

    if (st != null)
    {
      if (st.iCount > 0)
        stTemp.stItem = new rockColorChartStruct[st.iCount+1];

      for (i=0; i<st.iCount; i++)
      {
        stTemp.stItem[iCount] = new rockColorChartStruct();
        stTemp.stItem[iCount] = copy(st.stItem[i]);
        iCount++;
      }

      st.delete();
      st = null;
    }

    if (stNew != null)
    {
      stTemp.stItem[iCount] = new rockColorChartStruct();
      stTemp.stItem[iCount] = copy(stNew);
      iCount++;
    }

    st        = new rockColorChartListStruct();
    st.iCount = iCount;
    st.stItem = new rockColorChartStruct[st.iCount];

    for (i=0; i<st.iCount; i++)
    {
      st.stItem[i] = new rockColorChartStruct();
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();
    stTemp = null;

    return (st);
  }

  /** Method copyList( rockColorChartListStruct stOld )
   * <p> This method will copy one Color list structure to another
   * @param  stOld = Old Color list data structure
   * @return stNew = New Color unit list data structure
   */

  public static rockColorChartListStruct copyList(
                                           rockColorChartListStruct stOld )
  {
    rockColorChartListStruct stNew = null;

    if (stOld != null)
    {
      if (stOld.iCount > 0)
      {
        stNew        = new rockColorChartListStruct();
        stNew.iCount = stOld.iCount;
        stNew.stItem = new rockColorChartStruct[stOld.iCount];

        for (int i=0; i<stOld.iCount; i++)
        {
          stNew.stItem[i] = copy(stOld.stItem[i]);
        }
      }
    }

    return (stNew);
  }

  /** Method copy( rockColorChartStruct stOld )
   * <p> This method will add a Color Block to an existing list
   * @param  stOld = The Rock Color Data Structure
   * @return stNew = The New Rock Color Data Structure
   */

  public static rockColorChartStruct copy ( rockColorChartStruct stOld )
  {
    rockColorChartStruct stNew = null;

    if (stOld != null)
    {
      stNew = new rockColorChartStruct();

      // Munsell Code Variables

      stNew.sMunsell = new String( stOld.sMunsell ); // Code
      stNew.sHue     = new String( stOld.sHue );     // Hue
      stNew.dHue     = stOld.dHue;                   // Hue Number
      stNew.sColor   = new String( stOld.sColor );   // Color
      stNew.dValue   = stOld.dValue;    // Value (Lightness) 0 ... 10
      stNew.dChroma  = stOld.dChroma;   // Chroma color purity or colorfulness

      // Measured & Interpolated xyY Color Values

      stNew.x  = stOld.x;
      stNew.y  = stOld.y;
      stNew.Yo = stOld.Yo;

      // Color Name & computed rgb value

      stNew.sName = new String( stOld.sName ); // Color Name
      stNew.sHex  = new String( stOld.sHex );  // Hex Color Value
      stNew.iRGB  = new int[3];                // RGB equivalent value

      for (int i=0; i<3; i++)
        stNew.iRGB[i] = stOld.iRGB[i];
    }

    return (stNew);
  }

  /** Method getHEX( int R, int G, int B )
   *  <p> This method will convert the RGB values to a Hex Code
   * @param R = RGB Red Value
   * @param G = RGB Green Value
   * @param B = RGB Blue Value
   * @return sHex = the Hex value of the RGB Color
   */


  public static String getHEX( int R, int G, int B )
  {
    String sHex  = "#FFFFFF";
    String sHexR = "FF";
    String sHexG = "FF";
    String sHexB = "FF";
    int    iVal1 = 0;
    int    iVal2 = 0;
    String sVal[] = { "0", "1", "2", "3", "4", "5", "6", "7",
                      "8", "9", "A", "B", "C", "D", "E", "F" };

    // Red Hex Value

    iVal1 = R/16;
    if (iVal1 < 0) iVal1 = 0;

    sHexR = sVal[iVal1];

    iVal1 = R - (16 * iVal1);
    sHexR = new String( sHexR + sVal[iVal1] );

    // Green Hex Value

    iVal1 = G/16;
    if (iVal1 < 0) iVal1 = 0;

    sHexG = sVal[iVal1];

    iVal1 = G - (16 * iVal1);
    sHexG = new String( sHexG + sVal[iVal1] );

    // Blue Hex Value

    iVal1 = B/16;
    if (iVal1 < 0) iVal1 = 0;

    sHexB = sVal[iVal1];

    iVal1 = B - (16 * iVal1);
    sHexB = new String( sHexB + sVal[iVal1] );

    // Hex Value

    sHex = new String( "#" + sHexR + sHexG + sHexB );

    return (sHex);
  }

  /** Method getColor()
   * <p> This method will retrieve the color from the RGB Hex Value Text
   *     NOTE: Format "#RRGGBB"
   * @param  iClr = The Color Requested 0 = Red, 1 = Green, 2 = Blue
   * @param  sClr = The RGB Hex String, i.e. #RRGGBB
   * @return iRGB = The Color required
   */

  public static int getColor(int iClr, String sClr)
  {
    int    iRGB = -1;
    char   c[];
    String sHex1 = "";
    String sHex0 = "";
    int    iHex1 = 0;
    int    iHex0 = 0;

    c = sClr.toCharArray();

    switch (iClr)
    {
      case 0: // Red
        sHex1 = new String("" + c[1]);
        sHex0 = new String("" + c[2]);
        break;
      case 1: // Green
        sHex1 = new String("" + c[3]);
        sHex0 = new String("" + c[4]);
        break;
      case 2: // Blue
        sHex1 = new String("" + c[5]);
        sHex0 = new String("" + c[6]);
        break;
    }

    if (sHex1.equals("A")) iHex1 = 10;
    else if (sHex1.equals("B")) iHex1 = 11;
    else if (sHex1.equals("C")) iHex1 = 12;
    else if (sHex1.equals("D")) iHex1 = 13;
    else if (sHex1.equals("E")) iHex1 = 14;
    else if (sHex1.equals("F")) iHex1 = 15;
    else iHex1 = cmn.cmnString.stringToInt(sHex1);

    if (sHex0.equals("A")) iHex0 = 10;
    else if (sHex0.equals("B")) iHex0 = 11;
    else if (sHex0.equals("C")) iHex0 = 12;
    else if (sHex0.equals("D")) iHex0 = 13;
    else if (sHex0.equals("E")) iHex0 = 14;
    else if (sHex0.equals("F")) iHex0 = 15;
    else iHex0 = cmn.cmnString.stringToInt(sHex0);

    iRGB = iHex1 * 16 + iHex0;

    return (iRGB);
  }

  /** Method checkPhrase( String str, rockColorChartListStruct st )
   * <p> This method will determine if a valid Munsell code exists in the phrase
   * @param  str    = phrase passed in
   * @param  st     = rock color list data structure
   * @return bValue = true it is a Munsell Code and false if it is not
   */

  public static boolean checkPhrase( String str, rockColorChartListStruct st )
  {
    boolean bValue     = false;
    String  tokens[]   = null;
    String  sDelimiter = new String("[ ,;:=-]+");
    String  sPhrase    = "";

    sPhrase = new String( str.toUpperCase() );
    tokens  = sPhrase.split( sDelimiter );

    for (int i=0; i<tokens.length; i++)
    {
      if (isMunsell( tokens[i], st ))
      {
        bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isMunsell( String str, rockColorChartListStruct st )
   * <p> This method will determine if the string passed in is a valid Munsell
   *     code.
   * @param  str    = string passed in
   * @param  st     = rock color list data structure
   * @return bValue = true it is a Munsell Code and false if it is not
   */

  public static boolean isMunsell( String str, rockColorChartListStruct st )
  {
    boolean bValue = false;
    String  sText  = new String( removeSpace( str ) );

    sText = new String( removeParenthesis( sText ));

    for (int i=0; i<st.iCount; i++)
    {
      if (sText.equals(st.stItem[i].sMunsell))
        bValue = true;
    }

    return (bValue);
  }

  /** Method getMunsell( String str, rockColorChartListStruct st )
   * <p> This method will determine if the string passed in is a valid Munsell
   *     code.
   * @param  str    = string passed in
   * @param  st     = rock color list data structure
   * @return iValue = row id
   */

  public static int getMunsell( String str, rockColorChartListStruct st )
  {
    int    iValue = -1;
    String sText  = new String( removeSpace( str ) );

    sText = new String( removeParenthesis( sText ));

    for (int i=0; i<st.iCount; i++)
    {
      if (sText.equals(st.stItem[i].sMunsell))
        iValue = i;
    }

    return (iValue);
  }

  /** Method parseColor()
   * <p> This method will parse the text string into rock color
   * @param  depthStart   = the starting depth
   * @param  depthEnd     = the ending depth
   * @param  str          = the phrase
   * @param  stColors     = Color Thesaurus
   * @return st           = rock color data structure
   */

  public static  rockColorStruct parseColor(
                     double depthStart, double depthEnd, String str,
                     rockColorChartListStruct stColors )
  {
    rockColorStruct st         = null;
    int             iNew       = 0;
    int             iTotal     = 0;
    String          tokens[]   = null;
    String          sDelimiter = new String("[ ,;:=-]+");
    String          sPhrase    = "";
    int             iClr       = -1;

    sPhrase = new String( str.toUpperCase() );
    tokens  = sPhrase.split( sDelimiter );

    for (int i=0; i<tokens.length; i++)
    {
      if (isMunsell( tokens[i], stColors ))
      {
        iClr = getMunsell( tokens[i], stColors );

        if (iTotal < 3)
        {
          if (iNew == 0)
          {
            iNew = 1;
            st      = new rockColorStruct();
            st.sKEY = new String( cmn.cmnString.UniqueName() +
                                  "_" + iTotal + "_" +
                                  ( (int) depthStart) ); // Create a Unique KEY
            st.depthStart = depthStart; // Starting Depth
            st.depthEnd   = depthEnd;   // Ending Depth
            st.iTotal     = 0;
          }

          for (int j=0; j<3; j++)
          {
            st.iRGB[iTotal][j] = stColors.stItem[iClr].iRGB[j];
          }

          st.sColor[iTotal]   = new String(stColors.stItem[iClr].sName);
          st.sMunsell[iTotal] = new String(stColors.stItem[iClr].sMunsell);
          st.sRGBHex[iTotal]  = new String(stColors.stItem[iClr].sHex);

          iTotal++;
        }

        st.iTotal = iTotal;  // Total Number of Colors for section
      }
    }

    return (st);
  }

  /* ======================================================================== */
  /** Method removeSpace()
   * <p> This method will remove the spaces in the String
   * @param  str  = The string to be converted
   * @return sNew = The text converted
   */

  public static String removeSpace( String str )
  {
    String sNew = new String("");
    char   c[]  = null;

    if (str != null)
    {
      c = str.toCharArray();

      for (int i=0; i<c.length; i++)
      {
        if (c[i] != ' ')
          sNew = new String(sNew + c[i]);
      }
    }

    return (sNew);
  }

  /** Method removeParenthesis()
   * <p> This method will remove the parenthesis in the String
   * @param  str  = The string to be converted
   * @return sNew = The text converted
   */

  public static String removeParenthesis( String str )
  {
    String sNew = new String("");
    char   c[]  = null;

    if (str != null)
    {
      c = str.toCharArray();

      for (int i=0; i<c.length; i++)
      {
        if ((c[i] != '(') && (c[i] != ')'))
          sNew = new String(sNew + c[i]);
      }
    }

    return (sNew);
  }

  /** Method setLegend( rockColorListStruct st)
   * <p> This method will create a unique list of Colors from the rock
   *     color List data structure
   * @param  st     = rock color list data structure
   * @return stList = rock color chart list data structure
   */

  public static rockColorChartListStruct setLegend( rockColorListStruct st)
  {
    rockColorChartListStruct stList = null;
    rockColorChartStruct     stTemp = null;
    int                      iFound = -1;

    if (st != null)
    {
//      stList = new rockColorChartListStruct();

      for (int i=0; i<st.iCount; i++)
      {
        for (int k=0; k<3; k++)
        {
          if (st.stItem[i].sMunsell[k].length() > 0)
          {
            iFound = -1;
            if (stList != null)
            {
              for (int j=0; j<stList.iCount; j++)
              {
                if (st.stItem[i].sMunsell[k].equals(stList.stItem[j].sMunsell))
                {
                  iFound = 1;
                }
              }
            }

            if (iFound == -1)
            {
              stTemp          = new rockColorChartStruct();
              stTemp.sMunsell = new String(st.stItem[i].sMunsell[k]);
              stTemp.sName    = new String(st.stItem[i].sColor[k]);
              stTemp.sHex     = new String(st.stItem[i].sRGBHex[k]);
              for (int p=0; p<3; p++)
                stTemp.iRGB[p]   = st.stItem[i].iRGB[k][p];

              stList = add( stTemp, stList );
            }
          }
        }
      }
    }

    return (stList);
  }

  /** Method print( rockColorChartListStruct st )
   * <p> This method will print the contents of the color chart data structure
   * @param  st = Color list data structure
   */

  public static void print( rockColorChartListStruct st )
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        System.out.println( st.stItem[i].sMunsell + " " +
                            st.stItem[i].sName    + " " +
                            st.stItem[i].sHex     + " {" +
                            st.stItem[i].iRGB[0]  + " " +
                            st.stItem[i].iRGB[1]  + " " +
                            st.stItem[i].iRGB[2]  + "}" );
      }
    }
  }
}