/*
 * @rockColorChartPlot.java Version 1.1 11/10/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.color.plot;

import java.util.Observable;
import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;

import iqstrat.iqstratStruct;
import rock.color.rockColorChartStruct;
import rock.color.rockColorChartListStruct;

/** CLASS rockColorChartPlot
 *  <p> This Class will create the Rock Color Chart Plot Panel
 *
 *  @version 1.1 11/10/2009
 *  @author  John R. Victorine
 */

public class rockColorChartPlot extends Canvas implements MouseListener
{
  private Observable    notifier = null; // Observable Object
  private iqstratStruct stStruct = null; // Global IQSTRAT Data Structure

  private int    iColor       = -1; // Color Identifier
  private String sMunsellCode = ""; // Munsell Color Code
  private String sColor       = ""; // ISCC Color Name
  private String sHex         = "";
  private int    iRGB[]       = { 255, 255, 255 };

  private rockColorChartStruct st = null; // Selected Color Value

  // Color Numbers

  public static final int    _R  = 0;
  public static final int    _YR = 1;
  public static final int    _Y  = 2;
  public static final int    _GY = 3;
  public static final int    _G  = 4;
  public static final int    _BG = 5;
  public static final int    _B  = 6;
  public static final int    _PB = 7;
  public static final int    _P  = 8;
  public static final int    _RP = 9;
  public static final int    _N  = 10;

  private int iHue = _R;

  public static final String _HUE[] = { "R", "YR", "Y", "GY", "G",
                                        "BG", "B", "PB", "P", "RP", "N" };

  public static final String HUE[]  = { "Red", "Yellow Red", "Yellow",
                                        "Green Yellow", "Green", "Blue Green",
                                        "Blue", "Purple Blue", "Purple",
                                        "Red Purple"};

  public static final int    _2p5 = 0;
  public static final int    _5p0 = 1;
  public static final int    _7p5 = 2;
  public static final int    _10p = 3;

  public static final double HUE_NO[] = { 2.5, 5.0, 7.5, 10.0 };

  private int iHueNo = _5p0;

  public static final double CHROMA[] = { 1, 2, 3, 4, 6, 8 };

  public static final double ROCK_VALUE[] =
      { 9.0, 8.0, 7.0, 6.0, 5.0, 4.0, 3.0, 2.0, 1.0 };

  public static final double SOIL_VALUE[] =
      { 8.0, 7.0, 6.0, 5.0, 4.0, 3.0, 2.5, 2.0 };

  public static final int _ROCK_CATALOG = 0;
  public static final int _SOIL_CATALOG = 1;

  public int iCatalog = _ROCK_CATALOG;
/*
  public static final int _ISCC_NO        = rock.color.rockMunsellColorStruct._ISCC_NO;
  public static final int _ISCC_COLOR     = rock.color.rockMunsellColorStruct._ISCC_COLOR;
  public static final int _MUNSELL_NO     = rock.color.rockMunsellColorStruct._MUNSELL_NO;
  public static final int _MUNSELL_HUE    = rock.color.rockMunsellColorStruct._MUNSELL_HUE;
  public static final int _MUNSELL_VALUE  = rock.color.rockMunsellColorStruct._MUNSELL_VALUE;
  public static final int _MUNSELL_CHROMA = rock.color.rockMunsellColorStruct._MUNSELL_CHROMA;
  public static final int _MUNSELL_CODE   = rock.color.rockMunsellColorStruct._MUNSELL_CODE;
  public static final int _RGB_HEX        = rock.color.rockMunsellColorStruct._RGB_HEX;
*/
  private static final int WIDTH         = 360; //600;
  private static final int HEIGHT        = 675; //720;

  private static final int HUE_WIDTH     = 300;
  private static final int HUE_START     = 36;
  private static final int COL_START     = 60;

  private static final int TEXT_LARGE    = 12;
  private static final int TEXT          = 10;
  private static final int ROW           = 75;
  private static final int COLUMN        = 50;

  private static final int COLOUR_WIDTH  = 40;
  private static final int COLOUR_HEIGHT = 20;

//  private static final int _ROWS    = 9;
//  private static final int _COLUMNS = 5;//4;

  /** CONSTRUCTOR rockColorChartPlot()
   *  <p> This is the constructor for this class.
   *  @param notifier   = Observable Object
   *  @param stStruct   = The Global IQSTRAT Data Structure
   */

  public rockColorChartPlot( Observable notifier, iqstratStruct stStruct )
  {
    this.notifier = notifier;
    this.stStruct = stStruct;

    this.setBackground(Color.white);
    addMouseListener(this);
  }

  /** Method close()
   * <p> This method will force Java to deallocate memory
   */

  public void close()
  {
    notifier     = null;
    sMunsellCode = null;
    sColor       = null;
    sHex         = null;
    iRGB         = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Width of the plot panel.
   * @return WIDTH = the width of the plot panel
   */

  public int getPlotWidth() { return (WIDTH); }

  /** Method getPlotHeight()
   * <p> This method will return the Width of the plot panel.
   * @return the height of the plot panel
   */

  public int getPlotHeight() { return (HEIGHT); }

  /** Method getColorID()
   * <p> This method will return color identifier
   * @return iColor = the color identifier or the row number in the lookup table
   */

  public int    getColorID() { return (iColor); }

  /** Method getMunsellCode()
   * <p> This method will return the Munsell Code
   * @return sMunsellCode = the Munsell Code
   */

  public String getMunsellCode() { return (sMunsellCode); }

  /** Method getColorName()
   * <p> This method will return the ISCC Color Name.
   * @return sColor = The ISCC Color Name
   */

  public String getColorName() { return (sColor); }

  /** Mthod getRGBHex()
   * <p> This method will return the RGB Hex String
   * @return sHex = The RGB Hex String
   */

  public String getRGBHex() { return (sHex); }

  /** Mthod getRGB()
   * <p> This method will return the RGB Values
   * @return iRGB = The RGB Values
   */

  public int[] getRGB() { return (iRGB); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setCatalog( int iCatalog )
   * <p> This method will set the type of catalog to display
   * @param iCatalog = the catalog to display
   */

  public void setCatalog( int iCatalog )
  {
    this.iCatalog = iCatalog;
    repaint();
  }

  /** Method setHue(int iHue)
   * <p> This method will set the Hue to plot
   * @param iHue = the color hue identifier
   */

  public void setHue( int iHue )
  {
    this.iHue = iHue;
    repaint();
  }

  /** Method setHueNo(int iHueNo)
   * <p> This method will set the Hue Number to plot
   * @param iHueNo = the color hue number
   */

  public void setHueNo( int iHueNo )
  {
    this.iHueNo = iHueNo;
    repaint();
  }

  /* =============================================================== *
   * -------------------- MouseListener Methods -------------------- *
   * =============================================================== */

  public void mouseMoved(MouseEvent event) { }
  public void mouseEntered(MouseEvent event) { }
  public void mouseExited(MouseEvent event) { }
  public void mousePressed(MouseEvent event) { }
  public void mouseDragged(MouseEvent event) {}
  public void mouseReleased(MouseEvent event) {}

  /** Method mouseClicked()
   *  <p> Mouse Clicked Method retrieve the location of the mouse pointer
   *      location.
   * @param event - Mouse Clicked Event
   */

  public void mouseClicked(MouseEvent event)
  {
    int    iXm     = event.getX();
    int    iYm     = event.getY();
    int    iX      = 0;
    int    iY      = 0;

    rockColorChartListStruct stList = null;
    double      dValue[] = null;

    switch (iCatalog)
    {
      case _ROCK_CATALOG:
        dValue = ROCK_VALUE;
        stList = stStruct.stRockColor;
        break;

      case _SOIL_CATALOG:
        dValue = SOIL_VALUE;
        stList = stStruct.stSoilColor;
        break;
    }

    if (stList != null)
    {
      // Color Catalog List
      for (int m=0; m<stList.iCount; m++)
      {
        if ((HUE_NO[iHueNo] == stList.stItem[m].dHue) &&
            (_HUE[iHue]).equals(stList.stItem[m].sColor))
        {
          // FOR EACH Row in the Hue Panel
          for (int i=0; i<dValue.length; i++)
          {
            // Does Value Match?
            if (dValue[i] == stList.stItem[m].dValue)
            {
              // FOR EACH Column in the Row
              for (int j=0; j<CHROMA.length; j++)
              {
                // Does Chroma Match?
                if (CHROMA[j] == stList.stItem[m].dChroma)
                {
                  iY = HUE_START + i * ROW;
                  iX = COL_START + j * COLUMN;

                  if ((iXm > iX) && (iXm < iX+COLOUR_WIDTH) &&
                      (iYm > iY) && (iYm < iY+COLOUR_HEIGHT))
                  {
                    iColor = m;
                    sMunsellCode = new String( stList.stItem[m].sMunsell );
                    sColor       = new String( stList.stItem[m].sName );
                    sHex         = new String( stList.stItem[m].sHex );
                    for (int p=0; p<3; p++)
                      iRGB[p] = stList.stItem[m].iRGB[p];

                    if (notifier != null)
                      notifier.notifyObservers(new String("Color Selected"));
                  }
                }
              }
            }
          }
        }
      }
    }
/*
    int    iXm     = event.getX();
    int    iYm     = event.getY();
    int    iCol    = 9;
    int    jCol    = 0;
    int    iX      = 0;
    int    iY      = 0;
    String sHue[]  = { "5", "10" };
    String sValue  = "";
    String sChroma = "";

    for (int m=0; m<rock.color.rockMunsellColorStruct.COLORS.length; m++)
    {
      // FOR EACH HUES PANEL
      for (int k=0; k<2; k++)
      {
        iCol = 9;  // Set the Column Number to the top of the panel

        // FOR EACH Row in the Hue Panel
        if ((sHue[k].equals(
               rock.color.rockMunsellColorStruct.COLORS[m][_MUNSELL_NO])) &&
            (_HUE[iHue].equals(
               rock.color.rockMunsellColorStruct.COLORS[m][_MUNSELL_HUE])))
        {
          for (int i=0; i<_ROWS; i++)
          {
            sValue = "" + iCol;
            if (sValue.equals(
                rock.color.rockMunsellColorStruct.COLORS[m][_MUNSELL_VALUE]))
            {
              // FOR EACH Column in the Row
              for (int j=0; j<_COLUMNS; j++)
              {
                jCol = (j+1)*2;
                sChroma = "" + jCol;

                // Check Chroma value

                if (sChroma.equals(
                    rock.color.rockMunsellColorStruct.COLORS[m][_MUNSELL_CHROMA]))
                {
                   iY = HUE_START + i * ROW;
                   iX = COL_START + j * COLUMN + k * HUE_WIDTH;

                   if ((iXm > iX) && (iXm < iX+COLOUR_WIDTH) &&
                       (iYm > iY) && (iYm < iY+COLOUR_HEIGHT))
                   {

System.out.println(
"Munsell=" + rock.color.rockMunsellColorStruct.COLORS[m][_MUNSELL_CODE] +
" Text=" + rock.color.rockMunsellColorStruct.COLORS[m][_ISCC_COLOR]);

                     iColor = m;
                     sMunsellCode = new String(
                       rock.color.rockMunsellColorStruct.COLORS[m][_MUNSELL_CODE]);
                     sColor = new String(
                       rock.color.rockMunsellColorStruct.COLORS[m][_ISCC_COLOR]);
                     sHex   = new String(
                       rock.color.rockMunsellColorStruct.COLORS[m][_RGB_HEX]);

                     if (notifier != null)
                       notifier.notifyObservers(new String("Color Selected"));
                   }
                }  // END Check Chroma value
              }  // END FOR EACH Column in the Row
            }
            iCol--;
          }
        }  // FOR EACH Row in the Hue Panel
      }  // END FOR EACH HUES PANEL

    }
*/
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawBlock()
   * <p> This method will draw and fill a rectangular block
   *  @param g          = Graphics Handle;
   *  @param iRed       = Red Value for RGB Color
   *  @param iGreen     = Green Value for RGB Color
   *  @param iBlue      = Blue Value for RGB Color
   *  @param iXp        = X Starting Position
   *  @param iYp        = Y Starting Position
   *  @param iWide      = Width  of block
   *  @param iHigh      = Height of block
   */

  public void drawBlock(Graphics g,
                        int iRed, int iGreen, int iBlue,
                        int iXp,   int iYp,
                        int iWide, int iHigh)
  {
    g.setColor(new Color(iRed, iGreen, iBlue));
    g.fillRect(iXp, iYp, iWide, iHigh);
  }

  /** Method draw(Graphics g)
   * <p> This method will create a LAS Plot Track
   * @param g - Graphics Parameter.
   */

  public void draw(Graphics g)
  {
    rockColorChartListStruct stList = null;
    double      dValue[] = null;
    int         iX       = 0;
    int         iY       = 0;
    int         len      = 0;
    String      sText    = "";
    String      sMunsell = "";
    String      sText5   = "";
    String      delims   = "[ ]+";
    String[]    tokens   = null;
    Font        fsb      = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm     = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    switch (iCatalog)
    {
      case _ROCK_CATALOG:
        dValue = ROCK_VALUE;
        stList = stStruct.stRockColor;
        break;

      case _SOIL_CATALOG:
        dValue = SOIL_VALUE;
        stList = stStruct.stSoilColor;
        break;
    }

    // Draw Hue Title

    sText = "Hue: " + HUE_NO[iHueNo] + " " + HUE[iHue];
    len   = 6*sText.length();
    g.drawString( sText, (HUE_WIDTH - len)/2,  TEXT_LARGE );

    // Draw Chroma Axis Title

    g.drawString( "Chroma (Saturation)", COL_START + COLUMN,   2*TEXT_LARGE );

    // Draw Value Axis Title

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Value (Lightness)", 200, -1*(20));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    fsb  = new Font("Serif", Font.PLAIN, 11);
    fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    for (int j=0; j<CHROMA.length; j++)
    {
      g.drawString( "" + CHROMA[j],
                    COL_START + j * COLUMN + COLOUR_WIDTH/2,
                    3*TEXT_LARGE-2 );
    }

    for (int i=0; i<dValue.length; i++)
    {
      g.drawString( ""+dValue[i], COL_START-20, HUE_START + TEXT_LARGE + i*ROW );
    }

    fsb  = new Font("Serif", Font.PLAIN, 10);
    fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    if (stList != null)
    {
      // Color Catalog List
      for (int m=0; m<stList.iCount; m++)
      {
        // IS Color have the same Hue?
        if ((HUE_NO[iHueNo] == stList.stItem[m].dHue) &&
            (_HUE[iHue]).equals(stList.stItem[m].sColor))
        {
          // FOR EACH Row in the Hue Panel
          for (int i=0; i<dValue.length; i++)
          {
            // Does Value Match?
            if (dValue[i] == stList.stItem[m].dValue)
            {
              // FOR EACH Column in the Row
              for (int j=0; j<CHROMA.length; j++)
              {
                // Does Chroma Match?
                if (CHROMA[j] == stList.stItem[m].dChroma)
                {
                  iY = HUE_START + i * ROW;
                  iX = COL_START + j * COLUMN;

                  drawBlock( g,
                             stList.stItem[m].iRGB[0],
                             stList.stItem[m].iRGB[1],
                             stList.stItem[m].iRGB[2],
                             iX, iY,
                             COLOUR_WIDTH, COLOUR_HEIGHT );

                  sMunsell = stList.stItem[m].sMunsell;
                  sText5   = stList.stItem[m].sName;
                  tokens   = sText5.split( delims );

                  g.setColor(Color.black);

                  len = tokens.length;
                  for (int n=0; n<len; n++)
                  {
                    g.drawString( tokens[n], iX, COLOUR_HEIGHT+iY+(n+1)*TEXT );
                  }
                  g.drawString( sMunsell, iX, COLOUR_HEIGHT+iY+(len+1)*TEXT );

                } // END CHROMA?
              }   // END FOR EACH Column in the Row
            }     // END VALUE?
          }       // END FOR EACH Row in the Hue Panel
        }         // END HUE?
      }           // END CATALOG
    }
  }

  /** Method draw(Graphics g)
   * <p> This method will create a LAS Plot Track
   * @param g - Graphics Parameter.
   *

  public void draw(Graphics g)
  {
    int    iCol       = 9;
    int    jCol       = 0;
    int    iRed       = 0;
    int    iGreen     = 0;
    int    iBlue      = 0;
    int    iX         = 0;
    int    iY         = 0;
    int    len        = 0;
    String sHue[]     = { "5", "10" };
    String sValue     = "";
    String sChroma    = "";
    String sText5     = "";
    String sMunsell   = "";
    String sText10    = "";
    String delims     = "[ ]+";
    String[] tokens   = null;
    String sText      = "";

    Font        fsb   = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm  = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    sText = "Hue: 5 " + HUE[iHue];
    len   = 6*sText.length();
    g.drawString(sText,
                 (HUE_WIDTH - len)/2,  TEXT_LARGE);
    g.drawString("Chroma (Saturation)",
                 COL_START + COLUMN,   2*TEXT_LARGE);

    sText = "Hue: 10 " + HUE[iHue];
    len   = 4*sText.length();
    g.drawString(sText,
                 (HUE_WIDTH - len)/2 + HUE_WIDTH, TEXT_LARGE);
    g.drawString("Chroma (Saturation)",
                 COL_START + COLUMN + HUE_WIDTH, 2*TEXT_LARGE);

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Value (Lightness)", 200, -1*(20));
    g2.drawString("Value (Lightness)", 200, -1*(20+HUE_WIDTH));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    fsb   = new Font("Serif", Font.PLAIN, 11);
    fsbm  = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (int j=0; j<_COLUMNS; j++)
    {
      jCol = (j+1)*2;
      sChroma = "" + jCol;
      g.drawString(sChroma,
                   COL_START + j * COLUMN + COLOUR_WIDTH/2,
                   3*TEXT_LARGE-2);
      g.drawString(sChroma,
                   COL_START + j * COLUMN + COLOUR_WIDTH/2 + HUE_WIDTH,
                   3*TEXT_LARGE-2);
    }

    iCol = 9;  // Set the Column Number to the top of the panel
    for (int i=0; i<_ROWS; i++)
    {
      sValue = "" + iCol;
      g.drawString( sValue, COL_START-20, HUE_START + TEXT_LARGE + i*ROW );
      g.drawString( sValue, COL_START-20+HUE_WIDTH, HUE_START + TEXT_LARGE + i*ROW );
      iCol--;
    }

    fsb   = new Font("Serif", Font.PLAIN, 10);
    fsbm  = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (int m=0; m<rock.color.rockMunsellColorStruct.COLORS.length; m++)
    {
      // FOR EACH HUES PANEL
      for (int k=0; k<2; k++)
      {
        iCol = 9;  // Set the Column Number to the top of the panel

        // FOR EACH Row in the Hue Panel
        if ((sHue[k].equals(
                         rock.color.rockMunsellColorStruct.COLORS[m][_MUNSELL_NO])) &&
                        (_HUE[iHue].equals(
                         rock.color.rockMunsellColorStruct.COLORS[m][_MUNSELL_HUE])))
        {
          for (int i=0; i<_ROWS; i++)
          {
            sValue = "" + iCol;
            if (sValue.equals(
                rock.color.rockMunsellColorStruct.COLORS[m][_MUNSELL_VALUE]))
            {
              // FOR EACH Column in the Row
              for (int j=0; j<_COLUMNS; j++)
              {
                jCol = (j+1)*2;
                sChroma = "" + jCol;

                // Check Chroma value

                if (sChroma.equals(
                    rock.color.rockMunsellColorStruct.COLORS[m][_MUNSELL_CHROMA]))
                {
                   iRed = rock.color.rockColorUtility.getColor(0,
                         rock.color.rockMunsellColorStruct.COLORS[m][_RGB_HEX]);
                   iGreen = rock.color.rockColorUtility.getColor(1,
                         rock.color.rockMunsellColorStruct.COLORS[m][_RGB_HEX]);
                   iBlue = rock.color.rockColorUtility.getColor(2,
                         rock.color.rockMunsellColorStruct.COLORS[m][_RGB_HEX]);

                   iY = HUE_START + i * ROW;
                   iX = COL_START + j * COLUMN + k * HUE_WIDTH;

                   drawBlock( g, iRed, iGreen, iBlue,
                              iX, iY, COLOUR_WIDTH, COLOUR_HEIGHT);

                   switch (k)
                   {
                     case 0: // Hue 5 Panel
                       sMunsell =
                           rock.color.rockMunsellColorStruct.COLORS[m][_MUNSELL_CODE];

                       sText5 =
                          rock.color.rockMunsellColorStruct.COLORS[m][_ISCC_COLOR];

                       tokens = sText5.split(delims);
                       break;

                     case 1: // Hue 10 Panel
                       sMunsell =
                           rock.color.rockMunsellColorStruct.COLORS[m][_MUNSELL_CODE];

                       sText10 =
                          rock.color.rockMunsellColorStruct.COLORS[m][_ISCC_COLOR];

                       tokens = sText10.split(delims);
                       break;
                   }

                   g.setColor(Color.black);

                   len = tokens.length;
                   for (int n=0; n<len; n++)
                   {
                     g.drawString(tokens[n], iX, COLOUR_HEIGHT+iY+(n+1)*TEXT);
                   }
                   g.drawString(sMunsell, iX, COLOUR_HEIGHT+iY+(len+1)*TEXT);

                }  // END Check Chroma value
              }  // END FOR EACH Column in the Row
            }
            iCol--;
          }
        }  // FOR EACH Row in the Hue Panel
      }  // END FOR EACH HUES PANEL

    }
  }
*/
  /** Method paint(Graphics g)
   * <p> This method will paint Porosity & Permeability Plot
   * @param g - Graphics Parameter.
   */

  public void paint(Graphics g)
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, WIDTH, HEIGHT);

    draw(g);
  }
}