/*
 * @ReadColorChartXML.java Version 1.0 05/14/2009
 *
 * Copyright (c) 2009 John R. Victorine
 * All Rights Reserved.
 */

package rock.color.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import rock.color.rockColorChartListStruct;
import rock.color.rockColorChartStruct;

/** Class ReadColorChartXML()
 * <p> This Class will parse the Rock Color XML Data Stream
 *     directly to the Rock Color List Data Structure.
 *
 *  @version 1.1 05/14/2009
 *  @author  John R. Victorine
 */

public class ReadColorChartXML extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  private int iType = FILE;

  private static final int _NONE     = 0;
  private static final int _COLOR    = 1;

  private int iData = _NONE;

  public static final String ROOTNAME       = "colors";  // ROOT NAME
  public static final String RECORDS        = "records"; // Number of Records
  public static final String REFERENCE      = "ref";     // Reference

  public static final String COLOR          = "color";   // ELEMENT

  public static final String COLOR_NAME     = "name";    // Color Name
  public static final String HUE            = "H";       // Hue Number
  public static final String HUE_NUMBER     = "Hn";      // Hue Number
  public static final String HUE_COLOR      = "Hc";      // Hue Color
  public static final String MUNSELL_VALUE  = "V";       // Saturation
  public static final String MUNSELL_CHROMA = "C";       // Chroma
  public static final String MUNSELL_CODE   = "munsell"; // Munsell  Code
  public static final String COLOR_x        = "x";       // x
  public static final String COLOR_y        = "y";       // y
  public static final String COLOR_Yo       = "Yo";      // Yo
  public static final String RGB_RED        = "r";       // RGB Red Value
  public static final String RGB_GREEN      = "g";       // RGB Green Value
  public static final String RGB_BLUE       = "b";       // RGB Blue Value

  private int                      iRows   = 0;
  private int                      iCount  = -1;
  private rockColorChartListStruct stChart = new rockColorChartListStruct();

  private int    iError = 0;
  private String sError = "";

  /** Method ReadColorChartXML()
   *  This is the Default Constructor for this class.
   */

  public ReadColorChartXML() { iType = FILE; }

  /** Method ReadColorChartXML()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadColorChartXML(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public rockColorChartListStruct Process(String filename)
  {
    Read(filename);

    stChart.iCount = iCount+1;

    return (stChart);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println("ReadColorChartXML.Read() " + e);
          e.printStackTrace();
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println("ReadColorChartXML.Read() " + e);
          e.printStackTrace();
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement( String namespaceURI,
   *                       String lName,
   *                       String qName,
   *                       Attributes attrs )
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement( String namespaceURI,
                            String lName, // local name
                            String qName, // qualified name
                            Attributes attrs ) throws SAXException
  {
    String eName = lName; // element name

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    // If the ELEMENT Well is detected then initialize the structure.

    if (eName.equals(COLOR))
    {
      iCount++;
      stChart.stItem[iCount] = new rockColorChartStruct();
      iData  = _COLOR;
    }

    // Add Attributes to the Well Header Data List Structure

    if (attrs != null)
    {
      for (int i=0; i < attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          iRows   = cmn.cmnString.stringToInt( sOut.trim() );
          if (iRows > 0)
            stChart.stItem = new rockColorChartStruct[iRows];
        }

        // Parse each attribute and add it to the structure

        if (iCount > -1)
        {
          switch (iData)
          {
            case _NONE:
              break;

            case _COLOR:
              if (iCount < iRows)
              {
                stChart.stItem[iCount] =
                    ParseData( aName, sOut.trim(), stChart.stItem[iCount] );
              }
              break;
          }
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement( String namespaceURI,
                          String sName, // simple name
                          String qName  // qualified name
                         ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters( char buf[], int offset, int len )
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters( char buf[], int offset, int len ) throws SAXException
  {
    // Ignore it
  }

  /** Method ignorableWhitespace( char buf[], int offset, int len )
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace( char buf[], int offset, int len )
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction( String target, String data )
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction( String target, String data )
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess( String sin )
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess( String sin )
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** ParseData( String sIdentifier, String sData, rockColorChartListStruct st )
   * <p> This method will parse the Data Stream
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData      = Measured Core Data String
   * @param  st         = Rock Color Data Structure
   * @return st         = Rock Color Data Structure
   */

  private rockColorChartStruct ParseData( String sIdentifier,
                                          String sData,
                                          rockColorChartStruct st )
  {
    if (sIdentifier.equals(COLOR_NAME))   { st.sName = new String( sData ); }

    if (sIdentifier.equals(MUNSELL_CODE)) { st.sMunsell = new String( sData ); }

    if (sIdentifier.equals(HUE))          { st.sHue = new String( sData ); }

    if (sIdentifier.equals(HUE_NUMBER))
    {
      st.dHue = cmn.cmnString.stringToDouble( sData );
    }

    if (sIdentifier.equals(HUE_COLOR))  { st.sColor = new String( sData ); }

    if (sIdentifier.equals(MUNSELL_VALUE))
    {
      st.dValue = cmn.cmnString.stringToDouble( sData );
    }

    if (sIdentifier.equals(MUNSELL_CHROMA))
    {
      st.dChroma = cmn.cmnString.stringToDouble( sData );
    }

    if (sIdentifier.equals(COLOR_x))
    {
      st.x = cmn.cmnString.stringToDouble( sData );
    }

    if (sIdentifier.equals(COLOR_y))
    {
      st.y = cmn.cmnString.stringToDouble( sData );
    }

    if (sIdentifier.equals(COLOR_Yo))
    {
      st.Yo = cmn.cmnString.stringToDouble( sData );
    }

    if (sIdentifier.equals(RGB_RED))
    {
      st.iRGB[0] = cmn.cmnString.stringToInt( sData );
    }

    if (sIdentifier.equals(RGB_GREEN))
    {
      st.iRGB[1] = cmn.cmnString.stringToInt( sData );
    }

    if (sIdentifier.equals(RGB_BLUE))
    {
      st.iRGB[2] = cmn.cmnString.stringToInt( sData );
      st.sHex    = new String( rock.color.rockColorChartUtility.getHEX(
                                  st.iRGB[0], st.iRGB[1], st.iRGB[2] ));
    }

    return (st);
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}