/*
 * @rockColorChartPlotFrame.java Version 1.1 11/10/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package rock.color.gui;

import java.util.Observer;
import java.util.Observable;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import iqstrat.iqstratStruct;

import rock.color.rockColorChartListStruct;
import rock.color.rockColorStruct;
import rock.color.gui.rockColorChartPlotPanel;
import rock.color.gui.rockColorDataEntryPanel;

import util.utilImageIO;

/** Class rockColorChartPlotFrame
 *  <p> This Class will allow the user to select colors from a color panel
 *
 *  @version 1.1 11/10/2009
 *  @author  John R. Victorine
 */

public class rockColorChartPlotFrame extends JFrame implements ActionListener, Observer
{
  // Input Variables

  private Observable    pNotifier = null; // Observable Object
  private iqstratStruct stStruct = null; // Global IQSTRAT Data Structure

  // Global Frame Variables

  private Observable              notifier = null; // Observable Object
  private rockColorChartPlotPanel plot     = null;
  private rockColorDataEntryPanel pColor   = null; // Color Data Panel

  private int iCatalog = rock.color.plot.rockColorChartPlot._ROCK_CATALOG;
  private int iHueNo   = rock.color.plot.rockColorChartPlot._5p0;
  private int iHue     = rock.color.plot.rockColorChartPlot._R;

  // Global Frame Widgets

  private static final int RGB[][] =
  {
    { 255,   0,   0 }, // R
    { 255, 128,   0 }, // YR
    { 255, 255,   0 }, // Y
    {  64, 255,   0 }, // GY
    {   0, 220,   0 }, // G
    {  80, 180, 180 }, //{   0, 220, 128 }, // BG
    {   0,   0, 220 }, // B
    { 180, 180, 255 }, //{   0,  64, 255 }, // PB
    { 220,   0, 220 }, // P
    { 255,   0, 128 }  // RP
  };

  // Type of Color Catalog

  private JRadioButton rbRock = new JRadioButton();  // Rock Color Catalog
  private JRadioButton rbSoil = new JRadioButton();  // Soil Color Catalog

  // Hue Number Buttons

  private JRadioButton rb2p5 = new JRadioButton();   // 2.5
  private JRadioButton rb5p0 = new JRadioButton();   // 5.0
  private JRadioButton rb7p5 = new JRadioButton();   // 7.5
  private JRadioButton rb10p = new JRadioButton();   // 10.0

  // Neutral Color Buttons

  private JButton btnN9 = new JButton();
  private JButton btnN8 = new JButton();
  private JButton btnN7 = new JButton();
  private JButton btnN6 = new JButton();
  private JButton btnN5 = new JButton();
  private JButton btnN4 = new JButton();
  private JButton btnN3 = new JButton();
  private JButton btnN2 = new JButton();
  private JButton btnN1 = new JButton();

  // Hue Color Buttons

  private JButton btnR  = new JButton();  // Red
  private JButton btnYR = new JButton();  // Yellow Red
  private JButton btnY  = new JButton();  // Yellow
  private JButton btnGY = new JButton();  // Green Yellow
  private JButton btnG  = new JButton();  // Green
  private JButton btnBG = new JButton();  // Blue Green
  private JButton btnB  = new JButton();  // Blue
  private JButton btnRP = new JButton();  // Red Purple
  private JButton btnPB = new JButton();  // Purple Blue
  private JButton btnP  = new JButton();  // Purple

  private JMenuItem mExit = null; // Exit Profile Plot

  // Dialog Position & Size

  private int iXStart = 0;
  private int iYStart = 0;
  private int iWidth  = 0;
  private int iHeight = 0;

  /** Constructor rockColorChartPlotFrame()
   *  <p> This is the Constructor for this class.
   *  @param stStruct = The Global IQSTRAT Data Structure
   */

  public rockColorChartPlotFrame( iqstratStruct stStruct )
  {
    try
    {
      this.stStruct = stStruct;
      jbInit();
      addWindowListener( new rockColorChartPlotFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Constructor rockColorChartPlotFrame()
   *  <p> This is the Constructor for this class.
   *  @param pNotifier = Observable Object
   *  @param stStruct = The Global IQSTRAT Data Structure
   */

  public rockColorChartPlotFrame( Observable    pNotifier,
                                  iqstratStruct stStruct )
  {
    try
    {
      this.pNotifier = pNotifier;
      this.stStruct  = stStruct;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Constructor rockColorChartPlotFrame()
   *  <p> This is the Constructor for this class.
   *  @param pNotifier = Observable Object
   *  @param stStruct = The Global IQSTRAT Data Structure
   *  @param iXStart  = the x-coordinate of Data entry dialog
   *  @param iYStart  = the y-coordinate of Data entry dialog
   *  @param iWidth   = the width of Data entry dialog
   *  @param iHeight  = the height of Data entry dialog
   */

  public rockColorChartPlotFrame( Observable pNotifier,
                                  iqstratStruct stStruct,
                                  int iXStart, int iYStart,
                                  int iWidth,  int iHeight )
  {
    try
    {
      this.pNotifier = pNotifier;
      this.stStruct = stStruct;
      this.iXStart  = iXStart;
      this.iYStart  = iYStart;
      this.iWidth   = iWidth;
      this.iHeight  = iHeight;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlEast     = new JPanel();
    JPanel pnlNeutrals = new JPanel();

    JPanel pnlWest     = new JPanel();
    JPanel pnlCatalog  = new JPanel();
    JPanel pnlNumber   = new JPanel();
    JPanel pnlColor    = new JPanel();

    JPanel pnlBase     = new JPanel();

    ButtonGroup groupNo      = new ButtonGroup();
    ButtonGroup groupCatalog = new ButtonGroup();

    TitledBorder titledBorderCatalog = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Color");
    titledBorderCatalog.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderHueNumber = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Hue #:");
    titledBorderHueNumber.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderHueColor = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Hue:");
    titledBorderHueColor.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderNeutral = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Gray:");
    titledBorderNeutral.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    JMenuBar    menuBar  = new JMenuBar();
    JMenu       menuFile = new JMenu("Close Dialog");

    // Build Menu

    mExit = new JMenuItem("Close");
    mExit.addActionListener(this);

    menuFile.add( mExit );

    menuBar.add(menuFile);

    this.setJMenuBar(menuBar);

    // Create an Observable

    notifier = new rockColorChartPlotFrameObservable();

    // Add this Dialog to the notifier as observer

    notifier.addObserver(this);

    // Build Frame

    this.setTitle("Rock Color Chart");
    this.getContentPane().setLayout(new BorderLayout());

    // East Panel

    pnlEast.setLayout(new BorderLayout());

    // Neutral Color Button Panel

    pnlNeutrals.setBorder(titledBorderNeutral);
    pnlNeutrals.setLayout(new GridLayout(9,1));
    pnlNeutrals.setPreferredSize(new Dimension(61, 450));

    btnN9.setBackground(new Color(228, 228, 228));
    btnN9.setPreferredSize(new Dimension(50, 30));
    btnN9.setText("N9");
    btnN9.addActionListener(this);

    btnN8.setBackground(new Color(202, 202, 202));
    btnN8.setPreferredSize(new Dimension(50, 30));
    btnN8.setText("N8");
    btnN8.addActionListener(this);

    btnN7.setBackground(new Color(175, 175, 175));
    btnN7.setPreferredSize(new Dimension(50, 30));
    btnN7.setText("N7");
    btnN7.addActionListener(this);

    btnN6.setBackground(new Color(158, 158, 158));
    btnN6.setPreferredSize(new Dimension(50, 30));
    btnN6.setText("N6");
    btnN6.addActionListener(this);

    btnN5.setBackground(new Color(123, 123, 123));
    btnN5.setForeground(Color.white);
    btnN5.setPreferredSize(new Dimension(50, 30));
    btnN5.setText("N5");
    btnN5.addActionListener(this);

    btnN4.setBackground(new Color(97, 97, 97));
    btnN4.setForeground(Color.white);
    btnN4.setPreferredSize(new Dimension(50, 30));
    btnN4.setText("N4");
    btnN4.addActionListener(this);

    btnN3.setBackground(new Color(72, 72, 72));
    btnN3.setForeground(Color.white);
    btnN3.setPreferredSize(new Dimension(50, 30));
    btnN3.setText("N3");
    btnN3.addActionListener(this);

    btnN2.setBackground(new Color(49, 49, 49));
    btnN2.setForeground(Color.white);
    btnN2.setPreferredSize(new Dimension(50, 30));
    btnN2.setText("N2");
    btnN2.addActionListener(this);

    btnN1.setBackground(new Color(29, 29, 29));
    btnN1.setForeground(Color.white);
    btnN1.setPreferredSize(new Dimension(50, 30));
    btnN1.setText("N1");
    btnN1.addActionListener(this);

    // West Panel

    pnlWest.setLayout(new BorderLayout());

    // -- Color Catalog Panel

    pnlCatalog.setBorder(titledBorderCatalog);
    pnlCatalog.setLayout(new GridLayout(2,1));

    rbRock.setFont(new java.awt.Font("Dialog", 1, 11));
    rbRock.setSelected(true);
    rbRock.setText("Rock");
    rbRock.addActionListener(this);

    rbSoil.setFont(new java.awt.Font("Dialog", 1, 11));
    rbSoil.setText("Soil");
    rbSoil.addActionListener(this);

    // -- Hue Number Selection Panel

    pnlNumber.setBorder(titledBorderHueColor);
    pnlNumber.setLayout(new GridLayout(4,1));

    rb2p5.setFont(new java.awt.Font("Dialog", 1, 11));
    rb2p5.setText("2.5");
    rb2p5.addActionListener(this);

    rb5p0.setFont(new java.awt.Font("Dialog", 1, 11));
    rb5p0.setSelected(true);
    rb5p0.setText("5.0");
    rb5p0.addActionListener(this);

    rb7p5.setFont(new java.awt.Font("Dialog", 1, 11));
    rb7p5.setText("7.5");
    rb7p5.addActionListener(this);

    rb10p.setFont(new java.awt.Font("Dialog", 1, 11));
    rb10p.setText("10.0");
    rb10p.addActionListener(this);

    // -- Hue Color Selection Panel

    pnlColor.setBorder(titledBorderHueColor);
    pnlColor.setLayout(new GridLayout(10,1));
    pnlColor.setPreferredSize(new Dimension(61, 450));

    btnR.setBackground(new Color(RGB[0][0], RGB[0][1], RGB[0][2]));
    btnR.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnR.setPreferredSize(new Dimension(50, 50));
    btnR.setText("R");
    btnR.addActionListener(this);

    btnYR.setBackground(new Color(RGB[1][0], RGB[1][1], RGB[1][2]));
    btnYR.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnYR.setPreferredSize(new Dimension(50, 50));
    btnYR.setText("YR");
    btnYR.addActionListener(this);

    btnY.setBackground(new Color(RGB[2][0], RGB[2][1], RGB[2][2]));
    btnY.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnY.setPreferredSize(new Dimension(50, 50));
    btnY.setText("Y");
    btnY.addActionListener(this);

    btnGY.setBackground(new Color(RGB[3][0], RGB[3][1], RGB[3][2]));
    btnGY.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnGY.setPreferredSize(new Dimension(50, 50));
    btnGY.setText("GY");
    btnGY.addActionListener(this);

    btnG.setBackground(new Color(RGB[4][0], RGB[4][1], RGB[4][2]));
    btnG.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnG.setPreferredSize(new Dimension(50, 50));
    btnG.setText("G");
    btnG.addActionListener(this);

    btnBG.setBackground(new Color(RGB[5][0], RGB[5][1], RGB[5][2]));
    btnBG.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnBG.setPreferredSize(new Dimension(50, 50));
    btnBG.setText("BG");
    btnBG.addActionListener(this);

    btnB.setBackground(new Color(RGB[6][0], RGB[6][1], RGB[6][2]));
    btnB.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnB.setPreferredSize(new Dimension(50, 50));
    btnB.setText("B");
    btnB.addActionListener(this);

    btnPB.setBackground(new Color(RGB[7][0], RGB[7][1], RGB[7][2]));
    btnPB.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnPB.setPreferredSize(new Dimension(50, 50));
    btnPB.setText("PB");
    btnPB.addActionListener(this);

    btnP.setBackground(new Color(RGB[8][0], RGB[8][1], RGB[8][2]));
    btnP.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnP.setPreferredSize(new Dimension(50, 50));
    btnP.setText("P");
    btnP.addActionListener(this);

    btnRP.setBackground(new Color(RGB[9][0], RGB[9][1], RGB[9][2]));
    btnRP.setFont(new java.awt.Font("Dialog", 1, 11));
//    btnRP.setPreferredSize(new Dimension(50, 50));
    btnRP.setText("RP");
    btnRP.addActionListener(this);

    setButtons(); // Set Buttons depending on data

    // Create Rock Color Chart Plot Panel

    plot = new rockColorChartPlotPanel( this, notifier, stStruct );

    // Create Rock Color Data Entry Panel

    pnlBase.setLayout(new BorderLayout());

    pColor = new rockColorDataEntryPanel( pNotifier, stStruct );

    // Attach Widgets to Frame

    this.getContentPane().add(pnlEast,  BorderLayout.EAST);
    pnlEast.add(pnlNeutrals,  BorderLayout.SOUTH);
    pnlNeutrals.add(btnN1, null);
    pnlNeutrals.add(btnN2, null);
    pnlNeutrals.add(btnN3, null);
    pnlNeutrals.add(btnN4, null);
    pnlNeutrals.add(btnN5, null);
    pnlNeutrals.add(btnN6, null);
    pnlNeutrals.add(btnN7, null);
    pnlNeutrals.add(btnN8, null);
    pnlNeutrals.add(btnN9, null);


    this.getContentPane().add(pnlWest,  BorderLayout.WEST);
    pnlWest.add(pnlNumber,  BorderLayout.CENTER);
    pnlWest.add(pnlColor,   BorderLayout.SOUTH);
    pnlColor.add(btnR,      null);
    pnlColor.add(btnYR,     null);
    pnlColor.add(btnY,      null);
    pnlColor.add(btnGY,     null);
    pnlColor.add(btnG,      null);
    pnlColor.add(btnBG,     null);
    pnlColor.add(btnB,      null);
    pnlColor.add(btnPB,     null);
    pnlColor.add(btnP,      null);
    pnlColor.add(btnRP,     null);

    pnlWest.add(pnlCatalog, BorderLayout.NORTH);
    pnlCatalog.add(rbRock,  null);
    pnlCatalog.add(rbSoil,  null);

    groupCatalog.add(rbRock);
    groupCatalog.add(rbSoil);

    this.getContentPane().add(plot, BorderLayout.CENTER);
    pnlNumber.add(rb2p5,    null);
    pnlNumber.add(rb5p0,    null);
    pnlNumber.add(rb7p5,    null);
    pnlNumber.add(rb10p,    null);

    groupNo.add(rb2p5);
    groupNo.add(rb5p0);
    groupNo.add(rb7p5);
    groupNo.add(rb10p);

    this.getContentPane().add(pnlBase,  BorderLayout.SOUTH);
    pnlBase.add(pColor,                 BorderLayout.CENTER);

    // Display the Frame

    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
//    this.setSize(new Dimension(450, d.height-50)); //700));
    this.setSize(new Dimension(550, 850));

    int iXPos = (d.width  - this.getSize().width) / 2;
    if (iWidth > 0) iXPos = iXStart + iWidth;
    if (this.getSize().width > d.width - iXPos)
      iXPos = d.width  - this.getSize().width;
    this.setLocation(iXPos, (d.height - this.getSize().height) / 2);

//    this.setResizable(false);
    this.setVisible(true);
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getColorID()
   * <p> This method will return color identifier
   * @return iColor = the color identifier or the row number in the lookup table
   */

  public int    getColorID() { return (plot.getColorID()); }

  /** Method getMunsellCode()
   * <p> This method will return the Munsell Code
   * @return sMunsellCode = the Munsell Code
   */

  public String getMunsellCode() { return (plot.getMunsellCode()); }

  /** Method getColorName()
   * <p> This method will return the ISCC Color Name.
   * @return sColor = The ISCC Color Name
   */

  public String getColorName() { return (plot.getColorName()); }

  /** Mthod getRGBHex()
   * <p> This method will return the RGB Hex String
   * @return sHex = The RGB Hex String
   */

  public String getRGBHex() { return (plot.getRGBHex()); }

  /** Mthod getRGB()
   * <p> This method will return the RGB Values
   * @return iRGB = The RGB Values
   */

  public int[] getRGB() { return (plot.getRGB()); }

  /** Method getData()
   * <p> This method will return the selected colors
   * @return the color data structure
   */

  public rockColorStruct getData() { return ( pColor.getData() ); }

  /** Method clear()
   * <p> This method will clear the textfields
   */

  public void clear()  { if (pColor != null) pColor.clear(); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setButtons()
   * <p> This method will set the buttons depending on the data and button
   *     configuration.
   */

  private void setButtons()
  {
    rockColorChartListStruct stList = null;
    int iColor[]  = { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 };
    int iNumber[] = { -1, -1, -1, -1 };

    switch (iCatalog)
    {
      case rock.color.plot.rockColorChartPlot._ROCK_CATALOG:
        this.setTitle("Rock Color Chart");
        stList = stStruct.stRockColor;
        break;

      case rock.color.plot.rockColorChartPlot._SOIL_CATALOG:
        this.setTitle("Munsell Soil Color Chart");
        stList = stStruct.stSoilColor;
        break;
    }

    if (stList != null)
    {
      rb2p5.setEnabled(false);   // 2.5
      rb5p0.setEnabled(false);   // 5.0
      rb7p5.setEnabled(false);   // 7.5
      rb10p.setEnabled(false);   // 10.0

      btnR.setEnabled(false);    // Red
      btnYR.setEnabled(false);   // Yellow Red
      btnY.setEnabled(false);    // Yellow
      btnGY.setEnabled(false);   // Green Yellow
      btnG.setEnabled(false);    // Green
      btnBG.setEnabled(false);   // Blue Green
      btnB.setEnabled(false);    // Blue
      btnPB.setEnabled(false);   // Purple Blue
      btnP.setEnabled(false);    // Purple
      btnRP.setEnabled(false);   // Red Purple

      for (int i=0; i<stList.iCount; i++)
      {
        for (int j=0; j<rock.color.plot.rockColorChartPlot.HUE_NO.length; j++)
        {
          if (stList.stItem[i].dHue ==
                rock.color.plot.rockColorChartPlot.HUE_NO[j])
          {
            iNumber[j] = 1;
          }
        }

        for (int j=0; j<rock.color.plot.rockColorChartPlot._HUE.length; j++)
        {
          if ((stList.stItem[i].dHue ==
                 rock.color.plot.rockColorChartPlot.HUE_NO[iHueNo]) &&
              (stList.stItem[i].sColor.equals(
                 rock.color.plot.rockColorChartPlot._HUE[j])))
          {
            iColor[j] = 1;
          }
        }
      }

      for (int j=0; j<rock.color.plot.rockColorChartPlot._HUE.length; j++)
      {
        switch (j)
        {
          case rock.color.plot.rockColorChartPlot._R:
            if (iColor[j] > -1) btnR.setEnabled(true);    // Red
            break;
          case rock.color.plot.rockColorChartPlot._YR:
            if (iColor[j] > -1) btnYR.setEnabled(true);   // Yellow Red
            break;
          case rock.color.plot.rockColorChartPlot._Y:
            if (iColor[j] > -1) btnY.setEnabled(true);    // Yellow
            break;
          case rock.color.plot.rockColorChartPlot._GY:
            if (iColor[j] > -1) btnGY.setEnabled(true);   // Green Yellow
            break;
          case rock.color.plot.rockColorChartPlot._G:
            if (iColor[j] > -1) btnG.setEnabled(true);    // Green
            break;
          case rock.color.plot.rockColorChartPlot._BG:
            if (iColor[j] > -1) btnBG.setEnabled(true);   // Blue Green
            break;
          case rock.color.plot.rockColorChartPlot._B:
            if (iColor[j] > -1) btnB.setEnabled(true);    // Blue
            break;
          case rock.color.plot.rockColorChartPlot._PB:
            if (iColor[j] > -1) btnPB.setEnabled(true);   // Purple Blue
            break;
          case rock.color.plot.rockColorChartPlot._P:
            if (iColor[j] > -1) btnP.setEnabled(true);    // Purple
            break;
          case rock.color.plot.rockColorChartPlot._RP:
            if (iColor[j] > -1) btnRP.setEnabled(true);   // Red Purple
            break;
        }
      }

      for (int j=0; j<rock.color.plot.rockColorChartPlot.HUE_NO.length; j++)
      {
        switch (j)
        {
          case rock.color.plot.rockColorChartPlot._2p5:
            if (iNumber[j] > -1) rb2p5.setEnabled(true);   // 2.5
            break;
          case rock.color.plot.rockColorChartPlot._5p0:
            if (iNumber[j] > -1) rb5p0.setEnabled(true);   // 5.0
            break;
          case rock.color.plot.rockColorChartPlot._7p5:
            if (iNumber[j] > -1) rb7p5.setEnabled(true);   // 7.5
            break;
          case rock.color.plot.rockColorChartPlot._10p:
            if (iNumber[j] > -1) rb10p.setEnabled(true);   // 10.0
            break;
        }
      }
    }
  }

  /** Method setCatalog( int iCatalog )
   * <p> This method will set the type of catalog to display
   * @param iCatalog = the catalog to display
   */

  public void setCatalog( int iCatalog )
  {
    this.iCatalog = iCatalog;
    plot.setCatalog( iCatalog );
  }

  /** Method setHue(int iHue)
   * <p> This method will set the Hue to plot
   * @param iHue = the color hue identifier
   */

  public void setHue(int iHue)
  {
    this.iHue = iHue;
    plot.setHue( iHue );
  }

  /** Method setHueNo(int iHueNo)
   * <p> This method will set the Hue Number to plot
   * @param iHueNo = the color hue number
   */

  public void setHueNo( int iHueNo )
  {
    this.iHueNo = iHueNo;
    plot.setHueNo( iHueNo);
  }

  /** Method setDepthRange(double depthStart, double depthEnd)
   * <p> This method will set the depth range of the data
   * @param depthStart = the starting depth
   * @param depthEnd   = the ending depth
   */

  public void setDepthRange(double depthStart, double depthEnd)
  {
	if (pColor != null) { pColor.setDepthRange( depthStart, depthEnd ); }
  }

  /** Method setData()
   * <p> This method will set the Color Data Structure to be modified
   * @param stColor = the color data structure
   */

  public void setData(rockColorStruct stColor)
  {
    if (pColor != null) { pColor.setData( stColor ); }
  }

  /** Method setNeutralColor()
   * <p> This method will retrieve the Neutral color selection from the color plot
   * @param iColor = the Color ID
   */

  public void setNeutralColor(int iColor)
  {
    pColor.setNeutralColor( iColor );
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** METHOD rockColorChartPlotFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class rockColorChartPlotFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
  	  if (notifier != null)
        notifier.notifyObservers(new String("Cancel Edit"));
    }
  }

  /** Method close()
   * <p> This method will cancel the Rock Color Dialog
   */

  public void close()
  {
    pNotifier = null;
    notifier  = null;

    if (pColor != null)
      pColor.close();
    pColor = null;

    // Type of Color Catalog

    rbRock = null;  // Rock Color Catalog
    rbSoil = null;  // Soil Color Catalog

    // Hue Number Buttons

    rb2p5 = null;   // 2.5
    rb5p0 = null;   // 5.0
    rb7p5 = null;   // 7.5
    rb10p = null;   // 10.0

    // Hue Color Buttons

    btnR  = null;
    btnYR = null;
    btnY  = null;
    btnGY = null;
    btnG  = null;
    btnBG = null;
    btnB  = null;
    btnRP = null;
    btnPB = null;
    btnP  = null;

    mExit = null; // Exit Profile Plot

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    // Type of Color Catalog

    if (event.getSource() == rbRock)
    {
      setCatalog( rock.color.plot.rockColorChartPlot._ROCK_CATALOG );
    }
    if (event.getSource() == rbSoil)
    {
      setCatalog( rock.color.plot.rockColorChartPlot._SOIL_CATALOG );
    }

    // Hue Number Buttons

    if (event.getSource() == rb2p5)
    {
      setHueNo( rock.color.plot.rockColorChartPlot._2p5 );
    }
    if (event.getSource() == rb5p0)
    {
      setHueNo( rock.color.plot.rockColorChartPlot._5p0 );
    }
    if (event.getSource() == rb7p5)
    {
      setHueNo( rock.color.plot.rockColorChartPlot._7p5 );
    }
    if (event.getSource() == rb10p)
    {
      setHueNo( rock.color.plot.rockColorChartPlot._10p );
    }

    // Hue Color Buttons

    if (event.getSource() == btnR)
    {
      setHue( rock.color.plot.rockColorChartPlot._R );
    }
    if (event.getSource() == btnYR)
    {
      setHue( rock.color.plot.rockColorChartPlot._YR );
    }
    if (event.getSource() == btnY)
    {
      setHue( rock.color.plot.rockColorChartPlot._Y );
    }
    if (event.getSource() == btnGY)
    {
      setHue( rock.color.plot.rockColorChartPlot._GY );
    }
    if (event.getSource() == btnG)
    {
      setHue( rock.color.plot.rockColorChartPlot._G );
    }
    if (event.getSource() == btnBG)
    {
      setHue( rock.color.plot.rockColorChartPlot._BG );
    }
    if (event.getSource() == btnB)
    {
      setHue( rock.color.plot.rockColorChartPlot._B );
    }
    if (event.getSource() == btnPB )
    {
      setHue( rock.color.plot.rockColorChartPlot._PB );
    }
    if (event.getSource() == btnP)
    {
      setHue( rock.color.plot.rockColorChartPlot._P );
    }
    if (event.getSource() == btnRP)
    {
      setHue( rock.color.plot.rockColorChartPlot._RP );
    }

    if ( event.getSource() == btnN9 ) { setNeutralColor(8); }
    if ( event.getSource() == btnN8 ) { setNeutralColor(7); }
    if ( event.getSource() == btnN7 ) { setNeutralColor(6); }
    if ( event.getSource() == btnN6 ) { setNeutralColor(5); }
    if ( event.getSource() == btnN5 ) { setNeutralColor(4); }
    if ( event.getSource() == btnN4 ) { setNeutralColor(3); }
    if ( event.getSource() == btnN3 ) { setNeutralColor(2); }
    if ( event.getSource() == btnN2 ) { setNeutralColor(1); }
    if ( event.getSource() == btnN1 ) { setNeutralColor(0); }

    setButtons(); // Set Buttons depending on data

    // Close Dialog

    if (event.getSource() == mExit) { close(); }
  }

  /* ===================================================================== *
   * ----------------------- OBSERVABLE ACTIONS -------------------------- *
   * ===================================================================== */

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String sArg      = new String((String) arg);
    int    iColor    = -1;
    int    iRGBClr[] = { 255, 255, 255 };

    if (sArg.equals("Color Selected"))
    {
      pColor.getColor( plot.getColorID(),
	                   plot.getRGB(),
                       plot.getMunsellCode(),
	                   plot.getColorName(),
	                   plot.getRGBHex() );
    }
  }
}

/** CLASS rockColorChartPlotFrameObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class rockColorChartPlotFrameObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 11/10/2009
 *  @author  John R. Victorine
 */
