/*
 * @pfefferPlotTrack.java Version 1.1 03/09/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package pfeffer.plot;

import java.awt.*;
import java.awt.geom.*;

import pfeffer.pfefferDataListStruct;
import pfeffer.pfefferDataStruct;

import las.lasFileDataStruct;
import lith.texture.textureListStruct;

import lith.lithology.lithologySymbolsListStruct;
import lith.lithology.plot.lithologyPlotSymbol;

/** CLASS pfefferPlotTrack
 *  <p> This Class will create the PfEFFER Plot Tracks.
 *
 *  @version 1.1 03/09/2010
 *  @author  John R. Victorine
 */

public class pfefferPlotTrack extends Canvas
{
  // Input Variables

  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables

  public static final Color cRED     = new Color( 180,   0,   0 );
  public static final Color cGREEN   = new Color(  0,  180,   0 );
  public static final Color cBLUE    = new Color(  0,    0, 180 );
  public static final Color cCYAN    = new Color(  0,  180, 180 );
  public static final Color cMAGENTA = new Color( 180,   0, 180 );
  public static final Color cYELLOW  = new Color( 180, 180,   0 );
  public static final Color cORANGE  = new Color( 204, 102,   0 );
  public static final Color cBROWN   = new Color( 102, 102,   0 );
//  public static final Color cBROWN   = new Color(  51,  51,   0 );

  public static final Color COLORS[] =  // Color for Log Curves
      { cRED, cGREEN, cBLUE, cBROWN, cMAGENTA, cORANGE, cYELLOW, cCYAN };

  public static final Color cFlow = new Color( 255, 255, 220 );//102 );

  public static final Color cBVW  = new Color(   0, 255, 255 );
  public static final Color cCBW  = new Color(  65, 105, 225 );
  public static final Color cBVH  = new Color(   0,   0,   0 );
  public static final Color c2nd  = new Color( 255, 204,   0 );

  private double                dNull     = 0.0;  // LAS File Null Value

  private String                sZone     = "";   // Name of the Zone to be plotted

  private pfefferDataStruct     stData    = null; // PfEFFER Data Structure
  private pfefferDataListStruct stList    = null; // PfEFFER Data List Structure
  private textureListStruct     stTexture = null; // LAS Texture

  private lithologySymbolsListStruct stLithology = null;
  private lithologyPlotSymbol        plotLith    = null;

  private double dLIMITS[][] = null; // Minimum & Maximum Limits to PfEFFER Plot Track

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR pfefferPlotTrack( int iDataType,
   *                                double depthStart,
   *                                double depthEnd,
   *                                int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   *  @param stLithology = Texture Symbols List Data Structure
   */

  public pfefferPlotTrack( int    iDataType,
                           double depthStart,
                           double depthEnd,
                           int    iScale,
                           lithologySymbolsListStruct stLithology  )
  {
    this.iDataType = iDataType;

    this.stLithology = stLithology;
    plotLith = new lithologyPlotSymbol( stLithology );

    setPlotHeight(iScale, depthEnd, depthStart);

    builidDataStructure();

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
    stLithology = null;
    stData      = null;
    stList      = null;

	dLIMITS     = null; // Minimum & Maximum Limits to PfEFFER Plot

    sZone       = null;

    if (plotLith != null)
      plotLith.close();
    plotLith = null;

    if (stTexture != null)
      stTexture.delete();
    stTexture = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setZone()
   *  <p> This method will set the name of the  zone that is to be plotted
   *  @param sZone = the zone name
   */

  public void setZone(String sZone) { this.sZone = new String(sZone); }

  /** Method setData( pfefferDataStruct st )
   * <p> This method will set the Default PfEFFER Data Structure
   * @param st - PfEFFER Data Structure
   */

  public void setData( pfefferDataStruct st ) { this.stData = st; }

  /** Method setData( pfefferDataListStruct st )
   * <p> This method will set the Default PfEFFER Data List Structure
   * @param st - PfEFFER Data List Structure
   */

  public void setData( pfefferDataListStruct st ) { this.stList = st; }

  /** Method setLASFileData( lasFileDataStruct st )
   * <p> This method will set the LAS File Data Structure
   * @param st - LAS File Data Structure
   */

  public void setLASFileData( lasFileDataStruct st )
  {
    if (st != null)
      dNull = st.dNull;

    builidDataStructure();

    stTexture = lith.texture.textureUtility.computeLASPlotTexture(
                   st, stTexture, stLithology);
    stTexture = lith.texture.textureUtility.fillEmptySpaces(stTexture);
  }

  /** Method setPlotLimits()
   *  <p> This method will override the default Plot Limits
   *  @param dLIMITS = Array of Minimum & Maximum Plot Limits
   */

  public void setPlotLimits( double dLIMITS[][] ) {	this.dLIMITS = dLIMITS; }

  /* =============================================================== *
   * --------------------------- Compute --------------------------- *
   * =============================================================== */

  /** Method buildDataStructure()
   * <p> This method will allocate space for the depth range that is passed in
   *     the total number of records will be based on 2 * the depth difference
   */

  private void builidDataStructure()
  {
    if (stTexture != null)
      stTexture.delete();
    stTexture = null;

    stTexture = lith.texture.textureUtility.buildNew(
        iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT, depthStart, depthEnd);
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g, int iLAS, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iLAS        = The LAS Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g, int iLAS, int iStartTrack, int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iCycle = 1;
    double dXinc  = 0.0;
    double dTemp  = 0.0;
    int    iIncrement  = iWidthTrack/4;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iTrack      = iLAS - iqstrat.iqstratTracksStruct.PFEFFER_START;
    int    iCurve      = iqstrat.iqstratTracksStruct.PFEFFER_TRACK[iTrack][0];
    int    iValue      = 0;
    int    iY1         = 0;
    int    iY2         = 0;

    double dMinimum = cmn.cmnString.stringToDouble(
                        pfeffer.pfefferStandardTools.CURVES[iCurve][4]);
    double dMaximum = cmn.cmnString.stringToDouble(
                        pfeffer.pfefferStandardTools.CURVES[iCurve][5]);

    if (dLIMITS != null)
    {
	  dMinimum = dLIMITS[iCurve][pfeffer.pfefferPlotLimitsUtility.MINIMUM];
	  dMaximum = dLIMITS[iCurve][pfeffer.pfefferPlotLimitsUtility.MAXIMUM];
	}

    if (pfeffer.pfefferStandardTools.CURVES[iCurve][3].equals("OHM-M"))
    {
       iLog   = 1;
       dMinimum = Math.round(Math.log(dMinimum) / Math.log(10));
       dMaximum = Math.round(Math.log(dMaximum) / Math.log(10));
       iCycle = (int) dMaximum - (int) dMinimum;
    }

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES) && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    if (iLog == 0)
    {
      g.setColor(Color.gray);
      g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

      for (i=0; i<=iWidthTrack; i+=iIncrement)
      {
        if ((i==0) || (i==iWidthTrack))
        {
          g.drawLine(i+iStartTrack, LABELSTART,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }
        else
        {
          g.drawLine(i+iStartTrack, PLOT_TITLES,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }
      }
    }
    else
    {
      iIncrement = iWidthTrack / iCycle;

      g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

      for (i=0; i<iWidthTrack; i+=iIncrement)
      {
        g.setColor(Color.gray);
        if (i==0)
        {
          g.drawLine(i+iStartTrack, LABELSTART,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }

        g.setColor(Color.gray);
        g.drawLine(i+iStartTrack, PLOT_TITLES,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);

        g.setColor(Color.lightGray);
        for (j=1; j<5; j++)
        {
          dXinc = (double) j * 2.0;
          dTemp = (double) iIncrement * Math.log(dXinc) / Math.log(10.0);
          jXinc = i + iStartTrack + (int) dTemp;

          if (jXinc <= iEndTrack)
          {
            g.drawLine(jXinc, PLOT_TITLES,
                       jXinc, PLOT_TITLES+iLogHeight); //iHeight);
          }
        }
      }
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawTrackLabels( Graphics g,
   *                          int iLAS,
   *                          int iStartTrack,
   *                          int iWidthTrack )
   *  <p> This method will draw the labels on the LAS Plot.
   *   @param g           = Graphics Parameter.
   *   @param iLAS        = The LAS Track to draw
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTrackLabels( Graphics g,
                               int iLAS,
                               int iStartTrack,
                               int iWidthTrack )
  {
    int    i=0;
    int    iCurve      = -1;
    int    iContinue   = 1;
    int    iLine       = -1;
    int    length      = 0;
    String sLabel      = "";
    String sTemp       = "";
    String sUnits      = "";
    double dMinimum    = 0.0;
    double dMaximum    = 0.0;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iTrack      = iLAS - iqstrat.iqstratTracksStruct.PFEFFER_START;

    for (i=0; i<5; i++)
    {
      iLine  = i;
      iCurve = iqstrat.iqstratTracksStruct.PFEFFER_TRACK[iTrack][i];

      if ((iCurve > -1) && (iContinue == 1))
      {
        sLabel = new String(pfeffer.pfefferStandardTools.CURVES[iCurve][1]);
        length = sLabel.length();

        sUnits = new String(pfeffer.pfefferStandardTools.CURVES[iCurve][3]);
        if (sUnits.equals("PU"))
        {
          dMaximum = cmn.cmnString.stringToDouble(
              pfeffer.pfefferStandardTools.CURVES[iCurve][4]);
          dMinimum = cmn.cmnString.stringToDouble(
              pfeffer.pfefferStandardTools.CURVES[iCurve][5]);

          if (dLIMITS != null)
          {
      	    dMaximum = dLIMITS[iCurve][pfeffer.pfefferPlotLimitsUtility.MINIMUM];
	        dMinimum = dLIMITS[iCurve][pfeffer.pfefferPlotLimitsUtility.MAXIMUM];
	      }
        }
        else
        {
          dMinimum = cmn.cmnString.stringToDouble(
              pfeffer.pfefferStandardTools.CURVES[iCurve][4]);
          dMaximum = cmn.cmnString.stringToDouble(
              pfeffer.pfefferStandardTools.CURVES[iCurve][5]);

          if (dLIMITS != null)
          {
      	    dMinimum = dLIMITS[iCurve][pfeffer.pfefferPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][pfeffer.pfefferPlotLimitsUtility.MAXIMUM];
	      }
        }

        Font fsb = new Font("Serif", Font.BOLD, 10);
        g.setFont( fsb );

        if (iLine > -1)
        {
          g.setColor(COLORS[i]);
          g.drawString(sLabel,
                       (iStartTrack+iWidthTrack/2 - length*4),
                       LABELSTART + (iLine+2)*14);

          g.drawString("" + dMinimum,
                         iStartTrack+1,
                         LABELSTART + (iLine+2)*14);

          sTemp = new String("" + dMaximum);
          length = sTemp.length();
          g.drawString("" + dMaximum, iEndTrack-length*5, LABELSTART + (iLine+2)*14);
        }
      }
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);

    sLabel = new String( iqstrat.iqstratTracksStruct.PFEFFER_DESC[iTrack] );
    length = sLabel.length();
    g.drawString(sLabel, iStartTrack+iWidthTrack/2-(9*length)/4, LABELSTART+14);
  }

  /** Method drawTrackLabelM( Graphics g,
   *                          int iLAS,
   *                          int iStartTrack,
   *                          int iWidthTrack )
   *  <p> This method will draw the labels on the LAS Plot.
   *   @param g           = Graphics Parameter.
   *   @param iLAS        = The LAS Track to draw
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTrackLabelM( Graphics g,
                               int iLAS,
                               int iStartTrack,
                               int iWidthTrack )
  {
    int    length = 0;
    String sLabel = "";
    double Ma     = 0.0;  // Ma is the modification to M
    double Mb     = 0.0;  // Mb is generally the M value

    Font fsb = new Font("Serif", Font.BOLD, 10);
    g.setFont( fsb );
    g.setColor(COLORS[1]);

    if (stData != null)
    {
      if (stList != null)
      {
        for (int n=0; n<stList.iCount; n++)
        {
          if ((depthStart == stList.stItem[n].depthStart) &&
              (depthEnd   == stList.stItem[n].depthEnd) &&
              (stList.stItem[n].sZone.equals(sZone)))
          {
	        Ma = stList.stItem[n].Mp;
	        Mb = stList.stItem[n].M;

            sLabel = new String("M = " + Ma + " * PHI + " + Mb);
            length = sLabel.length();

            g.drawString(sLabel,
                   (iStartTrack)+4, //+iWidthTrack/2 - length*4),
                    LABELSTART + (5)*14);
	      }
		}
	  }
	}
  }

  /** Method drawTrackCurves( Graphics g,
   *                          int iLAS,
   *                          int iStartTrack,
   *                          int iWidthTrack )
   *  <p> This method will draw the curves on the LAS Plot.
   *   @param g           = Graphics Parameter.
   *   @param iLAS        = The LAS Track to draw
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTrackCurves( Graphics g,
                               int iLAS,
                               int iStartTrack,
                               int iWidthTrack )
  {
    int    iColumns    = 0;
    int    iPlot       = 0;
    int    iX1         = 0;
    int    iX2         = 0;
    int    iY1         = 0;
    int    iY2         = 0;
    int    iLog        = 0;
    int    iColor      = 0;
    double dRatio      = 0.0;
    double dMinimum    = 0.0;
    double dMaximum    = 0.0;
    String sUnits      = "";
    double dLog[]      = null;
    double dStart      = 0.0;
    double dEnd        = 0.0;
    double data        = 0.0;
    double data1       = 0.0;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iTrack      = iLAS - iqstrat.iqstratTracksStruct.PFEFFER_START;
    int    iCurve      = iqstrat.iqstratTracksStruct.PFEFFER_TRACK[iTrack][0];

    if (pfeffer.pfefferStandardTools.CURVES[iCurve][3].equals("OHM-M"))
       iLog   = 1;

    if (stData != null)
    {
      for (int i=0; i<5; i++)
      {
        dLog   = null;
        iCurve = iqstrat.iqstratTracksStruct.PFEFFER_TRACK[iTrack][i];

        if (iCurve > -1)
        {
          iColor = i;
          iColumns = stData.iRows-1;

          dLog = stData.getData( iCurve );

          if (dLog != null)
          {
            for (int j=0; j<iColumns-1; j++)
            {
              if (stData != null)
              {
                dStart = stData.depth[j];
                dEnd   = stData.depth[j+1];

                data       = dLog[j];
                data1      = dLog[j+1];

                if (stList != null)
                {
                  for (int n=0; n<stList.iCount; n++)
                  {
                    if ((dStart >= stList.stItem[n].depthStart) &&
                        (dEnd   <= stList.stItem[n].depthEnd) &&
                        (stList.stItem[n].sZone.equals(sZone)))
                    {
                      for (int m=0; m<stList.stItem[n].iRows-1; m++)
                      {
                        if (dStart == stList.stItem[n].depth[m])
                        {
                          data   = stList.stItem[n].getData( iCurve, m );
                          data1  = stList.stItem[n].getData( iCurve, m+1 );
                        }
                      }
                    }
                  }
                }

                dMinimum = cmn.cmnString.stringToDouble(
                    pfeffer.pfefferStandardTools.CURVES[iCurve][4]);
                dMaximum = cmn.cmnString.stringToDouble(
                    pfeffer.pfefferStandardTools.CURVES[iCurve][5]);
                sUnits = new String(pfeffer.pfefferStandardTools.CURVES[iCurve][3]);

                if (dLIMITS != null)
                {
            	  dMinimum = dLIMITS[iCurve][pfeffer.pfefferPlotLimitsUtility.MINIMUM];
            	  dMaximum = dLIMITS[iCurve][pfeffer.pfefferPlotLimitsUtility.MAXIMUM];
             	}

                if (iLog == 1)
                {
                  if (data <= 0.0)  data  = .00001;
                  if (data1 <= 0.0) data1 = .00001;

                  data     = Math.log(data) / Math.log(10);
                  data1    = Math.log(data1) / Math.log(10);
                  dMinimum = Math.log(dMinimum) / Math.log(10);
                  dMaximum = Math.log(dMaximum) / Math.log(10);
                }

                dRatio = (double) iLogHeight * (dStart - depthStart) /
                    (depthEnd - depthStart);
                iY1 = PLOT_TITLES + (int) dRatio;
                if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                  iY1 = iHeight - (int) dRatio;

                dRatio = (double) iLogHeight * (dEnd - depthStart) /
                    (depthEnd - depthStart);
                iY2 = PLOT_TITLES + (int) dRatio;
                if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                  iY2 = iHeight - (int) dRatio;

                iPlot = 0;
                if ( (iY1 > PLOT_TITLES) && (iY2 > PLOT_TITLES) &&
                    (iY1 < iHeight) && (iY2 < iHeight))
                {
                  iPlot = 1;
                }

                if (sUnits.equals("PU"))
                {
                  dRatio = (double) iWidthTrack *
                      (dMaximum - data) / (dMaximum - dMinimum);
                  iX1 = iStartTrack + (int) dRatio;

                  dRatio = (double) iWidthTrack *
                      (dMaximum - data1) / (dMaximum - dMinimum);
                  iX2 = iStartTrack + (int) dRatio;

                  if ( (iX1 < iStartTrack) && (iX2 < iStartTrack))
                  {
                    while (iX1 < iStartTrack) { iX1 = iX1 + iWidthTrack; }
                    while (iX2 < iStartTrack) { iX2 = iX2 + iWidthTrack; }
                  }
                  else if ( (iX1 >= iStartTrack) && (iX2 < iStartTrack))
                  {
                    g.setColor(COLORS[iColor]);
                    while (iX2 < iStartTrack) { iX2 = iX2 + iWidthTrack; }
                    if (iPlot == 1)
                      g.drawLine(iEndTrack, iY1, iX2, iY2);
                    iX2 = iStartTrack;
                  }
                  else if ( (iX1 < iStartTrack) && (iX2 >= iStartTrack))
                  {
                    g.setColor(COLORS[iColor]);
                    while (iX1 < iStartTrack) { iX1 = iX1 + iWidthTrack; }
                    if (iPlot == 1)
                      g.drawLine(iX1, iY1, iEndTrack, iY2);
                    iX1 = iStartTrack;
                  }
                }
                else
                {
                  dRatio = (double) iWidthTrack *
                      (data - dMinimum) / (dMaximum - dMinimum);
                  iX1 = iStartTrack + (int) dRatio;

                  dRatio = (double) iWidthTrack *
                      (data1 - dMinimum) / (dMaximum - dMinimum);
                  iX2 = iStartTrack + (int) dRatio;

                  if ((iX1 > iEndTrack) && (iX2 > iEndTrack))
                  {
                    while (iX1 > iEndTrack) { iX1 = iX1 - iWidthTrack; }
                    while (iX2 > iEndTrack) { iX2 = iX2 - iWidthTrack; }
                  }
                  else if ((iX1 <= iEndTrack) && (iX2 > iEndTrack))
                  {
                    g.setColor(COLORS[iColor]);
                    while (iX2 > iEndTrack) { iX2 = iX2 - iWidthTrack; }
                    if (iPlot == 1)
                      g.drawLine(iStartTrack, iY1, iX2, iY2);
                    iX2 = iEndTrack;
                  }
                  else if ((iX1 > iEndTrack) && (iX2 <= iEndTrack))
                  {
                    g.setColor(COLORS[iColor]);
                    while (iX1 > iEndTrack) { iX1 = iX1 - iWidthTrack; }
                    if (iPlot == 1)
                      g.drawLine(iX1, iY1, iStartTrack, iY2);
                    iX1 = iEndTrack;
                  }
                }

                if ((iX1 >= iStartTrack) && (iX2 >= iStartTrack) &&
                    (iX1 <= iEndTrack)    && (iX2 <= iEndTrack))
                {
                  g.setColor(COLORS[iColor]);
                  if (iPlot == 1)
                    g.drawLine(iX1, iY1, iX2, iY2);
                }
              }
            }
          }
        }
      }
    }
  }

  /** Method drawTexture( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method draw the texture from the Gamma Ray after determining if
   *     it is a carbonate or a siliclastic rock.
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTexture( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int               k          = 0;
    int               n          = 0;
    int               m          = 0;
    int               iPlus      = 0;
    int               iCount     = 0;
    int               iDepth1    = 0;
    int               iDepth2    = 0;
    int               iDepth     = 0;
    int               iGroup     = 0;
    double            depth1     = 0.0;
    double            depth2     = 0.0;
    double            diff       = 0;
    double            dRatio     = 0.0;
    int               iCells     = 0;
    int               iRows      = 0;
    int               iColumns   = 0;
    int               iRGB[]     = { 255, 255, 255 };
    int               iSymbol   = -1;
    String            symbol[][] = null;
    textureListStruct st         = stTexture;
    int               iNew       = 0;
    int               iSegment   = 0;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        if (st.stItem[0] != null)
        {
          depth1 = st.stItem[0].depthStart;
          depth2 = st.stItem[0].depthEnd;
          diff   = depth2-depth1;

          iPlus = (int) (( 5.0/diff ) *
                         ( iqstrat.iqstratTracksStruct.SCALE[iScale] / 100.0 ));
          if (iPlus < 1) iPlus = 1;

          iCount = -1;
          for (k=0; k<st.iCount; k++)
          {
            if (st.stItem[k] != null)
            {
              depth1 = st.stItem[k].depthStart;
              depth2 = depth1 + diff*iPlus;

              for (int i=1; i<iPlus; i++)
                if (k<st.iCount-1)
                  k++;

              iSymbol  = st.stItem[k].iSymbol;

/* 27 Feb 2012 JRV Removed set lithology from flow units
              if (stList != null)
              {
                for (n=0; n<stList.iCount; n++)
                {
                  if ((depth1 >= stList.stItem[n].depthStart) &&
                      (depth2 <= stList.stItem[n].depthEnd))
                  {
                    for (m=0; m<stList.stItem[n].iRows; m++)
                    {
                      if (depth1 == stList.stItem[n].depth[m])
                      {
                        iSymbol = stList.stItem[n].iLithology;
                      }
                    }
                  }
                }
              }
*/
              dRatio  = (double) iLogHeight * (depthStart - depth1) /
                                              (depthStart - depthEnd);
              iDepth1 = PLOT_TITLES + (int) dRatio;
              if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                iDepth1 = iHeight - (int) dRatio;

              dRatio  = (double) iLogHeight * (depthStart - depth2) /
                                              (depthStart - depthEnd);
              iDepth2 = PLOT_TITLES + (int) dRatio;
              if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                iDepth2 = iHeight - (int) dRatio;

              iDepth = Math.abs( iDepth2 - iDepth1 );

              if (iNew == 0)
              {
                iSegment = iDepth;
                iNew = 1;
              }

              if (iDepth > iSegment) iSegment = iDepth;

              if (iDepth < iSegment)
              {
                iDepth  = iSegment;
                iDepth2 = iDepth1 + iSegment;
              }

              if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
                  (iDepth1 < iHeight)      && (iDepth2 < iHeight))
              {
                if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                  iDepth1 =  iDepth2;

                 iGroup = iDepth / 5;
                 if (iGroup == 0) iGroup = 1;

                 iRows    = 5;
                 iColumns = 10;
                 iCells   = 10;

                 if (iSymbol > -1)
                 {
                   iRGB     = st.stItem[k].iRGB;
//                   iRGB   = lith.lithology.lithologySymbolsUtility.getColor( //st.stItem[k].iRGB;
//                              iSymbol, stLithology);
                   symbol = lith.lithology.lithologySymbolsUtility.getSymbol(
                              iSymbol, stLithology);

                   for (n=0; n<iGroup; n++)
                   {
                     iCount++;
                     for (int i=0; i<iCells; i++)
                     {
                       plotLith.drawBlock( g, iRGB[0], iRGB[1], iRGB[2],
                                           iStartTrack+i*iColumns, iDepth1 + n*5,
                                           iColumns, iRows);

                       plotLith.drawLithologySymbolMask(
                          g, iRows, plotLith.getOrder(iCount), symbol,
                          iStartTrack+i*iColumns, iDepth1 + n*5 );
                     }
                   }
                 }
              }
            }
          }
        }
      }
    }
  }

  /** Method drawFluid( Graphics g, int iStartTrack, int iWidthTrack )
   *  <p> This method will draw the background for the porosity.
   *   @param g           = Graphics Parameter.
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawFluid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iColumns    = 0;
    int    iPlot       = 0;
    int    iY1         = 0;
    int    iY2         = 0;
    double dRatio      = 0.0;
    double dMinimum    = 0.0;
    double dMaximum    = 0.0;
    double dStart      = 0.0;
    double dEnd        = 0.0;
    double data        = 0.0;
    double data1       = 0.0;
    int    iEndTrack   = iStartTrack+iWidthTrack;

    double Rt[]        = null;
    double BVW[]       = null;
    double PHIt[]      = null;
    double Vsh[]       = null;
    double Sw[]        = null;

    double Rt1         = 0.0;
    double BVW1        = 0.0;
    double PHIt1       = 0.0;
    double Vsh1        = 0.0;
    double Sw1         = 0.0;

    double Rt2         = 0.0;
    double BVW2        = 0.0;
    double PHIt2       = 0.0;
    double Vsh2        = 0.0;
    double Sw2         = 0.0;

    double CBW1        = 0.0;
    double CBW2        = 0.0;
    double BVH1        = 0.0;
    double BVH2        = 0.0;

    int    iBVW1       = 0;
    int    iBVW2       = 0;
    int    iCBW1       = 0;
    int    iCBW2       = 0;
    int    iBVH1       = 0;
    int    iBVH2       = 0;

    int    ix[]        = { 0, 0, 0, 0 };
    int    iy[]        = { 0, 0, 0, 0 };

    Font   fsb        = new Font("Serif", Font.BOLD, 10);
    String sTemp      = "";
    int    length     = 0;

    g.setFont( fsb );

    if (stData != null)
    {
      Rt   = stData.getData( pfeffer.pfefferStandardTools._RT );
      BVW  = stData.getData( pfeffer.pfefferStandardTools._BVW );
      PHIt = stData.getData( pfeffer.pfefferStandardTools._PHIT );
      Vsh  = stData.getData( pfeffer.pfefferStandardTools._VSH );
      Sw   = stData.getData( pfeffer.pfefferStandardTools._SW );

      if ((BVW != null) && (PHIt != null) && (Vsh != null))
      {
        iColumns = stData.iRows-1;
        dMinimum = 0.0;
        dMaximum = 0.5;

        if (dLIMITS != null)
        {
  	      dMinimum = dLIMITS[
  	                  pfeffer.pfefferStandardTools._PHIT][
  	                  pfeffer.pfefferPlotLimitsUtility.MINIMUM];
	      dMaximum = dLIMITS[
	                  pfeffer.pfefferStandardTools._PHIT][
	                  pfeffer.pfefferPlotLimitsUtility.MAXIMUM];
	    }

        for (int j=0; j<iColumns-1; j++)
        {
          if ((Rt[j] > dNull) && (PHIt[j] > dNull))
          {
            dStart = stData.depth[j];
            dEnd   = stData.depth[j+1];

            dRatio = (double) iLogHeight * (dStart - depthStart) /
                (depthEnd - depthStart);
            iY1 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iY1 = iHeight - (int) dRatio;

            dRatio = (double) iLogHeight * (dEnd - depthStart) /
                (depthEnd - depthStart);
            iY2 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iY2 = iHeight - (int) dRatio;

            iPlot = 0;
            if ((iY1 > PLOT_TITLES) && (iY2 > PLOT_TITLES) &&
                (iY1 < iHeight)     && (iY2 < iHeight))
            {
              iPlot = 1;

              iy[0] = iY1;
              iy[1] = iY1;
              iy[2] = iY2;
              iy[3] = iY2;
            }

            if (iPlot == 1)
            {
              PHIt1 = PHIt[j];
              Vsh1  = Vsh[j];
              Sw1   = Sw[j];
              BVW1  = BVW[j];

              PHIt2 = PHIt[j+1];
              Vsh2  = Vsh[j+1];
              Sw2   = Sw[j+1];
              BVW2  = BVW[j+1];

              if (stList != null)
              {
                for (int n=0; n<stList.iCount; n++)
                {
                  if ((dStart >= stList.stItem[n].depthStart) &&
                      (dEnd   <= stList.stItem[n].depthEnd) &&
                      (stList.stItem[n].sZone.equals(sZone)))
                  {
                    for (int m=0; m<stList.stItem[n].iRows-1; m++)
                    {
                      if (dStart == stList.stItem[n].depth[m])
                      {
                        PHIt1 = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._PHIT, m );
                        Vsh1  = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._VSH,  m );
                        Sw1   = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._SW,   m );
                        BVW1  = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._BVW,  m );

                        PHIt2 = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._PHIT, m+1 );
                        Vsh2  = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._VSH,  m+1 );
                        Sw2   = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._SW,   m+1 );
                        BVW2  = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._BVW,  m+1 );
                      }
                    }
                  }
                }
              }

              if (Vsh1 > 1.0)
                Vsh1 = 1.0;

              if (Sw1 > 1.0)
              {
                Sw1  = 1.0;
                BVW1 = PHIt1;
              }

              if (Vsh2 > 1.0)
                Vsh2 = 1.0;

              if (Sw2 > 1.0)
              {
                Sw2  = 1.0;
                BVW2 = PHIt2;
              }

              if (Sw1+Vsh1 > 1.0)
              {
                if (Sw1 <= Vsh1)
                {
                  Vsh1 = Vsh1 - Sw1;
                  Vsh2 = Vsh2 - Sw2;
                }
                else
                {
                  Vsh1 = 1.0 - Sw1;
                  Vsh2 = 1.0 - Sw2;
                }
              }

              CBW1  = PHIt1 * Vsh1;
              CBW2  = PHIt2 * Vsh2;

              BVH1  = PHIt1 - BVW1 - CBW1;
              BVH2  = PHIt2 - BVW2 - CBW2;

              if ((BVH1 > 0.0) && (BVH2 > 0.0))
              {
                data   = BVH1 + CBW1 + BVW1;
                data1  = BVH2 + CBW2 + BVW2;

                dRatio = (double) iWidthTrack *
                         (dMaximum - data) / (dMaximum - dMinimum);
                iBVH1 = iStartTrack + (int) dRatio;

                dRatio = (double) iWidthTrack *
                         (dMaximum - data1) / (dMaximum - dMinimum);
                iBVH2 = iStartTrack + (int) dRatio;

                if ((iBVH1 >= iStartTrack) && (iBVH2 >= iStartTrack) &&
                    (iBVH1 <= iEndTrack)   && (iBVH2 <= iEndTrack))
                {
                  ix[0] = iBVH1;
                  ix[1] = iEndTrack;
                  ix[2] = iEndTrack;
                  ix[3] = iBVH2;

                  g.setColor( cBVH );
                  g.fillPolygon(ix, iy, 4);
                }
              }

              data   = CBW1 + BVW1;
              data1  = CBW2 + BVW2;

              dRatio = (double) iWidthTrack *
                       (dMaximum - data) / (dMaximum - dMinimum);
              iCBW1 = iStartTrack + (int) dRatio;

              dRatio = (double) iWidthTrack *
                       (dMaximum - data1) / (dMaximum - dMinimum);
              iCBW2 = iStartTrack + (int) dRatio;

              if (iCBW1 < iStartTrack) iCBW1 = iStartTrack;
              if (iCBW2 < iStartTrack) iCBW2 = iStartTrack;

              if ((iCBW1 >= iStartTrack) && (iCBW2 >= iStartTrack) &&
                  (iCBW1 <= iEndTrack)   && (iCBW2 <= iEndTrack))
              {
                ix[0] = iCBW1;
                ix[1] = iEndTrack;
                ix[2] = iEndTrack;
                ix[3] = iCBW2;

                g.setColor( cCBW );
                g.fillPolygon(ix, iy, 4);
              }

              data   = BVW1;
              data1  = BVW2;

              dRatio = (double) iWidthTrack *
                  (dMaximum - data) / (dMaximum - dMinimum);
              iBVW1 = iStartTrack + (int) dRatio;

              dRatio = (double) iWidthTrack *
                  (dMaximum - data1) / (dMaximum - dMinimum);
              iBVW2 = iStartTrack + (int) dRatio;

              if ((iBVW1 >= iStartTrack) && (iBVW2 >= iStartTrack) &&
                  (iBVW1 <= iEndTrack)   && (iBVW2 <= iEndTrack))
              {
                ix[0] = iBVW1;
                ix[1] = iEndTrack;
                ix[2] = iEndTrack;
                ix[3] = iBVW2;

                g.setColor( cBVW );
                g.fillPolygon(ix, iy, 4);
              }
            }
          }
        }
      }
    }

    g.setColor( cBVH );
    g.fillRect(iStartTrack, PLOT_TITLES - 30, iWidthTrack, 30);

    g.setColor(Color.white);
    sTemp = new String("BVH");
    length = sTemp.length();
    g.drawString( sTemp,
                  (iStartTrack+iWidthTrack/2 - length*4),
                  PLOT_TITLES-21);

    g.setColor( cCBW );
    g.fillRect(iStartTrack, PLOT_TITLES - 20, iWidthTrack, 20);

    g.setColor(Color.lightGray);
    sTemp = new String("CBW");
    length = sTemp.length();
    g.drawString( sTemp,
                  (iStartTrack+iWidthTrack/2 - length*4),
                  PLOT_TITLES-11);

    g.setColor( cBVW );
    g.fillRect(iStartTrack, PLOT_TITLES - 10, iWidthTrack, 10);

    g.setColor(Color.black);
    sTemp = new String("BVW");
    length = sTemp.length();
    g.drawString( sTemp,
                  (iStartTrack+iWidthTrack/2 - length*4),
                  PLOT_TITLES-1);
  }

  /** Method draw2ndPhi( Graphics g, int iStartTrack, int iWidthTrack )
   *  <p> This method will draw the 2nd Porosity background.
   *   @param g           = Graphics Parameter.
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw2ndPhi( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iColumns    = 0;
    int    iPlot       = 0;
    int    iY1         = 0;
    int    iY2         = 0;
    double dRatio      = 0.0;
    double dMinimum    = 0.0;
    double dMaximum    = 0.5;
    double dStart      = 0.0;
    double dEnd        = 0.0;
    double data        = 0.0;
    double data1       = 0.0;
    int    iEndTrack   = iStartTrack+iWidthTrack;

    double PHIt[]      = null;
    double PHI1st[]    = null;
    double PHI2nd[]    = null;

    double PHIt1       = 0.0;
    double PHI1st1     = 0.0;
    double PHI2nd1     = 0.0;

    double PHIt2       = 0.0;
    double PHI1st2     = 0.0;
    double PHI2nd2     = 0.0;

    int    iPHI1st1    = 0;
    int    iPHI1st2    = 0;
    int    iPHIt1      = 0;
    int    iPHIt2      = 0;

    int    ix[]        = { 0, 0, 0, 0 };
    int    iy[]        = { 0, 0, 0, 0 };

    Font   fsb        = new Font("Serif", Font.BOLD, 10);
    String sTemp      = "";
    int    length     = 0;

    g.setFont( fsb );

    if (dLIMITS != null)
    {
      dMinimum = dLIMITS[pfeffer.pfefferStandardTools._PHIT][
  	                     pfeffer.pfefferPlotLimitsUtility.MINIMUM];
      dMaximum = dLIMITS[pfeffer.pfefferStandardTools._PHIT][
	                     pfeffer.pfefferPlotLimitsUtility.MAXIMUM];
    }

    if (stData != null)
    {
      PHIt   = stData.getData( pfeffer.pfefferStandardTools._PHIT );
      PHI1st = stData.getData( pfeffer.pfefferStandardTools._PHI1 );
      PHI2nd = stData.getData( pfeffer.pfefferStandardTools._PHI2 );

      if ((PHIt != null) && (PHI1st != null) && (PHI2nd != null))
      {
        iColumns = stData.iRows-1;

        for (int j=0; j<iColumns-1; j++)
        {
          if ((PHI1st[j] > dNull) && (PHIt[j] > dNull) && (PHI2nd[j] > dNull))
          {
            dStart = stData.depth[j];
            dEnd   = stData.depth[j+1];

            dRatio = (double) iLogHeight * (dStart - depthStart) /
                (depthEnd - depthStart);
            iY1 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iY1 = iHeight - (int) dRatio;

            dRatio = (double) iLogHeight * (dEnd - depthStart) /
                (depthEnd - depthStart);
            iY2 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iY2 = iHeight - (int) dRatio;

            iPlot = 0;
            if ((iY1 > PLOT_TITLES) && (iY2 > PLOT_TITLES) &&
                (iY1 < iHeight)     && (iY2 < iHeight))
            {
              iPlot = 1;
            }

            if (iPlot == 1)
            {
              PHIt1   = PHIt[j];
              PHI1st1 = PHI1st[j];

              PHIt2   = PHIt[j+1];
              PHI1st2 = PHI1st[j+1];

              if (stList != null)
              {
                for (int n=0; n<stList.iCount; n++)
                {
                  if ((dStart >= stList.stItem[n].depthStart) &&
                      (dEnd   <= stList.stItem[n].depthEnd) &&
                      (stList.stItem[n].sZone.equals(sZone)))
                  {
                    for (int m=0; m<stList.stItem[n].iRows-1; m++)
                    {
                      if (dStart == stList.stItem[n].depth[m])
                      {
                        PHIt1 = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._PHIT, m );
                        PHI1st1 = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._PHI1, m );

                        PHIt2 = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._PHIT, m+1 );
                        PHI1st2 = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._PHI1, m+1 );
                      }
                    }
                  }
                }
              }

              data    = PHI1st1;
              data1   = PHI1st2;

              dRatio = (double) iWidthTrack *
                    (dMaximum - data) / (dMaximum - dMinimum);
              iPHI1st1 = iStartTrack + (int) dRatio;

              dRatio = (double) iWidthTrack *
                  (dMaximum - data1) / (dMaximum - dMinimum);
              iPHI1st2 = iStartTrack + (int) dRatio;

              data   = PHIt1;
              data1  = PHIt2;

              dRatio = (double) iWidthTrack *
                    (dMaximum - data) / (dMaximum - dMinimum);
              iPHIt1 = iStartTrack + (int) dRatio;

              dRatio = (double) iWidthTrack *
                  (dMaximum - data1) / (dMaximum - dMinimum);
              iPHIt2 = iStartTrack + (int) dRatio;

              if (iPHIt1   < iStartTrack) iPHIt1   = iStartTrack;
              if (iPHIt2   < iStartTrack) iPHIt2   = iStartTrack;
              if (iPHI1st1 < iStartTrack) iPHI1st1 = iStartTrack;
              if (iPHI1st2 < iStartTrack) iPHI1st2 = iStartTrack;

              if (iPHIt1   > iEndTrack) iPHIt1   = iEndTrack;
              if (iPHIt2   > iEndTrack) iPHIt2   = iEndTrack;
              if (iPHI1st1 > iEndTrack) iPHI1st1 = iEndTrack;
              if (iPHI1st2 > iEndTrack) iPHI1st2 = iEndTrack;

//              if ((PHI2nd[j] > 0.0) || (PHI2nd[j+1] > 0.0))
              if (iPHIt1 < iPHI1st1)
              {
                ix[0] = iPHIt1;
                iy[0] = iY1;

                ix[1] = iPHI1st1;
                iy[1] = iY1;

                ix[2] = iPHI1st2;
                iy[2] = iY2;

                ix[3] = iPHIt2;
                iy[3] = iY2;

                g.setColor( c2nd );
//                g.fillRect(iPHIt1, iY1, iPHI1st1-iPHIt1, iY2-iY1);
                g.fillPolygon(ix, iy, 4);
              }
            }
          }
        }
      }
    }

    g.setColor( c2nd );
    g.fillRect(iStartTrack, PLOT_TITLES - 10, iWidthTrack, 10);

    g.setColor(Color.black);
    sTemp = new String("2nd Porosity");
    length = sTemp.length();
    g.drawString( sTemp,
                  (iStartTrack+iWidthTrack/2 - length*2),
                  PLOT_TITLES-1);
  }

  /** Method drawPHIr( Graphics g, int iStartTrack, int iWidthTrack )
   *  <p> This method will draw the Electrical connected Porosity background.
   *   @param g           = Graphics Parameter.
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawPHIr( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iColumns    = 0;
    int    iPlot       = 0;
    int    iY1         = 0;
    int    iY2         = 0;
    double dRatio      = 0.0;
    double dMinimum    = 0.0;
    double dMaximum    = 0.5;
    double dStart      = 0.0;
    double dEnd        = 0.0;
    double data        = 0.0;
    double data1       = 0.0;
    int    iEndTrack   = iStartTrack+iWidthTrack;

    double PHIt[]      = null;
    double PHIr[]      = null;

    double PHIt1       = 0.0;
    double PHIr1       = 0.0;

    double PHIt2       = 0.0;
    double PHIr2       = 0.0;

    int    iPHIr1      = 0;
    int    iPHIr2      = 0;
    int    iPHIt1      = 0;
    int    iPHIt2      = 0;

    int    ix[]        = { 0, 0, 0, 0 };
    int    iy[]        = { 0, 0, 0, 0 };

    Font   fsb         = new Font("Serif", Font.BOLD, 10);
    String sTemp       = "";
    int    length      = 0;

    g.setFont( fsb );

    if (dLIMITS != null)
    {
      dMinimum = dLIMITS[pfeffer.pfefferStandardTools._PHIT][
  	                     pfeffer.pfefferPlotLimitsUtility.MINIMUM];
      dMaximum = dLIMITS[pfeffer.pfefferStandardTools._PHIT][
	                     pfeffer.pfefferPlotLimitsUtility.MAXIMUM];
    }

    if (stData != null)
    {
      PHIt = stData.getData( pfeffer.pfefferStandardTools._PHIT );
      PHIr = stData.getData( pfeffer.pfefferStandardTools._PHIR );

      if ((PHIt != null) && (PHIr != null))
      {
        iColumns = stData.iRows-1;

        for (int j=0; j<iColumns-1; j++)
        {
          if ((PHIr[j] > dNull) && (PHIt[j] > dNull))
          {
            dStart = stData.depth[j];
            dEnd   = stData.depth[j+1];

            dRatio = (double) iLogHeight * (dStart - depthStart) /
                (depthEnd - depthStart);
            iY1 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iY1 = iHeight - (int) dRatio;

            dRatio = (double) iLogHeight * (dEnd - depthStart) /
                (depthEnd - depthStart);
            iY2 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iY2 = iHeight - (int) dRatio;

            iPlot = 0;
            if ((iY1 > PLOT_TITLES) && (iY2 > PLOT_TITLES) &&
                (iY1 < iHeight)     && (iY2 < iHeight))
            {
              iPlot = 1;
            }

            if (iPlot == 1)
            {
              PHIt1 = PHIt[j];
              PHIr1 = PHIr[j];

              PHIt2 = PHIt[j+1];
              PHIr2 = PHIr[j+1];

              if (stList != null)
              {
                for (int n=0; n<stList.iCount; n++)
                {
                  if ((dStart >= stList.stItem[n].depthStart) &&
                      (dEnd   <= stList.stItem[n].depthEnd) &&
                      (stList.stItem[n].sZone.equals(sZone)))
                  {
                    for (int m=0; m<stList.stItem[n].iRows-1; m++)
                    {
                      if (dStart == stList.stItem[n].depth[m])
                      {
                        PHIt1 = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._PHIT, m );
                        PHIr1 = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._PHIR, m );

                        PHIt2 = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._PHIT, m+1 );
                        PHIr2 = stList.stItem[n].getData(
                            pfeffer.pfefferStandardTools._PHIR, m+1 );
                      }
                    }
                  }
                }
              }

              data    = PHIr1;
              data1   = PHIr2;

              dRatio = (double) iWidthTrack *
                    (dMaximum - data) / (dMaximum - dMinimum);
              iPHIr1 = iStartTrack + (int) dRatio;

              dRatio = (double) iWidthTrack *
                  (dMaximum - data1) / (dMaximum - dMinimum);
              iPHIr2 = iStartTrack + (int) dRatio;

              data   = PHIt1;
              data1  = PHIt2;

              dRatio = (double) iWidthTrack *
                    (dMaximum - data) / (dMaximum - dMinimum);
              iPHIt1 = iStartTrack + (int) dRatio;

              dRatio = (double) iWidthTrack *
                  (dMaximum - data1) / (dMaximum - dMinimum);
              iPHIt2 = iStartTrack + (int) dRatio;

              if (iPHIt1 < iStartTrack) iPHIt1 = iStartTrack;
              if (iPHIt2 < iStartTrack) iPHIt2 = iStartTrack;
              if (iPHIr1 < iStartTrack) iPHIr1 = iStartTrack;
              if (iPHIr2 < iStartTrack) iPHIr2 = iStartTrack;

              if (iPHIt1 > iEndTrack) iPHIt1 = iEndTrack;
              if (iPHIt2 > iEndTrack) iPHIt2 = iEndTrack;
              if (iPHIr1 > iEndTrack) iPHIr1 = iEndTrack;
              if (iPHIr2 > iEndTrack) iPHIr2 = iEndTrack;

              if (iPHIr1 < iPHIt1)
              {
                ix[0] = iPHIr1;
                iy[0] = iY1;

                ix[1] = iPHIt1;
                iy[1] = iY1;

                ix[2] = iPHIt2;
                iy[2] = iY2;

                ix[3] = iPHIr2;
                iy[3] = iY2;

                g.setColor( c2nd );
                g.fillPolygon(ix, iy, 4);
//                g.fillRect(iPHIr1, iY1, iPHIt1-iPHIr1, iY2-iY1);
              }
            }
          }
        }
      }
    }

    g.setColor( c2nd );
    g.fillRect(iStartTrack, PLOT_TITLES - 10, iWidthTrack, 10);

    g.setColor(Color.black);
    sTemp = new String("OHM Porosity");
    length = sTemp.length();
    g.drawString( sTemp,
                  (iStartTrack+iWidthTrack/2 - length*2),
                  PLOT_TITLES-1);
  }

  /** Method drawFlowUnits( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will write the flow unit name and mark the depth range
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawFlowUnits( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iY1         = 0;
    int    iY2         = 0;
    double dRatio      = 0.0;
    double dStart      = 0.0;
    double dEnd        = 0.0;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    String sLabel      = "";
    int    length      = 0;

    Font        fsb  = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    if (stList != null)
    {
      for (int i=0; i<stList.iCount; i++)
      {
		if (stList.stItem[i] != null)
		{
          dStart = stList.stItem[i].depthStart;
          dEnd   = stList.stItem[i].depthEnd;

          dRatio = (double) iLogHeight * (dStart - depthStart) /
              (depthEnd - depthStart);
          iY1 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iY1 = iHeight - (int) dRatio;

          dRatio = (double) iLogHeight * (dEnd - depthStart) /
              (depthEnd - depthStart);
          iY2 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iY2 = iHeight - (int) dRatio;

          if ((iY1 >= PLOT_TITLES) && (iY2 >= PLOT_TITLES) &&
              (iY1 <= iHeight)     && (iY2 <= iHeight))
          {
            g.drawLine( iStartTrack, iY1, iEndTrack, iY1 );
            if (stList.stItem[i].sZone != null)
              g.drawString( "  " + stList.stItem[i].sZone, iStartTrack, iY1+10);
          }
	    }
      }
    }
  }

  /** Method drawBackground( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will write the flow unit name and mark the depth range
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawBackground( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iY1         = 0;
    int    iY2         = 0;
    double dRatio      = 0.0;
    double dStart      = 0.0;
    double dEnd        = 0.0;
    int    iEndTrack   = iStartTrack+iWidthTrack;

    if (stList != null)
    {
      for (int i=0; i<stList.iCount; i++)
      {
        dStart = stList.stItem[i].depthStart;
        dEnd   = stList.stItem[i].depthEnd;

        dRatio = (double) iLogHeight * (dStart - depthStart) /
            (depthEnd - depthStart);
        iY1 = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY1 = iHeight - (int) dRatio;

        dRatio = (double) iLogHeight * (dEnd - depthStart) /
            (depthEnd - depthStart);
        iY2 = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight - (int) dRatio;

        if ((iY1 >= PLOT_TITLES) && (iY2 >= PLOT_TITLES) &&
            (iY1 <= iHeight)     && (iY2 <= iHeight))
        {
          g.setColor( cFlow );
          g.fillRect(iStartTrack, iY1, iWidthTrack, iY2-iY1);
        }
      }
    }
  }

  /** Method drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g,  int iStartTrack, int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iIncrement  = iWidthTrack/4;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iY1         = 0;
    int    iY2         = 0;
    String sLabel      = "Flow Units";
    int    length      = 0;

    Font        fsb  = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    length = sLabel.length();
    g.drawString(sLabel, iStartTrack+iWidthTrack/2-(9*length)/4, LABELSTART+14);

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES) && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    if (iLog == 0)
    {
      g.setColor(Color.gray);
      g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

      for (i=0; i<=iWidthTrack; i+=iIncrement)
      {
        if ((i==0) || (i==iWidthTrack))
        {
          g.drawLine(i+iStartTrack, LABELSTART,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }
        else
        {
          g.drawLine(i+iStartTrack, PLOT_TITLES,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }
      }
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method draw( Graphics g, int iLAS, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iLAS        = The LAS Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw( Graphics g, int iLAS, int iStartTrack, int iWidthTrack )
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_PFEFFER;

    switch (iLAS)
    {
      case iqstrat.iqstratTracksStruct._PFEFFER_RESISTIVITY:
      case iqstrat.iqstratTracksStruct._PFEFFER_POROSITY:
      case iqstrat.iqstratTracksStruct._PFEFFER_2ND_POROSITY:
      case iqstrat.iqstratTracksStruct._PFEFFER_POROSITY_OHM:
      case iqstrat.iqstratTracksStruct._PFEFFER_OTHER:
      case iqstrat.iqstratTracksStruct._PFEFFER_M:
      case iqstrat.iqstratTracksStruct._PFEFFER_FLOW_UNITS:
        g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                               iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                               iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
        g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

        drawBackground( g, iStartTrack, iWidthTrack );
        break;
    }

    switch (iLAS)
    {
      case iqstrat.iqstratTracksStruct._PFEFFER_POROSITY:
        drawTexture(g, iStartTrack, iWidthTrack);
        drawFluid( g, iStartTrack, iWidthTrack );
        break;
      case iqstrat.iqstratTracksStruct._PFEFFER_2ND_POROSITY:
        draw2ndPhi( g, iStartTrack, iWidthTrack );
        break;
      case iqstrat.iqstratTracksStruct._PFEFFER_POROSITY_OHM:
        drawPHIr( g, iStartTrack, iWidthTrack );
        break;
      case iqstrat.iqstratTracksStruct._PFEFFER_RESISTIVITY:
      case iqstrat.iqstratTracksStruct._PFEFFER_OTHER:
      case iqstrat.iqstratTracksStruct._PFEFFER_M:
        break;
      case iqstrat.iqstratTracksStruct._PFEFFER_FLOW_UNITS:
        break;
    }

    switch (iLAS)
    {
      case iqstrat.iqstratTracksStruct._PFEFFER_RESISTIVITY:
      case iqstrat.iqstratTracksStruct._PFEFFER_2ND_POROSITY:
      case iqstrat.iqstratTracksStruct._PFEFFER_POROSITY:
      case iqstrat.iqstratTracksStruct._PFEFFER_POROSITY_OHM:
      case iqstrat.iqstratTracksStruct._PFEFFER_OTHER:
      case iqstrat.iqstratTracksStruct._PFEFFER_M:
        drawGrid( g, iLAS, iStartTrack, iWidthTrack );
        drawTrackLabels( g, iLAS, iStartTrack, iWidthTrack );
        drawTrackCurves( g, iLAS, iStartTrack, iWidthTrack );
        break;
      case iqstrat.iqstratTracksStruct._PFEFFER_FLOW_UNITS:
        drawGrid( g, iStartTrack, iWidthTrack );
        drawFlowUnits( g, iStartTrack, iWidthTrack );
        break;
    }

    switch (iLAS)
    {
      case iqstrat.iqstratTracksStruct._PFEFFER_M:
        drawTrackLabelM( g, iLAS, iStartTrack, iWidthTrack );
        break;
	}
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}
