/*
 * @pfefferStandardTools.java Version 1.1 03/09/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package pfeffer;

/** Class pfefferStandardTools
 *  <p> This Class holds the standard tool information to map the LAS log
 *      curves to standard names for plotting.
 */

public class pfefferStandardTools
{
  public static final int _C_ID     = 0;
  public static final int _C_MNEM   = 1;
  public static final int _C_NAME   = 2;
  public static final int _C_UNIT   = 3;
  public static final int _C_MIN    = 4;
  public static final int _C_MAX    = 5;
  public static final int _C_FORMAT = 6;

  // PfEFFER Flow Unit/Zone Identifier

  public static final int _KEY     = 0; // Unique Identifiery of Zone
  public static final int _ZONE    = 1; // Flow Unit Name
  public static final int _START   = 2; // Starting Depth
  public static final int _END     = 3; // Ending Depth

  // -- Lithology Symbol

  public static final int _ROCK    = 4; // Rock Matrix (Lithology)

  // Water Model Used

  public static final int _WATER   = 5; // Water Model
                                        // - Default = NONE
                                        // - Archie
                                        // - Simandoux
                                        // - Dual Water

  // Archie Equation Parameters

  public static final int _A       = 6; // Archie Constant
  public static final int _M       = 7; // Archie Cementation Factor
  public static final int _N       = 8; // Archie Saturation Exponent
  public static final int _RW      = 9; // Formation water resistivity
  public static final int _RSH     = 10; // Shale resistivity
  public static final int _PHISH   = 11; // Shale porosity

  // Total Resistivity Log Curve Value

  public static final int _L_Rt    = 12; // Log curve - Resistivity

  // Medium Resistivity Log Curve Value

  public static final int _L_Rxo   = 13; // Log curve - Medium Resistivity

  // V-Shale Log Curve Values

  public static final int _L_Vsh   = 14; // Log curve - Shale Factor
  public static final int _CLEAN   = 15; // Vsh - Clean Formation Value
  public static final int _SHALE   = 16; // Vsh - Shale Value

  // Total Porosity Log Curve Values

  public static final int _L_PHIt  = 17; // Porosity Type
                                         // - Neutron
                                         // - Density
                                         // - Average Neutron/Density
                                         // - Sonic
  public static final int _L_PHI1  = 18; // Log curve - Porosity
  public static final int _L_PHI2  = 19; // Log curve - Porosity - Only for Avg

  public static final int _GRAIN   = 20; // Rock Matrix Grain Density
  public static final int _FLUID   = 21; // Fluid Density

  // -- Total Porosity V-Shale Effect

  public static final int _PHI_VSH = 22; // Porosity V-Shale Effect
  public static final int _PHI_SH  = 23; // 1st Log Curve Value for Shale
  public static final int _PHI_SH2 = 24; // 2nd Log Curve Value for Shale

  // -- 2nd Porosity Curve ID

  public static final int _L_2ND   = 25; // Log curve - Primary Porosity
  public static final int _2_GRAIN = 26; // Rock Matrix Grain Density
  public static final int _2_FLUID = 27; // Fluid Density of 2nd Porosity

  // -- Total Porosity V-Shale Effect

  public static final int _2_VSH   = 28; // 2nd Porosity V-Shale Effect
  public static final int _2_SH    = 29; // 2nd Porosity Log Curve Shale Value

  // Cut-Offs

  public static final int _C_PHI   = 30; // Porosity Cut Off
  public static final int _C_SW    = 31; // Water Saturation Cut Off
  public static final int _C_VSH   = 32; // Fractional Shale Cut Off
  public static final int _C_BVW   = 33; // Bulk Volume Water Cut Off

  // Wyllie-Rose Equation Parameters

  public static final int _P       = 34;
  public static final int _Q       = 35;
  public static final int _R       = 36;

  // Volumetrics Data

  public static final int _V_THK   = 37; // Columns as Thickness
  public static final int _V_FT    = 38; // Oil or Gas Feet
  public static final int _V_PAY   = 39; // Amount of Pay in Feet
  public static final int _V_PHI   = 40; // Average Porosity
  public static final int _V_SW    = 41; // Average Water Saturation

  public static final int TOTAL_FLOW = 42;

  public static final String P_CURVES[][] =
  {
    {"Pf-KEY",  "KEY",   "Unique Identifier",           "",      "0.0",  "0.0",  "S" },

    {"Pf-ZONE", "ZONE",  "Flow Unit Name",              "",      "0.0",  "0.0",  "S" },
    {"Pf-STRT", "STRT",  "Starting Depth",              "F",     "0.0",  "0.0",  "F" },
    {"Pf-STOP", "STOP",  "Ending Depth",                "F",     "0.0",  "0.0",  "F" },

    {"Pf-ROCK", "ROCK",  "Rock Matrix",                 "",      "0.0",  "0.0",  "S" },
    {"Pf-H2O",  "H2O",   "Water Model Used",            "",      "0.0",  "0.0",  "S" },

    {"Pf-A",    "A",     "Archie Constant",             "FRAC",  "0.0",  "0.0",  "F" },
    {"Pf-M",    "M",     "Archie Cementation Factor",   "FRAC",  "0.0",  "0.0",  "F" },
    {"Pf-N",    "N",     "Archie Saturation Exponent",  "FRAC",  "0.0",  "0.0",  "F" },
    {"Pf-RW",   "RW",    "Formation water resistivity", "OHM-M", "0.01", "1000", "F" },
    {"Pf-RSH",  "RSH",   "Shale resistivity",           "OHM-M", "0.01", "1000", "F" },
    {"Pf-PHISH","PHISH", "Shale porosity",              "PU",    "0.0",  "0.5",  "F" },

    {"Pf-L-R",    "L_RT",   "Resistivity Log Curve",       "",  "0.0",  "0.0",  "S" },
    {"Pf-L-X",    "L_RXO",  "Rxo Resistivity Log Curve",   "",  "0.0",  "0.0",  "S" },

    {"Pf-L-V",    "L_VSH",  "V-Shale Log Curve",           "",  "0.0",  "0.0",  "S" },
    {"Pf-CLEAN",  "CLEAN",  "Vsh - Clean Formation Value", "",  "0.0",  "0.0",  "F" },
    {"Pf-SHALE",  "SHALE",  "Vsh - Shale Value",           "",  "0.0",  "0.0",  "F" },

    {"Pf-L-PHIT", "L_PHIT", "Total Porosity Type",         "",  "0.0",  "0.0",  "S" },
    {"Pf-P-PHI1", "L_PHI1", "1st Log curve - Porosity",    "",  "0.0",  "0.0",  "S" },
    {"Pf-P-PHI2", "L_PHI2", "2nd Log curve - Porosity",    "",  "0.0",  "0.0",  "S" },
    {"Pf-GRAIN",  "GRAIN",  "Rock Matrix Grain Density",   "GM/CC",  "1.0",  "4.0",  "F" },
    {"Pf-FLUID",  "FLUID",  "Fluid Density",               "GM/CC",  "0.0",  "2.0",  "F" },
    {"Pf-P-VSH",  "PHI_VSH", "Porosity V-Shale Effect",    "",  "NO",   "YES",  "S" },
    {"Pf-P-SH",   "PHI_SH",  "1st Log Curve Shale Value",  "",  "0.0",  "0.0",  "F" },
    {"Pf-P-SH2",  "PHI_SH2", "2nd Log Curve Shale Value",  "",  "0.0",  "0.0",  "F" },

    {"Pf-L-2ND",  "L_2ND",   "Log curve - Primary Porosity",  "",  "0.0",  "0.0",  "S" },
    {"Pf-GRAIN2", "2_GRAIN", "Rock Matrix Grain Density",     "GM/CC",  "1.0",  "4.0",  "F" },
    {"Pf-FLUID2", "2_FLUID", "Fluid Density of 2nd Porosity", "GM/CC",  "0.0",  "2.0",  "F" },
    {"Pf-2-VSH",  "2_VSH",   "2nd Porosity V-Shale Effect",   "",  "NO",   "YES",  "S" },
    {"Pf-2-SH",   "2_SH",    "2nd Porosity Shale Value", "",  "0.0",  "0.0",  "F" },

    {"Pf-C-PHI",   "C_PHI",  "Porosity Cut Off",          "PU",   "0.0",  "1.0",  "F" },
    {"Pf-C-SW",    "C_SW",   "Water Saturation Cut Off",  "FRAC", "0.0",  "1.0",  "F" },
    {"Pf-C-VSH",   "C_VSH",  "Fractional Shale Cut Off",  "FRAC", "0.0",  "1.0",  "F" },
    {"Pf-C-BVW",   "C_BVW",  "Bulk Volume Water Cut Off", "PU",   "0.0",  "1.0",  "F" },

    {"Pf-P",    "P", "Wyllie-Rose Equation Parameter - P", "FRAC", "0.0",  "1.0",  "F" },
    {"Pf-Q",    "Q", "Wyllie-Rose Equation Parameter - Q", "FRAC", "0.0",  "1.0",  "F" },
    {"Pf-R",    "R", "Wyllie-Rose Equation Parameter - R", "FRAC", "0.0",  "1.0",  "F" },

    {"Pf-V_THK",  "V_THK",  "Columns as Thickness",     "F",    "0.0",  "0.0",  "F" },
    {"Pf-V_FT",   "V_FT",   "Oil or Gas Feet",          "F",    "0.0",  "0.0",  "F" },
    {"Pf-V_PAY",  "V_PAY",  "Amount of Pay in Feet",    "F",    "0.0",  "0.0",  "F" },
    {"Pf-V_PHI",  "V_PHI",  "Average Porosity",         "PU",   "0.0",  "5.0",  "F" },
    {"Pf-V_SW",   "V_SW",   "Average Water Saturation", "FRAC", "0.0",  "1.0",  "F" },
  };

  public static final int _FKEY  = 0;

  public static final int _DEPTH = 1;
  public static final int _THK   = 2;
  public static final int _RT    = 3;
  public static final int _PHIT  = 4;
  public static final int _VSH   = 5;

  public static final int _PHI1  = 6;
  public static final int _PHI2  = 7;

  public static final int _RWA   = 8;
  public static final int _RO    = 9;
  public static final int _M_    = 10;
  public static final int _MA    = 11;
  public static final int _SW    = 12;
  public static final int _BVW   = 13;
  public static final int _PAY   = 14;

  public static final int _PHIR  = 15;

  public static final int _RXO   = 16;
  public static final int _SXO   = 17;
  public static final int _BVF   = 18;

  public static final int _PHIE  = 19;
  public static final int _CBW   = 20;
  public static final int _BVH   = 21;

  public static final int TOTAL_PF = 19;
  public static final int TOTAL  = 22;

  public static final String CURVES[][] =
  {
    {"Pf-FKEY",  "FKEY",  "Unique Identifier",       "",      "0.0",  "0.0",  "S" },

    {"PF-DEPTH", "DEPTH", "Depth",                   "F",     "0.0",  "0.0",  "F" },
    {"PF-THK",   "THK",   "Thickness",               "F",     "0.0",  "0.0",  "F" },
    {"PF-RT",    "RT",    "Total Resistivity",       "OHM-M", "0.01", "1000", "F" },
    {"PF-PHIT",  "PHIT",  "Total Porosity",          "PU",    "0.0",  "0.5",  "F" },
    {"PF-VSH",   "VSH",   "V-Shale",                 "FRAC",  "0.0",  "1.0",  "F" },

    {"PF-PHI1",  "PHI1",  "1st Porosity",            "PU",    "0.0",  "0.5",  "F" },
    {"PF-PHI2",  "PHI2",  "2nd Porosity",            "PU",    "0.0",  "0.5",  "F" },

    {"PF-RWA",   "RWA",   "Water Resistivity",       "OHM-M", "0.01", "1000", "F" },
    {"PF-RO",    "RO",    "Water Saturated Rock Resistivity", "OHM-M", "0.01", "1000", "F" },
    {"PF-Mc",    "M",     "Archie Cementation",      "FRAC",  "1.0",  "4.0",  "F" },
    {"PF-MA",    "MA",    "Archie Cementation",      "FRAC",  "1.0",  "4.0",  "F" },
    {"PF-SW",    "SW",    "Water Saturation",        "FRAC",  "0.0",  "1.0",  "F" },
    {"PF-BVW",   "BVW",   "Bulk Volume Water",       "PU",    "0.0",  "0.5",  "F" },
    {"PF-PAY",   "PAY",   "Pay",                     "F",     "0.0",  "0.5",  "F" },

    {"PF-PHIR",  "PHIR",  "Resistivity Porosity",    "PU",    "0.0",  "0.5",  "F" },

    {"PF-RXO",   "RXO",   "Medium Resistivity",      "OHM-M", "0.01", "1000", "F" },
    {"PF-SXO",   "SXO",   "Moveable Saturation",     "FRAC",  "0.0",  "1.0",  "F" },
    {"PF-BVF",   "BVF",   "Bulk Volume Fluid",       "PU",    "0.0",  "0.5",  "F" },

    {"PF-PHIE",  "PHIE",  "Effective Porosity",      "PU",    "0.0",  "0.5",  "F" },
    {"PF-CBW",   "CBW",   "Clay Bound Water",        "PU",    "0.0",  "0.5",  "F" },
    {"PF-BVH",   "BVH",   "Bulk Volumn Hydrocarbon", "PU",    "0.0",  "0.5",  "F" },
  };


  /** Method isPCurve()
   * <p> This method will determine if the curve is part of the standard
   *     stratigraphic units curves from the LAS File.
   * @param  iColumn = The column number
   * @param  str     = the string being compared.
   * @return bValue  = true it is a valid curve, false it is not
   */

  public static boolean isPCurve( int iColumn, String str )
  {
    boolean bValue = false;

    if (iColumn==1) str = new String( str.toUpperCase() );

    for (int i=0; i<TOTAL_FLOW; i++)
    {
      if (str.equals(P_CURVES[i][iColumn]))
      {
        bValue = true;
      }
    }

    return (bValue);
  }

  /** Method getPRow()
   * <p> This method will return the row number of the Stratigraphic Unit Curve
   * @param  iColumn = The column number
   * @param  str     = the string being compared.
   * @return iRow = row number
   */

  public static int getPRow( int iColumn, String str )
  {
    int iRow = -1;

    if (iColumn==1) str = new String( str.toUpperCase() );

    for (int i=0; i<TOTAL_FLOW; i++)
    {
      if (str.equals(P_CURVES[i][iColumn]))
      {
        iRow = i;
      }
    }

    return (iRow);
  }

  /** Method isCurve()
   * <p> This method will determine if the curve is part of the standard
   *     stratigraphic units curves from the LAS File.
   * @param  iColumn = The column number
   * @param  str     = the string being compared.
   * @return bValue  = true it is a valid curve, false it is not
   */

  public static boolean isCurve( int iColumn, String str )
  {
    boolean bValue = false;

    if (iColumn==1) str = new String( str.toUpperCase() );

    for (int i=0; i<TOTAL; i++)
    {
      if (str.equals(CURVES[i][iColumn]))
      {
        bValue = true;
      }
    }

    return (bValue);
  }

  /** Method getRow()
   * <p> This method will return the row number of the Stratigraphic Unit Curve
   * @param  iColumn = The column number
   * @param  str     = the string being compared.
   * @return iRow = row number
   */

  public static int getRow( int iColumn, String str )
  {
    int iRow = -1;

    if (iColumn==1) str = new String( str.toUpperCase() );

    for (int i=0; i<TOTAL; i++)
    {
      if (str.equals(CURVES[i][iColumn]))
      {
        iRow = i;
      }
    }

    return (iRow);
  }
}