/*
 * @pfefferPlotLimitsUtility.java Version 1.1 12/10/2013
 *
 * Copyright (c) 2013 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package pfeffer;

/** Class pfefferPlotLimitsUtility
 *  <p> This Class are utilities to build the PfEFFER Plot Track
 *      Limits Data Array.  This will allow the user to change
 *      the plot limits.
 */

public class pfefferPlotLimitsUtility
{
  public static final int _FKEY   = pfeffer.pfefferStandardTools._FKEY;

  public static final int _DEPTH  = pfeffer.pfefferStandardTools._DEPTH;
  public static final int _THK    = pfeffer.pfefferStandardTools._THK;
  public static final int _RT     = pfeffer.pfefferStandardTools._RT;
  public static final int _PHIT   = pfeffer.pfefferStandardTools._PHIT;
  public static final int _VSH    = pfeffer.pfefferStandardTools._VSH;

  public static final int _PHI1   = pfeffer.pfefferStandardTools._PHI1;
  public static final int _PHI2   = pfeffer.pfefferStandardTools._PHI2;

  public static final int _RWA    = pfeffer.pfefferStandardTools._RWA;
  public static final int _RO     = pfeffer.pfefferStandardTools._RO;
  public static final int _M_     = pfeffer.pfefferStandardTools._M_;
  public static final int _MA     = pfeffer.pfefferStandardTools._MA;
  public static final int _SW     = pfeffer.pfefferStandardTools._SW;
  public static final int _BVW    = pfeffer.pfefferStandardTools._BVW;
  public static final int _PAY    = pfeffer.pfefferStandardTools._PAY;

  public static final int _PHIR   = pfeffer.pfefferStandardTools._PHIR;

  public static final int _PHIE   = pfeffer.pfefferStandardTools._PHIE;
  public static final int _CBW    = pfeffer.pfefferStandardTools._CBW;
  public static final int _BVH    = pfeffer.pfefferStandardTools._BVH;

  public static final int ROWS    = pfeffer.pfefferStandardTools.TOTAL;

  public static final int MINIMUM = 0;
  public static final int MAXIMUM = 1;
  public static final int GROUP   = 2;
  public static final int DATA    = 3;
  public static final int COLUMNS = 4;

  /** Method initialize()
   *  <p> This method will initialize the LAS Plot Limits data Array
   *      There are 3 columns Minimum, Maximum, Group, Log Curve Present.
   *      The Group is used to force other curves that reside on the same
   *      plot track, i.e. resistivity, permeability, porosity and
   *      the Log Curve Present is an indicator to include this curve.
   *
   * @return  dOUT = Array of data limits
   */

  public static double[][] initialize()
  {
    double dOUT[][] = null;
    double dMinimum = 0.0;
    double dMaximum = 0.0;

    dOUT = new double[ROWS][COLUMNS];

    for (int i=0; i<ROWS; i++)
    {
      dOUT[i][MINIMUM] = cmn.cmnString.stringToDouble(
                           pfeffer.pfefferStandardTools.CURVES[i][4]);
      dOUT[i][MAXIMUM] = cmn.cmnString.stringToDouble(
                           pfeffer.pfefferStandardTools.CURVES[i][5]);
      dOUT[i][GROUP]   = -1.0;
      dOUT[i][DATA]    = -1.0;

      // Identify Groups for Curves

      switch (i)
      {
        case  _FKEY:

        case  _DEPTH:
        case  _THK:
        case  _VSH:
          break;

        case  _RT:
        case  _RWA:
        case  _RO:
          dOUT[i][GROUP] = (double) _RT;
          break;

        case  _PHIT:
        case  _PHI1:
        case  _PHI2:
        case  _PHIR:

        case  _BVW:
          dOUT[i][GROUP] = (double) _PHIT;
          break;

        case  _SW:
        case  _PAY:
          break;

        case  _MA:
        case  _M_:
          dOUT[i][GROUP] = (double) _MA;
          break;

        case  _PHIE:
        case  _CBW:
        case  _BVH:
          dOUT[i][GROUP] = (double) _PHIT;
          break;
      }

      // Set Curves with Data that will allow change of Limits

      switch (i)
      {
        case  _FKEY:
          break;

        case  _DEPTH:
        case  _THK:
          break;

        case  _RT:
        case  _VSH:

        case  _PHIT:
        case  _PHI1:
        case  _PHI2:
        case  _PHIR:

        case  _RWA:
        case  _RO:
        case  _BVW:
        case  _SW:
        case  _PAY:
        case  _M_:
        case  _MA:
          dOUT[i][DATA] = 1.0;
          break;

        case  _PHIE:
        case  _CBW:
        case  _BVH:
          break;
  	  }
    }

    return (dOUT);
  }

  /** Method copy()
   * <p>  This method will copy the LAS Plot Limits data Array
   *      There are 3 columns Minimum, Maximum, Group, Log Curve Present.
   *      The Group is used to force other curves that reside on the same
   *      plot track, i.e. resistivity, permeability, porosity and
   *      the Log Curve Present is an indicator to include this curve.
   *
   * @param   dIN  = Array of data limits
   * @return  dOUT = Array of data limits copied from dIN
   */

  public static double[][] copy( double dIN[][] )
  {
    double dOUT[][] = null;
    double iColumns = 3;

    if (dIN != null)
    {
      dOUT = new double[ROWS][COLUMNS];

      for (int i=0; i<ROWS; i++)
      {
		for (int j=0; j<COLUMNS; j++)
		{
		  dOUT[i][j] = dIN[i][j];
		}
      }
    }

    return (dOUT);
  }

  /** Method setByGroup()
   * <p>  This method will copy the LAS Plot Limits data Array
   *      There are 3 columns Minimum, Maximum, Group, Log Curve Present.
   *      The Group is used to force other curves that reside on the same
   *      plot track, i.e. resistivity, permeability, porosity and
   *      the Log Curve Present is an indicator to include this curve.
   *
   * @param   dGroup = Group ID
   * @param   dMin   = Minimum Value for Group
   * @param   dMax   = Maximum Value for Group
   * @param   data   = Array of data limits
   * @return  data   = Array of data limits normalized by group
   */

  public static double[][] setByGroup( double dGroup, double dMin, double dMax, double data[][] )
  {
    if (data != null)
    {
      for (int i=0; i<ROWS; i++)
      {
		if ((dGroup == data[i][GROUP]) && (data[i][GROUP] > -1))
		{
          data[i][MINIMUM] = dMin;
          data[i][MAXIMUM] = dMax;
		}
      }
    }

    return (data);
  }
}

/*
 *  @version 1.1 12/10/2013
 *  @author  John Victorine
 */
