/*
 * @pfefferDataUtility.java Version 1.1 01/11/2010
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package pfeffer;

import iqstrat.iqstratHeadersStruct;
import pfeffer.pfefferDataListStruct;
import pfeffer.pfefferDataStruct;

import las.lasFileDataStruct;
import lith.texture.textureListStruct;
import lith.lithology.lithologySymbolsListStruct;

/** Class pfefferDataUtility
 *  <p> This Class will provide basic utilities for the pfeffer Data Structures
 *
 *  @version 1.1 01/11/2010
 *  @author  John R. Victorine
 */

public class pfefferDataUtility
{
  /** Method addHeaderData()
   * <p> This method will add the header data to the PfEFFER data List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The PfEFFER List Data Structure
   * @return st       = The PfEFFER List Data Structure
   */

  public static pfefferDataListStruct addHeaderData(
      iqstratHeadersStruct stHeader, pfefferDataListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      st.stHeader = iqstrat.iqstratHeadersUtility.copy( stHeader );
    }

    return (st);
  }

  /** Method computeDepthRange()
   * <p> This method will compute the depth range of the core data
   * @param   st = the PfEFFER data list structure
   * @return  st = the PfEFFER data list structure
   */

  public static pfefferDataListStruct computeDepthRange(pfefferDataListStruct st)
  {
    double depthStart = 0.0;
    double depthEnd   = 0.0;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        depthStart = st.stItem[i].depthStart;
        depthEnd   = st.stItem[i].depthEnd;

        if (i==0)
        {
          st.depthStart = depthStart;
          st.depthEnd   = depthEnd;
        }

        if (st.depthStart > depthStart)
          st.depthStart = depthStart;

        if (st.depthEnd < depthEnd)
          st.depthEnd   = depthEnd;
      }
    }

    return (st);
  }

  /** Method add()
   * <p> This method will add the pfeffer Data Structure to the list
   * @param   stBlock = the PfEFFER data structure to add to the list
   * @param   st      = the PfEFFER data list structure
   * @return  st      = the PfEFFER data list structure
   */

  public static pfefferDataListStruct add( pfefferDataStruct stBlock,
                                           pfefferDataListStruct st )
  {
    int i = 0;
    int iRecords = 0;
    int iCount   = 0;
    pfefferDataListStruct stTemp   = null;
    String sKGS      = "YES"; // KGS Saved Data Indicator
    String source    = "";    // Source of Data
    String sCreated  = "";    // Created Date
    String savedLast = "";    // Latest Save Date

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp          = new pfefferDataListStruct();
    stTemp.stItem   = new pfefferDataStruct[iRecords];

    if (st != null)
    {
      if (st.stHeader != null)
        stTemp.stHeader = iqstrat.iqstratHeadersUtility.copy( st.stHeader );

      stTemp.source    = new String( st.source );    // Source of Data
      stTemp.sCreated  = new String( st.sCreated );  // Created Date
      stTemp.savedLast = new String( st.savedLast ); // Latest Save Date

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy( st.stItem[i] );
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new pfefferDataStruct();
    stTemp.stItem[iCount] = copy( stBlock );
    iCount++;

    stTemp.iCount = iCount;

    st          = new pfefferDataListStruct();
    if (stTemp.stHeader != null)
      st.stHeader = iqstrat.iqstratHeadersUtility.copy( stTemp.stHeader );

    st.sKGS      = new String("YES");              // KGS Saved Data Indicator
    st.source    = new String( stTemp.source );    // Source of Data
    st.sCreated  = new String( stTemp.sCreated );  // Created Date
    st.savedLast = new String( stTemp.savedLast ); // Latest Save Date

    st.stItem   = new pfefferDataStruct[iCount];
    st.iCount   = iCount;

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy( stTemp.stItem[i] );
    }

    stTemp.delete();

    st = computeDepthRange( st );

    return (st);
  }

  /** Method modify()
   * <p> This method will modify the pfeffer Data Structure in the list
   * @param   sKEY    = Unique Identifier for Flow Unit Zone
   * @param   stBlock = the PfEFFER data structure to add to the list
   * @param   st      = the PfEFFER data list structure
   * @return  st      = the PfEFFER data list structure
   */

  public static pfefferDataListStruct modify( String sKEY,
                                              pfefferDataStruct stBlock,
                                              pfefferDataListStruct st )
  {
    int i=0;
    pfefferDataListStruct stTemp = null;
    String sKGS      = "YES"; // KGS Saved Data Indicator
    String source    = "";    // Source of Data
    String sCreated  = "";    // Created Date
    String savedLast = "";    // Latest Save Date

    if (st != null)
    {
      stTemp          = new pfefferDataListStruct();
      if (st.stHeader != null)
        stTemp.stHeader = iqstrat.iqstratHeadersUtility.copy( st.stHeader );

      stTemp.source    = new String( st.source );    // Source of Data
      stTemp.sCreated  = new String( st.sCreated );  // Created Date
      stTemp.savedLast = new String( st.savedLast ); // Latest Save Date

      stTemp.stItem   = new pfefferDataStruct[st.iCount];
      stTemp.iCount   = st.iCount;

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            stTemp.stItem[i] = copy( stBlock );
          }
          else
          {
            stTemp.stItem[i] = copy( st.stItem[i] );
          }
        }
      }

      st.delete();

      st          = new pfefferDataListStruct();
      if (stTemp.stHeader != null)
        st.stHeader = iqstrat.iqstratHeadersUtility.copy( stTemp.stHeader );

      st.sKGS      = new String("YES");              // KGS Saved Data Indicator
      st.source    = new String( stTemp.source );    // Source of Data
      st.sCreated  = new String( stTemp.sCreated );  // Created Date
      st.savedLast = new String( stTemp.savedLast ); // Latest Save Date

      st.stItem   = new pfefferDataStruct[stTemp.iCount];
      st.iCount   = stTemp.iCount;

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy( stTemp.stItem[i] );
      }

      stTemp.delete();
    }

    st = computeDepthRange( st );

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a pfeffer Data Structure from the list
   * @param  sKEY = The Primary Key of the data to be removed
   * @param  st   = The Old pfeffer List Data Structure
   * @return st   = The new pfeffer List.
   */

  public static pfefferDataListStruct remove( String sKEY,
                                              pfefferDataListStruct st )
  {
    int i=0;
    int iRecords = 0;
    int iCount   = 0;
    int iFound   = -1;
    pfefferDataListStruct stTemp = null;
    String sKGS      = "YES"; // KGS Saved Data Indicator
    String source    = "";    // Source of Data
    String sCreated  = "";    // Created Date
    String savedLast = "";    // Latest Save Date

    if (st.iCount == 1)
    {
      st.delete();
      st = null;
    }
    else
    {
      if (st != null)
      {
        iRecords = st.iCount-1;

        stTemp          = new pfefferDataListStruct();
        if (st.stHeader != null)
          stTemp.stHeader = iqstrat.iqstratHeadersUtility.copy( st.stHeader );

        stTemp.source    = new String( st.source );    // Source of Data
        stTemp.sCreated  = new String( st.sCreated );  // Created Date
        stTemp.savedLast = new String( st.savedLast ); // Latest Save Date

        stTemp.stItem   = new pfefferDataStruct[iRecords];

        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            iFound = i;
          }
        }
      }

      if (st != null)
      {
        if (st.iCount > 0)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (iCount < iRecords)
            {
              if (i != iFound)
              {
                stTemp.stItem[iCount] = copy( st.stItem[i] );
                iCount++;
              }
            }
          }
        }

        st.delete();
      }

      stTemp.iCount = iCount;

      st          = new pfefferDataListStruct();
      if (stTemp.stHeader != null)
        st.stHeader = iqstrat.iqstratHeadersUtility.copy( stTemp.stHeader );

      st.sKGS      = new String("YES");              // KGS Saved Data Indicator
      st.source    = new String( stTemp.source );    // Source of Data
      st.sCreated  = new String( stTemp.sCreated );  // Created Date
      st.savedLast = new String( stTemp.savedLast ); // Latest Save Date

      st.stItem   = new pfefferDataStruct[iCount];
      st.iCount   = iCount;

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy( stTemp.stItem[i] );
      }

      stTemp.delete();

      st = computeDepthRange( st );
    }

    return (st);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static pfefferDataListStruct transfer(pfefferDataListStruct stOld)
  {
    pfefferDataListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copyList( pfefferDataListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static pfefferDataListStruct copyList( pfefferDataListStruct stOld )
  {
    pfefferDataListStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new pfefferDataListStruct();

      if (stOld.stHeader != null)
        stNew.stHeader = iqstrat.iqstratHeadersUtility.copy(stOld.stHeader);

      stNew.depthStart = stOld.depthStart; // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth

      stNew.sKGS       = new String( stOld.sKGS ); // KGS Saved Data Indicator
      stNew.source     = new String( stOld.source );    // Source of Data
      stNew.sCreated   = new String( stOld.sCreated );  // Created Date
      stNew.savedLast  = new String( stOld.savedLast ); // Latest Save Date

      stNew.iCount     = stOld.iCount;
      stNew.stItem     = new pfefferDataStruct[stOld.iCount];

      for (int i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy data from one structure to another
   * @param  stOld = The original PfEFFER Data Structure
   * @return stNew = The PfEFFER Data Structure
   */

  public static pfefferDataStruct copy( pfefferDataStruct stOld )
  {
    pfefferDataStruct stNew = new pfefferDataStruct();

    if (stOld != null)
    {
      stNew.sKEY       = new String(stOld.sKEY);  // Unique KEY
      stNew.sZone      = new String(stOld.sZone); // Name of the Zone
      stNew.depthStart = stOld.depthStart;        // Starting Depth (Feet)
      stNew.depthEnd   = stOld.depthEnd;          // Ending Depth (Feet)

      // Volumetrics Data

      stNew.dThickness   = stOld.dThickness;   // Columns as Thickness
      stNew.dHydroCarbon = stOld.dHydroCarbon; // Oil or Gas Feet
      stNew.dPay         = stOld.dPay;         // Amount of Pay in Feet
      stNew.dPorosity    = stOld.dPorosity;    // Average Porosity
      stNew.dSaturation  = stOld.dSaturation;  // Average Water Saturation

      // -- Resistivity Track Variables

      stNew.iRt = stOld.iRt;  // Log curve Selected

      // -- Vsh Track Variables

      stNew.iVsh   = stOld.iVsh; // Log Curve used to compute Vsh
      stNew.dClean = stOld.dClean;   // Clean Value
      stNew.dShale = stOld.dShale;   // Shale Value

      // -- Porosity Track Variables

      stNew.iPHIt = stOld.iPHIt;  // Type of Porosity
      stNew.iPHI1 = stOld.iPHI1;  // 1st Porosity Curve
      stNew.iPHI2 = stOld.iPHI2;  // 2nd Porosity Curve

      // -- Lithology Symbol

      stNew.iLithology = stOld.iLithology;

      // -- Used with Bulk Density to compute porosity

      stNew.dGrain = stOld.dGrain;  // Grain
      stNew.dFluid = stOld.dFluid;  // Fluid

      // -- Used with porosity to compute Vsh Effect

      stNew.iPHIVsh    = stOld.iPHIVsh;
      stNew.dPHIShale1 = stOld.dPHIShale1; // Log Curve Value for Shale
      stNew.dPHIShale2 = stOld.dPHIShale2; // Log Curve Value for Shale - 2nd Curve

      // -- 2nd Porosity Curve ID

      stNew.iPHI2nd = stOld.iPHI2nd;  // 1st Porosity Curve

      // -- Used with 2nd Porosity to compute porosity

      stNew.dtGrain = stOld.dtGrain;  // Grain
      stNew.dtFluid = stOld.dtFluid;  // Fluid

      // -- Used with Primary Porosity to compute Vsh Effect

      stNew.iPHI2ndVsh   = stOld.iPHI2ndVsh;
      stNew.dPHI2ndShale = stOld.dPHI2ndShale; // Log Curve Value for Shale

      // Water Model Used

      stNew.iWaterModel = stOld.iWaterModel;   // Water Model Identifier
      stNew.sWaterModel = new String(stOld.sWaterModel);  // Water Model Used

      // Archie Equation Parameters

      stNew.A     = stOld.A;     // Archie multiplication factor
      stNew.M     = stOld.M;     // Cementation factor
      stNew.Mp    = stOld.Mp;    // Modify Cementation factor (M) by adding Mp*PHI
      stNew.N     = stOld.N;     // Saturation exponent
      stNew.Rw    = stOld.Rw;    // Formation water resistivity
      stNew.Rsh   = stOld.Rsh;   // Shale resistivity
      stNew.Phish = stOld.Phish; // Shale porosity

      // Moveable Hydrocarbon Parameters

      stNew.iRxo  = stOld.iRxo;  // Rxo Log curve Selected

      stNew.Rmf   = stOld.Rmf;   // Mud Filtrate Resistivity (Rmf)
      stNew.Rmft  = stOld.Rmft;  // Mud Filtrate Resistivity Temperature (Rmft)
      stNew.ST    = stOld.ST;    // Surface Temperature (ST)
      stNew.BHT   = stOld.BHT;   // Bore Hole Temperature (BHT)
      stNew.TD    = stOld.TD;    // Total Depth (TD)

      stNew.iTemp = stOld.iTemp; // Temperature Unit

      // Cut-Offs

      stNew.dPhiCut = stOld.dPhiCut; // Porotiy Cut Off
      stNew.dSwCut  = stOld.dSwCut;  // Water Saturation Cut Off
      stNew.dVshCut = stOld.dVshCut; // Fractional Shale Cut Off
      stNew.dBvwCut = stOld.dBvwCut; // Bulk Volume Water Cut Off

      // Wyllie-Rose Equation Parameters

      stNew.P = stOld.P;
      stNew.Q = stOld.Q;
      stNew.R = stOld.R;

      stNew.iRows = stOld.iRows;

      // Log Values

      if (stOld.iRows > 0)
      {
        stNew.depth = new double[stOld.iRows]; // Depth
        stNew.thick = new double[stOld.iRows]; // Thickness

        stNew.Rt    = new double[stOld.iRows]; // True Resistivity
        stNew.Vsh   = new double[stOld.iRows]; // V-Shale
        stNew.PHIt  = new double[stOld.iRows]; // True Porosity

        // 2nd Porosity

        stNew.PHI1st = new double[stOld.iRows]; // 1st Porosity (Sonic)
        stNew.PHI2nd = new double[stOld.iRows]; // 2nd Porosity

        // Computed Variables

        stNew.Rtc   = new double[stOld.iRows]; // Corrected resistivity
        stNew.Rwa   = new double[stOld.iRows]; // Apparent form water resistivity
        stNew.Ro    = new double[stOld.iRows]; // Resistivity at 100% saturation
        stNew.Mo    = new double[stOld.iRows]; // computed cementation exponent
        stNew.Ma    = new double[stOld.iRows]; // Apparent cementation exponent
        stNew.Sw    = new double[stOld.iRows]; // Water saturation itself
        stNew.BVW   = new double[stOld.iRows]; // Bulk volume water (Sw*porosity)

        stNew.Pay   = new double[stOld.iRows]; // Pay

        stNew.PHIr  = new double[stOld.iRows]; // Electrically Connected Porosity

        stNew.Rxo   = new double[stOld.iRows]; // Filtrate Resistivity (Rxo)
        stNew.Sxo   = new double[stOld.iRows]; // Saturation of total moveable fluid
        stNew.BVF   = new double[stOld.iRows]; // Bulk volume (moveable) fluid (Sxo*porosity)

        for (int i=0; i<stOld.iRows; i++)
        {
          stNew.depth[i]  = stOld.depth[i];  // Depth
          stNew.thick[i]  = stOld.thick[i];  // Thickness

          stNew.Rt[i]     = stOld.Rt[i];     // True Resistivity
          stNew.Vsh[i]    = stOld.Vsh[i];    // V-Shale
          stNew.PHIt[i]   = stOld.PHIt[i];   // True Porosity

          // 2nd Porosity

          stNew.PHI1st[i] = stOld.PHI1st[i]; // 1st Porosity (sonic)
          stNew.PHI2nd[i] = stOld.PHI2nd[i]; // 2nd Porosity (total - 1st)

          // Computed Variables

          stNew.Rtc[i]    = stOld.Rtc[i];    // Corrected resistivity
          stNew.Rwa[i]    = stOld.Rwa[i];    // Apparent form water resistivity
          stNew.Ro[i]     = stOld.Ro[i];     // Resistivity at 100% saturation
          stNew.Mo[i]     = stOld.Mo[i];     // computed cementation exponent
          stNew.Ma[i]     = stOld.Ma[i];     // Apparent cementation exponent
          stNew.Sw[i]     = stOld.Sw[i];     // Water saturation itself
          stNew.BVW[i]    = stOld.BVW[i];    // Bulk volume water (Sw*porosity)
          stNew.Pay[i]    = stOld.Pay[i];    // Pay

          stNew.PHIr[i]   = stOld.PHIr[i];   // Electrically Connected Porosity

          stNew.Rxo[i]    = stOld.Rxo[i];    // Filtrate Resistivity (Rxo)
          stNew.Sxo[i]    = stOld.Sxo[i];    // Saturation of total moveable fluid
          stNew.BVF[i]    = stOld.BVF[i];    // Bulk volume (moveable) fluid (Sxo*porosity)
        }
      }
    }

    return (stNew);
  }

  /** Method getData()
   * <p> This method will retrieve the default PfEFFER Log Curves and fill
   *     the PfEFFER Values for the  whole log as default. This may be modified
   *     by the user during the interactive session.
   * @param  stLAS     = LAS File Data Structure
   * @param  stTexture = Texture Data Structure
   * @param  stSymbols = Lithology Symbols Data List Structure
   * @return stP       = The PfEFFER Data Structure
   */

  public static pfefferDataStruct getData( lasFileDataStruct stLAS,
                                           textureListStruct stTexture,
                                           lithologySymbolsListStruct stSymbols )
  {
    int i=0;
    int               icont  = 0;
    double            data[] = null;
    lasFileDataStruct st     = null;
    textureListStruct stT    = null;
    pfefferDataStruct stP    = null;

    if ((stLAS != null) && (stTexture != null))
    {
      st  = stLAS;
      stT = stTexture;

      stP = new pfefferDataStruct();

      // Get the Depth Arrays

      if (st != null)
      {
        if (st.iRows > 0)
        {
          stP.iRows  = pfeffer.math.pfefferMath.getRowCount(
                        st.depthStart, st.depthEnd, st.depthStep );
          stP.depth  = pfeffer.math.pfefferMath.getDepth(
                        st.depthStart, st.depthEnd, st.depthStep, st );
          stP.thick = pfeffer.math.pfefferMath.getThickness(
                        st.depthStart, st.depthEnd, st.depthStep );

          stP.Rt    = new double[st.iRows];
          stP.PHIt  = new double[st.iRows];
          stP.Vsh   = new double[st.iRows];

          stP.PHI1st = new double[st.iRows];
          stP.PHI2nd = new double[st.iRows];

          stP.Rtc   = new double[st.iRows];
          stP.Rwa   = new double[st.iRows];
          stP.Ro    = new double[st.iRows];
          stP.Mo    = new double[st.iRows];
          stP.Ma    = new double[st.iRows];
          stP.Sw    = new double[st.iRows];
          stP.BVW   = new double[st.iRows];

          stP.Pay   = new double[st.iRows];

          stP.PHIr  = new double[st.iRows];

          stP.Rxo   = new double[st.iRows];
          stP.Sxo   = new double[st.iRows];
          stP.BVF   = new double[st.iRows];
        }
      }

      for (i=0; i<stP.iRows; i++)
      {
        stP.Rt[i]    = st.dNull;
        stP.PHIt[i]  = st.dNull;
        stP.Vsh[i]   = st.dNull;

        stP.PHI1st[i] = st.dNull; // 1st Porosity (sonic)
        stP.PHI2nd[i] = 0.0;      // 2nd Porosity (total - 1st)

        stP.Rtc[i]    = 0.0;
        stP.Rwa[i]    = 0.0;
        stP.Ro[i]     = 0.0;
        stP.Mo[i]     = 2.0;
        stP.Ma[i]     = 0.0;
        stP.Sw[i]     = 0.0;
        stP.BVW[i]    = 0.0;

        stP.Pay[i]    = 0.0;

        stP.PHIr[i]   = 0.0;

        stP.Rxo[i]    = st.dNull;
        stP.Sxo[i]    = 0.0;
        stP.BVF[i]    = 0.0;
      }

      // Get the Total Resistivity ( Rt ) data array

      stP.iRt = getRt( st );
      if (stP.iRt > -1)
      {
        stP.Rt = pfeffer.math.pfefferMath.getRt(
                   st.depthStart, st.depthEnd, st.depthStep, stP.iRt, st );
      }

      // Get the Medium Resistivity ( Rxo ) data array

      stP.iRxo = getRxo( st );

      if (stP.iRxo > -1)
      {
        stP.Rxo = pfeffer.math.pfefferMath.getRt(
                   st.depthStart, st.depthEnd, st.depthStep, stP.iRxo, st );
      }

      // Get the V-Shale Factor

      stP.iVsh   = getVsh( st );
      if ((stP.iVsh == las.lasStandardTools._GR) ||
          (stP.iVsh == las.lasStandardTools._CGR))
      {
        stP.dClean = iqstrat.iqstratShaleStruct.GAMMA_MIN;
        stP.dShale = iqstrat.iqstratShaleStruct.GAMMA_MAX;
      }
      else if (stP.iVsh == las.lasStandardTools._SP)
      {
        stP.dClean = st.dSPMin;
        stP.dShale = st.dSPMax;
      }

      data = st.getData(stP.iVsh);
      for (i=0; i<st.iRows; i++)
      {
        if (data[i] > st.dNull)
        {
          if (icont == 0)
          {
            stP.dClean = data[i];
            icont = 1;
          }

          if (stP.dClean > data[i])
            stP.dClean = data[i];
        }
      }

      if (stP.dClean < 0.0) stP.dClean = 0.0;

      if (stP.iVsh > -1)
      {
        stP.Vsh = pfeffer.math.pfefferMath.getVsh(
                    st.depthStart, st.depthEnd, st.depthStep,
                    stP.iVsh, stP.dClean, stP.dShale, st );
      }

      for (i=0; i<stP.iRows; i++)
      {
        if (stP.Vsh[i] > 1.0)
          stP.Vsh[i] = 1.0;

        if (stP.Vsh[i] < 0.0)
          stP.Vsh[i] = 0.0;
      }

      // Get the Total Porosity ( PhiT ) data array

      stP.iPHIt = getPhit( st );
      if (stP.iPHIt > -1)
      {
        switch (stP.iPHIt)
        {
          case pfeffer.math.pfefferMath._NONE: // Porosity from Bulk Density
            break;
          case pfeffer.math.pfefferMath._RHOB:  // Porosity from Bulk Density
            stP.iPHI1 = las.lasStandardTools._DPHI;  // 1st Porosity Curve
            if (st.iRHOB > -1) stP.iPHI1 = las.lasStandardTools._RHOB;
            stP.iPHI2 = -1;                       // 2nd Porosity Curve
            break;
          case pfeffer.math.pfefferMath._NPHI:  // Porosity from Neutron Porosity
            stP.iPHI1 = las.lasStandardTools._NPHI;  // 1st Porosity Curve
            stP.iPHI2 = -1;                       // 2nd Porosity Curve
            break;
          case pfeffer.math.pfefferMath._PHIAvg:  // Average of Neutron & Density Porosity
            stP.iPHI1 = las.lasStandardTools._NPHI;  // 1st Porosity Curve
            stP.iPHI2 = las.lasStandardTools._DPHI;  // 2nd Porosity Curve
            if (st.iRHOB > -1) stP.iPHI2 = las.lasStandardTools._RHOB;
            break;
          case pfeffer.math.pfefferMath._SPHI:  // Porosity from Sonic
            stP.iPHI1 = las.lasStandardTools._SPHI;  // 1st Porosity Curve
            if (st.iDT > -1) stP.iPHI1 = las.lasStandardTools._DT;
            stP.iPHI2 = -1;                       // 2nd Porosity Curve
            break;
        }

        stP = getPHI( st, stT, stP, stSymbols );

        if ((stP.iRt > -1) && (stP.iPHIt > -1))
        {
          stP = pfeffer.math.pfefferMath.compute( stP, st.dNull );
        }

        for (i=0; i<stP.iRows; i++)
        {
          if (stP.Sw[i] > 1.0)
          {
            stP.Sw[i]  = 1.0;
            stP.BVW[i] = stP.PHIt[i];
          }
        }

        stP.iPHI2nd = getPHI1st( st );
        stP         = getPHI2( st, stT, stP );
      }

//      st.delete();
//      st = null;

//      stT.delete();
//      stT =null;
    }

    return (stP);
  }

  /** Method getRt()
   * <p> This method will return the data array for the Total Resistivity Data
   * @param  st  = LAS File Data Structure
   * @return iRt = the Resistivity Identifier
   */

  public static int getRt( lasFileDataStruct st )
  {
    int i       = 0;
    int iVal    = -1;
    int iTool[] = { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 };
    int iRt     = -1;

    for (i=las.lasStandardTools._RES; i<las.lasStandardTools._AHT90+1; i++)
    {
      if (st.checkData(i))
      {
        iVal        = i - las.lasStandardTools._RES;
        iTool[iVal] = 1;
      }
    }

    iVal = las.lasStandardTools._ILD - las.lasStandardTools._RES;
    if (iTool[iVal] == 1)
      iRt = las.lasStandardTools._ILD;

    iVal = las.lasStandardTools._AHT90 - las.lasStandardTools._RES;
    if ((iTool[iVal] == 1) && (iRt == -1))
      iRt = las.lasStandardTools._AHT90;

    iVal = las.lasStandardTools._LL - las.lasStandardTools._RES;
    if ((iTool[iVal] == 1) && (iRt == -1))
      iRt = las.lasStandardTools._LL;

    iVal = las.lasStandardTools._RDEP - las.lasStandardTools._RES;
    if ((iTool[iVal] == 1) && (iRt == -1))
      iRt = las.lasStandardTools._RDEP;

    if (iRt == -1)
    {
      for (i=0; i<iTool.length; i++)
      {
        if ((iTool[i] > -1) && (iRt == -1))
          iRt = i + las.lasStandardTools._RES;
      }
    }

    return (iRt);
  }

  /** Method getRxo()
   * <p> This method will return the data array for the Mud Filtrate Resistivity Data
   * @param  st  = LAS File Data Structure
   * @return iRt = the Resistivity Identifier
   */

  public static int getRxo( lasFileDataStruct st )
  {
    int i       = 0;
    int iVal    = -1;
    int iTool[] = { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
                    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1 };
    int iRt     = -1;

    for (i=las.lasStandardTools._RES; i<las.lasStandardTools._AHT90+1; i++)
    {
      if (st.checkData(i))
      {
        iVal        = i - las.lasStandardTools._RES;
        iTool[iVal] = 1;
      }
    }

    iVal = las.lasStandardTools._SFLU - las.lasStandardTools._RES;
    if (iTool[iVal] == 1)
      iRt = las.lasStandardTools._SFLU;

    iVal = las.lasStandardTools._AHT10 - las.lasStandardTools._RES;
    if ((iTool[iVal] == 1) && (iRt == -1))
      iRt = las.lasStandardTools._AHT10;

    iVal = las.lasStandardTools._LL8 - las.lasStandardTools._RES;
    if ((iTool[iVal] == 1) && (iRt == -1))
      iRt = las.lasStandardTools._LL8;

    iVal = las.lasStandardTools._RSHAL - las.lasStandardTools._RES;
    if ((iTool[iVal] == 1) && (iRt == -1))
      iRt = las.lasStandardTools._RSHAL;

    iVal = las.lasStandardTools._MSFL - las.lasStandardTools._RES;
    if ((iTool[iVal] == 1) && (iRt == -1))
      iRt = las.lasStandardTools._MSFL;

    if (iRt == -1)
    {
      for (i=0; i<iTool.length; i++)
      {
        if ((iTool[i] > -1) && (iRt == -1))
          iRt = i + las.lasStandardTools._RES;
      }
    }

    return (iRt);
  }

  /** Method getVsh()
   * <p> This method will return the data array for the Total Porosity Data
   * @param  st  = LAS File Data Structure
   * @return iVsh = the Shale Identifier
   */

  public static int getVsh( lasFileDataStruct st )
  {
    int i       = 0;
    int iVal    = -1;
    int iTool[] = { -1, -1, -1 };
    int iVsh   = -1;

    for (i=las.lasStandardTools._GR; i<las.lasStandardTools._SP+1; i++)
    {
      if (st.checkData(i))
      {
        iVal        = i - las.lasStandardTools._GR;
        iTool[iVal] = 1;
      }
    }

    if (iTool[0] > -1)
      iVsh = las.lasStandardTools._GR;
    else if (iTool[1] > -1)
      iVsh = las.lasStandardTools._CGR;
    else if (iTool[2] > -1)
      iVsh = las.lasStandardTools._SP;

    return (iVsh);
  }

  /** Method getPhit()
   * <p> This method will return the data array for the Total Porosity Data
   * @param  st  = LAS File Data Structure
   * @return iPhit = the Porosity Identifier
   */

  public static int getPhit( lasFileDataStruct st )
  {
    int i       = 0;
    int iVal    = -1;
    int iTool[] = { -1, -1, -1, -1, -1, -1 };
    int iPhit   = -1;

    for (i=las.lasStandardTools._RHOB; i<las.lasStandardTools._DT+1; i++)
    {
      if (st.checkData(i))
      {
        iVal        = i - las.lasStandardTools._RHOB;
        iTool[iVal] = 1;
      }
    }

    if (((iTool[0] > -1) || (iTool[2] > -1)) && (iTool[3] > -1))
      iPhit = pfeffer.math.pfefferMath._PHIAvg;
    else if (iTool[3] > -1)
      iPhit = pfeffer.math.pfefferMath._NPHI;
    else if ((iTool[0] > -1) || (iTool[2] > -1))
      iPhit = pfeffer.math.pfefferMath._RHOB;
    else if ((iTool[4] > -1) || (iTool[5] > -1))
      iPhit = pfeffer.math.pfefferMath._SPHI;

    return (iPhit);
  }

  /** Method getPHI()
   * <p> This method will calculate the PHI Value dynamically depending on the
   *     lithology of the matrix.
   * @param  stL       = LAS File Data Structure
   * @param  stT       = Texture Data Structure
   * @param  stP       = The PfEFFER Data Structure
   * @param  stSymbols = Lithology Symbols Data List Structure
   * @return data      = the Porosity data array
   */

  public static pfefferDataStruct getPHI( lasFileDataStruct stL,
                                          textureListStruct stT,
                                          pfefferDataStruct stP,
                                          lithologySymbolsListStruct stSymbols )
  {
    int iLAS    = -1;
    int iTEXT   = -1;
    int iLITH   = -1;

    if ((stL != null) && (stT != null) && (stP != null))
    {
      for (int i=0; i<stP.iRows; i++)
      {
        for (int k=0; k<stL.iRows; k++)
        {
          if (stL.depths[k] == stP.depth[i])
          {
            iLAS = k;
          }
        }

        iTEXT = lith.lithology.lithologySymbolsStruct._NONE;
        for (int j=0; j<stT.iCount; j++)
        {
          if (stT.stItem[j].depthStart == stP.depth[i])
          {
            iTEXT = stT.stItem[j].id;
          }
        }

        switch (iTEXT)
        {
          // Siliciclastic Lithotypes
          case lith.lithology.lithologySymbolsStruct._GRAVEL:
          case lith.lithology.lithologySymbolsStruct._SAND:
          case lith.lithology.lithologySymbolsStruct._SANDSTONE:
          case lith.lithology.lithologySymbolsStruct._SILT:
          case lith.lithology.lithologySymbolsStruct._SHALY_SAND:
            iLITH = lith.math.lithMath._QUARTZ;
            break;
          case lith.lithology.lithologySymbolsStruct._ARKOSE:
          case lith.lithology.lithologySymbolsStruct._FELDSPAR:
            iLITH = lith.math.lithMath._FELDSPAR;
            break;
          // Evaporites
          case lith.lithology.lithologySymbolsStruct._SALT_GENERAL:
          case lith.lithology.lithologySymbolsStruct._HALITE:
            iLITH = lith.math.lithMath._HALITE;
            break;
          // Organic-Rich Rocks
          case lith.lithology.lithologySymbolsStruct._COAL:
            iLITH = lith.math.lithMath._COAL;
            break;
          case lith.lithology.lithologySymbolsStruct._CLAY:
          case lith.lithology.lithologySymbolsStruct._SHALE:
            iLITH = lith.math.lithMath._QUARTZ;
            break;
          // Carbonate Lithotypes
          case lith.lithology.lithologySymbolsStruct._LIMESTONE:
          case lith.lithology.lithologySymbolsStruct._SHALY_CARB:
            iLITH = lith.math.lithMath._CALCITE;
            break;
          case lith.lithology.lithologySymbolsStruct._DOLOMITE:
          case lith.lithology.lithologySymbolsStruct._SHALY_DOLO:
          case lith.lithology.lithologySymbolsStruct._IRONSTONE:
            iLITH = lith.math.lithMath._DOLOMITE;
            break;
            // Evaporites
          case lith.lithology.lithologySymbolsStruct._GYPSUM:
            iLITH = lith.math.lithMath._GYPSUM;
            break;
          case lith.lithology.lithologySymbolsStruct._ANHYDRITE:
            iLITH = lith.math.lithMath._ANHYDRITE;
            break;
          case lith.lithology.lithologySymbolsStruct._NONE:
          default:
            iLITH = lith.math.lithMath._CALCITE;
            break;
        }

        switch (stP.iPHIt)
        {
          case pfeffer.math.pfefferMath._NONE: // Porosity from Bulk Density
          case pfeffer.math.pfefferMath._NPHI:  // Porosity from Neutron Porosity
            break;
          case pfeffer.math.pfefferMath._RHOB:  // Porosity from Bulk Density
          case pfeffer.math.pfefferMath._PHIAvg:  // Average of Neutron & Density Porosity
            stP.dGrain = lith.math.lithMath._MINERAL[iLITH][1];
            stP.dFluid = cmn.cmnString.stringToDouble(
                           pfeffer.math.pfefferMath.FLUID[0][1] );
            break;
          case pfeffer.math.pfefferMath._SPHI:  // Porosity from Sonic
            stP.dGrain = lith.math.lithMath._MINERAL[iLITH][9];
            stP.dFluid = cmn.cmnString.stringToDouble(
                           pfeffer.math.pfefferMath.FLUID[0][2] );
            break;
        }

        stP.iPHIVsh = pfeffer.math.pfefferMath._NO;
        stP.PHIt[i] = pfeffer.math.pfefferMath.getPHI(
                          stP.depth[i],   stP.depth[i]+stP.thick[i],
                          stP.iPHIt, stP.iPHI1, stP.iPHI2,
                          stP.dGrain,     stP.dFluid,
                          stP.dPHIShale1, stP.dPHIShale2,
                          stP.iPHIVsh,    stP.Vsh[i],
                          stL );
      }
    }

    return (stP);
  }

  /** Method getPHI1st()
   * <p> This method will return the sonic porosity id for the 2nd Porosity Data
   * @param  st    = LAS File Data Structure
   * @return iPhit = the Porosity Identifier
   */

  public static int getPHI1st( lasFileDataStruct st )
  {
    int i       = 0;
    int iVal    = -1;
    int iTool[] = { -1, -1, -1, -1, -1, -1 };
    int iPhi    = -1;

    for (i=las.lasStandardTools._RHOB; i<las.lasStandardTools._DT+1; i++)
    {
      if (st.checkData(i))
      {
        iVal        = i - las.lasStandardTools._RHOB;
        iTool[iVal] = 1;
      }
    }

    if (iTool[5] > -1)
      iPhi = las.lasStandardTools._DT;
    else if (iTool[4] > -1)
      iPhi = las.lasStandardTools._SPHI;

    return (iPhi);
  }

  /** Method getPHI2()
   * <p> This method will calculate the PHI Value dynamically depending on the
   *     lithology of the matrix.
   * @param  stL   = LAS File Data Structure
   * @param  stT   = Texture Data Structure
   * @param  stP   = The PfEFFER Data Structure
   * @return data  = the Porosity data array
   */

  public static pfefferDataStruct getPHI2( lasFileDataStruct stL,
                                           textureListStruct stT,
                                           pfefferDataStruct stP )
  {
    int iLAS  = -1;
    int iTEXT = -1;
    int iLITH = -1;

    if ((stL != null) && (stT != null) && (stP != null))
    {
      if ((stP.iPHIt > -1) && (stP.iPHI2nd > 1))
      {
        for (int i=0; i<stP.iRows; i++)
        {
          for (int k=0; k<stL.iRows; k++)
          {
            if (stL.depths[k] == stP.depth[i])
            {
              iLAS = k;
            }
          }

          iTEXT = lith.lithology.lithologySymbolsStruct._NONE;
          for (int j=0; j<stT.iCount; j++)
          {
            if (stT.stItem[j].depthStart == stP.depth[i])
            {
              iTEXT = stT.stItem[j].id;
//              iTEXT = stT.stItem[j].iSymbol;
            }
          }

          switch (iTEXT)
          {
            // Siliciclastic Lithotypes
            case lith.lithology.lithologySymbolsStruct._GRAVEL:
            case lith.lithology.lithologySymbolsStruct._SAND:
            case lith.lithology.lithologySymbolsStruct._SANDSTONE:
            case lith.lithology.lithologySymbolsStruct._SILT:
            case lith.lithology.lithologySymbolsStruct._SHALY_SAND:
              iLITH = lith.math.lithMath._QUARTZ;
              break;
            case lith.lithology.lithologySymbolsStruct._ARKOSE:
            case lith.lithology.lithologySymbolsStruct._FELDSPAR:
              iLITH = lith.math.lithMath._FELDSPAR;
              break;
            // Evaporites
            case lith.lithology.lithologySymbolsStruct._SALT_GENERAL:
            case lith.lithology.lithologySymbolsStruct._HALITE:
              iLITH = lith.math.lithMath._HALITE;
              break;
            // Organic-Rich Rocks
            case lith.lithology.lithologySymbolsStruct._COAL:
              iLITH = lith.math.lithMath._COAL;
              break;
            case lith.lithology.lithologySymbolsStruct._CLAY:
            case lith.lithology.lithologySymbolsStruct._SHALE:
              iLITH = lith.math.lithMath._QUARTZ;
              break;
            // Carbonate Lithotypes
            case lith.lithology.lithologySymbolsStruct._LIMESTONE:
            case lith.lithology.lithologySymbolsStruct._SHALY_CARB:
              iLITH = lith.math.lithMath._CALCITE;
              break;
            case lith.lithology.lithologySymbolsStruct._DOLOMITE:
            case lith.lithology.lithologySymbolsStruct._SHALY_DOLO:
            case lith.lithology.lithologySymbolsStruct._IRONSTONE:
              iLITH = lith.math.lithMath._DOLOMITE;
              break;
            // Evaporites
            case lith.lithology.lithologySymbolsStruct._GYPSUM:
              iLITH = lith.math.lithMath._GYPSUM;
              break;
            case lith.lithology.lithologySymbolsStruct._ANHYDRITE:
              iLITH = lith.math.lithMath._ANHYDRITE;
              break;
            case lith.lithology.lithologySymbolsStruct._NONE:
            default:
              iLITH = lith.math.lithMath._CALCITE;
              break;
          }

          stP.dtGrain = lith.math.lithMath._MINERAL[iLITH][9];
          stP.dtFluid = cmn.cmnString.stringToDouble(
                              pfeffer.math.pfefferMath.FLUID[0][2] );

          stP.iPHI2ndVsh   = pfeffer.math.pfefferMath._NO;
          stP.dPHI2ndShale = 0.0;
          stP.PHI1st[i]    = pfeffer.math.pfefferMath.getPHI(
                               stP.depth[i],     stP.depth[i]+stP.thick[i],
                               pfeffer.math.pfefferMath._SPHI, stP.iPHI2nd, -1,
                               stP.dtGrain,      stP.dtFluid,
                               stP.dPHI2ndShale, 0.0,
                               stP.iPHI2ndVsh,   stP.Vsh[i],
                               stL );
          stP.PHI2nd[i]    = stP.PHIt[i] - stP.PHI1st[i];

          if (stP.PHI2nd[i] < 0.0) stP.PHI2nd[i] = 0.0;
        }
      }
    }

    return (stP);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = pfeffer data list structure
   * @return st = sorted pfeffer data list structure
   */

  public static pfefferDataListStruct bubbleSort( pfefferDataListStruct st )
  {
    boolean swappedOnPrevRun  = true;
    pfefferDataStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].depthStart > st.stItem[i+1].depthStart)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method print()
   * <p> This method will print the contents of the PfEFFER Data Structure
   * @param st = the PfEFFER Data Structure
   */

  public static void print( pfefferDataStruct st )
  {
    if (st != null)
    {
	  System.out.println(st.sKEY+" "+st.sZone+" "+st.depthStart+" "+st.depthEnd);

	  if (st.iRt != pfeffer.pfefferDataStruct._NONE)
	    System.out.println("OHM="+las.lasStandardTools.LAS_TOOLS[st.iRt]);

      if (st.iVsh != pfeffer.pfefferDataStruct._NONE)
      {
  	    System.out.println("Vsh="+las.lasStandardTools.LAS_TOOLS[st.iVsh]+
  	                       "Clean="+st.dClean+" Shale="+st.dShale);
	  }

	  if (st.iPHIt != pfeffer.pfefferDataStruct._NONE)
	  {
		if (st.iPHIVsh == pfeffer.pfefferDataStruct._NONE)
		  System.out.println("PHI="+pfeffer.math.pfefferMath.PHI_TYPE[st.iPHIt] +
		                     " Grain="+st.dGrain+" Fluid="+st.dFluid);
		else
		  System.out.println("PHI="+pfeffer.math.pfefferMath.PHI_TYPE[st.iPHIt] +
		                     " Grain="+st.dGrain+" Fluid="+st.dFluid+
		                     " 1sh Vsh="+st.dPHIShale1+" 2nd Vsh="+ st.dPHIShale2);
	  }

	  if (st.iPHI2nd != pfeffer.pfefferDataStruct._NONE)
	  {
		if (st.iPHI2ndVsh == pfeffer.pfefferDataStruct._NONE)
		  System.out.println("2nd PHI="+las.lasStandardTools.LAS_TOOLS[st.iPHI2nd]+
		                     " Grain="+st.dtGrain+" Fluid="+st.dtFluid);
		else
		  System.out.println("2nd PHI="+las.lasStandardTools.LAS_TOOLS[st.iPHI2nd]+
		                     " Grain="+st.dtGrain+" Fluid="+st.dtFluid+" Vsh="+st.dPHI2ndShale);
	  }

      System.out.println( "A="      + st.A   +
                          " M="     + st.M   +
                          " N="     + st.N   +
                          " Rw="    + st.Rw  +
                          " Rsh="   + st.Rsh +
                          " Phish=" + st.Phish);
      System.out.println( "Water Model = " + st.sWaterModel);

      if (st.iRows > 0)
      {
        System.out.println(
            "Depth, THK, RT, PHI, RWA, RO, MA, SW, BVW, VSH, PAY - PHI1, PHI2" );

        for (int i=0; i<st.iRows; i++)
        {
          System.out.println( st.depth[i] + ", " + st.thick[i]  + ", " +
                              st.Rt[i]    + ", " + st.PHIt[i]   + ", " +
                              st.Rwa[i]   + ", " + st.Ro[i]     + ", " +
                              st.Ma[i]    + ", " +  st.Sw[i]    + ", " +
                              st.BVW[i]   + ", " +  st.Vsh[i]   + ", " +
                              st.Pay[i]   + " - " +
                              st.PHI1st[i]   + ", " +  st.PHI2nd[i] );
        }
      }
    }
  }
}