/*
 * @pfefferDataStruct.java Version 1.1 01/07/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 *
 * Defines "structure" for parameters controlling water saturation computations.
 *   Archie equation uses A, M, N, Rw
 *   Simandoux uses A, M, N, Rw, Rsh, Phish?
 *   Dual water uses A, M, N, Rw, Rklkjlkjlkj?
 *
 * @original version 1.00 09 Jan 2000
 * @author Geoff Bohling
 */

package pfeffer;

public class pfefferDataStruct
{
  // PfEFFER Flow Unit/Zone Identifier

  public String sKEY       = "0"; // Unique KEY based on Data and Time

  public String sZone      = "";  // Name of the Zone that is being analyzed.
  public double depthStart = 0.0; // Starting Depth (Feet)
  public double depthEnd   = 0.0; // Ending Depth (Feet)

  // PfEFFER Identifiers & Constants

  // -- No Value is selected

  public static final int _NONE  = -1;

  // -- Resistivity Track Variables

  public int    iRt = _NONE;  // Log curve Selected

  // -- Vsh Track Variables

  public int    iVsh   = _NONE; // Log Curve used to compute Vsh
  public double dClean = 0.0;   // Clean Value
  public double dShale = 0.0;   // Shale Value

  // -- Porosity Track Variables

  public int    iPHIt = _NONE;  // Type of Porosity
  public int    iPHI1 = _NONE;  // 1st Porosity Curve
  public int    iPHI2 = _NONE;  // 2nd Porosity Curve

  // -- Lithology Symbol

  public int    iLithology = lith.lithology.lithologySymbolsStruct._NONE;

  // -- Used with Bulk Density to compute porosity

  public double dGrain = 0.0;  // Grain
  public double dFluid = 0.0;  // Fluid

  // -- Used with porosity to compute Vsh Effect

  public int    iPHIVsh    = _NONE;
  public double dPHIShale1 = 0.0; // Log Curve Value for Shale
  public double dPHIShale2 = 0.0; // Log Curve Value for Shale - 2nd Curve

  // -- 2nd Porosity Curve ID

  public int    iPHI2nd = _NONE;  // Primary Porosity Curve

  // -- Used with 2nd Porosity to compute porosity

  public double dtGrain = 0.0;  // Grain
  public double dtFluid = 0.0;  // Fluid

  // -- Used with Primary Porosity to compute Vsh Effect

  public int    iPHI2ndVsh   = _NONE;
  public double dPHI2ndShale = 0.0; // Log Curve Value for Shale

  // Water Model Used

  public int    iWaterModel = 0;   // Water Model Identifier
  public String sWaterModel = "Archie";  // Water Model Used
                                   //   Default = NONE
                                   //   Archie
                                   //   Simandoux
                                   //   Dual Water

  // Archie Equation Parameters

  public double A     = 1.0;  // Archie multiplication factor
  public double M     = 2.0;  // Cementation factor
  public double Mp    = 0.0;  // Modify Cementation factor (M) by adding Ma*PHI
  public double N     = 2.0;  // Saturation exponent
  public double Rw    = 0.01; // Formation water resistivity
  public double Rsh   = 0.0;  // Shale resistivity
  public double Phish = 0.0;  // Shale porosity

  // Moveable Hydrocarbon Parameters

  public int    iRxo = _NONE;  // Rxo Log curve Selected

  public static final int FAHRENHEIT = 0;
  public static final int CELSIUS    = 1;

  public double Rmf    = 0.0; // Mud Filtrate Resistivity (Rmf)
  public double Rmft   = 0.0; // Mud Filtrate Resistivity Temperature (Rmft)
  public double ST     = 0.0; // Surface Temperature (ST)
  public double BHT    = 0.0; // Bore Hole Temperature (BHT)
  public double TD     = 0.0; // Total Depth (TD)

  public int    iTemp = FAHRENHEIT; // Temperature Unit

  // Cut-Offs

  public double dPhiCut = 0.0; // Porosity Cut Off
  public double dSwCut  = 1.0; // Water Saturation Cut Off
  public double dVshCut = 1.0; // Fractional Shale Cut Off
  public double dBvwCut = 1.0; // Bulk Volume Water Cut Off

  // Wyllie-Rose Equation Parameters

  public double P = 8581.0;
  public double Q =    4.4;
  public double R =    2.0;

  public int iRows = 0;

  // Log Values

  public double depth[]  = null; // Depth
  public double thick[]  = null; // Thickness

  public double Rt[]     = null; // True Resistivity
  public double Vsh[]    = null; // V-Shale
  public double PHIt[]   = null; // True Porosity

  // 2nd Porosity

  public double PHI1st[] = null; // 1st Porosity (Sonic)
  public double PHI2nd[] = null; // 2nd Porosity (Total - Sonic)

  // Computed Variables

  public double Rtc[]    = null; // Corrected resistivity (same as input Rt for Archie)
  public double Rwa[]    = null; // Apparent formation water resistivity (for Sw=100%)
  public double Ro[]     = null; // Resistivity at 100% saturation
  public double Ma[]     = null; // Apparent cementation exponent (for Sw=100%)
  public double Mo[]     = null; // Computed cementation exponent Mo = Ma*PHI+M
  public double Sw[]     = null; // Water saturation itself
  public double BVW[]    = null; // Bulk volume water (Sw*porosity)

  public double Pay[]    = null; // Pay

  public double PHIr[]   = null; // Electrically Connected Porosity

  public double Rxo[]    = null; // Filtrate Resistivity (Rxo)
  public double Sxo[]    = null; // Saturation of total moveable fluid
  public double BVF[]    = null; // Bulk volume (moveable) fluid (Sxo*porosity)

  // Volumetrics Data

  public double dThickness   = 0.0; // Columns as Thickness
  public double dHydroCarbon = 0.0; // Oil or Gas Feet
  public double dPay         = 0.0; // Amount of Pay in Feet
  public double dPorosity    = 0.0; // Average Porosity
  public double dSaturation  = 0.0; // Average Water Saturation

  /** Method delete()
   * <p> This method will set the string variable to null and initialize
   *     all the other variables.
   */

  public void delete()
  {
    sKEY        = null;  // Unique KEY based on Data and Time
    sZone       = null;  // Name of the Zone that is being analyzed.
    sWaterModel = null;  // Water Model Used

    // Log Values

    depth  = null; // Depth
    thick  = null; // Thickness

    Rt     = null; // True Resistivity
    Vsh    = null; // V-Shale
    PHIt   = null; // True Porosity

    // 2nd Porosity

    PHI1st = null; // 1st Porosity (Sonic)
    PHI2nd = null; // 2nd Porosity

    // Computed Variables


    Rtc    = null; // Corrected resistivity (same as input Rt for Archie)
    Rwa    = null; // Apparent formation water resistivity (for Sw=100%)
    Ro     = null; // Resistivity at 100% saturation
    Ma     = null; // Apparent cementation exponent (for Sw=100%)
    Mo     = null; // Computed cementation exponent Mo = Ma*PHI+M
    Sw     = null; // Water saturation itself
    BVW    = null; // Bulk volume water (Sw*porosity)

    Pay    = null; // Pay

    PHIr   = null; // Electrically Connected Porosity

    Rxo    = null; // Filtrate Resistivity (Rxo)
    Sxo    = null; // Saturation of total moveable fluid
    BVF    = null; // Bulk volume (moveable) fluid (Sxo*porosity)
  }

  /** Method getData(int iCurve)
   * <p> This method will retrieve the curve data requested
   * @param  iCurve = the requested curve type
   * @return data   = the data requested.
   */

  public double[] getData(int iCurve)
  {
    double data[] = null;
    int    i=0;

    if (iRows > 0)
    {
      data = new double[iRows];

      switch (iCurve)
      {
        case pfeffer.pfefferStandardTools._DEPTH:
          data = depth;
          break;
        case pfeffer.pfefferStandardTools._THK:
          data = thick;
          break;
        case pfeffer.pfefferStandardTools._RT:
          data = Rt;
          break;
        case pfeffer.pfefferStandardTools._PHIT:
          data = PHIt;
          break;
        case pfeffer.pfefferStandardTools._VSH:
          data = Vsh;
          break;

        case pfeffer.pfefferStandardTools._PHI1:
          data = PHI1st;
          break;
        case pfeffer.pfefferStandardTools._PHI2:
          data = PHI2nd;
          break;

        case pfeffer.pfefferStandardTools._RWA:
          data = Rwa;
          break;
        case pfeffer.pfefferStandardTools._RO:
          data = Ro;
          break;
        case pfeffer.pfefferStandardTools._M_:
          data = Mo;
          break;
        case pfeffer.pfefferStandardTools._MA:
          data = Ma;
          break;
        case pfeffer.pfefferStandardTools._SW:
          data = Sw;
          break;
        case pfeffer.pfefferStandardTools._BVW:
          data = BVW;
          break;
        case pfeffer.pfefferStandardTools._PAY:
          data = Pay;
          break;

        case pfeffer.pfefferStandardTools._PHIR:
          data = PHIr;
          break;

        case pfeffer.pfefferStandardTools._RXO:
          data = Rxo;
          break;
        case pfeffer.pfefferStandardTools._SXO:
          data = Sxo;
          break;
        case pfeffer.pfefferStandardTools._BVF:
          data = BVF;

        case pfeffer.pfefferStandardTools._PHIE:
          break;
        case pfeffer.pfefferStandardTools._CBW:
          break;
        case pfeffer.pfefferStandardTools._BVH:
          break;
      }
    }

    return (data);
  }

  /** Method getData( int iCurve, int iRow )
   * <p> This method will retrieve the curve data requested
   * @param  iCurve = the requested curve type
   * @param  iRow   = the row number
   * @return data   = the data requested.
   */

  public double getData( int iCurve, int iRow )
  {
    double data = 0.0;

    if (iRows > iRow)
    {
      switch (iCurve)
      {
        case pfeffer.pfefferStandardTools._DEPTH:
          data = depth[iRow];
          break;
        case pfeffer.pfefferStandardTools._THK:
          data = thick[iRow];
          break;
        case pfeffer.pfefferStandardTools._RT:
          data = Rt[iRow];
          break;
        case pfeffer.pfefferStandardTools._PHIT:
          data = PHIt[iRow];
          break;
        case pfeffer.pfefferStandardTools._VSH:
          data = Vsh[iRow];
          break;

        case pfeffer.pfefferStandardTools._PHI1:
          data = PHI1st[iRow];
          break;
        case pfeffer.pfefferStandardTools._PHI2:
          data = PHI2nd[iRow];
          break;

        case pfeffer.pfefferStandardTools._RWA:
          data = Rwa[iRow];
          break;
        case pfeffer.pfefferStandardTools._RO:
          data = Ro[iRow];
          break;
        case pfeffer.pfefferStandardTools._M_:
          data = Mo[iRow];
          break;
        case pfeffer.pfefferStandardTools._MA:
          data = Ma[iRow];
          break;
        case pfeffer.pfefferStandardTools._SW:
          data = Sw[iRow];
          break;
        case pfeffer.pfefferStandardTools._BVW:
          data = BVW[iRow];
          break;
        case pfeffer.pfefferStandardTools._PAY:
          data = Pay[iRow];
          break;

        case pfeffer.pfefferStandardTools._PHIR:
          data = PHIr[iRow];
          break;

        case pfeffer.pfefferStandardTools._RXO:
          data = Rxo[iRow];
          break;
        case pfeffer.pfefferStandardTools._SXO:
          data = Sxo[iRow];
          break;
        case pfeffer.pfefferStandardTools._BVF:
          data = BVF[iRow];

        case pfeffer.pfefferStandardTools._PHIE:
          break;
        case pfeffer.pfefferStandardTools._CBW:
          break;
        case pfeffer.pfefferStandardTools._BVH:
          break;
      }
    }

    return (data);
  }
}
