/*
 * @(#)pfefferSwModels.java Version 1.0 01/09/01
 *
 * Copyright (c) 2001 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, USA
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of the
 * Kansas Geological Survey.  ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with the Kansas Geological Survey.
 *
 * Last change:  GCB  24 Jan 2001    2:51 pm
 * Last change:  JRV  07 Jan 2010
 */

package pfeffer.math;

import pfeffer.pfefferStruct;

/**
 * Implements water saturation models (Archie, Simandoux, Dual Water).
 *
 * @version 1.00 08 Jan 2000
 * @author Geoff Bohling
 */

public class pfefferSwModels
{
  public static final int _ARCHIE     = 0;
  public static final int _SIMANDOUX  = 1;
  public static final int _DUAL_WATER = 2;

  public static final String WATER_MODEL[] = { "Archie",
                                               "Simandoux",
                                               "Dual Water" };

  /** Method pfefferArchie()
   * Computes results for Archie water saturation model.
   * @param st  = structure containing saturation model
   *              parameters (uses only A,M,N,Rw).
   * @param Rt   = true resistivity
   * @param Phi  = true porosity
   * @returns st = structure containing saturation-related output values.
   * @see "Doveton, J. H., 1994, Geologic Log Analysis Using Computer
   *       Methods, AAPG."
   * @see "Archie, G. E., 1942, The electrical resistivity log as an aid in
   *       determining some reservoir characteristics: Transactions of the
   *       AIME, v. 146, no. 1, p. 54-62."
   */

  public static pfefferStruct pfefferArchie( pfefferStruct st,
                                             double Rt,
					                         double Phi )
  {
    double A  = st.A;
    double M  = st.M;
    double N  = st.N;
    double Rw = st.Rw;

    if (Phi == 0.0) Phi = .01;
    if (Rt  == 0.0) Rt  = .01;

    double F = A / Math.pow(Phi,M);	   // Formation factor

    st.Rtc = Rt;  // "Corrected" resistivity same as input for Archie
    st.Rwa = Rt / F;
    st.Ro  = Rw * F;
    st.Ma  = Math.log( A*Rw / Rt ) / Math.log( Phi );
    st.Sw  = Math.pow( F*Rw/Rt, 1.0/N );
    st.BVW = st.Sw*Phi;

    return( st );
  }

  /** Method pfefferSimandoux
   * Computes results for Simandoux water saturation model.
   * @param st   = structure containing saturation model parameters.
   * @param Rt   = true resistivity
   * @param Phi  = true porosity
   * @param Vsh  = V-Shale
   * @returns st = structure containing saturation-related output values.
   * @see "Aguilera, R., 1990, Extensions of picket plots for the analysis of
   *       shaly formations by well logs, The Log Analyst, September-October
   *       1990, p. 304-313."
   * @see "Simandoux, P., 1963, Mesures dialectriques en milieu poreux . . .,
   *       Revue de l'Institute Francais du Petrole, v. 18, supplementary issue,
   *       p. 193-215."
   */

  public static pfefferStruct pfefferSimandoux( pfefferStruct st,
                                                double Rt,
                                                double Phi,
                                                double Vsh )
  {
    double A   = st.A;
    double M   = st.M;
    double N   = st.N;
    double Rw  = st.Rw;
    double Rsh = st.Rsh;

    if (Phi == 0.0) Phi = .01;
    if (Rt  == 0.0) Rt  = .01;

    double F = A / Math.pow(Phi,M); // Formation factor
    double Ro = F*Rw;               // (Corrected) resistivity at Sw=100%

    // Coefficients in polynomial equation for Sw ( a*Sw^N + b*Sw + c = 0 ):

    double ao = 1.0/Ro;
    double bo = Vsh/Rsh;
    double co = -1.0/Rt;

    // Initial estimate assuming N = 2.0 (quadratic equation):

    double Sw = ( -bo + Math.sqrt(bo*bo - 4.0*ao*co) ) / (2.0*ao);
    if ( Sw < 0.0 ) Sw = 2.0;

    // Iterative solution (Newton's method) for root of polynomial:

    double SwDelta = 0.0;
    for (int i=0; i<100; i++)
    {
      if ( Sw < 0.0 ) Sw = 2.0;

      SwDelta = ( ao*Math.pow(Sw,N) + bo*Sw + co ) /
                ( N*ao*Math.pow(Sw,(N-1.0)) + bo );
      Sw = Sw - SwDelta;

      if ( Math.abs(SwDelta) < 0.001*Math.abs(Sw) )	break;
    }

    double Rtc = Ro / Math.pow( Sw, N );    // Corrected resistivity

    st.Rtc = Rtc;
    st.Rwa = Rtc / F;
    st.Ro  = Ro;
    st.Ma  = Math.log( A*Rw / Rtc ) / Math.log( Phi );
    st.Sw  = Sw;
    st.BVW = st.Sw*Phi;

    return( st );
  }

  /** pfefferDualWater
   * Computes results for Dual Water water saturation model.
   * @param st   = structure containing saturation model parameters.
   * @param Rt   = true resistivity
   * @param Phi  = true porosity
   * @param Vsh  = V-Shale
   * @returns st = structure containing saturation-related output values.
   * @see "Aguilera, R., 1990, Extensions of picket plots for the analysis of
   *       shaly formations by well logs, The Log Analyst, September-October
   *       1990, p. 304-313."
   * @see "Clavier, C., Coates, G., and Dumanoir, J., 1984, The theoretical and
   *       experimental bases for the "Dual Water" model for interpretation of
   *       shaly sands, Society of Petroleum Engineers Journal, v. 24,
   *       p. 153-169."
   */

  public static pfefferStruct pfefferDualWater( pfefferStruct st,
                                                double Rt,
                                                double Phi,
                                                double Vsh )
  {
    double A     = st.A;
    double M     = st.M;
    double N     = st.N;
    double Rw    = st.Rw;
    double Rsh   = st.Rsh;
    double Phish = st.Phish;

    if (Phi == 0.0) Phi = .01;
    if (Rt  == 0.0) Rt  = .01;

    double F  = A / Math.pow(Phi,M);   // Formation factor
    double Ro = F*Rw;                  // (Corrected) resistivity at Sw=100%

    double Rb    = Rsh*Phish*Phish;    // Bound water resistivity
    double Phit  = Phi + Vsh*Phish;    // Total porosity
    double Swb   = Vsh*Phish/Phit;     // Bound water saturation

    // Coefficients in polynomial equation for total saturation, Swt
    //          ( a*Swt^N + b*Swt + c = 0 ):

    double ao = 1.0;
    double bo = -Swb*(1.0 - A*Rw/Rb);
    double co = -A*Rw/(Rt*Math.pow(Phit,M));

    // Initial estimate assuming N = 2.0 (quadratic equation):

    double Swt = ( -bo + Math.sqrt(bo*bo - 4.0*ao*co) ) / (2.0*ao);
    if ( Swt < 0.0 ) Swt = 2.0;

    // Iterative solution (Newton's method) for root of polynomial:

    double SwtDelta = 0.0;
    for (int i=0; i<100; i++)
    {
      if ( Swt < 0.0 ) Swt = 2.0;

      SwtDelta = ( ao*Math.pow(Swt,N) + bo*Swt + co )/
                 ( N*ao*Math.pow(Swt,(N-1.0)) + bo );

      Swt = Swt - SwtDelta;

      if ( Math.abs(SwtDelta) < 0.001*Math.abs(Swt) ) break;
    }

    double Sw = ( Swt - Swb ) / ( 1.0 - Swb );  // Effective saturation
    double Rtc = Ro / Math.pow( Sw, N );        // Corrected resistivity

    st.Rtc = Rtc;
    st.Rwa = Rtc / F;
    st.Ro  = Ro;
    st.Ma  = Math.log( A*Rw / Rtc ) / Math.log( Phi );
    st.Sw  = Sw;
    st.BVW = st.Sw*Phi;

    return( st );
  }

  /** Method pfefferArchieSxo()
   * Computes results for Archie water saturation model.
   * @param st  = structure containing saturation model
   *              parameters (uses only A,M,N,Rmf).
   * @param Rxo  = medium resistivity
   * @param Phi  = true porosity
   * @returns st = structure containing saturation-related output values.
   * @see "Doveton, J. H., 1994, Geologic Log Analysis Using Computer
   *       Methods, AAPG."
   * @see "Archie, G. E., 1942, The electrical resistivity log as an aid in
   *       determining some reservoir characteristics: Transactions of the
   *       AIME, v. 146, no. 1, p. 54-62."
   */

  public static pfefferStruct pfefferArchieSxo( pfefferStruct st,
                                                double Rxo,
					                            double Phi )
  {
    double A     = st.A;
    double M     = st.M;
    double N     = st.N;

    double depth = st.depth;
    double Rmf   = st.Rmf;
    double Rmft  = st.Rmft;
    double ST    = st.ST;
    double BHT   = st.BHT;
    double TD    = st.TD;
    int    iTemp = st.iTemp;

    double F     = 0.0;
    double Rmfc  = 0.0;
    double T     = 0.0;

    if (Phi == 0.0) Phi = .01;
    if (Rxo == 0.0) Rxo = .01;

    F = A / Math.pow(Phi,M);	   // Formation factor
    T = ST + depth * (BHT - ST ) / TD;

    switch (iTemp)
    {
      case pfeffer.pfefferDataStruct.FAHRENHEIT:
        Rmfc = Rmf * (Rmft + 6.77) / (T + 6.77);
        break;
      case pfeffer.pfefferDataStruct.CELSIUS:
        Rmfc = Rmf * (Rmft + 21.5) / (T + 21.5);
        break;
    }

    st.Sxo = Math.pow( F*Rmfc/Rxo, 1.0/N );
    st.BVF = st.Sxo*Phi;

    return( st );
  }

}

