/*
 * @pfefferMath.java Version 1.1 01/05/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package pfeffer.math;

import las.lasFileDataStruct;

import pfeffer.pfefferStruct;
import pfeffer.pfefferDataStruct;

/** Class pfefferMath
 *  <p> This Class will perform the math for the PfEFFER Data Tables
 *
 *  @version 1.1 01/05/2010
 *  @author  John R. Victorine
 */

public class pfefferMath
{
  public static final int _NONE   = -1; // No Type selected

  // Porosity Types

  public static final int _RHOB   = 0;  // Porosity from Bulk Density
  public static final int _NPHI   = 1;  // Porosity from Neutron Porosity
  public static final int _PHIAvg = 2;  // Average of Neutron & Density Porosity
  public static final int _SPHI   = 3;  // Porosity from Sonic

  public static final String PHI_TYPE[] = { "RHOB", "NPHI", "AVERAGE", "SPHI" };
  public static final String UNITS[]    = { "[gm/cc]", "[gm/cc]", "[gm/cc]", "[usec/ft]" };

  // Compute Indicator

  public static final int _NO     = 0;
  public static final int _YES    = 1;

  // Lithology Values

  public static final int LITHOLOGY[] =
  {
    lith.lithology.lithologySymbolsStruct._SANDSTONE,
    lith.lithology.lithologySymbolsStruct._LIMESTONE,
    lith.lithology.lithologySymbolsStruct._DOLOMITE
  };

  // Column Values

  public static final int _LITHOLOGY = 0;
  public static final int _DENSITY   = 1;
  public static final int _SONIC     = 2;

  // Grain Matrix Values

  public static final int _QUARTZ   = 0;
  public static final int _CALCITE  = 1;
  public static final int _DOLOMITE = 2;

  public static final String MATRIX[][] =
  { { "Quartz",   "2.65", "55.5"},
    { "Calcite",  "2.71", "47.5"},
    { "Dolomite", "2.87", "43.5"} };

  // Fluid Values

  public static final int _FRESH    = 0;
  public static final int _SALT     = 1;
  public static final int _OIL      = 2;

  public static final String FLUID[][] =
  { { "Fresh Water", "1.0",   "189.0"},
    { "Salty Water", "1.1",   "185.0"},
    { "Oil Mud",     "0.985", "204.5"} };

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getRowCount()
   * <P> This method will compute the number of rows
   * @param  depthStart = Starting Depth
   * @param  depthEnd   = Ending Depth
   * @param  depthIncr  = Increment Depth
   * @return iRows      = The number of rows of data
   */

  public static int getRowCount( double depthStart,
                                 double depthEnd,
                                 double depthIncr )
  {
    int iRows = 0;

    if ((depthEnd > 0.0) && (depthIncr > 0.0))
      iRows  = (int) ( ( depthEnd - depthStart ) / depthIncr );

    return (iRows);
  }

  /** Method getDepth()
   * <p> This method will return the array of depth values for a specific
   *     depth range
   * @param  depthStart = Starting Depth
   * @param  depthEnd   = Ending Depth
   * @param  depthIncr  = Increment Depth
   * @param  st         = LAS File Data Structure
   * @return depth      = array of depth values
   */

  public static double[] getDepth( double depthStart,
                                   double depthEnd,
                                   double depthIncr,
                                   lasFileDataStruct st )
  {
    int    i        = 0;
    int    iRows    = 0;
    int    iCount   = 0;
    double depth[]  = null;

    if ((depthEnd > 0.0) && (depthIncr > 0.0) && (st != null))
    {
      iRows = getRowCount( depthStart, depthEnd, depthIncr );
      if (iRows > 0)
      {
        depth  = new double[iRows];

        for (i=0; i<iRows; i++)
          depth[i] = 0.0;

        for (i=0; i<st.iRows; i++)
        {
          if ((st.depths[i] >= depthStart) && (st.depths[i] < depthEnd))
          {
            if (iCount < iRows)
            {
              depth[iCount]  = st.depths[i];
              iCount++;
            }
          }
        }
      }
    }

    return (depth);
  }

  /** Method getThickness()
   * <p> This method will return the array of depth thickness values for a
   *     specific depth range
   * @param  depthStart = Starting Depth
   * @param  depthEnd   = Ending Depth
   * @param  depthIncr  = Increment Depth
   * @return dThick     = array of thickness values
   */

  public static double[] getThickness( double depthStart,
                                       double depthEnd,
                                       double depthIncr )
  {
    int    i       = 0;
    int    iRows   = 0;
    double dThick[] = null;

    if ((depthEnd > 0.0) && (depthIncr > 0.0))
    {
      iRows = getRowCount( depthStart, depthEnd, depthIncr );

      if (iRows > 0)
      {
        dThick  = new double[iRows];

        for (i=0; i<iRows; i++)
          dThick[i] = 0.0;

        for (i=0; i<iRows; i++)
        {
          dThick[i] = depthIncr;
        }
      }
    }

    return (dThick);
  }

  /** Method getRt()
   * <p> This method will return the array of true resistivity values for a
   *     specific depth range
   * @param  depthStart = Starting Depth
   * @param  depthEnd   = Ending Depth
   * @param  depthIncr  = Increment Depth
   * @param  iRt        = Resistivity Identifier
   * @param  st         = LAS File Data Structure
   * @return data       = array of true resistivity values
   */

  public static double[] getRt( double depthStart,
                                double depthEnd,
                                double depthIncr,
                                int    iRt,
                                lasFileDataStruct st )
  {
    int    i        = 0;
    int    iRows    = 0;
    int    iCount   = 0;
    double dtmp[]  = null;
    double data[]  = null;

    if ((depthEnd > 0.0) && (depthIncr > 0.0) && (st != null) && (iRt != -1))
    {
      dtmp  = st.getData( iRt );
      iRows = getRowCount( depthStart, depthEnd, depthIncr );

      if (iRows > 0)
      {
        data  = new double[iRows];

        for (i=0; i<iRows; i++)
          data[i] = 0.0;

        for (i=0; i<st.iRows; i++)
        {
          if ((st.depths[i] >= depthStart) && (st.depths[i] < depthEnd))
          {
            if (iCount < iRows)
            {
              data[iCount]   = dtmp[i];
              iCount++;
            }
          }
        }
      }
    }

    return (data);
  }

  /** Method getVsh()
   * <p> This method will return the Vsh Data Column
   * @param  depthStart = Starting Depth
   * @param  depthEnd   = Ending Depth
   * @param  depthIncr  = Increment Depth
   * @param  iVsh       = Vsh Correction Type Indicator
   * @param  dClean     = Vsh Clean Value
   * @param  dShale     = Vsh Shale Value
   * @param  st         = LAS File Data Structure
   * @return data       = the Vsh Data Column
   */

  public static double[] getVsh( double depthStart,
                                 double depthEnd,
                                 double depthIncr,
                                 int    iVsh,
                                 double dClean,
                                 double dShale,
                                 lasFileDataStruct st )
  {
    int    i        = 0;
    int    iRows    = 0;
    int    iCount   = 0;
    int    itmp     = 0;
    double dtmp[]  = null;
    double data[]  = null;

    if ((depthEnd > 0.0) && (depthIncr > 0.0))
    {
      iRows = getRowCount( depthStart, depthEnd, depthIncr );

      data  = new double[iRows];
      for (i=0; i<iRows; i++)
        data[i] = 0.0;
	}

    if ((st != null) && (iVsh != -1))
    {
      dtmp  = st.getData( iVsh );

      if (iRows > 0)
      {
        for (i=0; i<st.iRows; i++)
        {
          if ((st.depths[i] >= depthStart) && (st.depths[i] < depthEnd))
          {
            if (iCount < iRows)
            {
              itmp = (int) (1000.0 * (dtmp[i] - dClean) / (dShale - dClean));
              data[iCount] = (double) itmp / 1000.0;
              iCount++;
            }
          }
        }
      }
    }

    return (data);
  }

  /** Method getPHIt()
   * <p> This method will return the total porosity from the LAS File Curves
   * @param  depthStart = Starting Depth
   * @param  depthEnd   = Ending Depth
   * @param  depthIncr  = Increment Depth
   * @param  iPHIt      = the Porosity Type identifier
   * @param  iPHI1      = the Porosity Log Identification
   * @param  iPHI2      = the Porosity Log Identification
   * @param  dMatrix    = Matrix Formation Value
   * @param  dFluid     = Fluid Value
   * @param  dShale     = Shale Value
   * @param  dShale2    = 2nd Curve Shale Value
   * @param  iVsh       = Vsh Correction Indicator for Porosity
   * @param  dVsh       = Array of Vsh Values
   * @param  st         = LAS File Data Structure
   * @return data       = the total porosity array
   */

  public static double[] getPHIt( double depthStart,
                                  double depthEnd,
                                  double depthIncr,
                                  int    iPHIt,
                                  int    iPHI1,
                                  int    iPHI2,
                                  double dMatrix,
                                  double dFluid,
                                  double dShale,
                                  double dShale2,
                                  int    iVsh,
                                  double dVsh[],
                                  lasFileDataStruct st )
  {
    int    i        = 0;
    int    iRows    = 0;
    int    iCount   = 0;
    int    itmp     = 0;
    int    iSwitch  = iPHIt;
    double dPHI    = 0.0;
    double dtmp[]  = null;
    double dtmp2[] = null;
    double data[]  = null;

//    if ((depthEnd > 0.0) && (depthIncr > 0.0) && (st != null) && (iVsh != -1))
    if ((depthEnd > 0.0) && (depthIncr > 0.0) && (st != null))
    {
      iRows = getRowCount( depthStart, depthEnd, depthIncr );

      switch (iPHIt)
      {
        case _RHOB:    // Porosity from Bulk Density
        case _NPHI:    // Porosity from Neutron Porosity
        case _SPHI:    // Porosity from Sonic
          dtmp = st.getData( iPHI1 );
          if ((iPHI1 == las.lasStandardTools._DPHI) ||
              (iPHI1 == las.lasStandardTools._SPHI))
              iSwitch = _NPHI;
          break;
        case _PHIAvg:  // Average of Neutron & Density Porosity
          dtmp  = st.getData( iPHI1 );
          dtmp2 = st.getData( iPHI2 );
          if (iPHI2 == las.lasStandardTools._RHOB)
          {
            dShale2  = computePHI(dShale2,  dMatrix, dFluid);
            for (i=0; i<st.iRows; i++)
            {
              dtmp2[i] = computePHI(dtmp2[i], dMatrix, dFluid);
            }
          }
          break;
      }

      if (iRows > 0)
      {
        data  = new double[iRows];

        for (i=0; i<iRows; i++)
          data[i] = 0.0;

        for (i=0; i<st.iRows; i++)
        {
          if ((st.depths[i] >= depthStart) && (st.depths[i] < depthEnd))
          {
            if (iCount < iRows)
            {
              switch (iSwitch)
              {
                case _RHOB:    // Porosity from Bulk Density
                case _SPHI:    // Porosity from Sonic
                  dPHI = computePHI(
                         dtmp[i], dMatrix, dFluid, iVsh, dShale, dVsh[iCount]);
                  break;
                case _NPHI:    // Porosity from Neutron Porosity
                  if (iVsh == _NO)
                    dPHI = dtmp[i];
                  else
                    dPHI = dtmp[i] - dVsh[iCount] * dShale;
                  break;
                case _PHIAvg:  // Average of Neutron & Density Porosity
                  if (iVsh == _NO)
                  {
                    dPHI = Math.sqrt( (dtmp[i]*dtmp[i] + dtmp2[i]*dtmp2[i]) /2.0);
                  }
                  else
                  {
                    dPHI = ( dtmp[i] - (dShale/dShale2) * dtmp2[i] ) /
                           ( 1.0 - dShale/dShale2 );
				  }
                  break;
              }

              itmp = (int) (1000.0 * dPHI);
              data[iCount] = (double) itmp / 1000.0;
              iCount++;
            }
          }
        }
      }
    }

    return (data);
  }

  /** Method get2ndPHI()
   * <p> This method will retrieve the 2nd Porosity Column
   * @param  iRows   = The total rows
   * @param  dPHI1st = The Primary Porosity
   * @param  dPHI    = The Total Porosity
   * @return dPHI2nd = the 2nd Porosity Column
   */

  public static double[] get2ndPHI( int iRows, double dPHI1st[], double dPHI[] )
  {
    double dPHI2nd[] = null;
    int    iTemp     = 0;

    if (iRows > 0)
    {
      dPHI2nd = new double[iRows];

      for (int i=0; i<iRows; i++)
      {
        dPHI2nd[i] = dPHI[i] - dPHI1st[i];
        iTemp     = (int) (1000.0 * dPHI2nd[i]);
        dPHI2nd[i] = (double) iTemp / 1000.0;

        if (dPHI2nd[i] < 0.0) dPHI2nd[i] = 0.0;
      }
    }

    return (dPHI2nd);
  }

  /** Method getPHI()
   * <p> This method will return the total porosity from the LAS File Curves
   * @param  depthStart = Starting Depth
   * @param  depthEnd   = Ending Depth
   * @param  iPHIt      = the Porosity Type identifier
   * @param  iPHI1      = the Porosity Log Identification
   * @param  iPHI2      = the Porosity Log Identification
   * @param  dMatrix    = Matrix Formation Value
   * @param  dFluid     = Fluid Value
   * @param  dShale     = Shale Value
   * @param  dShale2    = 2nd Curve Shale Value
   * @param  iVsh       = Vsh Correction Indicator for Porosity
   * @param  dVsh       = Array of Vsh Values
   * @param  st         = LAS File Data Structure
   * @return data       = the total porosity array
   */

  public static double getPHI( double depthStart,
                               double depthEnd,
                               int    iPHIt,
                               int    iPHI1,
                               int    iPHI2,
                               double dMatrix,
                               double dFluid,
                               double dShale,
                               double dShale2,
                               int    iVsh,
                               double dVsh,
                               lasFileDataStruct st )
  {
    int    i        = 0;
    int    itmp     = 0;
    int    iSwitch  = iPHIt;
    double dPHI     = 0.0;
    double dtmp     = 0.0;
    double dtmp2    = 0.0;
    double data     = 0.0;

    if ((depthEnd > 0.0) && (st != null))
    {
      dShale2 = computePHI(dShale2, dMatrix, dFluid);
      for (i=0; i<st.iRows; i++)
      {
        if ((st.depths[i] >= depthStart) && (st.depths[i] < depthEnd))
        {
          switch (iPHIt)
          {
            case _RHOB:    // Porosity from Bulk Density
            case _NPHI:    // Porosity from Neutron Porosity
            case _SPHI:    // Porosity from Sonic
              dtmp = st.getRow( iPHI1, i );
              if ((iPHI1 == las.lasStandardTools._DPHI) ||
                  (iPHI1 == las.lasStandardTools._SPHI))
                  iSwitch = _NPHI;
              break;
            case _PHIAvg:  // Average of Neutron & Density Porosity
              dtmp  = st.getRow( iPHI1, i );
              dtmp2 = st.getRow( iPHI2, i );
              if (iPHI2 == las.lasStandardTools._RHOB)
              {
                dtmp2   = computePHI(dtmp2,   dMatrix, dFluid);
              }
              break;
          }

          switch (iSwitch)
          {
            case _RHOB:    // Porosity from Bulk Density
            case _SPHI:    // Porosity from Sonic
              dPHI = computePHI( dtmp, dMatrix, dFluid, iVsh, dShale, dVsh );
              break;
            case _NPHI:    // Porosity from Neutron Porosity
              if (iVsh == _NO)
                dPHI = dtmp;
              else
                dPHI = dtmp - dVsh * dShale;
              break;
            case _PHIAvg:  // Average of Neutron & Density Porosity
              if (iVsh == _NO)
              {
                dPHI = Math.sqrt(( dtmp*dtmp + dtmp2*dtmp2 ) / 2.0);
			  }
              else
              {
                dPHI = ( dtmp - (dShale/dShale2) * dtmp2 ) /
                       ( 1.0 - dShale/dShale2 );
			  }
              break;
          }

          itmp = (int) (1000.0 * dPHI);
          data = (double) itmp / 1000.0;
        }
      }
    }

    return (data);
  }

  /* ===================================================================== *
   * ------------------------ COMPUTE METHODS ---------------------------- *
   * ===================================================================== */

  /** Method computePHI()
   * <p> This method will compute the Porosity from Log Curve
   * @param  dLog    = Log Curve Value
   * @param  dMatrix = Matrix Value
   * @param  dFluid  = Fluid Value
   * @return dPHI    = Porosity
   */

  public static double computePHI( double dLog,  double dMatrix, double dFluid )
  {
    double dPHI = 0.0;

    dPHI = ( dMatrix - dLog ) / ( dMatrix - dFluid );

    return (dPHI);
  }

  /** Method computePHI()
   * <p> This method will compute the Porosity from Log Curve
   * @param  dLog    = Log Curve Value
   * @param  dMatrix = Matrix Value
   * @param  dFluid  = Fluid Value
   * @param  iVsh    = Make Vsh Correction
   * @param  dShale  = Shale Value
   * @param  dVsh    = Vsh Value
   * @return dPHI    = Porosity
   */

  public static double computePHI( double dLog,  double dMatrix, double dFluid,
                                   int iVsh,     double dShale,  double dVsh )
  {
    double dPHI = 0.0;

    if (iVsh == _NO)
    {
      dPHI = ( dMatrix - dLog ) / ( dMatrix - dFluid );
    }
    else
    {
      dPHI = ( dMatrix - dLog - dVsh * ( dMatrix-dShale )) / ( dMatrix-dFluid );
    }

    return (dPHI);
  }

  /** Method compute()
   * <p> This method will make pfeffer computations
   * @param  st    = the PfEFFER Data Structure
   * @param  dNull = the Null Value
   * @return st    = the PfEFFER Data Structure
   */

  public static pfefferDataStruct compute( pfefferDataStruct st, double dNull )
  {
    int i=0;
    pfefferStruct stData     = new pfefferStruct();
    double        dRt        = 0.0;
    double        dRxo       = 0.0;
    double        dPhi       = 0.0;
    double        dVsh       = 0.0;
    double        dThickness = 0.0;
    double        dThick     = 0.0;
    double        dAvgPhi    = 0.0;
    double        dAvgSw     = 0.0;
    double        dPayFeet   = 0.0;
    double        dFtOil     = 0.0;
    double        dTemp      = 0.0;
    int           iTemp      = 0;
    int           iCount     = 0;

    if (st != null)
    {
      if (st.iRows > 0)
      {
        stData.A     = st.A;     // Archie multiplication factor
        stData.M     = st.M;     // Cementation factor
        stData.N     = st.N;     // Saturation exponent
        stData.Rw    = st.Rw;    // Formation water resistivity
        stData.Rsh   = st.Rsh;   // Shale resistivity
        stData.Phish = st.Phish; // Shale porosity

        stData.Rmf   = st.Rmf;   // Mud Filtrate (Rmf)
        stData.Rmft  = st.Rmft;  // Mud Filtrate Resistivity Temperature (Rmft)
        stData.ST    = st.ST;    // Surface Temperature (ST)
        stData.BHT   = st.BHT;   // Bore Hole Temperature (BHT)
        stData.TD    = st.TD;    // Total Depth (TD)

        stData.Rtc = 0.0; // Corrected resistivity (same as input Rt for Archie)
        stData.Rwa = 0.0; // Apparent formation water resistivity (for Sw=100%)
        stData.Ro  = 0.0; // Resistivity at 100% saturation
        stData.Ma  = 0.0; // Apparent cementation exponent (for Sw=100%)
        stData.Sw  = 0.0; // Water saturation itself
        stData.BVW = 0.0; // Bulk volume water (Sw*porosity)

        stData.Sxo = 0.0; // Fluid (moveable) saturation
        stData.BVF = 0.0; // Bulk volume fluid (Sxo*porosity)

        st.Rtc  = new double[st.iRows];
        st.Rwa  = new double[st.iRows];
        st.Ro   = new double[st.iRows];
        st.Mo   = new double[st.iRows];
        st.Ma   = new double[st.iRows];
        st.Sw   = new double[st.iRows];
        st.BVW  = new double[st.iRows];
        st.Pay  = new double[st.iRows];

        st.PHIr = new double[st.iRows];

        st.Sxo  = new double[st.iRows];
        st.BVF  = new double[st.iRows];

        for (i=0; i<st.iRows; i++)
        {
          st.Rtc[i]  = 0.0;
          st.Rwa[i]  = 0.0;
          st.Ro[i]   = 0.0;
          st.Mo[i]   = 0.0;
          st.Ma[i]   = 0.0;
          st.Sw[i]   = 0.0;
          st.BVW[i]  = 0.0;
          st.Pay[i]  = 0.0;

          st.PHIr[i] = 0.0;

          st.Sxo[i]  = 0.0;
          st.BVF[i]  = 0.0;
        }

        st.dThickness   = 0.0; // Columns as Thickness
        st.dHydroCarbon = 0.0; // Oil or Gas Feet
        st.dPay         = 0.0; // Amount of Pay in Feet
        st.dPorosity    = 0.0; // Average Porosity
        st.dSaturation  = 0.0; // Average Water Saturation

        for (i=0; i<st.iRows; i++)
        {
          stData.depth = st.depth[i]; // Depth of formation

          dRt          = st.Rt[i];
          dRxo         = st.Rxo[i];
          dPhi         = st.PHIt[i];
          dVsh         = st.Vsh[i];
          dThickness   = st.thick[i];

          stData.M     = st.M + st.Mp * dPhi;  // Assume that M may be dependant on PHI

          if ((dRt != dNull) && (dPhi != dNull) && (dVsh != dNull))
          {
            switch (st.iWaterModel)
            {
              case pfeffer.math.pfefferSwModels._ARCHIE:
                stData = pfeffer.math.pfefferSwModels.pfefferArchie(
                           stData, dRt, dPhi );
                break;

              case pfeffer.math.pfefferSwModels._SIMANDOUX:
                stData = pfeffer.math.pfefferSwModels.pfefferSimandoux(
                           stData, dRt, dPhi, dVsh );
                break;

              case pfeffer.math.pfefferSwModels._DUAL_WATER:
                stData = pfeffer.math.pfefferSwModels.pfefferDualWater(
                           stData, dRt, dPhi, dVsh );
                break;
            }


            // Corrected resistivity (same as input Rt for Archie)
            iTemp     = (int) (1000.0 * stData.Rtc);
            st.Rtc[i] = (double) iTemp / 1000.0;

            // Apparent formation water resistivity (for Sw=100%)
            iTemp     = (int) (1000.0 * stData.Rwa);
            st.Rwa[i] = (double) iTemp / 1000.0;

            // Resistivity at 100% saturation
            iTemp     = (int) (1000.0 * stData.Ro);
            st.Ro[i]  = (double) iTemp / 1000.0;

            // Electrically connected Porosity
            if (st.Ro[i] > 0.0)
            {
              st.PHIr[i] = Math.sqrt(stData.Rwa / stData.Ro);
              iTemp      = (int) (1000.0 * st.PHIr[i]);
              st.PHIr[i] = (double) iTemp / 1000.0;
			}

            // Apparent cementation exponent (for Sw=100%)
            iTemp     = (int) (1000.0 * stData.M);
            st.Mo[i]  = (double) iTemp / 1000.0;

            // Apparent cementation exponent (for Sw=100%)
            iTemp     = (int) (1000.0 * stData.Ma);
            st.Ma[i]  = (double) iTemp / 1000.0;

            // Water saturation itself
            iTemp     = (int) (1000.0 * stData.Sw);
            st.Sw[i]  = (double) iTemp / 1000.0;

            // Bulk volume water (Sw*porosity)
            iTemp     = (int) (1000.0 * stData.BVW);
            st.BVW[i] = (double) iTemp / 1000.0;

            dThick  = dThick  + dThickness;

            if ((dPhi > st.dPhiCut) && (stData.Sw < st.dSwCut) &&
                (dVsh < st.dVshCut) && (stData.BVW < st.dBvwCut))
            {
              dTemp     = dThickness * dPhi * (1.0 - stData.Sw);
              iTemp     = (int) (1000.0 * dTemp);
              st.Pay[i] = (double) iTemp / 1000.0;

              dPayFeet = dPayFeet + dThickness;
              dFtOil   = dFtOil   + st.Pay[i];

              dAvgPhi  = dAvgPhi  + dPhi;
              dAvgSw   = dAvgSw   + st.Sw[i];

              iCount++;
            }
          }

          if ((dRxo != dNull) && (dPhi != dNull))
          {
            stData = pfeffer.math.pfefferSwModels.pfefferArchieSxo(
                           stData, dRxo, dPhi );

            // Fluid (moveable) saturation itself
            iTemp     = (int) (1000.0 * stData.Sxo);
            st.Sxo[i] = (double) iTemp / 1000.0;

            // Bulk volume fluid (Sxo*porosity)
            iTemp     = (int) (1000.0 * stData.BVF);
            st.BVF[i] = (double) iTemp / 1000.0;
		  }

          stData.Rtc = 0.0; // Corrected resistivity (same as input Rt for Archie)
          stData.Rwa = 0.0; // Apparent formation water resistivity (for Sw=100%)
          stData.Ro  = 0.0; // Resistivity at 100% saturation
          stData.Ma  = 0.0; // Apparent cementation exponent (for Sw=100%)
          stData.Sw  = 0.0; // Water saturation itself
          stData.BVW = 0.0; // Bulk volume water (Sw*porosity)

          stData.Sxo = 0.0; // Fluid (moveable) saturation
          stData.BVF = 0.0; // Bulk volume fluid (Sxo*porosity)
        }
      }

      // Volumetrics Data

      st.dThickness   = dThick;   // Columns as Thickness

      iTemp           = (int) (100.0 * dFtOil);
      dFtOil          = (double) iTemp / 100.0;
      st.dHydroCarbon = dFtOil;   // Oil or Gas Feet

      st.dPay         = dPayFeet; // Amount of Pay in Feet

      if (iCount > 0)
      {
        iTemp           = (int) (100.0 * dAvgPhi / (double) iCount);
        dAvgPhi         = (double) iTemp / 100.0;
        st.dPorosity    = dAvgPhi;  // Average Porosity

        iTemp           = (int) (100.0 * dAvgSw / (double) iCount);
        dAvgSw          = (double) iTemp / 100.0;
        st.dSaturation  = dAvgSw;   // Average Water Saturation
      }
    }

    return (st);
  }
}

/*
 *  @version 1.1 01/05/2010
 *  @author  John Victorine
 */
