/*
 * @parseTextureUtility.java Version 1.1 09/28/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package parse;

import parse.parseTextureListStruct;
import parse.parseTextureStruct;
import parse.parseMineralsListStruct;

//import kgs.kgsMeasSectListStruct;
import iqstrat.iqstratRemarkListStruct;

import lith.lithology.lithologySymbolsListStruct;

import lith.texture.textureListStruct;
import lith.texture.textureStruct;

/** Class parseTextureUtility
 *  <p> This Class will provide basic utilities for the textures parser
 *      data structures.
 *
 *  @version 1.1 09/28/2009
 *  @author  John R. Victorine
 */

public class parseTextureUtility
{
  public static final int _MAIN = 0;
  public static final int _SUB  = 1;

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getColor( String str, parseColorsStruct st )
   *  <p> This method will return the color number identifier from the
   *      color name that is passed in.
   * @param  str  = Color Name
   * @param  st   = Texture Data List Structure
   * @return iClr = the Color Identifier
   */

  public static int[] getColor( String str, parseTextureListStruct st )
  {
    int iRGB[] = { 255, 255, 255 };

    if (st != null)
    {
      for (int i=1; i<st.iColors; i++)
      {
        if (str.equals(st.sColors[i]))
        {
           for (int j=0; j<3; j++)
             iRGB[j] = st.iRGB[i][j];
        }
      }
    }

    return (iRGB);
  }

  /** Method getAbbrevWord()
   * <p> This method will determine if the word is an abbreviation and
   *     replace the abbreviation with a correct word or phrase
   * @param  str  = Abbreviation
   * @param  st   = Texture Data List Structure
   * @return word = the word or phrase to represent the abbreviation
   */

  public static String getAbbrevWord(String str, parseTextureListStruct st)
  {
    String word = "";

    str = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iAbbrev; i++)
      {
        if (str.equals(st.sAbbrev[i][0].toLowerCase()))
        {
          word = new String( st.sAbbrev[i][1] );
        }
      }
    }

    return (word);
  }

  /** Method getKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static String[] getKeyword( String str, parseTextureListStruct st )
  {
    String sWords[] = { "", "" };
    String sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase()))
        {
          sWords[0] = new String( st.sKeywords[i][0] );
          sWords[1] = new String( st.sKeywords[i][1] );
        }
      }
    }

    return (sWords);
  }

  /** Method getTextureStruct()
   * <p> This method will return the texture data structure
   * @param  iLevel = MAIN (Mineral) or SUB (Mineral Descriptor)
   * @param  iRow   = The Row Number for the Level
   * @param  str    = The Mineral Name or the Descriptor Name
   * @param  st     = The Parse texture List Data Structure
   * @return stT    = parse texture data structure
   */

  public static parseTextureStruct getTextureStruct(
      int iLevel, int iRow, String str, parseTextureListStruct st )
  {
    int i=0;
    String sWord = new String(str.toLowerCase());
    parseTextureStruct stT = null;

    if (st != null)
    {
      switch (iLevel)
      {
        case _MAIN:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[i].sName.toLowerCase()) )
            {
              stT = copy( st.stItem[i] );
            }
          }
          break;
        case _SUB:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[iRow].descriptors[i].sName.toLowerCase()) )
            {
              stT = copy( st.stItem[iRow].descriptors[i] );
            }
          }
          break;
      }
    }

    return (stT);
  }

  /** Method getTextureRow()
   * <p> This method will return the texture data structure
   * @param  iLevel = MAIN (Mineral) or SUB (Mineral Descriptor)
   * @param  iRow   = The Row Number for the Level
   * @param  str    = The Mineral Name or the Descriptor Name
   * @param  st     = The Parse texture List Data Structure
   * @return stT    = parse texture data structure
   */

  public static int getTextureRow(
           int iLevel, int iRow, String str, parseTextureListStruct st )
  {
    int iTexture = -1;
    int i=0;
    String sWord = new String(str.toLowerCase());

    if (st != null)
    {
      switch (iLevel)
      {
        case _MAIN:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[i].sName.toLowerCase()) )
            {
              iTexture = i;
            }
          }
          break;
        case _SUB:
          for (i=0; i<st.stItem[iRow].iCount; i++)
          {
            if (sWord.equals( st.stItem[iRow].descriptors[i].sName.toLowerCase()) )
            {
              iTexture = i;
            }
          }
          break;
      }
    }

    return (iTexture);
  }

  /* ======================================================================== *
   * ---------------------------- SET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method setRhomaaUmaa()
   * <p> This method will set the Rhomaa-Umaa Values depending on the primary,
   *     secondary and other setting.
   * @param  stMinerals = Common Minerals Data List Structure
   * @param  st         = Parse Texture List Data Structure
   * @return st         = Parse Texture List Data Structure
   */

  public static parseTextureListStruct setRhomaaUmaa(
      parseMineralsListStruct stMinerals, parseTextureListStruct st )
  {
    int i=0;
    int j=0;
    int k=0;
    int iType         = 0;
                      //  1st, 2nd, 3rd
    double dRhomaa[]  = { 0.0, 0.0, 0.0 };
    double dUmaa[]    = { 0.0, 0.0, 0.0 };
    double dGR[]      = { 0.0, 0.0, 0.0 };
    double dPer       = 0.0;
    String sMineral   = "";
    String sPrimary   = "";
    String secondary  = "";
    String sOther     = "";

    if ((stMinerals != null) && (st != null))
    {
      for (i=0; i<st.iCount; i++)
      {
        iType     = st.stItem[i].iType;
        sPrimary  = new String( st.stItem[i].sPrimary.toLowerCase() );
        secondary = new String( st.stItem[i].sSecondary.toLowerCase() );

        for (j=0; j<stMinerals.iCount; j++)
        {
          sMineral = new String( stMinerals.stItem[j].sName.toLowerCase() );

          // Set Primary Mineral Rhomaa, Umaa and GR data Values

          if (sPrimary.equals(sMineral))
          {
            dRhomaa[0] = stMinerals.stItem[j].dRhomaa;
            dUmaa[0]   = stMinerals.stItem[j].dUmaa;
            dGR[0]     = stMinerals.stItem[j].dGR;

            if ((st.stItem[i].dGR > 0.0) &&
                (st.stItem[i].sSecondary.length() == 0))
              dGR[0] = st.stItem[i].dGR;
          }

          // Set Secondary Mineral Rhomaa, Umaa and GR data Values

          if (secondary.equals(sMineral))
          {
            dRhomaa[1] = stMinerals.stItem[j].dRhomaa;
            dUmaa[1]   = stMinerals.stItem[j].dUmaa;
            dGR[1]     = stMinerals.stItem[j].dGR;

            if (st.stItem[i].dGR > 0.0)
              dGR[1] = st.stItem[i].dGR;
          }
        }

        // IF there is more than one mineral to describe the rock type THEN
        //   Compute the Rhomaa, Umaa and Gamma Ray by proportion:
        //     X = 60% of 1st Mineral * X1 + 40% of 2nd Mineral * X2
        //   Otherwise X = X1

        if (secondary.length() > 0)
        {
          st.stItem[i].density = 0.6 * dRhomaa[0] + 0.4 * dRhomaa[1];
          st.stItem[i].dU      = 0.6 * dUmaa[0]   + 0.4 * dUmaa[1];
          st.stItem[i].dGR     = 0.6 * dGR[0]     + 0.4 * dGR[1];

          // Redefine the Primary Rhomaa, Umaa and GR

          dRhomaa[0] = st.stItem[i].density;
          dUmaa[0]   = st.stItem[i].dU;
          dGR[0]     = st.stItem[i].dGR;
        }
        else
        {
          st.stItem[i].density = dRhomaa[0];
          st.stItem[i].dU      = dUmaa[0];
          st.stItem[i].dGR     = dGR[0];
        }

        for (k=0; k<st.stItem[i].iCount; k++)
        {
          st.stItem[i].descriptors[k].iType = st.stItem[i].iType;
          sOther = new String(st.stItem[i].descriptors[k].sOther.toLowerCase());

          for (j=0; j<stMinerals.iCount; j++)
          {
            sMineral = new String( stMinerals.stItem[j].sName.toLowerCase() );

            if (sOther.equals(sMineral))
            {
              dRhomaa[2] = stMinerals.stItem[j].dRhomaa;
              dUmaa[2]   = stMinerals.stItem[j].dUmaa;
              dGR[2]     = stMinerals.stItem[j].dGR;

              if (st.stItem[i].descriptors[k].dGR > 0.0)
              {
                dGR[2] = st.stItem[i].descriptors[k].dGR;
              }
            }
          }

          if (!sPrimary.equals(sOther))
          {
            dPer = st.stItem[i].descriptors[k].dPercent;

            st.stItem[i].descriptors[k].density =
                (1.0 - dPer) * dRhomaa[0] + dPer * dRhomaa[2];
            st.stItem[i].descriptors[k].dU =
                (1.0 - dPer) * dUmaa[0] + dPer * dUmaa[2];
            st.stItem[i].descriptors[k].dGR =
                (1.0 - dPer) * dGR[0] + dPer * dGR[2];
          }
          else
          {
            st.stItem[i].descriptors[k].density = dRhomaa[2];
            st.stItem[i].descriptors[k].dU      = dUmaa[2];
            st.stItem[i].descriptors[k].dGR     = dGR[2];
          }
        }
      }
    }

    return (st);
  }

  /* ======================================================================== *
   * --------------------------- PARSE METHODS ------------------------------ *
   * ======================================================================== */

  /** Method parse()
   * <p> This method will parse the Remarks string for Texture/Lithology.
   * @param  stList     = Remarks List Data Structure
   * @param  stSymbols  = Lithology Symbols Data List Structure
   * @param  stTextures = Texture Thesaurus
   * @return stTexture  = Rock texture Data List Structure
   */

  public static textureListStruct parse( iqstratRemarkListStruct    stList,
                                         lithologySymbolsListStruct stSymbols,
                                         parseTextureListStruct     stTextures )
  {
    textureListStruct stTexture  = null;
    int               iUnit      = iqstrat.iqstratTracksStruct._SCALE_HALF_FOOT; //_SCALE_10TH_FOOT; //._SCALE_1_FOOT;
    double            dUnit      = 0.1; //1.0;
    double            depth      = 0.0;
    double            depthStart = 0.0;
    double            depthEnd   = 0.0;
    String            sKEY       = cmn.cmnString.UniqueName();
    String            str        = "";
    String            tokens[]   = null;
//    String            sDelimiter = new String("[,-;.:= ]+");
    String            sDelimiter = new String("[ /,;:.=-]+");
    textureStruct     stText     = null;

    if (stList != null)
    {
      stTexture = new textureListStruct();

      for (int i=0; i<stList.iCount; i++)
      {
        if (i == 0)
        {
          depthStart = stList.stItem[i].depthStart;
          depthEnd   = stList.stItem[i].depthEnd;
        }

        if (depthStart > stList.stItem[i].depthStart)
          depthStart = stList.stItem[i].depthStart;

        if (depthEnd < stList.stItem[i].depthEnd)
          depthEnd   = stList.stItem[i].depthEnd;

        depth = Math.abs( stList.stItem[i].depthEnd -
                          stList.stItem[i].depthStart );

        iUnit = lith.texture.textureUtility.getUnit( iUnit, depth );
        dUnit = iqstrat.iqstratTracksStruct._SCALE[iUnit];
/*
        if (dUnit > iqstrat.iqstratTracksStruct._SCALE[
                      lith.texture.textureUtility.getUnit( depth )])
        {
          iUnit = lith.texture.textureUtility.getUnit( depth );
          dUnit = iqstrat.iqstratTracksStruct._SCALE[iUnit];
        }
*/
      }

      stTexture.depthStart = depthStart;
      stTexture.depthEnd   = depthEnd;
      stTexture.iUnit      = iUnit;
//System.out.println("TEXTURE: "+iUnit+" "+dUnit);

      for (int i=0; i<stList.iCount; i++)
      {
        str    = new String( stList.stItem[i].sText.replace('\t', ' ') );
        str    = new String( str.replace('\n', ' ') );

        tokens = str.split( sDelimiter );

        stText = parseTexture( stList.stItem[i].depthStart,
                               stList.stItem[i].depthEnd,
                               tokens, stSymbols, stTextures );

        if (stText != null)
        {
          stText.sKEY = new String( sKEY + "_" + i );
          stTexture   = lith.texture.textureUtility.add( stText, stTexture );
        }

        tokens = null;

        if (stText != null)
          stText.delete();
        stText = null;
      }

      stList.delete();
      stList = null;
    }

    return (stTexture);
  }

  /** Method parseTexture()
   * <p> This method will parse the text string into rock texture
   * @param  depthStart = the starting depth
   * @param  depthEnd   = the ending depth
   * @param  sData      = the texture string phrase or word
   * @param  stSymbols  = Lithology Symbols Data List Structure
   * @param  stTextures = Texture Thesaurus
   * @return st         = rock texture data structure
   */

  public static  textureStruct parseTexture(
                     double depthStart, double depthEnd, String sData[],
                     lithologySymbolsListStruct stSymbols,
                     parseTextureListStruct     stTextures )
  {
    int i, j, k, m, p;
    textureStruct      st         = null;
    int                iCount     = 0;
    int                iTotal     = 10;
    String             sWords[][] = { {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""}};
    String             sTemp[]    = {"", ""};
    int                iTexture  = -1;
    int                iDescrip  = -1;
    int                iContinue = 0;
    int                iNext     = 0;
    parseTextureStruct stTexture = null;
    String             sLithology = "";
    int                id         = 0;
    int                iSymbol    = 0;
    int                iRGB[]     = { 255, 255, 255 };
    int                iEnergy    = 0;
    int                iType      = lith.texture.textureConstants.NONE;
    double             density    = 0.0;
    double             dU         = 0.0;
    double             dGR        = 0.0;
    int                iaEnergy   = 0;
    int                iSet       = 0;
    int                iSet_a     = 0;

    if (stTexture != null)
      stTexture.delete();

    // Modify Abbreviation to word

    for (i=0; i<sData.length; i++)
    {
      for (j=0; j<stTextures.iAbbrev; j++)
      {
        if (sData[i].toLowerCase().equals(stTextures.sAbbrev[j][0].toLowerCase()))
          sData[i] = new String(stTextures.sAbbrev[j][1]);
      }
    }

    // Find Texture Words up to 5

    for (j=0; j<sData.length; j++)
    {
      if (isKeyword( sData[j], stTextures ))
      {
        if (iCount < iTotal)
        {
          sTemp             = getKeyword( sData[j], stTextures );
          sWords[iCount][0] = new String(sTemp[0]);
          sWords[iCount][1] = new String(sTemp[1]);
          iCount++;
        }
      }
    }

    // Change a Modifier to a descriptor

    for (i=0; i<iCount; i++)
    {
      if ((sWords[i][1].equals("M")) || (sWords[i][1].equals("Mt")))
      {
        k = i+1;
        m = i+2;
        if (k<iCount)
        {
          if (isModified( sWords[i][0], sWords[k][0], stTextures ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String(sWords[k][1]);

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");

            if (m < iCount)
            {
		      if (sWords[m][1].equals("T"))
		      {
                sWords[i][0] = new String(sWords[i][0] + " " + sWords[m][0]);
                sWords[i][1] = new String(sWords[m][1]);

                sWords[m][0] = new String("");
                sWords[m][1] = new String("");
			  }
			}
          }
        }
      }
/*
      if (sWords[i][1].equals("Da"))
      {
		if ((i > 0) && (i<iCount-1))
		{
          k = i-1;
          m = i+1;
		  if ((!sWords[k][1].equals("L")) && (!sWords[m][1].equals("L")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
		else if (i<iCount-1)
		{
          m = i+1;
		  if (!sWords[m][1].equals("L"))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
		else if (i > 0)
		{
          k = i-1;
		  if (!sWords[k][1].equals("L"))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
      }
*/
      if (sWords[i][1].equals("Da"))
      {
	    if ((i > 0) && (i<iCount-1))
	 	{
          k = i-1;
          m = i+1;

		  if ((!sWords[k][1].equals("L")) && (!sWords[m][1].equals("L")) &&
		      (!sWords[k][1].equals("B")) && (!sWords[m][1].equals("B")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
		else if (i == 0)
		{
          m = i+1;
		  if ((!sWords[m][1].equals("L")) && (!sWords[m][1].equals("B")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
		else if (i==iCount-1)
		{
          k = i-1;
		  if ((!sWords[k][1].equals("L")) && (!sWords[k][1].equals("B")))
		  {
		    sWords[i][1] = new String("L");
		  }
		  else
		  {
		    sWords[i][1] = new String("T");
		  }
		}
      }

      if (sWords[i][1].equals("B"))
      {
        k = i+1;
        if (k<iCount)
        {
          if (isModified( sWords[i][0], sWords[k][0], stTextures ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String("L");

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }
    }

    // Identify the main lithology type

    for (i=0; i<iCount; i++)
    {
      if (sLithology.length() == 0)
      {
        if ((sWords[i][1].equals("L")) || (sWords[i][1].equals("B")) &&
            (iContinue == 0))
        {
          iTexture  = getTextureRow( _MAIN, -1, sWords[i][0], stTextures );
          iContinue = 1;

          if (iTexture > -1)
          {
            stTexture  = stTextures.stItem[iTexture];

            sLithology = new String( stTexture.sName );
            id         = stTexture.id;

            for (p=0; p<stSymbols.iCount; p++)
            {
              if (id == stSymbols.stItem[p].id)
              {
                iSymbol = p;
              }
            }

            for (j=0; j<3; j++)
              iRGB[j] = stTexture.iRGB[j];

            iType   = stTexture.iType;
            iEnergy = stTexture.iEnergy;

//            density = stTexture.density;
//            dU      = stTexture.dU;
//            dGR     = stTexture.dGR;
          }
        }
      }
    }

    // Modify Litholgy with a descriptor

    if (iContinue == 1)
    {
      iNext    = 0;
      iaEnergy = 0;
      iSet_a   = 0;

      for (i=0; i<iCount; i++)
      {
		iSet = 0;

        if ((sWords[i][1].equals("T")) || (sWords[i][1].equals("Ta")) || (sWords[i][1].equals("Mt")))
        {
          if (iTexture > -1)
          {
            iDescrip = getTextureRow(_SUB, iTexture, sWords[i][0], stTextures);

            if ((iDescrip > -1) && (iNext == 0))
            {
			  if (sWords[i][1].equals("Ta"))
			    iSet_a  = 1;

              iSet      = 1;
              stTexture = stTextures.stItem[iTexture].descriptors[iDescrip];
              iaEnergy  = stTexture.iEnergy;
              iEnergy   = stTexture.iEnergy;

              for (j=0; j<3; j++)
			     iRGB[j] = stTexture.iRGB[j];

              iType = stTexture.iType;

//              sLithology = new String( stTexture.sName + " " + sLithology); // JRV
              id         = stTexture.id;

              for (p=0; p<stSymbols.iCount; p++)
              {
                if (id == stSymbols.stItem[p].id)
                {
                  iSymbol = p;
                }
              }
		    }
          }
	    }

        if ((iSet_a == 0) && (iSet == 0) &&
            ((sWords[i][1].equals("D")) ||
             (sWords[i][1].equals("M")) ||
             (sWords[i][1].equals("Mt"))))
        {
          if (iTexture > -1)
          {
            iDescrip = getTextureRow(_SUB, iTexture, sWords[i][0], stTextures);

            if ((iDescrip > -1) && (iNext == 0))
            {
              iNext = 1;
              stTexture  = stTextures.stItem[iTexture].descriptors[iDescrip];
              if (sWords[i][1].equals("D"))  //JRV
                sLithology = new String( stTexture.sName + " " + sLithology);
              id         = stTexture.id;

              for (p=0; p<stSymbols.iCount; p++)
              {
                if (id == stSymbols.stItem[p].id)
                {
                  iSymbol = p;
                }
              }

              if (iaEnergy == 0)
              {
                for (j=0; j<3; j++)
                  iRGB[j] = stTexture.iRGB[j];
              }

              iType = stTexture.iType;

              if (iaEnergy == 0)
                iEnergy = stTexture.iEnergy;
              else
                iEnergy = iaEnergy;

//              density = stTexture.density;
//              dU      = stTexture.dU;
//              dGR     = stTexture.dGR;
            }
          }
        }
      }
    }

    if (iContinue == 1)
    {
      st = new textureStruct();

      st.sKEY       = new String( cmn.cmnString.UniqueName() +
                                  ((int)depthStart)); // Create a Unique KEY
      st.depthStart = depthStart; // Starting Depth
      st.depthEnd   = depthEnd;   // Ending Depth

      st.sTexture   = new String( sLithology );
      st.iTexture   = iType;
      st.iType      = iType;
      st.iEnergy    = iEnergy;
      st.iRGB       = iRGB;
      st.id         = id;
      st.iSymbol    = iSymbol;
    }

    return (st);
  }

  /** Method parse()
   * <p> This method will parse the Remarks string for Texture/Lithology.
   * @param  sRemarks   = Remarks List Data Structure
   * @param  stSymbols  = Lithology Symbols Data List Structure
   * @param  stTextures = Texture Thesaurus
   * @return dComments  = Rock texture Data List Structure
   */

  public static String parse( String                     sRemarks,
                              lithologySymbolsListStruct stSymbols,
                              parseTextureListStruct     stTextures )
  {
    int i, j, k, m, p, q;
	String             sComments = "";
    String             sDelimiter = new String("[ /,;:.=-]+");
    String             str        = "";
    String             sData[]     = null;
    int                iCount     = 0;
    int                iTotal     = 20;
    String             sWords[][] = { {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""},
                                      {"", ""}};
    String             sTemp[]    = {"", ""};
    int                iTexture  = -1;
    int                iDescrip  = -1;
    int                iContinue = 0;
    parseTextureStruct stTexture = null;
    String             sLithology = "";
    int                iNew      = -1;

    str    = new String( sRemarks.replace('\t', ' ') );
    str    = new String( str.replace('\n', ' ') );

    sData = str.split( sDelimiter );

    if (sData != null)
    {
      // Modify Abbreviation to word

      for (i=0; i<sData.length; i++)
      {
        for (j=0; j<stTextures.iAbbrev; j++)
        {
          if (sData[i].toLowerCase().equals(stTextures.sAbbrev[j][0].toLowerCase()))
            sData[i] = new String(stTextures.sAbbrev[j][1]);
        }
      }

      // Find Texture Words up to 5

      for (j=0; j<sData.length; j++)
      {
        if (isKeyword( sData[j], stTextures ))
        {
          if (iCount < iTotal)
          {
            sTemp             = getKeyword( sData[j], stTextures );
            sWords[iCount][0] = new String(sTemp[0]);
            sWords[iCount][1] = new String(sTemp[1]);
            iCount++;
          }
        }
      }

      sData = null;

      // Change a Modifier to a descriptor

      for (i=0; i<iCount; i++)
      {
        if ((sWords[i][1].equals("M")) || (sWords[i][1].equals("Mt")))
        {
          k = i+1;
          m = i+2;
          if (k<iCount)
          {
            if (isModified( sWords[i][0], sWords[k][0], stTextures ))
            {
              sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
              sWords[i][1] = new String(sWords[k][1]);

              sWords[k][0] = new String("");
              sWords[k][1] = new String("");

              if (m < iCount)
              {
  		        if (sWords[m][1].equals("T"))
		        {
                  sWords[i][0] = new String(sWords[i][0] + " " + sWords[m][0]);
                  sWords[i][1] = new String(sWords[m][1]);

                  sWords[m][0] = new String("");
                  sWords[m][1] = new String("");
			    }
			  }
            }
          }
        }

        if (sWords[i][1].equals("Da"))
        {
		  if ((i > 0) && (i<iCount-1))
		  {
            k = i-1;
            m = i+1;

		    if ((!sWords[k][1].equals("L")) && (!sWords[m][1].equals("L")) &&
		        (!sWords[k][1].equals("B")) && (!sWords[m][1].equals("B")))
		    {
		      sWords[i][1] = new String("L");
		    }
		    else
		    {
		      sWords[i][1] = new String("T");
		    }
		  }
		  else if (i == 0)
		  {
            m = i+1;

		    if ((!sWords[m][1].equals("L")) && (!sWords[m][1].equals("B")))
		    {
		      sWords[i][1] = new String("L");
		    }
		    else
		    {
		      sWords[i][1] = new String("T");
		    }
		  }
		  else if (i==iCount-1)
		  {
            k = i-1;

		    if ((!sWords[k][1].equals("L")) && (!sWords[k][1].equals("B")))
		    {
		      sWords[i][1] = new String("L");
		    }
		    else
		    {
		      sWords[i][1] = new String("T");
		    }
		  }
        }

        if (sWords[i][1].equals("B"))
        {
          k = i+1;
          if (k<iCount)
          {
            if (isModified( sWords[i][0], sWords[k][0], stTextures ))
            {
              sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
              sWords[i][1] = new String("L");

              sWords[k][0] = new String("");
              sWords[k][1] = new String("");
            }
          }
        }
      }

      // Identify the main lithology type

      for (i=0; i<iCount; i++)
      {
        if (sLithology.length() == 0)
        {
          if ((sWords[i][1].equals("L")) || (sWords[i][1].equals("B")) &&
              (iContinue == 0))
          {
            iTexture  = getTextureRow( _MAIN, -1, sWords[i][0], stTextures );
            iContinue = 1;

            if (iTexture > -1)
            {
              stTexture  = stTextures.stItem[iTexture];
              sLithology = new String( stTexture.sName );
            }
          }
        }
      }

      // Modify Litholgy with a descriptor

      if (iContinue == 1)
      {
		sComments = new String(sLithology + ": ");

        for (i=0; i<iCount; i++)
        {
          if ((sWords[i][1].equals("L")) || (sWords[i][1].equals("B")))
          {
            iDescrip = getTextureRow( _MAIN, -1, sWords[i][0], stTextures );

            if (iDescrip > -1)
            {
              stTexture = stTextures.stItem[iDescrip];

              iContinue = 0;
              sData     = sComments.split( sDelimiter );

              if (sData != null)
              {
			    for (q=0; q<sData.length; q++)
			    {
				  if (sData[q].equals(stTexture.sName))
				    iContinue = 1;
				}
			  }

              sData = null;

              if ((iContinue == 0) && (!sLithology.equals(stTexture.sName)))
              {
                if (iNew == -1)
                {
 		          sComments = new String( sComments + " " + stTexture.sName );
   		          iNew      = 1;
			    }
			    else
			    {
   		          sComments = new String( sComments + ", " + stTexture.sName );
			  	}
			  }
		    }
	      }

          if ((sWords[i][1].equals("T"))  ||
              (sWords[i][1].equals("Ta")) ||
              (sWords[i][1].equals("Mt")))
          {
            if (iTexture > -1)
            {
              iDescrip = getTextureRow(_SUB, iTexture, sWords[i][0], stTextures);

              if (iDescrip > -1)
              {
                stTexture = stTextures.stItem[iTexture].descriptors[iDescrip];

                iContinue = 0;
                sData     = sComments.split( sDelimiter );

                if (sData != null)
                {
			      for (q=0; q<sData.length; q++)
			      {
				    if (sData[q].equals(stTexture.sName))
				      iContinue = 1;
				  }
			    }

                sData = null;

                if (iContinue == 0)
                {
                  if (iNew == -1)
                  {
   		            sComments = new String( sComments + " " + stTexture.sName );
   		            iNew      = 1;
			      }
			      else
			      {
   		            sComments = new String( sComments + ", " + stTexture.sName );
			  	  }
			    }
              }
            }
          }

          if  ((sWords[i][1].equals("D")) || (sWords[i][1].equals("M")))
          {
            if (iTexture > -1)
            {
              iDescrip = getTextureRow(_SUB, iTexture, sWords[i][0], stTextures);

              if (iDescrip > -1)
              {
                stTexture = stTextures.stItem[iTexture].descriptors[iDescrip];

                iContinue = 0;
                sData     = sComments.split( sDelimiter );

                if (sData != null)
                {
			      for (q=0; q<sData.length; q++)
			      {
				    if (sData[q].equals(stTexture.sName))
				      iContinue = 1;
				  }
			    }

                sData = null;

                if (iContinue == 0)
                {
                  if (iNew == -1)
                  {
   		            sComments = new String( sComments + " " + stTexture.sName );
   		            iNew      = 1;
			      }
			      else
			      {
   		            sComments = new String( sComments + ", " + stTexture.sName );
				  }
			    }
              }
            }
          }
        }
  	  }
	}

    return (sComments);
  }

  /* ======================================================================== *
   * ----------------------------- IS METHODS ------------------------------- *
   * ======================================================================== */

  /** Method isKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isKeyword( String str, parseTextureListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isModified( String str, parseTextureListStruct st )
   *  <p> This method will determine if the string is part of descriptor set
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */


  public static boolean isModified( String str, parseTextureListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iModifiers; i++)
      {
        if (sWord.equals(st.sModifier[i][1].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isModified( String sin, String str, parseTextureListStruct st )
   *  <p> This method will determine if the string is part of descriptor set
   * @param  sin    = string or string phrase to be merged
   * @param  str    = string or string phrase being compared.
   * @param  st     = Texture Data List Structure
   * @return bValue = true it does and falls it does not
   */


  public static boolean isModified(
      String sin, String str, parseTextureListStruct st )
  {
    boolean bValue = false;
    String  sWord0 = new String(sin.toLowerCase());
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iModifiers; i++)
      {
        if ((sWord0.equals(st.sModifier[i][0].toLowerCase())) &&
            (sWord.equals(st.sModifier[i][1].toLowerCase())))
          bValue = true;
      }
    }

    return (bValue);
  }

  /* ======================================================================== *
   * ---------------------------- COPY METHODS ------------------------------ *
   * ======================================================================== */

  /** Method copyList( parseTextureListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static parseTextureListStruct copyList( parseTextureListStruct stOld )
  {
    parseTextureListStruct stNew = null;
    int i = 0;
    int j = 0;

    if (stOld != null)
    {
      stNew = new parseTextureListStruct();

      // Colors

      stNew.iColors = stOld.iColors;             // total number of records
      stNew.sColors = new String[stNew.iColors]; // Color description
      stNew.iRGB    = new int[stNew.iColors][3]; // rgb values

      for (i=0; i<stNew.iColors; i++)
      {
        stNew.sColors[i] = new String(stOld.sColors[i]);
        for (j=0; j<3; j++)
          stNew.iRGB[i][j] = stOld.iRGB[i][j];
      }

      // Abbreviations

      stNew.iAbbrev = stOld.iAbbrev;  // total number of records
      // Abreviation {Abbrev, word }
      stNew.sAbbrev = new String[stNew.iAbbrev][2];

      for (i=0; i<stNew.iAbbrev; i++)
      {
        for (j=0; j<2; j++)
          stNew.sAbbrev[i][j] = new String(stOld.sAbbrev[i][j]);
      }

      // Modifiers

      stNew.iModifiers    = stOld.iModifiers; // total number of records
      // Modifier List { Modifier, Modifies, phrase }
      stNew.sModifier = new String[stNew.iModifiers][3];

      for (i=0; i<stNew.iModifiers; i++)
      {
        for (j=0; j<3; j++)
          stNew.sModifier[i][j] = new String(stOld.sModifier[i][j]);
      }

      // Keywords

      stNew.iKeywords = 0; // Total Number of Keywords
      // Keyword List { pharase, type}
      stNew.sKeywords = new String[stNew.iKeywords][2];

      for (i=0; i<stNew.iKeywords; i++)
      {
        for (j=0; j<2; j++)
          stNew.sKeywords[i][j] = new String(stOld.sKeywords[i][j]);
      }

      // Lithologies

      stNew.iCount = stOld.iCount;
      stNew.stItem = new parseTextureStruct[stOld.iCount];

      for (i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy( parseTextureStruct stOld )
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static parseTextureStruct copy( parseTextureStruct stOld )
  {
    int i=0;
    int j=0;
    parseTextureStruct stNew = new parseTextureStruct();

    // Texture Lithology

    stNew.id     = stOld.id;                       // ID for Mask Symbol
    stNew.sID    = new String( stOld.sID );        // String ID for Mask
    stNew.sName  = new String( stOld.sName );      // Name
    stNew.sAbrev = new String( stOld.sAbrev );     // Abbreviation

    // Lithology Symbol & Background color

    stNew.iSymbol = stOld.iSymbol; // Lithology
    for (i=0; i<3; i++)
      stNew.iRGB[i] = stOld.iRGB[i];

    // Misc variables

    stNew.iEnergy    = stOld.iEnergy;  // Energy Level
    stNew.iType      = stOld.iType;    // Type of Texture

    stNew.sPrimary   = new String( stOld.sPrimary );
    stNew.sSecondary = new String( stOld.sSecondary );
    stNew.sOther     = new String( stOld.sOther );

    stNew.dPercent   = stOld.dPercent;

    stNew.density    = stOld.density;  // Grain Density (Rhomaa)
    stNew.dU         = stOld.dU;       // Photoelectric Factor (Umaa)
    stNew.dGR        = stOld.dGR;      // Gamma Ray (API)

    // Descriptors

    stNew.iCount      = stOld.iCount;
    stNew.descriptors = new parseTextureStruct[stOld.iCount];

    for (i=0; i<stOld.iCount; i++)
    {
      stNew.descriptors[i] = copy( stOld.descriptors[i] );
    }

    return (stNew);
  }

  /* ======================================================================== *
   * ---------------------------- PRINT METHODS ----------------------------- *
   * ======================================================================== */

  /** Method print( parseTextureListStruct st )
   * <p> This method will print the parse symbols structure
   * @param  st = Mask Symbols data list structure
   */

  public static void print( parseTextureListStruct st )
  {
    int i = 0;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        print(st.stItem[i]);
      }
    }
  }

  /** Method print( parseTextureStruct st )
   * <p> This method will print the symbols data structure
   * @param  st = Mask symbols data structure
   */

  public static void print( parseTextureStruct st )
  {
    String sType = "";

    if (st != null)
    {
      sType = "";
      switch (st.iType)
      {
        case lith.texture.textureConstants.SANDSTONE:
          sType = new String("SILICLASTIC");
          break;
        case lith.texture.textureConstants.CARBONATE:
          sType = new String("CARBONATE");
          break;
        case lith.texture.textureConstants.EVAPORITE:
          sType = new String("EVAPORITE");
          break;
        case lith.texture.textureConstants.ORGANIC:
          sType = new String("ORGANIC");
          break;
        case lith.texture.textureConstants.OTHER:
          sType = new String("OTHER");
          break;
      }

      System.out.println( sType     + ": " +
                          st.id     + " " + // ID for Mask Symbol
                          st.sID    + " " + // String ID for Mask Symbol
                          st.sName  + " " + // Name or Description of parse
                          st.sAbrev + " " + // Abbreviation of parse
                          st.iSymbol);      // Lithology Plot Symbol

      System.out.println( "  -- Color "   +
                          st.iRGB[0] + " " + // Background Color Red Number
                          st.iRGB[1] + " " + // Background Color Green Number
                          st.iRGB[2]);       // Background Color Blue Number

      System.out.println( "  -- Misc "     +
                          st.iEnergy  + " " + // Energy Level
                          st.density  + " " + // Density
                          st.dU       + " " + // Photoelectric factor
                          st.dGR );           // gamma ray

      if (st.descriptors != null)
      {
        System.out.println( " ---- Descriptors ---- ");

        for (int i=0; i<st.iCount; i++)
        {
          print(st.descriptors[i]);
        }

        System.out.println( " ---- End Descriptors ---- ");
      }

      System.out.println("");
    }
  }
}