/*
 * @parseSedStructUtility.java Version 1.1 10/01/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package parse;

import parse.parseSedStructListStruct;
import parse.parseSedStructStruct;

//import kgs.kgsMeasSectListStruct;
import iqstrat.iqstratRemarkListStruct;

import rock.sedimentary.sedimentaryListStruct;
import rock.sedimentary.sedimentaryStruct;

/** Class parseSedStructUtility
 *  <p> This Class will provide basic utilities for the sedimentarys parser
 *      data structures.
 *
 *  @version 1.1 10/01/2009
 *  @author  John R. Victorine
 */

public class parseSedStructUtility
{
  public static final int _MAIN = 0;
  public static final int _SUB  = 1;

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getAbbrevWord()
   * <p> This method will determine if the word is an abbreviation and
   *     replace the abbreviation with a correct word or phrase
   * @param  str  = Abbreviation
   * @param  st   = SedStruct Data List Structure
   * @return word = the word or phrase to represent the abbreviation
   */

  public static String getAbbrevWord(String str, parseSedStructListStruct st)
  {
    String word = "";

    str = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iAbbrev; i++)
      {
        if (str.equals(st.sAbbrev[i][0].toLowerCase()))
        {
          word = new String( st.sAbbrev[i][1] );
        }
      }
    }

    return (word);
  }

  /** Method getKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = SedStruct Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static String[] getKeyword( String str, parseSedStructListStruct st )
  {
    String sWords[] = { "", "" };
    String sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase()))
        {
          sWords[0] = new String( st.sKeywords[i][0] );
          sWords[1] = new String( st.sKeywords[i][1] );
        }
      }
    }

    return (sWords);
  }

  /** Method getSedStructStruct()
   * <p> This method will return the sedimentary data structure
   * @param  iLevel = MAIN (Mineral) or SUB (Mineral Descriptor)
   * @param  iRow   = The Row Number for the Level
   * @param  str    = The Mineral Name or the Descriptor Name
   * @param  st     = The Parse sedimentary List Data Structure
   * @return stT    = parse sedimentary data structure
   */

  public static parseSedStructStruct getSedStructStruct(
      int iLevel, int iRow, String str, parseSedStructListStruct st )
  {
    int i=0;
    String sWord = new String(str.toLowerCase());
    parseSedStructStruct stT = null;

    if (st != null)
    {
      switch (iLevel)
      {
        case _MAIN:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[i].sPhrase.toLowerCase()) )
            {
              stT = copy( st.stItem[i] );
            }
          }
          break;
        case _SUB:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals(
                  st.stItem[iRow].descriptors[i].sPhrase.toLowerCase()) )
            {
              stT = copy( st.stItem[iRow].descriptors[i] );
            }
          }
          break;
      }
    }

    return (stT);
  }

  /** Method getSedStructRow()
   * <p> This method will return the sedimentary data structure
   * @param  iLevel = MAIN (Mineral) or SUB (Mineral Descriptor)
   * @param  iRow   = The Row Number for the Level
   * @param  str    = The Mineral Name or the Descriptor Name
   * @param  st     = The Parse sedimentary List Data Structure
   * @return stT    = parse sedimentary data structure
   */

  public static int getSedStructRow(
           int iLevel, int iRow, String str, parseSedStructListStruct st )
  {
    int iSedStruct = -1;
    int i=0;
    String sWord = new String(str.toLowerCase());

    if (st != null)
    {
      switch (iLevel)
      {
        case _MAIN:
          for (i=0; i<st.iCount; i++)
          {
            if (sWord.equals( st.stItem[i].sPhrase.toLowerCase()) )
            {
              iSedStruct = i;
            }
          }
          break;
        case _SUB:
          for (i=0; i<st.stItem[iRow].iCount; i++)
          {
            if (sWord.equals(
                  st.stItem[iRow].descriptors[i].sPhrase.toLowerCase()) )
            {
              iSedStruct = i;
            }
          }
          break;
      }
    }

    return (iSedStruct);
  }

  /** Method parse()
   * <p> This method will parse the Measured Section into rock sedimentary list
   *     data structure
   * @param  stList       = Remarks List Data Structure
   * @param  stSedStructs = SedStruct Thesaurus
   * @return stSedStruct  = Rock sedimentary Data List Structure
   */

  public static sedimentaryListStruct parse(
                    iqstratRemarkListStruct  stList,
                    parseSedStructListStruct stSedStructs )
  {
    sedimentaryListStruct stSedStruct = null;
    String                sKEY       = cmn.cmnString.UniqueName();
    String                str        = "";
    String                tokens[]   = null;
//    String                sDelimiter = new String("[,-;.:= ]+");
    String                sDelimiter = new String("[ /,;:.=-]+");
    sedimentaryStruct     stText     = null;

    if (stList != null)
    {
      stSedStruct = new sedimentaryListStruct();

      for (int i=0; i<stList.iCount; i++)
      {
        str    = new String( stList.stItem[i].sText.replace('\t', ' ') );
        str    = new String( str.replace('\n', ' ') );

        tokens = str.split( sDelimiter );

        stText = parseSedStruct( stList.stItem[i].depthStart,
                                 stList.stItem[i].depthEnd,
                                 tokens, stSedStructs );

        if (stText != null)
        {
          stText.sKEY = new String( sKEY + "_" + i );
          stSedStruct = rock.sedimentary.sedimentaryUtility.add(
                          stText, stSedStruct );
        }

        tokens = null;

        if (stText != null)
          stText.delete();
        stText = null;
      }

      stList.delete();
      stList = null;
    }

    return (stSedStruct);
  }

  /** Method parse()
   * <p> This method will parse the Measured Section into rock sedimentary list
   *     data structure
   * @param  stList       = KGS Measured Section List Data Structure
   * @param  stSedStructs = SedStruct Thesaurus
   * @return stSedStruct  = Rock sedimentary Data List Structure
   *

  public static sedimentaryListStruct parse(
                    kgsMeasSectListStruct    stList,
                    parseSedStructListStruct stSedStructs )
  {
    sedimentaryListStruct stSedStruct = null;
    double                depth      = 0.0;
    double                depth0     = 0.0;
    String                tokens[]   = null;
    String                sDelimiter = new String("[,-;. ]+");
    sedimentaryStruct     stText     = null;

    if (stList != null)
    {
      for (int i=0; i<stList.iRows; i++)
      {
        if (stList.stData[i].sUnitNo.length() > 0)
        {
          depth = depth + stList.stData[i].dThickness;

          if (stList.stData[i].dThickness > 0.0)
          {
            tokens = stList.stData[i].sDescription.split(sDelimiter);
            stText = parse.parseSedStructUtility.parseSedStruct(
                       depth0, depth, tokens, stSedStructs );
            if (stText != null)
              stSedStruct =
                rock.sedimentary.sedimentaryUtility.add(stText, stSedStruct);
          }

          depth0 = depth;
        }

        tokens = null;

        if (stText != null)
          stText.delete();
        stText = null;
      }

      stList.delete();
      stList = null;
    }

    stSedStruct =
        rock.sedimentary.sedimentaryUtility.computeDepthRange( stSedStruct );

    return (stSedStruct);
  }
*/
  /** Method parseSedStruct()
   * <p> This method will parse the text string into rock sedimentary
   * @param  depthStart   = the starting depth
   * @param  depthEnd     = the ending depth
   * @param  sData        = the sedimentary string phrase or word
   * @param  stSedStructs = SedStruct Thesaurus
   * @return st           = rock sedimentary data structure
   */

  public static  sedimentaryStruct parseSedStruct(
                     double depthStart, double depthEnd, String sData[],
                     parseSedStructListStruct stSedStructs )
  {
    int i, j, k, m, m0, m1, m2, p;
    sedimentaryStruct    st         = null;
    int                  iCount     = 0;
    String sWords[][] = { {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""},
                          {"", ""}, {"", ""}, {"", ""}, {"", ""}, {"", ""} };
    int iRow[]   = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    int iOrder[] = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1};
    String               sTemp[]     = {"", ""};
    int                  iSedStruct  = -1;
    int                  iDescrip    = -1;
    int                  iContinue   = 0;
    int                  iNext       = 0;
    parseSedStructStruct stSedStruct = null;
    String               sID         = "";
    String               sName       = "";
    int                  iNew        = 0;

    if (stSedStruct != null)
      stSedStruct.delete();

    // Modify Abbreviation to word

    for (i=0; i<sData.length; i++)
    {
      for (j=0; j<stSedStructs.iAbbrev; j++)
      {
        if (sData[i].toLowerCase().equals(
              stSedStructs.sAbbrev[j][0].toLowerCase()))
          sData[i] = new String(stSedStructs.sAbbrev[j][1]);
      }
    }

    // Find SedStruct Words up to 10

    for (j=0; j<sData.length; j++)
    {
      if (isKeyword( sData[j], stSedStructs ))
      {
        if (iCount < 15)
        {
          sTemp             = getKeyword( sData[j], stSedStructs );
          sWords[iCount][0] = new String(sTemp[0]);
          sWords[iCount][1] = new String(sTemp[1]);
          iOrder[iCount]    = j;
          iCount++;
        }
      }
    }

    // Change a Modifier to a descriptor

    for (i=0; i<iCount; i++)
    {
      if (sWords[i][1].equals("M"))
      {
        k = i+1;
        if (k<iCount)
        {
          if (isModified( sWords[i][0], sWords[k][0], stSedStructs ))
          {
            sWords[i][0] = new String(sWords[i][0] + " " + sWords[k][0]);
            sWords[i][1] = new String(sWords[k][1]);
            iOrder[i]    = iOrder[k];

            sWords[k][0] = new String("");
            sWords[k][1] = new String("");
          }
        }
      }
    }

    // Identify the main lithology type

    for (i=0; i<iCount; i++)
    {
      if (sWords[i][1].equals("S"))
      {
        iSedStruct  = getSedStructRow( _MAIN, -1, sWords[i][0], stSedStructs );

        if (iSedStruct > -1)
        {
          iContinue    = 1;
          iRow[i]      = iSedStruct;
          sWords[i][1] = new String("X");
        }
      }
    }

    // Modify Litholgy with a descriptor

    if (iContinue == 1)
    {
      for (m=0; m<iCount; m++)
      {
        m1 = m+2;
        if (m1 > iCount) m1 = iCount;

        if (iRow[m] > -1)
        {
          stSedStruct = stSedStructs.stItem[iRow[m]];

          sName = new String( stSedStruct.sName );
          sID   = new String( stSedStruct.sID);

          iNext = 0;
//          for (i=0; i<iCount; i++)
          for (i=0; i<m1; i++)
          {
            if (Math.abs(iOrder[m] - iOrder[i]) <= 2)
            {
              if ((sWords[i][1].equals("D")) || (sWords[i][1].equals("M")))
              {
                iDescrip = getSedStructRow(
                                 _SUB, iRow[m], sWords[i][0], stSedStructs );

                if ((iDescrip > -1) && (iNext == 0))
                {
                  iNext = 1;
                  sWords[i][1] = new String("X");
                  stSedStruct = stSedStructs.stItem[iRow[m]].descriptors[iDescrip];

                  sName = new String( stSedStruct.sName );
                  sID   = new String( stSedStruct.sID );
                }
              }
            }
          }

          if (iNew == 0)
          {
            iNew = 1;
            st = new sedimentaryStruct();

            st.sKEY = new String(cmn.cmnString.UniqueName() +
                                 ( (int) depthStart)); // Create a Unique KEY
            st.depthStart = depthStart; // Starting Depth
            st.depthEnd   = depthEnd;   // Ending Depth
          }

          if (st != null)
          {
            st = rock.sedimentary.sedimentaryUtility.addStructure(sID, sName, st);
          }
        }
      }
    }

    return (st);
  }

  /* ======================================================================== *
   * ---------------------------- SET METHODS ------------------------------- *
   * ======================================================================== */

  /* ======================================================================== *
   * ----------------------------- IS METHODS ------------------------------- *
   * ======================================================================== */

  /** Method isKeyword()
   * <p> This method will determine if the string comes from the KEY_WORDS
   * @param  str    = string or string phrase being compared.
   * @param  st     = SedStruct Data List Structure
   * @return bValue = true it does and falls it does not
   */

  public static boolean isKeyword( String str, parseSedStructListStruct st )
  {
    boolean bValue = false;
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iKeywords; i++)
      {
        if (sWord.equals(st.sKeywords[i][0].toLowerCase())) bValue = true;
      }
    }

    return (bValue);
  }

  /** Method isModified( String sin, String str, parseSedStructListStruct st )
   *  <p> This method will determine if the string is part of descriptor set
   * @param  sin    = string or string phrase to be merged
   * @param  str    = string or string phrase being compared.
   * @param  st     = SedStruct Data List Structure
   * @return bValue = true it does and falls it does not
   */


  public static boolean isModified(
      String sin, String str, parseSedStructListStruct st )
  {
    boolean bValue = false;
    String  sWord0 = new String(sin.toLowerCase());
    String  sWord  = new String(str.toLowerCase());

    if (st != null)
    {
      for (int i=0; i<st.iModifiers; i++)
      {
        if ((sWord0.equals(st.sModifier[i][0].toLowerCase())) &&
            (sWord.equals(st.sModifier[i][1].toLowerCase())))
          bValue = true;
      }
    }

    return (bValue);
  }

  /* ======================================================================== *
   * ---------------------------- COPY METHODS ------------------------------ *
   * ======================================================================== */

  /** Method copyList( parseSedStructListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static parseSedStructListStruct copyList( parseSedStructListStruct stOld )
  {
    parseSedStructListStruct stNew = null;
    int i = 0;
    int j = 0;

    if (stOld != null)
    {
      stNew = new parseSedStructListStruct();

      // Abbreviations

      stNew.iAbbrev = stOld.iAbbrev;  // total number of records
      // Abreviation {Abbrev, word }
      stNew.sAbbrev = new String[stNew.iAbbrev][2];

      for (i=0; i<stNew.iAbbrev; i++)
      {
        for (j=0; j<2; j++)
          stNew.sAbbrev[i][j] = new String(stOld.sAbbrev[i][j]);
      }

      // Modifiers

      stNew.iModifiers    = stOld.iModifiers; // total number of records
      // Modifier List { Modifier, Modifies, phrase }
      stNew.sModifier = new String[stNew.iModifiers][3];

      for (i=0; i<stNew.iModifiers; i++)
      {
        for (j=0; j<3; j++)
          stNew.sModifier[i][j] = new String(stOld.sModifier[i][j]);
      }

      // Keywords

      stNew.iKeywords = 0; // Total Number of Keywords
      // Keyword List { pharase, type}
      stNew.sKeywords = new String[stNew.iKeywords][2];

      for (i=0; i<stNew.iKeywords; i++)
      {
        for (j=0; j<2; j++)
          stNew.sKeywords[i][j] = new String(stOld.sKeywords[i][j]);
      }

      // Lithologies

      stNew.iCount = stOld.iCount;
      stNew.stItem = new parseSedStructStruct[stOld.iCount];

      for (i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method copy( parseSedStructStruct stOld )
   * <p> This method will copy the data structure
   * @param  stOld = The old data structure
   * @return stNew = The new data structure
   */

  public static parseSedStructStruct copy( parseSedStructStruct stOld )
  {
    int i=0;
    int j=0;
    parseSedStructStruct stNew = new parseSedStructStruct();

    // SedStruct Lithology

    stNew.sID     = new String( stOld.sID );        // String ID for Mask
    stNew.sPhrase = new String( stOld.sPhrase );    // Phrase
    stNew.sName   = new String( stOld.sName );      // Name
    stNew.sAbrev  = new String( stOld.sAbrev );     // Abbreviation

    // Descriptors

    stNew.iCount      = stOld.iCount;
    stNew.descriptors = new parseSedStructStruct[stOld.iCount];

    for (i=0; i<stOld.iCount; i++)
    {
      stNew.descriptors[i] = copy( stOld.descriptors[i] );
    }

    return (stNew);
  }

  /* ======================================================================== *
   * ---------------------------- PRINT METHODS ----------------------------- *
   * ======================================================================== */

  /** Method print( parseSedStructListStruct st )
   * <p> This method will print the parse symbols structure
   * @param  st = Mask Symbols data list structure
   */

  public static void print( parseSedStructListStruct st )
  {
    int i = 0;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        print(st.stItem[i]);
      }
    }
  }

  /** Method print( parseSedStructStruct st )
   * <p> This method will print the symbols data structure
   * @param  st = Mask symbols data structure
   */

  public static void print( parseSedStructStruct st )
  {
    if (st != null)
    {
      System.out.println( st.sID     + " <" + // String ID for Mask Symbol
                          st.sPhrase + "> " +
                          st.sName   + " [" + // Name or Description of parse
                          st.sAbrev  + "]");  // Abbreviation of parse

      if (st.descriptors != null)
      {
        System.out.println( " ---- Descriptors ---- ");

        for (int i=0; i<st.iCount; i++)
        {
          print(st.descriptors[i]);
        }

        System.out.println( " ---- End Descriptors ---- ");
      }
    }
  }
}